package gov.va.med.hac.edi.ewv2.actions;

import gov.va.med.hac.edi.ewv2.beans.claim.*;
import gov.va.med.hac.edi.ewv2.beans.codes.*;
import gov.va.med.hac.edi.ewv2.beans.healthcareinfo.*;
import gov.va.med.hac.edi.ewv2.beans.ohi.OhiType;
import gov.va.med.hac.edi.ewv2.beans.ohi.linelevel.ServiceLineOhiAdjustment;
import gov.va.med.hac.edi.ewv2.beans.servicelines.*;
import gov.va.med.hac.edi.ewv2.persistence.*;
import gov.va.med.hac.edi.ewv2.persistence.base.*;
import gov.va.med.hac.edi.ewv2.utils.Ewv2Utils;
import gov.va.med.hac.edi.ewv2.exceptions.*;

import java.util.*;

import org.jboss.seam.log.Log;

public class ClaimDataBinder extends DTOBase
{
    public ClaimContainer populateClaimContainer(EwvClaim claimBean, Log log)
    throws NonUniqueRecordsException
    {
        ClaimType claimType = ClaimType.typeOf(claimBean.getClaimType());
        
        // Get the pieces from the claim
        Set<EwvProcedureCode> procedureCodes = claimBean.getEwvProcedureCodes();
        Set<EwvDiagnosisCode> diagnosisCodes = claimBean.getEwvDiagnosisCodes();
        Set<EwvECode5010> eCodes = claimBean.getEwvECodes5010();
        Set<EwvHealthInformation> healthInfo = claimBean.getEwvHealthInformations();
        
        // Present on Admission notes has multiple versions
        String presentOnAdmissionNotesLong = Ewv2Utils.addLineBreaks(claimBean.getHccodesPoaNotes());
        String presentOnAdmissionNotesShort = Ewv2Utils.shorten(presentOnAdmissionNotesLong, 10);
        boolean usePresentOnAdmissionModalPanel = false;
        if (presentOnAdmissionNotesShort.endsWith("...")) {
          usePresentOnAdmissionModalPanel = true;
        }
        
        String principalDiagnosisMasterCode = claimBean.getHccodesPrincipalDiagnosis();
        String principalDiagnosisPoa = claimBean.getHccodesPrincipalDiagnosisPoa();
        String admittingDiagnosisMasterCode = claimBean.getHccodesAdmittingDiagnosis();
        String reasonForVisitMasterCode = claimBean.getHccodesReasonForVists();
        String eCodeMasterCode = claimBean.getHccodesECode();
        
        BillingAndAdmissionInfo billingAndAdmissionInfo
          = populateBillingAndAdmissionInfo(claimType, claimBean);
        
        DiagnosisCodesContainer diagnosisCodesContainer
                 = populateDiagnosisCodesContainer(
                     presentOnAdmissionNotesLong,
                     presentOnAdmissionNotesShort,
                     usePresentOnAdmissionModalPanel,
                     principalDiagnosisMasterCode,
                     principalDiagnosisPoa,
                     admittingDiagnosisMasterCode,
                     reasonForVisitMasterCode,
                     eCodeMasterCode,
                     diagnosisCodes);
        
        ECodesContainer eCodesContainer = populateECodesContainer(eCodes);
      
        ProcedureCodesContainer procedureCodesContainer
          = populateProcedureCodesContainer(claimBean.getHccodesPrincipalProcedure(),
		                      claimBean.getHccodesPrincipalProcDate(),
		                      procedureCodes);
        
        List<ServiceLine> serviceLineList = populateServiceLines(claimBean);
        
        List<HealthCareInformationLine> healthCareValueInformationBeanList
          = createHealthCareInformationList(HealthInfoType.VALUE, healthInfo, log);
        List<HealthCareInformationLine> healthCareConditionInformationBeanList
          = createHealthCareInformationList(HealthInfoType.CONDITION, healthInfo, log);
        List<HealthCareInformationLine> healthCareOccurranceSpanInformationBeanList
          = createHealthCareInformationList(HealthInfoType.OCCUR_SPAN, healthInfo, log);
        List<HealthCareInformationLine> healthCareOccurranceInformationBeanList
        = createHealthCareInformationList(HealthInfoType.OCCUR, healthInfo, log);
        List<HealthCareInformationLine> healthCareDRGInformationBeanList
        = createHealthCareInformationList(HealthInfoType.DRG, healthInfo, log);
        
        // 5010 Stuff
        AmbulanceInfo claimLevelAmbulanceInfo = populateClaimLevelAmbulanceInfo(claimBean);
        
        ClaimContainer newClaimContainer =
                new ClaimContainer( billingAndAdmissionInfo,
                                    diagnosisCodesContainer,
    		                        procedureCodesContainer,
    		                        eCodesContainer,
    		                        serviceLineList,
    		                        healthCareValueInformationBeanList,
    		                        healthCareConditionInformationBeanList,
    		                        healthCareOccurranceSpanInformationBeanList,
    		                        healthCareOccurranceInformationBeanList,
    		                        healthCareDRGInformationBeanList,
    		                        claimLevelAmbulanceInfo);
        
        return newClaimContainer;
    }
    
    private List<HealthCareInformationLine> createHealthCareInformationList
        (HealthInfoType healthInfoType, Set<EwvHealthInformation> healthInfo, Log log)
    {
        ArrayList<HealthCareInformationLine> returnList = new ArrayList<HealthCareInformationLine>(0);
        
        for (EwvHealthInformation info : healthInfo)
        {
            // TODO Greg was indicating this as a temp solution, will need to be changed
            HealthInfoType infoType = HealthInfoType.typeOf(info.getQualifier());
            
            if ((infoType != null) && (infoType.equals(healthInfoType)))
            {
                returnList.add(populateHealthCareInfoLine(info,infoType, log));
                
            }
        }
        
        //if ((!accumulated) && (healthCareInformationLineList.size() > 0))
        if (returnList.size() > 0)
        {
            Collections.sort(returnList);
            
        }
        
        return returnList;
    }
    
    private HealthCareInformationLine populateHealthCareInfoLine
      (EwvHealthInformation info, HealthInfoType typeOf, Log log) 
    {
        String qualifier = info.getQualifier();
        String description = info.getDescription();
        String code = info.getCode();
        String date = info.getClaimDate();
        String format = info.getFormat();

//        log.info("Debugging: Formatting incoming date: " + date);
//        String formattedDate = Ewv2Utils.formatAsDate2(date, format, log);
//        log.info("Debugging: Formatted version: " + formattedDate);
//        
        String amount = info.getAmount();
        Long sequence = info.getHealthInformationSequence();
        
        HealthCareInformationLine bean
          = new HealthCareInformationLine(sequence,
                                          typeOf,
                                          qualifier, 
                                          description, 
                                          code, 
                                          date, 
                                          format, 
                                          amount);
        
        return bean;
    }
    
    private ProcedureCodesContainer populateProcedureCodesContainer
      (String inPrincipalProcedure,
       String inDateOfProcedure,
       Set<EwvProcedureCode> codes)
    {
        ProcedureCodesContainer ret = null;
        
        String principalProcedure = inPrincipalProcedure;
        String dateOfProcedure = inDateOfProcedure;
        
        if (codes != null)
        {
            List<ProcedureCode> listOfProcedureCodes
              = createListOfProcedureCodes(inPrincipalProcedure,
            		                       inDateOfProcedure,
            		                       codes);
            
            ret = new ProcedureCodesContainer(principalProcedure,
            		                          dateOfProcedure,
            		                          listOfProcedureCodes);
        }
        
        return ret;
    }
    

    // This method, new to Phase II, adds in the Principal Diagnosis and does not split the codes
    //   into four lists, for four columns.
    private List<DiagnosisCode> createListOfDxCodes(String principalDiagnosisMasterCode,
                                                    String principalDiagnosisPoa,
                                                    Set<EwvDiagnosisCode> codes)
    {
        List<DiagnosisCode> returnList = new ArrayList<DiagnosisCode>();
        
        // Add in the principal diagnosis
        returnList.add(new DiagnosisCode(1L, principalDiagnosisMasterCode, principalDiagnosisPoa));
        
        for (EwvDiagnosisCode codeBean : codes)
        {
            Long sequence = codeBean.getDiagnosisSequence();
            
            returnList.add(new DiagnosisCode(sequence,
                                             codeBean.getDiagnosisCode(),
                                             codeBean.getPoaIndicator_5010()));
        }
        
        Collections.sort(returnList);
        
        // Experience shows that the diagnosis codes in the database include sequence numbers
        //   that sometimes start with 1, and sometimes with 2.
        returnList = CodeBase.renumber(returnList);
        
        return returnList;
    }

    private List<ProcedureCode> createListOfProcedureCodes
      (String principalProcedureCode,
       String principalProcedureDate,
       Set<EwvProcedureCode> codes)
    {
        List<ProcedureCode> returnList = new ArrayList<ProcedureCode>();
        
        // Add in the principal procedure
        if (principalProcedureCode != null && !principalProcedureCode.isEmpty()) {
          ProcedureCode principalProcedureBean = new ProcedureCode
            (1L, principalProcedureCode, principalProcedureDate);
          returnList.add(principalProcedureBean);
        }
        
        // Add in the codes from the list
        for (EwvProcedureCode codeBean : codes) {
            ProcedureCode bean = new ProcedureCode(codeBean.getProcedureSequence(),
                                                   codeBean.getProcedureCode(),
                                                   codeBean.getProcedureDate());
            returnList.add(bean);
        }
        
        Collections.sort(returnList);
        
        // Experience shows that the diagnosis codes in the database include sequence numbers
        //   that sometimes start with 1, and sometimes with 2.
        returnList = CodeBase.renumber(returnList);

        return returnList;
    }
    
    private DiagnosisCodesContainer populateDiagnosisCodesContainer
            (String presentOnAdmissionNotesLong,
             String presentOnAdmissionNotesShort,
             boolean usePresentOnAdmissionModalPanel,
             String principalDiagnosisMasterCode,
             String principalDiagnosisPoa,
             String admittingDiagnosisMasterCode,
             String reasonForVisitMasterCode,
             String eCodeMasterCode,
             Set<EwvDiagnosisCode> dcodes)
    {
        // Phase 2 codes do not need to be split into four columnar lists.
        List<DiagnosisCode> diagnosisCodes
          =  createListOfDxCodes(principalDiagnosisMasterCode, principalDiagnosisPoa, dcodes);

        DiagnosisCodesContainer ret
          = new DiagnosisCodesContainer(presentOnAdmissionNotesShort,
        		                             presentOnAdmissionNotesLong,
        		                             usePresentOnAdmissionModalPanel,
        		                             principalDiagnosisMasterCode,
        		                             principalDiagnosisPoa,
        		                             admittingDiagnosisMasterCode,
        		                             reasonForVisitMasterCode,
        		                             eCodeMasterCode,
        		                             diagnosisCodes);
        
        boolean ok = Ewv2Utils.okToRender(presentOnAdmissionNotesLong, presentOnAdmissionNotesShort, principalDiagnosisMasterCode, admittingDiagnosisMasterCode, reasonForVisitMasterCode, eCodeMasterCode);
        
        ret.setShouldRender(ok);
        
        return ret;
    }
    
    // This method, new to Phase II, adds in the Principal Diagnosis and does not split the codes
    //   into four lists, for four columns.
    private List<ECode> createListOfECodes(Set<EwvECode5010> eCodes)
    {
        List<ECode> returnList = new ArrayList<ECode>();
        
        for (EwvECode5010 codeBean : eCodes)
        {
            Long sequence = codeBean.getSequenceNum();
            
            returnList.add(new ECode(sequence, codeBean.getECode(), codeBean.getPoaIndicator_5010()));
        }
        
        Collections.sort(returnList);
        
        // Experience shows that the codes in the database include sequence numbers
        //   that sometimes start with 1, and sometimes with 2.
        returnList = CodeBase.renumber(returnList);

        return returnList;
    }

    private ECodesContainer populateECodesContainer (Set<EwvECode5010> eCodesSet) {

      List<ECode> eCodes = createListOfECodes(eCodesSet);
      
      ECodesContainer ret = new ECodesContainer( eCodes);
      
      boolean ok = Ewv2Utils.okToRender(eCodes);
      
      ret.setShouldRender(ok);
      
      return ret;
      }

    private BillingAndAdmissionInfo populateBillingAndAdmissionInfo(ClaimType claimType, EwvClaim claimBean)
    {
        String patientAccountNumber = claimBean.getInvoicePatientAccountNum();
        
        String totalClaimCharges = claimBean.getSvcLineTotalClaimCharges();
        String typeOfBillOrFrequency = " ";
        String frequency = " ";
        
        if(claimType.equals(ClaimType.PROFESSIONAL))
        {
            // POS/Frequency without slash, UI handles the extra field
            typeOfBillOrFrequency = Ewv2Utils.defaultToSpace(claimBean.getPos());
            frequency = Ewv2Utils.defaultToSpace(claimBean.getInvoiceFrequency());
        }
        else if(claimType.equals(ClaimType.DENTAL))
        {
          // POS/Frequency without slash, UI handles the extra field
          typeOfBillOrFrequency = Ewv2Utils.defaultToSpace(claimBean.getPos());
          frequency = Ewv2Utils.defaultToSpace(claimBean.getInvoiceFrequency());
            // POS/Frequency with slash
            //typeOfBillOrFrequency = Ewv2Utils.defaultToSpace(claimBean.getPos())+"/"+ Ewv2Utils.defaultToSpace(claimBean.getInvoiceFrequency());
        }
        else if(claimType.equals(ClaimType.INSTITUTIONAL))
        {   
            // Type of Bill
            typeOfBillOrFrequency = Ewv2Utils.defaultToSpace(claimBean.getInvoiceTypeOfBill())+"/"+ Ewv2Utils.defaultToSpace(claimBean.getInvoiceFrequency());
        }
        
        String statementFromTo = Ewv2Utils.formatDateRangeEWV2(claimBean.getInvoiceStatementFromDate(),
                                                               claimBean.getInvoiceStatementToDate(),
                                                               " - ",
                                                               true);
        String statementFromToEWV2 = Ewv2Utils.formatDateRangeEWV2(
        		                     claimBean.getInvoiceStatementFromDate(),
        		                     claimBean.getInvoiceStatementToDate(),
        		                     " Thru ",
        		                     true);
            
        String previousPDIOrClaimNumber = claimBean.getInvoicePreviousPdi();
        
        String patientPaidAmount = claimBean.getInvoicePatientPaidAmt();
        String patientSignature = claimBean.getInvoicePatientSignature();
        String patientStatus = claimBean.getInvoicePatientStatus();
        String admissionType = claimBean.getInvoiceAdmissionType();
        String providerAcceptsAssignment = claimBean.getInvoiceProvAcceptsAssignmt();
        String admissionSource = claimBean.getInvoiceAdmissionSource();
        String assignmentOfBenefits = claimBean.getInvoiceAssignmentOfBenefits();
        
        // Admission date and hour are needed separate for EWV2
        String admissionDateOrHour = claimBean.getInvoiceAdmissionDateHour();
        String admissionDate = null;
        String admissionHour = null;

        if (admissionDateOrHour != null && !admissionDateOrHour.isEmpty()) {
          // Get the date portion of this field
          try {
            // Get the date portion
            if (admissionDateOrHour.length() >= 10) {
              admissionDate = admissionDateOrHour.substring(0,10);
            }
            
            // Get the time portion
            if (admissionDateOrHour.length() == 16) {
              admissionHour = admissionDateOrHour.substring(11,16);
            } else {
              admissionHour = "";
            }
            
            // Convert the date portion to a 2-digit date
            String twoDigitAdmissionDate = Ewv2Utils.formatAsDate(admissionDate);
            
            // Put the full string together again, with the 2-digit date
            admissionDateOrHour = twoDigitAdmissionDate + " " + admissionHour;
          } catch (StringIndexOutOfBoundsException e) {
            ClaimService.debug = "Caught an error";
            admissionDateOrHour = "Error";
          }
          
        }
        
        String dischargeDateHour = claimBean.getInvoiceDischargeDateHour();
        
        String medicareIcn = claimBean.getMedicareIcn();
        
        String accidentDate = claimBean.getInvoiceAccidentDate();
        String accidentIndicator = claimBean.getInvoiceAccidentIndicator();
        String priorAuthorizationReferral = claimBean.getInvoicePriorAuthorization1();
        String priorAuthorizationReferralShort = null;
        boolean usePriorAuthorizationModalPanel = false;
        priorAuthorizationReferralShort = Ewv2Utils.shorten(priorAuthorizationReferral, 20);
        if (priorAuthorizationReferralShort.endsWith("...")) {
          usePriorAuthorizationModalPanel = true;
        }
        
        // Added for Phase II
        String admittingDiagnosis = claimBean.getHccodesAdmittingDiagnosis();
        String reasonForVisit = claimBean.getHccodesReasonForVists();
        String eCode = claimBean.getHccodesECode();
        String poaNotesPlain = claimBean.getHccodesPoaNotes();
        String poaNotesLong = Ewv2Utils.addLineBreaks(poaNotesPlain);
        String poaNotesShort = Ewv2Utils.shorten(poaNotesLong, 15);
        boolean usePoaNotesModelPanel = false;
        if (poaNotesShort.endsWith("...")) {
          usePoaNotesModelPanel = true;
        }
        
        // Added for 5010
        String reasonForVisit2_5010 = claimBean.getHccodesReasonForVisit2_5010();
        String reasonForVisit3_5010 = claimBean.getHccodesReasonForVisit3_5010();
        String referralLong_5010 = claimBean.getInvoiceReferral_5010();
        String referralShort_5010 = null;
        boolean useReferralModalPanel_5010 = false;
        referralShort_5010 = Ewv2Utils.shorten(referralLong_5010, 20);
        if (referralShort_5010.endsWith("...")) {
          useReferralModalPanel_5010 = true;
        }

        BillingAndAdmissionInfo ret
          = new BillingAndAdmissionInfo(patientAccountNumber,
        		                       totalClaimCharges,
        		                       typeOfBillOrFrequency,
        		                       statementFromTo,
        		                       statementFromToEWV2,
        		                       previousPDIOrClaimNumber,
        		                       patientPaidAmount,
        		                       patientSignature,
        		                       patientStatus,
        		                       priorAuthorizationReferral,
        		                       admissionType,
        		                       providerAcceptsAssignment,
        		                       admissionSource,
        		                       assignmentOfBenefits,
        		                       admissionDateOrHour,
        		                       admissionDate,
        		                       admissionHour,
        		                       dischargeDateHour,
        		                       medicareIcn,
        		                       accidentDate,
        		                       accidentIndicator,
        		                       priorAuthorizationReferralShort,
        		                       usePriorAuthorizationModalPanel,
        		                       frequency,
        		                       
        		                       // Added for Phase II
        		                       admittingDiagnosis,
        		                       reasonForVisit,
        		                       eCode,
        		                       poaNotesShort,
        		                       poaNotesLong,
        		                       usePoaNotesModelPanel,
        		                       
                                       // Added for 5010
                                       reasonForVisit2_5010,
                                       reasonForVisit3_5010,
                                       referralLong_5010,
                                       referralShort_5010,
                                       useReferralModalPanel_5010);
        
        ret.setShouldRender(Ewv2Utils.okToRender(patientAccountNumber,
                                             totalClaimCharges,
                                             typeOfBillOrFrequency,
                                             statementFromTo,
                                             previousPDIOrClaimNumber,
                                             patientPaidAmount,
                                             patientSignature,
                                             patientStatus,
                                             priorAuthorizationReferral,
                                             admissionType,
                                             providerAcceptsAssignment,
                                             admissionSource,
                                             assignmentOfBenefits,
                                             admissionDateOrHour,
                                             dischargeDateHour,
                                             medicareIcn,
                                             accidentDate,
                                             accidentIndicator,
                                             priorAuthorizationReferralShort,
                                             frequency,
                                             
                                             // New for Phase II
                                             admittingDiagnosis,
                                             reasonForVisit,
                                             eCode,
                                             poaNotesPlain,
                                             
                                             // New for 5010
                                             reasonForVisit2_5010,
                                             reasonForVisit3_5010,
                                             referralLong_5010,
                                             referralShort_5010
                                             ));
        return ret;
    }

    private AmbulanceInfo populateClaimLevelAmbulanceInfo(EwvClaim claimBean)
    throws NonUniqueRecordsException
    {
      AmbulanceInfo returnInfo = null;
      boolean okToRender = false;
      
      // Get the ambulance info from the claim
      Set<EwvAmbulanceInfo5010> ambulanceInfoSet = claimBean.getEwvAmbulanceInfo5010();
      
      // There will be no more than one entry in the set
      int ambulanceLineCounter = 0;
      for (EwvAmbulanceInfo5010 ambulanceInfo : ambulanceInfoSet) {
        ambulanceLineCounter++;
        if (ambulanceLineCounter > 1) {
          throw new NonUniqueRecordsException ("Too many claim-level ambulance lines");
        }
        
        returnInfo = new AmbulanceInfo();
        
        // Pickup Address
        returnInfo.setPickupAddressLine1(ambulanceInfo.getPickupAddressLine1());
        returnInfo.setPickupAddressLine2(ambulanceInfo.getPickupAddressLine2());
        returnInfo.setPickupCity(ambulanceInfo.getPickupCity());
        returnInfo.setPickupState(ambulanceInfo.getPickupState());
        returnInfo.setPickupZip(ambulanceInfo.getPickupZip());

        // Dropoff Address
        returnInfo.setDropoffAddressLine1(ambulanceInfo.getDropoffAddressLine1());
        returnInfo.setDropoffAddressLine2(ambulanceInfo.getDropoffAddressLine2());
        returnInfo.setDropoffCity(ambulanceInfo.getDropoffCity());
        returnInfo.setDropoffState(ambulanceInfo.getDropoffState());
        returnInfo.setDropoffZip(ambulanceInfo.getDropoffZip());
        
        // Transport Fields
        returnInfo.setTransportReasonCode(ambulanceInfo.getTransportReasonCode());
        returnInfo.setTransportDistance(Ewv2Utils.formatAsInteger(ambulanceInfo.getTransportDistance()));

        returnInfo.setRoundTripPurpose(ambulanceInfo.getRoundTripPurpose());
        returnInfo.setRoundTripPurposeShort(Ewv2Utils.shorten(returnInfo.getRoundTripPurpose(), 50));
        returnInfo.setUseRoundTripPurposeModalPanel(returnInfo.getRoundTripPurposeShort().endsWith("..."));;

        returnInfo.setStretcherPurpose(ambulanceInfo.getStretcherPurpose());
        returnInfo.setStretcherPurposeShort(Ewv2Utils.shorten(returnInfo.getStretcherPurpose(), 50));
        returnInfo.setUseStretcherPurposeModalPanel(returnInfo.getStretcherPurposeShort().endsWith("..."));
        
        // Three possible Certs are slammed together into a single database row
        List<AmbulanceCert> certs = new ArrayList<AmbulanceCert>();
        
        // Cert #1
        populateAmbulanceCert(certs,
            ambulanceInfo.getCertIndicator1(),
            ambulanceInfo.getConditionCode1_1(),
            ambulanceInfo.getConditionCode1_2(),
            ambulanceInfo.getConditionCode1_3(),
            ambulanceInfo.getConditionCode1_4(),
            ambulanceInfo.getConditionCode1_5() );
        
        // Cert #2
        populateAmbulanceCert(certs,
            ambulanceInfo.getCertIndicator2(),
            ambulanceInfo.getConditionCode2_1(),
            ambulanceInfo.getConditionCode2_2(),
            ambulanceInfo.getConditionCode2_3(),
            ambulanceInfo.getConditionCode2_4(),
            ambulanceInfo.getConditionCode2_5() );
        
        // Cert #2
        populateAmbulanceCert(certs,
            ambulanceInfo.getCertIndicator3(),
            ambulanceInfo.getConditionCode3_1(),
            ambulanceInfo.getConditionCode3_2(),
            ambulanceInfo.getConditionCode3_3(),
            ambulanceInfo.getConditionCode3_4(),
            ambulanceInfo.getConditionCode3_5() );
        
        // Attach the certs to the ambulance info, if there are any certs
        if (certs.size() > 0) {
          returnInfo.setCerts(certs);
        }
      }
      
      return returnInfo;
    }

  private List<ServiceLine> populateServiceLines(EwvClaim claim)
          throws NonUniqueRecordsException
  {
    Set<EwvServiceLine> serviceLines = claim.getEwvServiceLines();

    List<EwvServiceLine> serviceLineList = new ArrayList(serviceLines);
    
    Collections.sort(serviceLineList);
    
    List<ServiceLine> listOfServiceLines = new ArrayList<ServiceLine>();
    
    for (EwvServiceLine line : serviceLineList)
    {
      ServiceLine bean = populateServiceLine(line, claim);
      listOfServiceLines.add(bean);
    }
    
    return listOfServiceLines;
  }

  protected ServiceLine populateServiceLine(EwvServiceLine line, EwvClaim claim) 
  throws NonUniqueRecordsException
  {
      String dateOfService = null;
      String beginDateOfService = null;
      String endDateOfService = null;
      String revCode = line.getRevenueCode();
      String procedureCode = line.getProcedureCode();
      
      String ndcCode = line.getNdc();
      String modifiers = Ewv2Utils.combineModifiers(", ", line.getModifier1(), line.getModifier2(), line.getModifier3(), line.getModifier4());
      String quantityOrType = getQuantityOrType(line);
      
      // Added quantity and type for phase II
      String quantity = line.getQuantity();
      String type = line.getType();
      
      String serviceTax = line.getServiceTax();
      String facilityTax = line.getFacilityTax();
      String charge = line.getCharge();
      
      String lineNumber = line.getLineNumber().toString();
      
      List<DrugLine> listOfDrugs = null;
      List<ToothLine> listOfTeeth = null;
      List<MedicalEquipmentLine> listOfEquipment = null;
      List<Provider> listOfProviderInfo = null;
      
      String dxDtPointers = Ewv2Utils.combineModifiers(", ", line.getDx1(),line.getDx2(),line.getDx3(),line.getDx4()); 
      String salesTax = line.getSalesTax();
      String pos = line.getPos();
      
      // Begin and End, new for Phase II, do not depend on claim type
      beginDateOfService = line.getBeginDateOfService();
      endDateOfService = line.getEndDateOfService();

      // 5010 Stuff
      AmbulanceInfo serviceLineLevelAmbulanceInfo = populateServiceLineLevelAmbulanceInfo(line);
      
      ClaimType claimType = ClaimType.typeOf(claim.getClaimType());
      if (claimType.equals(ClaimType.INSTITUTIONAL))
      {
          dateOfService = Ewv2Utils.formatDateRangeEWV2(line.getBeginDateOfService(),
                                                        line.getEndDateOfService(),
                                                        " - ",
                                                        true);
          
          // drugs accur in prof lines or inst
          listOfDrugs = createDrugLines(line.getEwvServiceLinesDrugs());
          
          listOfProviderInfo = createProviders(line.getEwvSlProviders());
      }
      else if (claimType.equals(ClaimType.PROFESSIONAL))
      {
          dateOfService = Ewv2Utils.formatDateRangeEWV2(line.getBeginDateOfService(),
                                                        line.getEndDateOfService(),
                                                        " - ",
                                                        true);
          
          // drugs accur in prof lines or inst
          listOfDrugs = createDrugLines(line.getEwvServiceLinesDrugs());
          
          // medical equipment occurs only on professional svc lines
          listOfEquipment = createMedicalEquipmentLines(line.getEwvSvcLineDmes());
          
          listOfProviderInfo = createProviders(line.getEwvSlProviders());
      }
      else if (claimType.equals(ClaimType.DENTAL))
      {
          // dental is it's own
          dateOfService = line.getBeginDateOfService();
          
          listOfTeeth = createToothLines(line.getEwvSvcLineTeeths());

          listOfProviderInfo = createProviders(line.getEwvSlProviders());
      }
      
      // Put together service line Ohi stuff
      String primaryPayerPaidAmount = null;
      List<ServiceLineOhiAdjustment> primaryOhiServiceLineAdjustments = null;
      String secondaryPayerPaidAmount = null;
      List<ServiceLineOhiAdjustment> secondaryOhiServiceLineAdjustments = null;
      
      // Go through the ohis for this service line
      Set<EwvServiceLineOhi> serviceLineOhis = line.getEwvServiceLineOhis();
      for (EwvServiceLineOhi lineOhi : serviceLineOhis) {
        // Is this a primary, secondary, or tertiary OHI?
        OhiType ohiType = determineOhiType(lineOhi, claim);
        
        // Check for primary
        if (ohiType.equals(OhiType.PRIMARY)) {
          primaryPayerPaidAmount = lineOhi.getOhiPaid();
          primaryOhiServiceLineAdjustments  = populateServiceLineAdjustments
            (lineOhi.getEwvServiceLineOhiAdjtmts());
        }

        // Check for secondary
        else if (ohiType.equals(OhiType.SECONDARY)) {
          secondaryPayerPaidAmount = lineOhi.getOhiPaid();
          secondaryOhiServiceLineAdjustments = populateServiceLineAdjustments
            (lineOhi.getEwvServiceLineOhiAdjtmts());
        }
        
      }
      
      ServiceLine newServiceLine =
         new ServiceLine(dateOfService,
                         beginDateOfService,
                         endDateOfService,
                         revCode,
                         procedureCode,
                         ndcCode,
                         modifiers,
                         quantityOrType,
                         quantity,
                         type,
                         primaryPayerPaidAmount,
                         primaryOhiServiceLineAdjustments,
                         secondaryPayerPaidAmount,
                         secondaryOhiServiceLineAdjustments,
                         serviceTax,
                         facilityTax,
                         charge,
                         lineNumber,
                         listOfDrugs,
                         listOfTeeth,
                         listOfEquipment,
                         dxDtPointers,
                         salesTax,
                         listOfProviderInfo,
                         pos,
                         serviceLineLevelAmbulanceInfo);
      
      return newServiceLine;
  }
  
  private AmbulanceInfo populateServiceLineLevelAmbulanceInfo(EwvServiceLine serviceLineBean)
  throws NonUniqueRecordsException
  {
    AmbulanceInfo returnInfo = null;
    boolean okToRender = false;
    
    // Get the ambulance info from the service line
    Set<EwvAmbulanceInfo5010> ambulanceInfoSet = serviceLineBean.getEwvAmbulanceInfo5010();
    
    // There will be no more than one entry in the set
    int ambulanceLineCounter = 0;
    for (EwvAmbulanceInfo5010 ambulanceInfo : ambulanceInfoSet) {
      ambulanceLineCounter++;
      if (ambulanceLineCounter > 1) {
        throw new NonUniqueRecordsException ("Too many service-line-level ambulance lines");
      }
      
      returnInfo = new AmbulanceInfo();
      
      // Pickup Address
      returnInfo.setPickupAddressLine1(ambulanceInfo.getPickupAddressLine1());
      returnInfo.setPickupAddressLine2(ambulanceInfo.getPickupAddressLine2());
      returnInfo.setPickupCity(ambulanceInfo.getPickupCity());
      returnInfo.setPickupState(ambulanceInfo.getPickupState());
      returnInfo.setPickupZip(ambulanceInfo.getPickupZip());

      // Dropoff Address
      returnInfo.setDropoffAddressLine1(ambulanceInfo.getDropoffAddressLine1());
      returnInfo.setDropoffAddressLine2(ambulanceInfo.getDropoffAddressLine2());
      returnInfo.setDropoffCity(ambulanceInfo.getDropoffCity());
      returnInfo.setDropoffState(ambulanceInfo.getDropoffState());
      returnInfo.setDropoffZip(ambulanceInfo.getDropoffZip());
      
      // Transport Fields
      returnInfo.setTransportReasonCode(ambulanceInfo.getTransportReasonCode());
      returnInfo.setTransportDistance(Ewv2Utils.formatAsInteger(ambulanceInfo.getTransportDistance()));

      returnInfo.setRoundTripPurpose(ambulanceInfo.getRoundTripPurpose());
      returnInfo.setRoundTripPurposeShort(Ewv2Utils.shorten(returnInfo.getRoundTripPurpose(), 50));
      returnInfo.setUseRoundTripPurposeModalPanel(returnInfo.getRoundTripPurposeShort().endsWith("..."));;

      returnInfo.setStretcherPurpose(ambulanceInfo.getStretcherPurpose());
      returnInfo.setStretcherPurposeShort(Ewv2Utils.shorten(returnInfo.getStretcherPurpose(), 50));
      returnInfo.setUseStretcherPurposeModalPanel(returnInfo.getStretcherPurposeShort().endsWith("..."));
      
//      // Transport Fields
//      returnInfo.setTransportReasonCode(ambulanceInfo.getTransportReasonCode());
//      returnInfo.setRoundTripPurpose(ambulanceInfo.getRoundTripPurpose());
//      returnInfo.setTransportDistance(ambulanceInfo.getTransportDistance());
//      returnInfo.setStretcherPurpose(ambulanceInfo.getStretcherPurpose());
//      
      // Three possible Certs are slammed together into a single database row
      List<AmbulanceCert> certs = new ArrayList<AmbulanceCert>();
      
      // Cert #1
      populateAmbulanceCert(certs,
          ambulanceInfo.getCertIndicator1(),
          ambulanceInfo.getConditionCode1_1(),
          ambulanceInfo.getConditionCode1_2(),
          ambulanceInfo.getConditionCode1_3(),
          ambulanceInfo.getConditionCode1_4(),
          ambulanceInfo.getConditionCode1_5() );
      
      // Cert #2
      populateAmbulanceCert(certs,
          ambulanceInfo.getCertIndicator2(),
          ambulanceInfo.getConditionCode2_1(),
          ambulanceInfo.getConditionCode2_2(),
          ambulanceInfo.getConditionCode2_3(),
          ambulanceInfo.getConditionCode2_4(),
          ambulanceInfo.getConditionCode2_5() );
      
      // Cert #2
      populateAmbulanceCert(certs,
          ambulanceInfo.getCertIndicator3(),
          ambulanceInfo.getConditionCode3_1(),
          ambulanceInfo.getConditionCode3_2(),
          ambulanceInfo.getConditionCode3_3(),
          ambulanceInfo.getConditionCode3_4(),
          ambulanceInfo.getConditionCode3_5() );
      
      // Attach the certs to the ambulance info, if there are any certs
      if (certs.size() > 0) {
        returnInfo.setCerts(certs);
      }
    }
    
    return returnInfo;
  }

  private List<MedicalEquipmentLine> createMedicalEquipmentLines(Set<EwvSvcLineDme> list)
  {
      List<MedicalEquipmentLine> equips = new ArrayList<MedicalEquipmentLine>();
      
      for (EwvSvcLineDme dme : list)
      {
          String lengthOfMedicalNecessity = dme.getDmeLengthMedicalNecessity();
          String frequency = dme.getDmeFrequency();
          String rentalAmt = dme.getDmeRentalAmt();
          String purchaseAmt = dme.getDmePurchaseAmt();
          
          equips.add(new MedicalEquipmentLine(lengthOfMedicalNecessity,
                                              frequency, 
                                              rentalAmt, 
                                              purchaseAmt));
      }
      
      return equips;
  }
  
}
