package gov.va.med.hac.edi.ewv2.actions;

import gov.va.med.hac.edi.ewv2.beans.claim.*;
import gov.va.med.hac.edi.ewv2.beans.demographics.*;
import gov.va.med.hac.edi.ewv2.beans.ohi.OhiInfoContainer;
import gov.va.med.hac.edi.ewv2.exceptions.*;
import gov.va.med.hac.edi.ewv2.persistence.EwvClaim;
import gov.va.med.hac.edi.ewv2.persistence.base.*;
import gov.va.med.hac.edi.ewv2.utils.*;

import java.io.*;

import javax.faces.application.FacesMessage;
import javax.faces.context.FacesContext;
import javax.persistence.*;

import org.jboss.seam.*;
import org.jboss.seam.annotations.*;
import org.jboss.seam.annotations.security.Restrict;
import org.jboss.seam.log.Log;

@Name("Ewv2AllClaimInfoAction")
@Scope(ScopeType.CONVERSATION)
public class AllClaimInfoAction implements Serializable {
  private static final long serialVersionUID = 5215329070049375734L;

  @In(create = true)
  protected EntityManager entityManager;
  
  @Logger
  protected Log log;
  
  @Out("Ewv2PatientHeaderInfo")
  private PatientHeaderInfo patientHeaderInfo;

  @Out("Ewv2CommonDemographicsInfo")
  private CommonDemographicsContainer commonDemographicsInfo;

  @Out("Ewv2ClaimService")
  private ClaimService claimService;

  @In("Ewv2PdiNumberBean")
  private PdiNumberBean pdiNumberBean;

  @Out("Ewv2ClaimContainer")
  private ClaimContainer claimContainer;

  @Out("Ewv2OhiInfoContainer")
  private OhiInfoContainer ohiInfoContainer;

  private boolean doDebug;
  
  @Restrict("#{s:hasRole('ewvUser')}")
  public void getAllClaimInfo() throws ActionErrorException, NonUniqueRecordsException {
    //log.info("getAllClaimInfo START");
    ClaimFormat claimFormat = null;
    ClaimType claimType = null;
    String pdiNumber = null;
    
    try {
      String unfilteredPdiNumber = pdiNumberBean.getPdiNumber();
      
      pdiNumber = filterPdiNumber(unfilteredPdiNumber);
    
      // Check the application properties file to see whether to carry out a test error.
      considerTestErrorAndDebug(pdiNumber);
      
      //log.info("getAllClaimInfo pdiNumber: " + pdiNumber);
      recordPdiNumber(pdiNumber);
  
      invalidate();
  
      //log.info("Claim Query Start");
      String errorMessage = "Unable to find submission for this PDI.";
      EwvClaim claimBean = getClaimRecord(pdiNumber);
  
      if (claimBean != null) {
      
        // Put together a Data Access Object for a few pieces common to both tabs
        //log.info("Demographics Start");
        commonDemographicsInfo = new DemographicsDataBinder().getCommonDemographics(claimBean);
        
        claimFormat = DemographicsDataBinder.determineClaimFormat(claimBean);
        claimType = DemographicsDataBinder.determineClaimType(claimBean);

        // The Patient Header appears near the top of the screen
        patientHeaderInfo = DemographicsDataBinder.populatePatientHeaderInfo(claimBean);

        //log.info("Claim Start");
        claimContainer = new ClaimDataBinder().populateClaimContainer(claimBean, log);
        
        // Most of the difficulties lie in the area of OHIs
        //log.info("OHI Start");
        ohiInfoContainer = new OhiDataBinder().getOhiData(claimBean, log);

        claimService = (ClaimService) Component.getInstance(ClaimService.class);
        if (claimService != null) {
          claimService.setClaimType(claimType);
          claimService.setClaimFormat(claimFormat);
          claimService.setIsClaimFound(true);
          claimService.setDebugOn(doDebug);
        }
        
      } else {
        // No claim was found
        FacesContext.getCurrentInstance().addMessage("ewvApplication",
            new FacesMessage(FacesMessage.SEVERITY_ERROR, errorMessage, ""));
      }

    } catch (Throwable e) {
      log.info("DEBUG: Caught a throwable: " + e);
      handleException(pdiNumber, e);
    }

  }

  protected void recordPdiNumber(String number)
  {
    MostRecentPdiNumbers recents = (MostRecentPdiNumbers) Component.getInstance("Ewv2RecentPdiNumbers");
    if (recents != null) {
        //log.info("recording PDI number: " + number);
        recents.addPdi(number);
    }
    else {
        log.error("Cannot get Instance of MostRecentPdiNumbers cache");
    }
  }
  
  protected EwvClaim getClaimRecord(String pdiNumber) throws ActionErrorException, NonUniqueRecordsException
  {
      EwvClaim ret = null;
      String queryString ="select claim from EwvClaim as claim where claim.pdi = :pdi";
      try {
          Query query = entityManager.createQuery(queryString);
          query.setParameter("pdi", pdiNumber);
          ret = (EwvClaim) query.getSingleResult();
      } catch (Throwable t) {
          if (t instanceof javax.persistence.NoResultException)
          {
              log.info("cannot get record for PDI: " + pdiNumber);
          }
          else if(t instanceof javax.persistence.NonUniqueResultException)
          {
              log.error("more than 1 claim exists for PDI: "+pdiNumber);
              
              // this means that there are crap claim records in DB
              throw new NonUniqueRecordsException("more than 1 claim exists for PDI: "+pdiNumber);
          }
          else
          {
              // log exception and route user to error page
              handleException(pdiNumber, t);
          }
      }
      
      return ret;
  }
  
  public String filterPdiNumber(String pdiNumber)
  {
    String returnVal = pdiNumber;
    
    if(returnVal != null) {
      returnVal = returnVal.replace(" ", "");
    }
    
    return returnVal;
  }
  
  public String filterPdiNumberDeprecatedButWorks(String pdiNumber)
  {
      CharSequence seq = new StringBuffer().append(" ").toString();
      
      if(pdiNumber != null && pdiNumber.contains(seq)) {
          pdiNumber = pdiNumber.replaceAll(" ", "");
      }
      
      return pdiNumber;
  }
  
  public void invalidate() {
    commonDemographicsInfo
        = new CommonDemographicsContainer(new PatientDataBean(),
          new ServiceLocationBean(),
          new VendorDataBean());

    // patient header panel
    patientHeaderInfo = new PatientHeaderInfo();

    // billing and ohi tab data...
    claimContainer = new ClaimContainer();
    ohiInfoContainer = new OhiInfoContainer();

    // the demographics tab data...
    PatientDataBean patientDataBean = new PatientDataBean();
    ServiceLocationBean serviceLocationDataBean = new ServiceLocationBean();
    VendorDataBean vendorDataBean = new VendorDataBean();

    vendorDataBean.setShouldRender(false);
    serviceLocationDataBean.setShouldRender(false);
    patientDataBean.setShouldRender(false);

    //log.info("PACKAGE START");

    commonDemographicsInfo
      = new CommonDemographicsContainer(patientDataBean,
                                        serviceLocationDataBean,
                                        vendorDataBean);

    claimService = (ClaimService) Component.getInstance(ClaimService.class);
    claimService.setIsClaimFound(false);

  }

  public void handleException(String pdiNumber, Throwable e) throws ActionErrorException
  {
      log.error("Could not obtain record for PDI: " + pdiNumber, e);
      
      StringWriter sw = new StringWriter();
      PrintWriter p = new PrintWriter(sw);
      
      e.printStackTrace(p);
      
      String error = sw.toString();
      
      String lastError = "PDI: " + pdiNumber + " Stacktrace Follows: \n\n"
                     + error.replaceAll("\\)", "\\)\n")
                     + "-End of Error stacktrace-";
      
      invalidate();
      
      claimService.setLastError(lastError);
      
      FacesContext context = FacesContext.getCurrentInstance();
      context.addMessage("ewvApplication",
              new FacesMessage(FacesMessage.SEVERITY_ERROR, "Exception Stack Trace", lastError));
      
      // this will be picked up by page navigation. (See pages.xml)
      throw new ActionErrorException(lastError,e);
  }

  public void considerTestErrorAndDebug(String inPdi)
  throws IOException {
    // Check the properties file for notice that we should throw a test error
    java.util.Properties props = getProperties("EWV2Application.properties");
    if (props != null && !props.isEmpty()) {
      // Turn on debug if specified.  This only controls a single field near the Build message.
      Object doDebugObject = props.get("allowDebug");
      if (doDebugObject != null) {
        String doDebugString = (String) doDebugObject;
        Boolean doDebugYN = Boolean.valueOf(doDebugString);
        if (doDebugYN) {
          doDebug = true;
        }
      }
      log.info("Debug specified?" + doDebug);
      
      // Throw an exception if specified, for testing purposes.
      Object sampleErrorObject = props.get("showSampleError");
       if (sampleErrorObject != null) {
         String sampleErrorString = (String) sampleErrorObject;
         Boolean sampleErrorYN = Boolean.valueOf(sampleErrorString);
         if (sampleErrorYN) {
           if (inPdi.equals("888888888888888")) {
             log.info("Producing a test error");
             ClaimType claimType = null;
             claimType.getValueOf();
           }
         }
       }
    }
    
    //log.info("Not producing a test error");
  }

  final public java.util.Properties getProperties(final String pFileName)
  throws IOException {
    // If the incoming parameter is null, throw an IOException rather than
    // waiting for a NullPointerException later
    if (pFileName == null) {
      throw new IOException("Properties file name may not be null");
    }
    
    // Put an input stream onto the specified properties file
    InputStream inputStream = this.getClass().getClassLoader().getResourceAsStream(pFileName);
    
    // Load the file into a properties object for return
    final java.util.Properties props = new java.util.Properties();
    if (inputStream != null) {
      props.load(inputStream);
    }
    
    return props;
  }

}
