

package gov.va.med.cds.socket.server.handler.audit;


import javax.sql.DataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.dao.DataAccessException;
import gov.va.med.cds.hapi.Hl7SupportHdrLegacy;

import org.springframework.jdbc.core.support.JdbcDaoSupport;
import gov.va.med.cds.util.DateTimeUtil;
import gov.va.med.cds.hapi.HL7SupportInterface;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.jdbc.core.JdbcTemplate;


/**
 * 
 * Socket adapter is a client to CDS application.
 * Handles the rare case scenario - audits the HL7 messages that fail the Socket adapter Validation checks.
 * There is no domain specific template information at the socket adapter
 * Uses the latest spring API that handles all the boilerplate code in order to keep the Socket adapter stack lean 
 * 
 */
public class HL7RequestAuditLogger
    extends
        JdbcDaoSupport
    implements
        HL7RequestAuditLoggerInterface
{

    private static Log logger = LogFactory.getLog( HL7RequestAuditLogger.class );

    private static final String SQL_SELECT_UNIQUE_AUDIT_CLOB_STRID = "SELECT AUDIT_CLOB_STR_SEQ_ID.NEXTVAL FROM DUAL";
    private static final String SQL_AUDIT = "INSERT INTO AUDIT_CLOB_STR (AUDIT_CLOB_STR_ID, FACILITY_ID, REQUEST_ID, HL7_CLOB, MESSAGE_DATE_TIME, MESSAGE_TYP, CLIENT_IDENTIFIER, CLIENT_REQUEST_TM ) VALUES (?, ?, ?, ?, ?, ?, ?, ? )";
    private static final String SOCKET_ADAPTER = "Socket Adapter";


    @Override
    public void auditWriteRequest( HL7SupportInterface aRequestHl7Support, String writeRequest, String msgCntrlId )
    {
        String sendingApplication = null;
        String sendingFacility = null;
        String messageDateTime = null;
        HL7SupportInterface hl7Support = null;
        try
        {
            if ( aRequestHl7Support == null )
            {
                hl7Support = new Hl7SupportHdrLegacy( writeRequest );

            }
            else
            {
                hl7Support = aRequestHl7Support;
            }
            sendingApplication = getRequestData( hl7Support, "/MSH-3", "<<sendingApp>>" );
            sendingFacility = getRequestData( hl7Support, "/MSH-4-2", "<<sendingFacility>>" );
            messageDateTime = getRequestData( hl7Support, "/MSH-7", "<<messageDateTime>>" );

        }
        catch ( Exception e )
        {
            logger.error( "Invalid HL7 meesage", e );
        }

        JdbcTemplate jdbcTemplate = getJdbcTemplate();

        try
        {
            
            jdbcTemplate.update( SQL_AUDIT,
                            new Object[] { jdbcTemplate.queryForObject( SQL_SELECT_UNIQUE_AUDIT_CLOB_STRID, Long.class ), 
                                            sendingFacility, 
                                            msgCntrlId,
                                            writeRequest, 
                                            createMessageDateTime(), 
                                            sendingApplication, 
                                            SOCKET_ADAPTER, 
                                            messageDateTime } );

        }
        catch ( Throwable e )
        {
            logger.error( "Invalid HL7 meesage", e );
        }

    }


    public String createMessageDateTime( )
    {
        return DateTimeUtil.getCurrentISOTime();
    }


    private String getRequestData( HL7SupportInterface hl7Support, String aTerserPath, String defaultValue )
    {
        String messageData = defaultValue;

        try
        {
            messageData = hl7Support.getMessageData( aTerserPath );
        }
        catch ( Exception ex )
        {
            /* Swallow the exception */
        }

        return messageData;
    }
}
