

package gov.va.med.cds.scheduler;


import java.util.Map;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import gov.va.med.cds.client.proxy.NetworkClientProxyInterface;
import gov.va.med.cds.exception.CensusTransformAckException;
import gov.va.med.cds.exception.ErrorCodeEnum;
import gov.va.med.cds.hapi.HL7SupportHAPI;
import gov.va.med.cds.hapi.HL7SupportInterface;
import gov.va.med.cds.hapi.HL7Utility;
import gov.va.med.cds.mediator.persistence.Hl7AuditDaoInterface;
import gov.va.med.cds.scheduler.persistence.CensusAcknowledgementResponse;
import gov.va.med.cds.scheduler.persistence.CensusCompleteWeek;
import gov.va.med.cds.scheduler.persistence.RoutingData;
import gov.va.med.cds.util.DateTimeUtil;


public class CensusAcknowledgementWork
    implements
        CensusAcknowledgementWorkInterface
{
    private static Log logger = LogFactory.getLog( CensusAcknowledgementWork.class );

    private static final String COMMIT_ACK = "CA";
    private static final String DOD_DATA = "DOD_DATA";
    private static final String VA_DATA = "VA_DATA";
    private static final String TRANSFROM_SUCCESS_STATUS = "COMPLETE";
    private static final String ERROR_MSG_DELIMITER = "\\|";
    private static final String AA_SUCCESS_MSG = "Census report has been successfully received";
    private static final String AR_EXPIRED_MSG = "Census Report segments missing or too many sent or not sent in a timely manner. Census start date: %s; Census End date: %s; Message Id: %s; Vendor Id: %s";

    private static final String HTH_CENSUS_SENDING_APP = "HTAPPL-CENSUS";
    private static final String WRITE_REQUEST_CENSUS_ALREADY_EXPIRED = "WRITE_REQUEST_CENSUS_ALREADY_EXPIRED";
    private static final String WRITE_REQUEST_CENSUS_EXPIRED = "WRITE_REQUEST_CENSUS_EXPIRED";
    private static final String WRITE_REQUEST_CENSUS_TOO_MANY_SEGMENTS = "WRITE_REQUEST_CENSUS_TOO_MANY_SEGMENTS";

    private Map<String, NetworkClientProxyInterface> clientProxiesMap;
    private Map<String, Hl7AuditDaoInterface> hl7AuditDaoMap = null;


    @Override
    public void sendCensusAcknowledgement( CensusAcknowledgementResponse aTransformResponse )
    {
        String sendingFacility = null;
        String sentFrom = null;
        String transformStatus = null; 
        String appAck = null;
        String errorMessage = null;
        
        try
        {
            if ( null != aTransformResponse )
            {
                sendingFacility = aTransformResponse.getSendingFacility();
                sentFrom = ( "200CH".equals( sendingFacility ) || "200HD".equals( sendingFacility ) ) ? DOD_DATA : VA_DATA;
                transformStatus = aTransformResponse.getTransformStatus();
                
                if ( ( null != transformStatus ) && transformStatus.equalsIgnoreCase( TRANSFROM_SUCCESS_STATUS ) )
                {
                    RoutingData routingData = ( RoutingData )aTransformResponse;
                    appAck = createAppAckNoErrors( routingData, aTransformResponse.getMessageControlId() );
                }
                else
                {
                    // elaborate error messages for error codes that are returned from Census_Ingest_Processing.Transform_Ack_Build for duplicate census messages
                	errorMessage = inspectErrors( aTransformResponse );
                                    
                    appAck = createAppAckWithErrorResponse( aTransformResponse, aTransformResponse.getMessageControlId(), errorMessage );
                }

                if ( appAck != null )
                {
                    sendAcknowledgementResponse( appAck, HTH_CENSUS_SENDING_APP, sendingFacility, sentFrom );
                }
            }
            else
            {
                logger.error( ErrorCodeEnum.CENSUS_ACKNOWLEDGEMENT_RESPONSE_IS_NULL );
                throw new CensusTransformAckException( ErrorCodeEnum.CENSUS_ACKNOWLEDGEMENT_RESPONSE_IS_NULL );
            }
        }
        catch ( Exception e )
        {
            logger.error( String.format( "Error sending acknowledement.\ncensusMasterId:\n%s", ( null != aTransformResponse ) ? aTransformResponse.getCensusMasterId() : "UNKNOWN" ) );
            throw e;
        }
    }


	@Override
    public void sendCensusAcknowledgement( CensusCompleteWeek aCensusCompleteWeek, Long aCensusMasterId )
    {
        String sendingFacility = null;
        String sentFrom = null;
        String appAck = null;

        try
        {
            if ( null != aCensusCompleteWeek )
            {
                sendingFacility = aCensusCompleteWeek.getSendingFacility();
                sentFrom = ( "200CH".equals( sendingFacility ) || "200HD".equals( sendingFacility ) ) ? DOD_DATA : VA_DATA;

                RoutingData routingData = ( RoutingData )aCensusCompleteWeek;
                String errorMessage = String.format( AR_EXPIRED_MSG, DateTimeUtil.convertDate2String( aCensusCompleteWeek.getReportStartDt() ),
                                DateTimeUtil.convertDate2String( aCensusCompleteWeek.getReportEndDt() ), aCensusCompleteWeek.getRequestId(),
                                aCensusCompleteWeek.getVendorFacility() );
                appAck = createAppAckWithErrorResponse( routingData, aCensusCompleteWeek.getRequestId(), errorMessage );

                if ( appAck != null )
                {
                    sendAcknowledgementResponse( appAck, HTH_CENSUS_SENDING_APP, sendingFacility, sentFrom );
                }
            }
            else
            {
                logger.error( ErrorCodeEnum.CENSUS_COMPLETE_WEEK_IS_NULL );
                throw new CensusTransformAckException( ErrorCodeEnum.CENSUS_COMPLETE_WEEK_IS_NULL );
            }
        }
        catch ( Exception e )
        {
            logger.error( String.format( "Error sending acknowledement.\ncensusMasterId:\n%s", aCensusMasterId ) );
            throw e;
        }
    }


    private String createAppAckWithErrorResponse( RoutingData aRoutingData, String aMessageControlId, String aErrorMessages )
    {
        return String.format( createAckResponse( aRoutingData ).concat( "AR^%s" ), aMessageControlId )
                        .concat( constructErrorMessages( aErrorMessages ) );
    }


    private String createAppAckNoErrors( RoutingData aRoutingData, String aMessageControlId )
    {
        return String.format( createAckResponse( aRoutingData ).concat( "AA^%s^%s" ), aMessageControlId, AA_SUCCESS_MSG );
    }


    private String createAckResponse( RoutingData aRoutingData )
    {
        return String.format( "MSH^~|\\&^%s^200HD~%s~DNS^%s^%s~%s~DNS^%s^^ACK^%s^P^2.4^^^AL^NE\rMSA^", aRoutingData.getReceivingApplication(),
                        aRoutingData.getReceivingFacility(), aRoutingData.getSendingApplication(), aRoutingData.getStationNumber(),
                        aRoutingData.getSendingFacility(), createMessageDateTime(), createMessageControlId() );
    }


    private String constructErrorMessages( String aErrorMessages )
    {
        StringBuilder sb = new StringBuilder();

        for ( String errorMessage : aErrorMessages.split( ERROR_MSG_DELIMITER ) )
        {
            if ( ( null != errorMessage ) && ( errorMessage.trim().length() > 0 ) )
            {
                sb.append( String.format( "\rERR^%s", errorMessage.trim() ) );
            }
        }

        return sb.toString();
    }


    private String createMessageDateTime( )
    {
        return HL7Utility.createMessageDateTime();
    }


    private String createMessageControlId( )
    {
        return HL7Utility.createMessageControlId();
    }


    private void sendAcknowledgementResponse( String anAcknowledgement, String aSendingApp, String aSendingFacility, String aSentFrom )
    {
        Hl7AuditDaoInterface hl7AuditDao = null;
        NetworkClientProxyInterface clientProxy = clientProxiesMap.get( aSendingApp );

        if ( clientProxy != null )
        {
            Properties properties = new Properties();
            properties.setProperty( NetworkClientProxyInterface.PROP_DNS_NAME, aSendingFacility );

            // write the acknowledgment to the HL7 Message Log
            HL7SupportInterface hl7supportACK = HL7SupportHAPI.createTranslationSupport( anAcknowledgement );

            if ( hl7AuditDaoMap != null )
            {
                hl7AuditDao = hl7AuditDaoMap.get( aSentFrom );
                hl7AuditDao.addMessage( hl7supportACK.getMessageControlId(), anAcknowledgement );
            }

            // send the acknowledgment
            String acknowledgmentCommitAck = clientProxy.sendMessage( anAcknowledgement, properties );

            // Write an error message to the CDS Application Log if other than CA.
            HL7SupportInterface hl7SupportCA = HL7SupportHAPI.createTranslationSupport( acknowledgmentCommitAck );
            if ( !hl7SupportCA.getAcknowledgementCode().equalsIgnoreCase( COMMIT_ACK ) )
            {
                if ( logger.isErrorEnabled() )
                {
                    logger.error( String.format( "Error sending acknowledement.\nCommit Ack:\n%s", acknowledgmentCommitAck ) );
                }
                if ( logger.isDebugEnabled() )
                {
                    logger.debug( String.format( "Error sending acknowledement.\nAcknowledgement: \n%s\n\nCommit Ack:\n%s", anAcknowledgement,
                                    acknowledgmentCommitAck ) );
                }
            }

            // write the CA message to the HL7 Message Log
            if ( hl7AuditDao != null )
            {
                hl7AuditDao.addMessage( hl7supportACK.getMessageControlId(), acknowledgmentCommitAck );
            }
        }
    }
    
    
    private String inspectErrors( CensusAcknowledgementResponse aTransformResponse ) 
    {
    	String errorMessage = null;
    	
    	errorMessage = ( ( null != aTransformResponse.getErrorMessage() ) && ( aTransformResponse.getErrorMessage().trim().length() > 0 ) ) ? aTransformResponse.getErrorMessage() : null;
                
    	// populate error message for duplicate expired census messages
        if ( null != errorMessage )
        {
        	if ( errorMessage.contentEquals(WRITE_REQUEST_CENSUS_ALREADY_EXPIRED) 
        			|| errorMessage.contentEquals(WRITE_REQUEST_CENSUS_EXPIRED) 
        			|| errorMessage.contentEquals(WRITE_REQUEST_CENSUS_TOO_MANY_SEGMENTS) )  
        	{
        		errorMessage = String.format( AR_EXPIRED_MSG, DateTimeUtil.convertDate2String( aTransformResponse.getReportStartDate() ),
        				DateTimeUtil.convertDate2String( aTransformResponse.getReportEndDate() ), aTransformResponse.getMessageControlId(),
        				aTransformResponse.getVendorFacility() );
        	}
        }
        
        return errorMessage;
	}


    public void setClientProxiesMap( Map<String, NetworkClientProxyInterface> aClientProxiesMap )
    {
        this.clientProxiesMap = aClientProxiesMap;
    }


    public void setHl7AuditDaoMap( Map<String, Hl7AuditDaoInterface> aHl7AuditDaoMap )
    {
        this.hl7AuditDaoMap = aHl7AuditDaoMap;
    }
}
