

package gov.va.med.cds.tools.cleanup.errorq;


import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import ca.uhn.hl7v2.HL7Exception;


public class MessagePreProcessor
{
    private static Log logger = LogFactory.getLog( MessagePreProcessor.class );


    public MessagePreProcessor()
    {
    }

    
    public static String[] getParsedMSHSegment( String aHL7Request )
        throws HL7Exception
    {
        String[] parsedMSHSegment = null;
        
        if ( ( aHL7Request.length() > 4 ) && aHL7Request.startsWith( "MSH" ) )
        {
            parsedMSHSegment = new String[25];
            char fieldSeparator = aHL7Request.charAt( 3 );
            parsedMSHSegment[0] = "MSH";
            parsedMSHSegment[1] = aHL7Request.substring( 3, 4 );
            int currentFieldIndex = 2;
            int fieldStartIndex = 4;
            for ( int i = 4; i < aHL7Request.length(); i++ )
            {
                if ( aHL7Request.charAt( i ) == fieldSeparator )
                {
                    parsedMSHSegment[currentFieldIndex++] = aHL7Request.substring( fieldStartIndex, i );
                    fieldStartIndex = i + 1;
                }
                
                if ( aHL7Request.charAt( i ) == '\r' )
                {
                    break;
                }
            }
        }
        else
        {
            throw new HL7Exception( "Message Invalid, HL7 messages must start with MSH");            
        }
        
        return parsedMSHSegment;
    }

    
    public static String preProcessHL7Message( String aHL7Request )
        throws HL7Exception
    {
        String[] parsedMSHSegment = getParsedMSHSegment( aHL7Request );
        if ( parsedMSHSegment.length > 3 )
        {
            String sendingApplication = parsedMSHSegment[3];
            if ( sendingApplication.equals( "LA7LAB" ) )
            {
                aHL7Request = handleEmptyLabObx5( aHL7Request );
//                aHL7Request = handleCRinNTE( aHL7Request ); // NTE to OBX (Lab)
            }
    
            if ( sendingApplication.equals( "HDRPREF" ) || sendingApplication.equals( "HDRPRES" ) || sendingApplication.equals( "HDRPPAR" ) )
            {
                aHL7Request = handleRxCRinRXEorNTE( aHL7Request ); // RXE to RXR (Pharmacy)
            }
/* Temporary comment out 
        if ( sendingApplication.equals( "HDRALGY" ) || sendingApplication.equals( "HDRADAS" ) )
        {
//            aHL7Request = handleCRinOBX( aHL7Request ); // OBX to end (Allergy)
        }
 */      
        }
        
        return aHL7Request;
    }
    
    
    private static String handleEmptyLabObx5( String hl7Message )
        throws HL7Exception
    {
        if ( logger.isDebugEnabled() )
        {
            logger.debug( " Begin." );
        }
        // sjb - non-java version
        // OBX\|\d*\|\|[^\|]*\|[^\|]*\|\"\"\|
        String regexPattern = "OBX\\|\\d*\\|\\|[^\\|]*\\|[^\\|]*\\|\"\"\\|"; // brings entire fragment that has all occurrences of ""

        Pattern pattern = Pattern.compile( regexPattern );
        Matcher matcher = pattern.matcher( hl7Message );

        String replacementString;
        while ( matcher.find() )
        {
            replacementString = matcher.toMatchResult().group();

            int idx = replacementString.lastIndexOf( '|' ) - 2;
            replacementString = replacementString.substring( 0, idx ) + "|";
            hl7Message = hl7Message.replaceFirst( regexPattern, replacementString );

            matcher = pattern.matcher( hl7Message );
        }
        
        if ( logger.isDebugEnabled() )
        {
            logger.debug( " End." );
        }

        return hl7Message;
    }


    // The method below removes the end of line terminator for OBR, NTE, NTE, OBX so that
    // the NTE, NTE, OBX segments look 
/*
 *  JLA temporarily commenting out. Fortify Quality Code Review flagged these methods as Dead Code: Unused Method
    private static String handleCRinNTE( String hl7Message )
        throws HL7Exception
    {
        // sjb - non-java version
        // \r\n(NTE(.*\r\n)*?)OBX
//        String regexPattern = "\\r(NTE(.*\\r(?!OBX))(.*\\r){1,}?(?=OBX))";
        String regexPattern = "\\R(NTE(.*\\R(?!OBX))(.*\\R)+?(?=OBX))";
        // \r\n(NTE(.*\r\n(?!OBX))(.*\r\n){1,}?(?=OBX)) // works great no OBX in
        // result
        // return handleCRinSegmentInMiddle( hl7Message, regexPattern );
        return handleCRinSegmentAtEnd( hl7Message, regexPattern );
    }


    private static String handleCRinOBX( String hl7Message )
        throws HL7Exception
    {
        // sjb - non-java version
        // OBX.*TX.COMMENT.*(\r\n.*)*
//        String regexPattern = "OBX.*TX.COMMENT.*(\\r.*)*";
        String regexPattern = "OBX.*TX.COMMENT.*(\\R.*)*";

        return handleCRinSegmentAtEnd( hl7Message, regexPattern );
    }
 */


    private static String handleRxCRinRXEorNTE( String hl7Message )
        throws HL7Exception
    {
        // sjb - non-java version
        // \r\n(RXE(.*\r\n)*?)RXR
//        String regexPattern = "\\r(RXE(.*\\r)*?)(RXR|FT1)";

//        String regexPattern = "\\R(RXE(.*\\R)*?)(RXR|FT1)";
        String regexPattern = "\\R((RXE(.*\\R(?!NTE))*?)(?=RXR|FT1)|(NTE(.*\\R(?!(NTE|ORC|FT1))(.*\\R)+?)(?=(NTE|ORC|FT1))))";

        return handleRxCRinSegmentRXEorNTE( hl7Message, regexPattern );
    }


    /*
     * This method splits the match from the regular expression passed in by an end of line terminator. All split lines append a ' ' and then
     * the last line appends a '\r'. Because of this, this method can only handle regular expressions that match one segment. If two 
     * or more segments are matched, this method will remove the end of line character between segments and return one segment for 
     * what should have been multiple segments.
     */
    private static String handleRxCRinSegmentRXEorNTE( String hl7Message, String regexPattern )
        throws HL7Exception
    {
        if ( logger.isDebugEnabled() )
        {
            logger.debug( " Begin.");
        }
        
        Pattern pattern = Pattern.compile( regexPattern );
        Matcher matcher = pattern.matcher( hl7Message );

        String regexMatch;
        while ( matcher.find() )
        {
            if ( logger.isDebugEnabled() )
            {
                logger.info( "Found RegEx match in HL7 message - checking for extra end of lines." );
            }
            regexMatch = matcher.group( 1 );
            if ( logger.isDebugEnabled() )
            {
                logger.debug( "RegEx match: '" + regexMatch + "'" );
            }
            int group1Start = matcher.start( 1 );
            int group1End = matcher.end( 1 );
            String badPart = hl7Message.substring( group1Start, group1End );
            if ( logger.isDebugEnabled() )
            {
                logger.debug( "group1 start/group1 end: " + group1Start + "/" + group1End );
                logger.debug( "badPart: '" + badPart + "'" );
            }
            
            // Look for any "end of line" characters that should not be here and replace them with a ' '(space).
            String[] lines = regexMatch.split( "\\R" );
            if ( lines.length > 1 )
            {
                if ( logger.isInfoEnabled() )
                {
                    logger.info( "Found CR in match - fix may be needed" );
                }
                StringBuilder newString = new StringBuilder();
                for ( String string : lines )
                {
                    if ( logger.isDebugEnabled() )
                    {
                        logger.debug( "splitline: '" + string + "'" );
                    }
                    newString.append( string ).append( ' ' );
                }
                // Change the last space back to a end of line to end the segment
                newString.setCharAt( newString.lastIndexOf( " " ), '\r' );
                
                if ( logger.isDebugEnabled() )
                {
                    logger.debug( "badpart replacement: '" + newString.toString() + "'" );
                }
                hl7Message = hl7Message.replace( badPart, newString.toString() );
                
                /*
                 *  Reset the matcher in case the replaced string is a different size than the original
                 *  as the matchers group start and ends would be off
                 */
//              matcher = pattern.matcher( hl7Message );
                matcher.reset( hl7Message );
                
                if ( logger.isInfoEnabled() )
                {
                    logger.info( "Found CR in segment - fixed" );
                }
            }
        }
        
        if ( logger.isDebugEnabled() )
        {
            logger.debug( " End.");
        }
        
        return hl7Message;
    }

/*
 *  JLA temporarily commenting out. Fortify Quality Code Review flagged these methods as Dead Code: Unused Method
    private static String handleCRinSegmentAtEnd( String hl7Message, String regexPattern )
        throws HL7Exception
    {
        if ( logger.isDebugEnabled() )
        {
            logger.debug( " Begin." );
        }
        Pattern pattern = Pattern.compile( regexPattern );
        Matcher matcher = pattern.matcher( hl7Message );

        String replacementString;
        while ( matcher.find() )
        {
            if ( logger.isInfoEnabled() )
            {
                logger.info( "Found CR in segment - fixing..." );
            }

            replacementString = matcher.toMatchResult().group( 0 );
            // int group1Start = matcher.start( 1 );
            // int group1End = matcher.end( 1 );
            // String badPart = hl7Message.substring( group1Start, group1End );

            // String[] lines = replacementString.split( "\r" );
            String[] lines = replacementString.split( "\\R" );
            StringBuilder newString = new StringBuilder();
            for ( String string : lines )
            {
                newString.append( string ).append( ' ' );
            }
            // newString.setCharAt( newString.lastIndexOf( " " ), '\r' );

            hl7Message = hl7Message.replace( replacementString, newString.toString() );
            if ( logger.isInfoEnabled() )
            {
                logger.info( "Found CR in segment - fixed" );
            }
        }
        
        if ( logger.isDebugEnabled() )
        {
            logger.debug( " End." );
        }

        return hl7Message;
    }
 */
}
