package gov.va.med.cds.tools.cleanup.errorq;

import java.util.ArrayList;
import java.util.Enumeration;

import javax.annotation.Resource;
import javax.jms.JMSException;
import javax.jms.Message;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.jdbc.core.JdbcTemplate;


/**
 * This method has all of the SQL for updating log records in the database (for the error queue cleanup process).
 * 
 * TODO Make some unit tests for this class.
 * 
 * @author vhaislbrushs
 *
 */
public class ErrorQueueCleanUpLogger
{
    private static Log logger = LogFactory.getLog( ErrorQueueCleanUpLogger.class );
    private JdbcTemplate hdrJdbcTemplate;
    private long logId = 0;
    

    public void logComplete( )
    {
        StringBuilder sqlQuery = new StringBuilder();
        sqlQuery.append( "update LOGGER.cds_errorq_process_log " );
        sqlQuery.append( " set success = 'T', last_process_date = sysdate " );
        sqlQuery.append( ", num_times_processed = num_times_processed + 1 " );
        sqlQuery.append( " where errorq_process_log_id = ? " );
        
        logUpdate( sqlQuery.toString(), logId );
    }

    
    private void logUpdate( String anUpdateString, Object... anArgs )
    {
        if ( logger.isDebugEnabled() )
        {
            logger.debug( "trying to log with SQL: " + anUpdateString + "(" + anArgs + ")" );
        }
        try
        {
            hdrJdbcTemplate.update( anUpdateString, anArgs );
        }
        catch ( DataIntegrityViolationException e )
        {
            // unique constraint violation - only occurs from logParseInfo
            // logParseInfo will retry if it gets this exception so throw it
            logger.fatal( "DataIntegrityViolationException at database.", e );
            throw e;
        }
        catch ( Exception e )  
        {
            logger.fatal( "Could not log to database.", e );
        }
    }

    
    private void deleteLog( long aLogId )
    {
        StringBuilder sqlQuery = new StringBuilder();
        sqlQuery.append( "delete from LOGGER.cds_errorq_process_log " );
        sqlQuery.append( " where errorq_process_log_id = ?" );

        logUpdate( sqlQuery.toString(), aLogId );
    }
    
    
    private void logUpdateProperty( String... aPropertyNameValues )
    {
        StringBuilder sqlQuery = new StringBuilder();
        sqlQuery.append( "update LOGGER.cds_errorq_process_log " );
        sqlQuery.append( " set last_process_date = sysdate" );

        ArrayList<Object> values = new ArrayList<Object>();
        for ( int i = 0; i < aPropertyNameValues.length; i = i+2 )
        {
            sqlQuery.append( ", " ).append( aPropertyNameValues[i] ).append( " = ? " );
        }
        for ( int i = 1; i < aPropertyNameValues.length; i = i+2 )
        {
            values.add( aPropertyNameValues[i] );
        }

        sqlQuery.append( " where errorq_process_log_id = ? " );
        values.add( logId );
        
        logUpdate( sqlQuery.toString(), values.toArray() );
    }

    
    public void logError( Throwable aThrowable )
    {
        String message = ( aThrowable == null ) ? "no exception given" : aThrowable.getMessage();
        if ( message.length() > 4000 )
        {
            message = message.substring( 0, 4000 );
        }
        
        StringBuilder sqlQuery = new StringBuilder();
        sqlQuery.append( "update LOGGER.cds_errorq_process_log " );
        sqlQuery.append( " set success = 'F', last_process_date = sysdate " );
        sqlQuery.append( ", num_times_processed = num_times_processed + 1 " );
        sqlQuery.append( ", eq_process_exception_detail = ?" );
        sqlQuery.append( " where errorq_process_log_id = ?" );

        logUpdate( sqlQuery.toString(), message, logId );
    }


    public void logCdsExceptionDetail( String aMessage )
    {
        aMessage = ( aMessage == null ) ? "no detail given" : aMessage.substring(0, Math.min(aMessage.length(), 4000));
        
        StringBuilder sqlQuery = new StringBuilder();
        sqlQuery.append( "update LOGGER.cds_errorq_process_log " );
        sqlQuery.append( " set cds_exception_detail = ?" );
        sqlQuery.append( " where errorq_process_log_id = ?" );

        logUpdate( sqlQuery.toString(), aMessage, logId );
    }

    
    public long logStart( Message aMessage )
    {
        // do something with properties
        if ( logger.isDebugEnabled() )
        {
            try
            {
                logger.debug( "message dest: " + aMessage.getJMSDestination() );
                Enumeration<?> names = aMessage.getPropertyNames();
                while ( names.hasMoreElements() )
                {
                    String name = names.nextElement().toString();
                    logger.debug( "property: " + name + ", value: " + aMessage.getObjectProperty( name ) );
                }
            }
            catch ( JMSException e )
            {
            }
        }

        String messageType = aMessage.getClass().getName();
        StringBuilder sqlQuery = new StringBuilder();
        sqlQuery.append( "insert into LOGGER.cds_errorq_process_log " );
        sqlQuery.append( " columns ( " );
        sqlQuery.append( "errorq_process_log_id, message_type, first_process_date" );
        sqlQuery.append( " ) " );
        sqlQuery.append( " values ( " );
        sqlQuery.append( "?, ?, sysdate" );
        sqlQuery.append( " ) " );

        try
        {
            logId = hdrJdbcTemplate.queryForObject( "select seq_errorq_process_log_id.nextval from dual", Long.class );
            String queryString = sqlQuery.toString();
            if ( logger.isDebugEnabled() )
            {
                logger.debug( "logId: " + logId );
                logger.debug( "trying to log with SQL: " + queryString );
            }
            hdrJdbcTemplate.update( queryString, new Object[] { logId, messageType } );
        }
        catch ( Exception e )
        {
            logger.fatal( "Could not log to database.", e );
        }

        return logId;
    }
    
    
    public void logMessageType( ca.uhn.hl7v2.model.Message aMessage )
    {
        String messageType = ( aMessage != null ) ? aMessage.getClass().getName() : "null message";
        StringBuilder sqlQuery = new StringBuilder();
        sqlQuery.append( "update LOGGER.cds_errorq_process_log " );
        sqlQuery.append( " set message_type = message_type || ' / ' || ? " );
        sqlQuery.append( " where errorq_process_log_id = ?" );
        
        logUpdate( sqlQuery.toString(), messageType, logId );
    }


    public void logStyle( String style )
    {
        logUpdateProperty( "message_style", style );
    }

    
    public void addLogStyle( String aStyle )
    {
        StringBuilder sqlQuery = new StringBuilder();
        sqlQuery.append( "update LOGGER.cds_errorq_process_log " );
        sqlQuery.append( " set message_style = message_style || ' / ' || ? " );
        sqlQuery.append( " where errorq_process_log_id = ?" );
        
        logUpdate( sqlQuery.toString(), aStyle, logId );
    }

    
    public void logDecoded( )
    {
        logUpdateProperty( "decoded", "T" );
    }

    
    public void logExceptionType( String aValue )
    {
        logUpdateProperty( "exception_type", aValue );
    }

    
    public void logDuplicate( boolean isDuplicate )
    {
        logUpdateProperty( "duplicate", isDuplicate ? "Y" : "N" );
    }

    
    public void logFacility( String aSendingFacilityId, String aSendingFacilityName )
    {
        logUpdateProperty( "facility_id", aSendingFacilityId, "facility_name", aSendingFacilityName );
    }
    
    
    public void logFacilityId( String aSendingFacilityId )
    {
        logUpdateProperty( "facility_id", aSendingFacilityId );
    }
    
    
    public void logPsimNationalId( String aNid )
    {
        StringBuilder sqlQuery = new StringBuilder();
        sqlQuery.append( "update LOGGER.cds_errorq_process_log " );
        sqlQuery.append( " set domain_name = domain_name || ' NID: ' || ? " );
        sqlQuery.append( " where errorq_process_log_id = ?" );
        
        logUpdate( sqlQuery.toString(), aNid, logId );
    }
    
    
    public void logPsimLocalId( String aLid )
    {
        StringBuilder sqlQuery = new StringBuilder();
        sqlQuery.append( "update LOGGER.cds_errorq_process_log " );
        sqlQuery.append( " set domain_name = domain_name || ' LID: ' || ? " );
        sqlQuery.append( " where errorq_process_log_id = ?" );
        
        logUpdate( sqlQuery.toString(), aLid, logId );
    }
    
    
    public void logPsimCorrelation( String aCorrelationStatus )
    {
        logUpdateProperty( "exception_type", aCorrelationStatus );
    }
    
    
    public void logParseInfo( String aMessageControlId, String aSendingApp, String aMessageDateTime )
    {
        logParseInfo( aMessageControlId, aSendingApp, aMessageDateTime, 1 );
    }
    
    
    /**
     * In order to reduce the messages in the log table, if the messageControlId exists, update that
     * record instead of creating a new one.  Because of multi-threading at the database, there
     * could be a slight race condition, which is resolved by trying again, up to 3 times.
     * 
     * @param aMessageControlId
     * @param aSendingApp
     * @param aMessageDateTime
     * @param aTryNum
     */
    private void logParseInfo( String aMessageControlId, String aSendingApp, String aMessageDateTime, int aTryNum )
    {
        StringBuilder sqlQuery = new StringBuilder();
        sqlQuery.append( "select count(errorq_process_log_id) " );
        sqlQuery.append( " from LOGGER.cds_errorq_process_log " );
        sqlQuery.append( " where REQUEST_ID = ? and SENDING_APP = ? and MESSAGE_DATE_TIME = ?" );
        
        try
        {   String queryString = sqlQuery.toString();
            if(logger.isDebugEnabled())
            {
                logger.debug( "trying to log with SQL: " + queryString );
                logger.debug( "trying to log parse info - try: " + aTryNum );
            }
            // getting count because query fails with org.springframework.dao.EmptyResultDataAccessException: if doesn't exist
            long logCount = hdrJdbcTemplate.queryForObject( queryString, new Object[] { aMessageControlId, aSendingApp, aMessageDateTime } , Long.class );
            
            if ( logCount == 0 )
            {
                // new log record - threading could cause contention at database
                try
                {
                    logUpdateProperty( "REQUEST_ID", aMessageControlId, "SENDING_APP", aSendingApp, "MESSAGE_DATE_TIME", aMessageDateTime );
                }
                catch ( DataIntegrityViolationException e )
                {
                    if ( aTryNum <= 3 )
                    {
                        logger.warn( "trying again to log parse info - try: " + aTryNum );
                        logParseInfo( aMessageControlId, aSendingApp, aMessageDateTime, aTryNum +1 );
                    }
                    else
                    {
                        logger.fatal( "Could not log parse info after " + aTryNum + " tries." );
                    }
                }
            }
            else if ( logCount == 1 )
            {
                if( logger.isDebugEnabled()){
                    logger.debug( "log parse info try " + aTryNum + ": record already exists: (requestId, sendingApp, messageDateTime) " + aMessageControlId + ", " + aSendingApp + ", " + aMessageDateTime );
                }
                
                sqlQuery = new StringBuilder();
                sqlQuery.append( "select errorq_process_log_id " );
                sqlQuery.append( " from LOGGER.cds_errorq_process_log " );
                sqlQuery.append( " where REQUEST_ID = ? and SENDING_APP = ? and MESSAGE_DATE_TIME = ?" );
                long existingLogId = hdrJdbcTemplate.queryForObject( sqlQuery.toString(), new Object[] { aMessageControlId, aSendingApp, aMessageDateTime }, Long.class  );
                
                deleteLog( logId );

                logId = existingLogId;
            }
            else if ( logCount > 1 )
            {
                logger.error( "multiple log records already exists: (requestId, sendingApp, messageDateTime) " + aMessageControlId + ", " + aSendingApp + ", " + aMessageDateTime );

                logUpdateProperty( "REQUEST_ID", aMessageControlId, "SENDING_APP", aSendingApp, "MESSAGE_DATE_TIME", aMessageDateTime ); 
            }
        }
        catch ( Exception e )  
        {
            logger.fatal( "Had some problem.", e );
        }
    }

    
    public void logRemovedFromQueue( )
    {
        logUpdateProperty( "queued_for_reprocess", "R" );
    }

    
    public void logRequeued( boolean isRequeued )
    {
        logUpdateProperty( "queued_for_reprocess", isRequeued ? "Y" : "N" );
    }
    


    
    
    // for the main running statistics
    
    public void logProcessFinish( long aProcessLogId, long aMessagesProcessed )
    {
        StringBuilder sqlQuery = new StringBuilder();
        sqlQuery.append( "update LOGGER.cds_eq_process_run_log " );
        sqlQuery.append( " set finish_date = sysdate " );
        sqlQuery.append( ", processed_count = ? " );
//        , num_duplicates = nd, num_success = ns, num_failed = nf" );
        sqlQuery.append( " where eq_process_run_log_id = ? " );
        
        logUpdate( sqlQuery.toString(), aMessagesProcessed, aProcessLogId );
    }

    
    public long logProcessStart( String aConfigFile )
    {
        long processLogId = 0;

        StringBuilder sqlQuery = new StringBuilder();
        sqlQuery.append( "insert into LOGGER.cds_eq_process_run_log " );
        sqlQuery.append( " columns ( " );
        sqlQuery.append( "eq_process_run_log_id, config_file, start_date " );
        sqlQuery.append( " ) " );
        sqlQuery.append( " values ( " );
        sqlQuery.append( "?, ?, sysdate" );
        sqlQuery.append( " ) " );
        
        try
        {
            processLogId = hdrJdbcTemplate.queryForObject( "select seq_eq_process_run_log_id.nextval from dual", Long.class  );
            String queryString = sqlQuery.toString();
            if(logger.isDebugEnabled()){
                logger.debug( "process logId: " + processLogId  + ", trying to log with SQL: " + queryString );
            }
            hdrJdbcTemplate.update( queryString, new Object[] { processLogId, aConfigFile } );
        }
        catch ( Exception e )  
        {
            logger.fatal( "Could not log to database.", e );
        }

        return processLogId;
    }

    
    @Resource
    public void setHdrJdbcTemplate( JdbcTemplate aHdrJdbcTemplate )
    {
        this.hdrJdbcTemplate = aHdrJdbcTemplate;
    }

}
