

package gov.va.med.cds.tools.cleanup.errorq;


import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;

import ca.uhn.hl7v2.HL7Exception;
import ca.uhn.hl7v2.model.Message;
import ca.uhn.hl7v2.model.v24.datatype.EI;
import ca.uhn.hl7v2.model.v24.message.ORU_R01;
import ca.uhn.hl7v2.model.v24.message.RDE_O11;
import ca.uhn.hl7v2.model.v24.message.RDS_O13;
import ca.uhn.hl7v2.parser.EncodingNotSupportedException;
import ca.uhn.hl7v2.parser.PipeParser;
import ca.uhn.hl7v2.util.Terser;
import ca.uhn.hl7v2.validation.impl.ValidationContextImpl;
import gov.va.med.cds.tools.cleanup.errorq.exception.ErrorQCleanupException;
import gov.va.med.cds.util.DateTimeUtil;


/**
 * @author susarlan
 */
public class ErrorQMessageParser
    implements
        BeanFactoryAware
{
    private static Log logger = LogFactory.getLog( ErrorQMessageParser.class );

    private static final String T_SENDING_APPLICATION = "/MSH-3";
    private static final String T_MESSAGE_DATE_TIME = "/MSH-7";
    private static final String T_STATION_NUMBER = "/MSH-4-1";
    private static final String T_SENDING_FACILITY_ID = "/MSH-4-2";
    private static final String T_IDENTIFIER_TYPE = "/.PID-3(2)-5";
    private static final String T_ASSIGNING_FACILITY = "/.PID-3(2)-6-2";
    private static final String T_MESSAGE_CONTROL_ID = "/MSH-10";

    private Map<String, ErrorQHandlerInformation> sendingApplicationToErrorQHandlerInfoMap;
    private List<String> knownRejectableSendingAppList;
    private Map<String, String> facilityIdMap;
    private Map<String, String> messageValidAgeMap;
    private BeanFactory beanFactory;


    /**
     * Used for tests
     * 
     * @param aDecodedHl7Message
     * @return
     */
    public String parseMessage( String aDecodedHl7Message )
    {
        return parseMessage( aDecodedHl7Message, new ErrorQueueCleanUpLogger() );
    }


    /**
     * 
     * @param aDecodedHl7Message
     * @param anErrorQueueCleanUpLogger
     * @return null indicates that message is consumed successfully and will be removed from the error queue by the WebLogic JMS
     *         otherwise an error message created containing info. generated while trying to process the message. 
     */
    public String parseMessage( String aDecodedHl7Message, ErrorQueueCleanUpLogger anErrorQueueCleanUpLogger )
    {
        String returnValue = null;
        String sendingApp = ErrorQueueConstant.DEFAULT_MESSAGE_SENDING_APP;

        if ( logger.isDebugEnabled() )
        {
            logger.debug( " Begin." );
        }

        GlobalValueObject globalValueObject = ErrorQueueThreadLocal.get();
        globalValueObject.setSendingApp( sendingApp );                    // set to default value, i.e, UNKNOWN
        ErrorQueueThreadLocal.set( globalValueObject );

        try
        {
            // Preprocess the decoded Hl7 message input request 
            aDecodedHl7Message = MessagePreProcessor.preProcessHL7Message( aDecodedHl7Message );
            
            // Parse the input request using HAPI
            Message message = parseUsingHapi( aDecodedHl7Message );
            
            // Log the message
            anErrorQueueCleanUpLogger.logMessageType( message );

            Terser terser = new Terser( message );
            String messageControlId = terser.get( T_MESSAGE_CONTROL_ID );
            sendingApp = terser.get( T_SENDING_APPLICATION ); // get the sending application info from HL7 (entry point)
            String messageDateTime = terser.get( T_MESSAGE_DATE_TIME );
            String sendingFacility = terser.get( T_SENDING_FACILITY_ID );

            globalValueObject = ErrorQueueThreadLocal.get();
            globalValueObject.setSendingApp( sendingApp );                // update to information from message itself
            globalValueObject.setMessageControlId( messageControlId );
            globalValueObject.setMessageDateTime( messageDateTime );
            globalValueObject.setSendingFacility( sendingFacility );
            if ( logger.isDebugEnabled() )
            {
                logger.debug( "Message properties: " + globalValueObject.toString() );
            }
            ErrorQueueThreadLocal.set( globalValueObject );

            anErrorQueueCleanUpLogger.logParseInfo( messageControlId, sendingApp, messageDateTime );

            if ( ( knownRejectableSendingAppList != null ) && knownRejectableSendingAppList.contains( sendingApp ) )
            {
                // Ignore (a.k.a. remove) useless ping messages, MPI Move/Link messages, and others
                anErrorQueueCleanUpLogger.logCdsExceptionDetail( "Discarding message based on Sending Application" );
            }
            else if ( ( sendingApp != null ) && isMessageOutOfDate( messageDateTime, DateTimeUtil.FORMAT_HL7, sendingApp ) )
            {
                /*
                 * Remove certain messages that are older than specified days from
                 * the error queue and save them to the database
                 */
                if ( logger.isDebugEnabled() )
                {
                    logger.debug( "messageControlId: " + messageControlId + " is out-of-date, backup message to database" );
                }

                ErrorQueueMessageDao dao = ( ErrorQueueMessageDao )beanFactory.getBean( "errorQueueMessageDao" );
                dao.setRequestId( messageControlId );
                dao.setHl7Clob( aDecodedHl7Message );
                dao.setMessageDateTime( messageDateTime );
                dao.setMessageStatus( ErrorQueueConstant.MESSAGE_BACKUP ); // set to "BK"
                dao.setMessageType( sendingApp );

                dao.save();
                
                if ( logger.isDebugEnabled() )
                {
                    logger.debug( "Message: messageControlId: " + messageControlId + " successfully saved " );
                }

                anErrorQueueCleanUpLogger.logRemovedFromQueue(); // set process status to "R" (Removed)

                if ( logger.isDebugEnabled() )
                {
                    logger.debug( "Message: messageControlId: " + messageControlId + " process log is updated" );
                }
            }
            else
            {
                // Process only clinical messages in order to fix the missing assigning facility
                if ( !aDecodedHl7Message.contains( "MSA" ) && ( sendingApp != null ) && !sendingApp.trim().contains( "MPIF TRIGGER" ) )
                {
                    aDecodedHl7Message = modifyHL7Msg( message, terser, messageControlId, anErrorQueueCleanUpLogger );
                }

                EntityIdentifier recordIdentifier;
                ErrorQHandlerInformation errorQHandlerInfo = sendingApplicationToErrorQHandlerInfoMap.get( sendingApp );

                try
                {
                    if ( errorQHandlerInfo != null )
                    {
                        recordIdentifier = getRecordIdentifier( message, errorQHandlerInfo.getHl7RecordIdSegment() );
                    }
                    else
                    {
                        logger.error( " Unknown sending app (" + sendingApp + ") in sendingApplicationToErrorQHandlerInfoMap.  Unable to get record identifier for messageControlId: "
                                + messageControlId );
                        throw new ErrorQCleanupException( "Unknown sending app (" + sendingApp
                                + ") in sendingApplicationToErrorQHandlerInfoMap.  Unable to get record identifier for messageControlId: "
                                + messageControlId );
                    }
                }
                catch ( ErrorQCleanupException e )
                {
                    throw e;
                }
                catch ( HL7Exception e )
                {
                    logger.error( " Unable to get record identifier for message: " + e.getMessage() );
                    logger.error( " messageControlId: " + messageControlId + ", sendingApp: " + sendingApp );
                    throw e;
                }
                catch ( RuntimeException e )
                {
                    logger.error( " Unable to get record identifier for message: " + e.getMessage() );
                    logger.error( " messageControlId: " + messageControlId + ", sendingApp: " + sendingApp );
                    throw e;
                }
                if ( logger.isDebugEnabled() )
                {
                    logger.debug( " Forward message to Mediator " );
                }

                returnValue = errorQHandlerInfo.getMessageHandler().handleMessage( aDecodedHl7Message, errorQHandlerInfo.getHdrTableName(),
                        recordIdentifier, messageDateTime, anErrorQueueCleanUpLogger );
               
            }
        }
        catch ( EncodingNotSupportedException ense )
        {
            anErrorQueueCleanUpLogger.logCdsExceptionDetail( aDecodedHl7Message );
            throw new ErrorQCleanupException( ense, ense.getMessage() );
        }
        catch ( HL7Exception e )
        {
            anErrorQueueCleanUpLogger.logCdsExceptionDetail( aDecodedHl7Message );
            throw new ErrorQCleanupException( e, e.getMessage() );
        }
        
        if ( logger.isDebugEnabled() )
        {
            logger.debug( " End with return value: '" + returnValue + "'" );
        }

        return returnValue;
    }
    
    
    public static Message parseUsingHapi( String aHL7Request )
        throws EncodingNotSupportedException, HL7Exception
    {
        Message returnMessage = null;

        if ( logger.isDebugEnabled() )
        {
            logger.debug( " Begin" );
        }

        // Parse using PipeParser
        PipeParser hapiParser = new PipeParser();
        hapiParser.setValidationContext( new ValidationContextImpl() );
        try
        {
            returnMessage = hapiParser.parse( aHL7Request ); // throws Encoding not supported on bad  messages

            if ( logger.isDebugEnabled() )
            {
                logger.debug( "Call hapiParser.parse() completed successfully" );
            }
        }
        catch ( Exception e )
        {
            // try again after fixing known possible problems with extra carriage returns
            // let fail this time if not successful
            if ( logger.isDebugEnabled() )
            {
                logger.debug( "Call hapiParser.parse( aHL7Request ) - Exception: " + e.getMessage() );
            }
            
            returnMessage = hapiParser.parse( aHL7Request );// throws Encoding not supported on bad messages
        }

        if ( logger.isDebugEnabled() )
        {
            logger.debug( " End." );
        }

        return returnMessage;
    }
   
     /**
     * Fix for avoiding CDS schema validation error when assigning facility id
     * of pid is missing. Populate it with sending facility id.
     * 
     * @param anHL7Message
     * @param aTerser
     * @return modified HL7 msg
     * @throws HL7Exception
     */
    private String modifyHL7Msg( Message anHL7Message, Terser aTerser, String aMessageControlId, ErrorQueueCleanUpLogger anErrorQueueCleanUpLogger )
        throws HL7Exception
    {
        String sendingFacilityId = null;
        String sendingFacilityName = null;
        String assigningFacility = null;
        String messageIdentifier = null;
        String modifiedHl7Msg = null;

        if ( logger.isDebugEnabled() )
        {
            logger.debug( "modifyHL7Msg(Message msg, Terser terser): Begin." );
        }

        sendingFacilityId = aTerser.get( T_STATION_NUMBER );
        sendingFacilityName = aTerser.get( T_SENDING_FACILITY_ID );
        anErrorQueueCleanUpLogger.logFacility( sendingFacilityId, sendingFacilityName );

        // fail fast
        if ( ( ( sendingFacilityId == null )   || sendingFacilityId.equals( "" )   ) && 
             ( ( sendingFacilityName == null ) || sendingFacilityName.equals( "" ) ) )
        {
            tryToLogMessage( anHL7Message, anErrorQueueCleanUpLogger );
            throw new ErrorQCleanupException( "MSH 4.1 and MSH 4.2 in the message are empty" );
        }

        assigningFacility = aTerser.get( T_ASSIGNING_FACILITY );
        messageIdentifier = aTerser.get( T_IDENTIFIER_TYPE );

        /*
         * TODO SJB - figure out this exception
         *  java.lang.NullPointerException at gov.va.med.cds.tools.cleanup.errorq.ErrorQMessageParser.modifyHL7Msg
         */

        if ( ( ( messageIdentifier != null ) && messageIdentifier.equals( "PI" ) ) &&
             ( ( assigningFacility == null ) || assigningFacility.equals( "" )   ) )
        {
            if ( ( ( sendingFacilityId == null ) || sendingFacilityId.equals( "" ) ) &&
                 ( ( sendingFacilityName != null ) && !sendingFacilityName.equals( "" ) ) )
            {
                logger.warn( "modifyHL7Msg(Message msg, Terser terser): MSH 4.1 is empty and getting it from facilityId Map" );

                sendingFacilityId = ( String )getKeyFromValue( facilityIdMap, sendingFacilityName );

                if ( ( sendingFacilityId == null ) || sendingFacilityId.equals( "" ) )
                {
                    tryToLogMessage( anHL7Message, anErrorQueueCleanUpLogger );
                    throw new ErrorQCleanupException(
                            "MSH 4.1 is empty and not in facilityId map and thus cannot populate the empty PID 3.6.2" );
                }
            }

            aTerser.set( T_ASSIGNING_FACILITY, sendingFacilityId );
        }
        else if ( ( ( messageIdentifier != null ) && messageIdentifier.equals( "PI" ) ) &&
                  ( ( sendingFacilityName == null ) || sendingFacilityName.equals( "" ) ) )
        {
            logger.warn( "modifyHL7Msg(Message msg, Terser terser): MSH 4.2 is empty and getting it from facilityId Map" );

            sendingFacilityName = facilityIdMap.get( sendingFacilityId );

            // SJB Check for 4.1 = "200" and pull real 4.1 from message control id and look it up and replace both.
            if ( ( sendingFacilityName == null ) && ( sendingFacilityId != null ) && sendingFacilityId.equals( "200" ) )
            {
                try
                {
                    // this is an old north chicago message
                    sendingFacilityId = aMessageControlId.substring( 0, 3 );
                    sendingFacilityName = facilityIdMap.get( sendingFacilityId );
                    aTerser.set( T_ASSIGNING_FACILITY, sendingFacilityId );
                }
                catch ( Exception e )
                {
                }
            }
            
            if ( ( sendingFacilityName == null ) || sendingFacilityName.equals( "" ) )
            {
                // CDS fails to store messages in audit clob store if there's not a valid 4.1 and 4.2 in MSH
                tryToLogMessage( anHL7Message, anErrorQueueCleanUpLogger );
                throw new ErrorQCleanupException( "MSH 4.2 is empty and 4.1 ( " + sendingFacilityId
                        + " ) is not in facilityId map and thus cannot populate the empty MSH 4.2" );
            }
            
            aTerser.set( T_SENDING_FACILITY_ID, sendingFacilityName );
        }

        PipeParser hapiParser = new PipeParser();
        hapiParser.setValidationContext( new ValidationContextImpl() );

        modifiedHl7Msg = hapiParser.encode( anHL7Message );

        if ( logger.isDebugEnabled() )
        {
            logger.debug( "modifyHL7Msg(Message msg, Terser terser): End." );
        }

        return modifiedHl7Msg;
    }


    private void tryToLogMessage( Message aMessage, ErrorQueueCleanUpLogger anErrorQueueCleanUpLogger )
    {
        String logMessage;
        
        try
        {
            PipeParser hapiParser = new PipeParser();
            hapiParser.setValidationContext( new ValidationContextImpl() );
            logMessage = hapiParser.encode( aMessage );
        }
        catch ( Exception e )
        {
            logMessage = "Exception trying to log parse message in order to log it: " + e.getMessage();
        }
        
        anErrorQueueCleanUpLogger.logCdsExceptionDetail( logMessage );
    }


    /**
     * Get the record identifier info from the hl7 segment OBR 3.1 to 3.4 for
     * allergies ORC 3.1 to 3.4 for Rx, Lab and Vitals
     * 
     * @param terser
     * @param anHL7Segment
     * @return
     */
    private EntityIdentifier getRecordIdentifier( Message aMessage, String anHL7Segment )
        throws HL7Exception
    {
        if ( logger.isDebugEnabled() )
        {
            logger.debug( "getRecordIdentifier( Message message, String hl7Segment ): Begin." );
        }

        EntityIdentifier recordIdentifier = new EntityIdentifier();

        EI ei = null;

        if ( aMessage instanceof ORU_R01 )
        {
            ORU_R01 oruR01 = ( ORU_R01 )aMessage;

            if ( anHL7Segment.equalsIgnoreCase( "/ORC-3" ) )
            {
                ei = oruR01.getPATIENT_RESULT().getORDER_OBSERVATION().getORC().getFillerOrderNumber();
            }
            else if ( anHL7Segment.equalsIgnoreCase( "/OBR-3" ) )
            {
                ei = oruR01.getPATIENT_RESULT().getORDER_OBSERVATION().getOBR().getFillerOrderNumber();
            }
        }
        else if ( aMessage instanceof RDS_O13 )
        {
            RDS_O13 rds13 = ( RDS_O13 )aMessage;
            ei = rds13.getORDER().getORC().getFillerOrderNumber();
        }
        else if ( aMessage instanceof RDE_O11 )
        {
            RDE_O11 rdeO11 = ( RDE_O11 )aMessage;
            ei = rdeO11.getORDER().getORC().getFillerOrderNumber();
        }

        if ( null != ei )
        {
            recordIdentifier.setIdentity( ei.getEntityIdentifier() != null ? ei.getEntityIdentifier().toString() : "" );
            recordIdentifier.setNamespaceId( ei.getNamespaceID() != null ? ei.getNamespaceID().toString() : "" );
            recordIdentifier.setUniversalId( ei.getUniversalID() != null ? ei.getUniversalID().toString() : "" );
            recordIdentifier.setUniversalIdType( ei.getUniversalIDType() != null ? ei.getUniversalIDType().toString() : "" );
        }
        
        if ( logger.isDebugEnabled() )
        {
            logger.debug( "getRecordIdentifier( Message message, String hl7Segment ): End." );
        }

        return recordIdentifier;
    }


    public void setSendingApplicationToErrorQHandlerInfoMap( Map<String, ErrorQHandlerInformation> aSendingApplicationToErrorQHandlerInfoMap )
    {
        this.sendingApplicationToErrorQHandlerInfoMap = aSendingApplicationToErrorQHandlerInfoMap;
    }


    public void setKnownRejectableSendingAppList( List<String> aKnownRejectableSendingAppList )
    {
        this.knownRejectableSendingAppList = aKnownRejectableSendingAppList;
    }


    public void setFacilityIdMap( Map<String, String> aFacilityIdMap )
    {
        this.facilityIdMap = aFacilityIdMap;
    }


    public static Object getKeyFromValue( Map<?, ?> aMap, String aValue )
    {
        Object keyVal = null;
        for ( Object o : aMap.keySet() )
        {
            if ( aMap.get( o ).equals( aValue ) )
            {
                keyVal = o;
            }
        }

        return keyVal;
    }


    public void setMessageValidAgeMap( Map<String, String> aMessageValidAgeMap )
    {
        this.messageValidAgeMap = aMessageValidAgeMap;
    }


    /**
     * Called by Spring framework.
     */
    public void setBeanFactory( BeanFactory aBeanFactory )
        throws BeansException
    {
        this.beanFactory = aBeanFactory;
    }


    // Throw ParseException if parse message datetime failed
    // message date time is in "yyyyMMddHHmmssZ" format; // example
    // 20170315120856-0800
    private boolean isMessageOutOfDate( String aStringMessageDatetime, String aFormat, String aSendingApp )
    {
        boolean returnValue = false;

        String ageMsgValid = messageValidAgeMap.get( aSendingApp );
        if ( ( null != ageMsgValid ) && ( ageMsgValid.length() > 0 ) )
        {
            long anAgeInDays = Long.parseLong( ageMsgValid );
            returnValue = MessageDatetimeUtil.isMessageOutOfDate( aStringMessageDatetime, aFormat, anAgeInDays );
            if ( logger.isDebugEnabled() )
            {
                logger.debug( "Message datetime: " + aStringMessageDatetime + ", ageInDays: " + anAgeInDays + ", message out of date? : "
                        + returnValue );
            }
        }

        return returnValue;
    }

}
