/**
 * 
 */


package gov.va.med.cds.tools.cleanup.errorq;


import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Properties;

import javax.xml.parsers.FactoryConfigurationError;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import gov.va.med.cds.properties.PropertiesUtil;
import gov.va.med.cds.util.DateTimeUtil;


/**
 * @author susarlan
 *
 */
public class ErrorQCleanUp
    extends
        Thread
    implements
        ErrorQCleanUpMBeanInterface
{
    private static Log logger = LogFactory.getLog( ErrorQCleanUp.class );
    private static ApplicationContext applicationContext;
    private ErrorQueueDefaultMessageListenerContainer listenerContainer = null;
    private static ErrorQCleanUp instance = null;
    private ErrorQueueCleanUpLogger errorQueueCleanUpLogger;
    private String configFilename = null;
    private long logId = 0;
    private AtomicCounter messageCounter;
    public static final String DEFAULT_JMS_TIMESTAMP_END = "default.jms.timestamp.end"; 
    public static final int MILLISECOND_PER_MINUTE = 60 * 1000;
 //   public final static String DATETIME_FORMAT = "yyyy-MM-dd HH:mm:ss";


    public enum RuntimeState
    {
        INITIALIZING, RUNNING, SHUTDOWN, TERMINATED
    };

    private RuntimeState runtimeState = RuntimeState.INITIALIZING;

    private static int errorQRunDurationMins = 35;


    /**
     * @param anArgs
     */
    public static void main( String[] anArgs )
    {
        logger.info( "ErrorQCleanup main() is starting at (in milli:) '" + System.currentTimeMillis() + "'" );
        
//        String currentDateTime = DateTimeUtil.getCurrentDateTime( ErrorQueueConstant.DATETIME_FORMAT );
        String currentZonedDatetimeDate = DateTimeUtil.getCurrentZonedDateTime( DateTimeUtil.FORMAT_HL7 );
        System.setProperty( DEFAULT_JMS_TIMESTAMP_END, currentZonedDatetimeDate );
        
        boolean processOldMessage = isTimeToProcessOldMessage();
        if ( processOldMessage )
        {
            System.setProperty( ErrorQueueConstant.IS_TIME_TO_PROCESS_OLD_MESSAGE, ErrorQueueConstant.YES );
            logger.info("ErrorQCleanup will start processing old messages ...." );
        }
        else
        {
            logger.info("ErrorQCleanup will start processing new messages ...." );
        }
        
        if ( ( anArgs != null ) && ( anArgs.length > 0 ) && anArgs[0].matches( ".*[xX][mM][lL]$" ) )
        {
            setup( anArgs[0] );
        }
        else
        {
            setup( "classpath:gov/va/med/cds/tools/cleanup/errorq/errorQCleanUpContext.xml" );
        }
        
        if ( processOldMessage )
        {
            updateLastOldMessageProcessTime();
        }
        
        logger.info( "ErrorQCleanup main() finished." );
    }
    

    public static boolean isTimeToProcessOldMessage( )
    {
        String strLastOldMessageProcessTime = getLastOldMessageProcessTime();
        if( strLastOldMessageProcessTime == null ){
            return true; // can't find last time old messages were processed record. Process old messages.
        }
        long frequency = 0;
        try
        {
            frequency = Long.parseLong(System.getProperty( ErrorQueueConstant.JMS_OLD_MESSAGE_PROCESS_FREQUENCY ) );
        }
        catch ( NumberFormatException dtpe )
        {
            logger.error( "Process old messages frequency (days): " + System.getProperty( ErrorQueueConstant.JMS_OLD_MESSAGE_PROCESS_FREQUENCY )  + " couldn't be parsed as integer");
            logger.error("Parse process old message frequency: " + dtpe.getMessage());
            frequency = 7;
        }
        
        if ( logger.isDebugEnabled() )
        {
            logger.debug( "Process old messages frequency (days): " + System.getProperty( ErrorQueueConstant.JMS_OLD_MESSAGE_PROCESS_FREQUENCY ) );
        }
        
        return MessageDatetimeUtil.isMessageOutOfDate( strLastOldMessageProcessTime, DateTimeUtil.FORMAT_HL7,frequency );    
    }
    
    
    /**
     * Get last old messages processed time
     * @return time as String in format of yyyy-MM-dd HH:mm:ss. null if time is not found
     */
    public static String getLastOldMessageProcessTime( )
    {
        String strLastProcessTime = null;
        FileInputStream fis = null;
        
        try
        {
            Properties props = new java.util.Properties();
            fis = new FileInputStream( "last_process_time.properties" );
            if ( fis != null )
            {
                props.load( fis );
                strLastProcessTime = props.getProperty( ErrorQueueConstant.JMS_OLD_MESSAGE_LAST_PROCESS_TIME );
                if( logger.isInfoEnabled())
                {   
                    logger.info( "property - jms.old.message.last.process.time=" + strLastProcessTime );
                }
            }
        }
        catch ( FileNotFoundException e )
        {
            logger.error( e.getMessage() );
            logger.error( "It's ok if this is the first time run this application after deployment. Will create a new later. " );
        }
        catch ( IOException e )
        {
            logger.error( e.getMessage() );
            e.printStackTrace();
        }
        finally
        {
            if ( fis != null )
            {
                try
                {
                    fis.close();
                }
                catch ( IOException e )
                {
                    e.printStackTrace();
                }
            }
        }

        return strLastProcessTime;
    }
    
    
    public static void updateLastOldMessageProcessTime( )
    {
        FileOutputStream fos = null;
        
        try
        {
            Properties props = new java.util.Properties();
            String strLastProcessTime = System.getProperty( ErrorQueueConstant.DEFAULT_JMS_TIMESTAMP_END );
            props.setProperty( ErrorQueueConstant.JMS_OLD_MESSAGE_LAST_PROCESS_TIME, strLastProcessTime );
            fos = new FileOutputStream( "last_process_time.properties" );
            props.store( fos, "Last old messages were processed time" );
            if( logger.isInfoEnabled())
            {
                logger.info( "Updated last old messages processed time to: " + strLastProcessTime );
            }
        }
        catch ( FileNotFoundException e )
        {
            logger.error( e.getMessage() );
            e.printStackTrace();
        }
        catch ( IOException e )
        {
            logger.error( e.getMessage() );
            e.printStackTrace();
        }
        finally
        {
            if ( fos != null )
            {
                try
                {
                    fos.close();
                }
                catch ( IOException e )
                {
                    // Do nothing
                    e.printStackTrace();
                }
            }
        }
    }  


    protected static void setup( String aConfigurationFile )
    {
    	String errorQRunDuration = System.getProperty( "errorq.cleanup.run.duration" );
        
        if ( ( null != errorQRunDuration ) && ( !errorQRunDuration.isEmpty() ) )
        {
            errorQRunDurationMins = Integer.parseInt( errorQRunDuration );
        }
        
        try
        {
            if ( logger.isInfoEnabled() )
            {
                logger.info( "setup( String cfg ): Starting up ErrorQ CleanUp application with config: " + aConfigurationFile );
            }

            // configure the runtime environment
            byte[] initializationVector = getInitializationVector();
            ErrorQCleanUp.configureLoggingAndProperties( System.getProperty( "log4j.configurationFile" ), initializationVector );
            
            applicationContext = new ClassPathXmlApplicationContext( aConfigurationFile );
            instance = ( ErrorQCleanUp )applicationContext.getBean( "errorQCleanUp" );
            instance.configFilename = aConfigurationFile;

            instance.start();
            instance.join( ( errorQRunDurationMins + 2 ) * MILLISECOND_PER_MINUTE );
        }
        catch ( InterruptedException e )
        {
            logger.error( "setup( String cfg ) join interrupted", e );
        }
        catch ( Exception e )
        {
            logger.error( "setup( String cfg )", e );
        }
        finally
        {
            instance.listenerContainer = null;
            instance = null;
        }
    }

    
    private static byte[] getInitializationVector()
    {
        String ivString = System.getProperty( "errorq.initialization.vector.string" );
        
        byte[] iv = null;
        if ( ( null != ivString ) && ( ivString.length() == 16 ) )
        {
            iv = ivString.getBytes();
            if( logger.isDebugEnabled() )
            {
                logger.debug( "Initialization Vector: '" + ivString + "', size: " + ivString.length() );
            }
        }
        else
        {
            iv = new byte[16];
            if( logger.isDebugEnabled() )
            {
                logger.debug( "Initialization Vector is null" );
            }
       }
        
        return iv;
    }

    
    public void run( )
    {
        while ( runtimeState != RuntimeState.TERMINATED )
        {
            switch ( runtimeState )
            {
            case INITIALIZING:
                executeInitStage();
                break;

            case RUNNING:
                executeRunningStage();
                break;

            case SHUTDOWN:
                if(logger.isInfoEnabled())
                {
                    logger.info( "run(): ErrorQ listenerContainer attempting to shutdown gracefully." );			
                }

                executeShutdownStage();
                break;
                
            case TERMINATED:
            	if(logger.isInfoEnabled())
            	{
            	    logger.info("ErrorQCleanUp has been terminated" );
            	}
            	break;
            }
        }

        if ( logger.isInfoEnabled() )
        {
            logger.info( "run(): End." );
        }
    }


    private void executeInitStage( )
    {
        runtimeState = RuntimeState.RUNNING;
    }


    private void executeRunningStage( )
    {
        logId = logProcessStart( configFilename );
        instance.listenerContainer.start();

        try
        {
            Thread.sleep( errorQRunDurationMins * MILLISECOND_PER_MINUTE );
        }
        catch ( InterruptedException e )
        {
            e.printStackTrace();
        }

        runtimeState = RuntimeState.SHUTDOWN;
    }


    private void executeShutdownStage( )
    {
        runtimeState = RuntimeState.TERMINATED;
        instance.listenerContainer.shutdown();
        logProcessFinish( logId );
    }


    protected static void configureLoggingAndProperties( String aLogfilePath, byte[] anInitializationVector ) 
            throws FactoryConfigurationError, IOException
    {
        //File logFile = new File( aLogfilePath );

        // DOMConfigurator.configure( logFile.getAbsolutePath() );

        try
        {
            //TODO : Fix this ASAP
            new PropertiesUtil().loadSystemRunProperties( anInitializationVector, true );
        }
        catch ( Exception e )
        {

            e.printStackTrace();
        }

//        if ( aLogger.isDebugEnabled() )
//        {
//            aLogger.debug( "configureLoggingAndProperties( Log logger, String logfilePath ): using log file: " + logFile.getAbsolutePath() );
//        }
    }


    private long logProcessStart( String aConfigFilename )
    {
        long logId = 0;
        
        if ( errorQueueCleanUpLogger != null )
        {
            logId = errorQueueCleanUpLogger.logProcessStart( aConfigFilename );
        }
        
        return logId;
    }

    
    private void logProcessFinish( long aLogId )
    {
        if ( errorQueueCleanUpLogger != null )
        {
            errorQueueCleanUpLogger.logProcessFinish( aLogId, messageCounter.getCount() );
        }
    }
    
    
    public void shutdown( )
    {
        instance.listenerContainer.shutdown();
        instance.interrupt();
    }


    public boolean isRunning( )
    {
        return instance.listenerContainer.isRunning();
    }


    public String getStatus( )
    {
        return "ErrorQCleanUp state is " + runtimeState;
    }


    public int getCurrentConsumerCount( )
    {
        int activeConsumerCount = 0;

        if ( null != instance )
        {
            activeConsumerCount = instance.listenerContainer.getActiveConsumerCount();
        }
        
        return activeConsumerCount;
    }

    
    public void setListenerContainer( ErrorQueueDefaultMessageListenerContainer aListenerContainer )
    {
        this.listenerContainer = aListenerContainer;        
    }

    
    public void setErrorQueueCleanUpLogger( ErrorQueueCleanUpLogger aErrorQueueCleanUpLogger )
    {
        this.errorQueueCleanUpLogger = aErrorQueueCleanUpLogger;       
    }

    
    public void setMessageCounter( AtomicCounter aMessageCounter )
    {
        this.messageCounter = aMessageCounter;  
    }
}
