

package gov.va.med.cds.util;


import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.TimeZone;

import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;

import gov.va.med.cds.junit.runners.Suite;
import gov.va.med.cds.junit.runners.SuiteAwareRunner;


@RunWith( SuiteAwareRunner.class )
public class DateTimeUtilTest
{
    long milliSeconds = 1000000000000L; // Sat Sep 09 01:46:40 GMT 2001
    TimeZone timeZone = TimeZone.getTimeZone( "GMT+0000" ); //.getID();
    Instant instant = Instant.ofEpochMilli( milliSeconds );
    ZoneId zoneId = ZoneId.of( "GMT" );


    @Test
    @Suite( groups = "checkintest" )
    public void testCurrentGetISOTime( )
    {
        String format = "yyyyMMddHHmmssZ";
        
        String currentISOTime = DateTimeUtil.getCurrentISOTime();
        assertTrue( currentISOTime.matches( "([0-9]{14}).([0-9]{3})([+|-])([0-9]{4})" ) );       
        
        currentISOTime = DateTimeUtil.getCurrentISOTime( format );
        assertTrue( currentISOTime.matches( "([0-9]{14})([+|-])([0-9]{4})" ) );       
        
        currentISOTime = DateTimeUtil.getCurrentISOTime( zoneId );
        assertTrue( currentISOTime.matches( "([0-9]{14}).([0-9]{3})([+|-])0000" ) );       
    }


    @Test
    @Suite( groups = "checkintest" )
    public void testGetISOTime( )
    {
        String isoTime = DateTimeUtil.getISOTime( milliSeconds, timeZone );
        assertEquals( "20010909014640.000+0000", isoTime );

        isoTime = DateTimeUtil.getISOTime( instant, zoneId );
        assertEquals( "20010909014640.000+0000", isoTime );
    }

    
    @Test
    @Suite( groups = "checkintest" )
    public void testGetPartialISOTime( )
    {
        String partialIsoTime = DateTimeUtil.getPartialISOTime( milliSeconds, null );

        partialIsoTime = DateTimeUtil.getPartialISOTime( milliSeconds, timeZone );
        assertEquals( "20019090146", partialIsoTime ); //yyyMddHHmm

        partialIsoTime = DateTimeUtil.getPartialISOTime( instant, zoneId );
        assertEquals( "20019090146", partialIsoTime ); //yyyMddHHmm
        
        partialIsoTime = DateTimeUtil.getPartialISOTime( null, zoneId );
        assertNull( partialIsoTime );
        
        partialIsoTime = DateTimeUtil.getPartialISOTime( instant, null );
        assertNotNull( partialIsoTime );
        
        partialIsoTime = DateTimeUtil.getPartialISOTime( null, null );
        assertNull( partialIsoTime );
    }


    @Test
    @Suite( groups = "checkintest" )
    public void testGetDateTimeDotSeparated( )
    {
        String partialIsoTime = DateTimeUtil.getDateTimeDotSeparated( milliSeconds, timeZone );
        assertEquals( "20010909.014640", partialIsoTime ); //yyyyMMdd.HHmmss
        
        partialIsoTime = DateTimeUtil.getDateTimeDotSeparated( instant, zoneId );
        assertEquals( "20010909.014640", partialIsoTime ); //yyyyMMdd.HHmmss
   }
    
    
    @Test
    @Suite( groups = "checkintest" )
    public void testIsDateValidHl7Format( )
    {
        assertTrue( DateTimeUtil.isDateValidHl7Format( "20170425123456-0600" ) );
        assertFalse( DateTimeUtil.isDateValidHl7Format( "20170425123456" ) );
        assertFalse( DateTimeUtil.isDateValidHl7Format( "2017042512345-0600" ) );
        assertFalse( DateTimeUtil.isDateValidHl7Format( "20170425123456-600" ) );
    }
    
    
    @Test
    @Suite( groups = "checkintest" )
    public void testGetCurrentDate( ) 
            throws ParseException
    {
        String format = "yyyyMMdd";
        String dateStr = DateTimeUtil.getCurrentDate( format );
        assertTrue( dateStr.matches( "([0-9]{8})" ) );
                
        dateStr= DateTimeUtil.getCurrentDate( null );
        assertTrue( dateStr.matches( "([0-9]{2})/([0-9]{2})/([0-9]{4})" ) );
    }
    
    
    @Test
    @Suite( groups = "checkintest" )
    public void testGetCurrentTime( ) 
            throws ParseException
    {
        String format = "HHmmss";
        String timeStr = DateTimeUtil.getCurrentTime( format );
        assertTrue( timeStr.matches( "([0-9]{6})" ) );
                
        timeStr= DateTimeUtil.getCurrentTime( null );
        assertTrue( timeStr.matches( "([0-9]{2}):([0-9]{2}):([0-9]{2})" ) );
    }
    
    
    @Test
    @Suite( groups = "checkintest" )
    public void testGetCurrentDateTime( ) 
            throws ParseException
    {
        String format = "yyyyMMddHHmmss";
        String dateTimeStr = DateTimeUtil.getCurrentDateTime( format );
        assertTrue( dateTimeStr.matches( "([0-9]{14})" ) );
                
        dateTimeStr= DateTimeUtil.getCurrentDateTime( null );
        assertTrue( dateTimeStr.matches( "([0-9]{14}).([0-9]{3})" ) );
    }
    
    
    @Test
    @Suite( groups = "checkintest" )
    public void testGetCurrentZonedDateTime( ) 
            throws ParseException
    {
        String format = "yyyyMMddHHmmss";
        String zonedDateTimeStr = DateTimeUtil.getCurrentDateTime( format );
        assertTrue( zonedDateTimeStr.matches( "([0-9]{14})" ) );
                
        zonedDateTimeStr= DateTimeUtil.getCurrentZonedDateTime( null );
        assertTrue( zonedDateTimeStr.matches( "([0-9]{14})-([0-9]{4})" ) );
    }
    
    
    @Test
    @Suite( groups = "checkintest" )
    public void testFormatLocalDate( ) 
            throws ParseException
    {
        String testDate = "12-Oct-2016";
        LocalDate localDate = LocalDate.parse( testDate, DateTimeFormatter.ofPattern( "d-MMM-yyyy" ) );
        
        String dateStr= DateTimeUtil.formatLocalDate( localDate );
        assertEquals( "10/12/2016", dateStr );
        
        dateStr= DateTimeUtil.formatLocalDate( null );
        assertNull( dateStr );
    }
    
    
    @Test
    @Suite( groups = "checkintest" )
    public void testFormatLocalTime( ) 
            throws ParseException
    {
        String testTime = "161214";
        LocalTime localTime = LocalTime.parse( testTime, DateTimeFormatter.ofPattern( "HHmmss" ) );
        
        String timeStr= DateTimeUtil.formatLocalTime( localTime );
        assertEquals( "16:12:14", timeStr );
        
        timeStr= DateTimeUtil.formatLocalTime( null );
        assertNull( timeStr );
    }
    
    
    @Test
    @Suite( groups = "checkintest" )
    public void testFormatLocalDateTime( ) 
            throws ParseException
    {
        String testDateTime = "12-Oct-2016,16:12:14";
        LocalDateTime localDateTime = LocalDateTime.parse( testDateTime, DateTimeFormatter.ofPattern( "d-MMM-yyyy,HH:mm:ss" ) );
        
        String dateTimeStr= DateTimeUtil.formatLocalDateTime( localDateTime );
        assertEquals( "20161012.161214", dateTimeStr );
        
        dateTimeStr= DateTimeUtil.formatLocalDateTime( null );
        assertNull( dateTimeStr );
    }
    
    
    @Test
    @Suite( groups = "checkintest" )
    public void testFormatZonedDateTime( ) 
            throws ParseException
    {
        String testDateTime = "12-Oct-2016,16:12:14-0600";
        ZonedDateTime zonedDateTime = ZonedDateTime.parse( testDateTime, DateTimeFormatter.ofPattern( "d-MMM-yyyy,HH:mm:ssZ" ) );
        
        String zonedDateTimeStr= DateTimeUtil.formatZonedDateTime( zonedDateTime );
        assertEquals( "20161012161214-0600", zonedDateTimeStr );
        
        zonedDateTimeStr= DateTimeUtil.formatZonedDateTime( null );
        assertNull( zonedDateTimeStr );
    }
    
    
    @Test
    @Suite( groups = "checkintest" )
    public void testConvertDate2String( ) 
            throws ParseException
    {
        String testDate = "12-Oct-2016,16:12:14 PM";
        DateFormat formatter = new SimpleDateFormat("d-MMM-yyyy,HH:mm:ss aaa");
        Date date = formatter.parse( testDate );
        
        String dateStr= DateTimeUtil.convertDate2String( date );
        assertEquals( "10/12/2016", dateStr );
        
        dateStr= DateTimeUtil.convertDate2String( null );
        assertNull( dateStr );
    }
    
    
    @Test
    @Suite( groups = "checkintest" )
    public void testAsLocalDate( )
    {
        LocalDate expectedDate = LocalDate.of(2017, 3, 19);
        Date utilDate = new GregorianCalendar(2017, 2, 19).getTime();

        LocalDate localDate = DateTimeUtil.asLocalDate( utilDate );
        assertTrue( expectedDate.isEqual( localDate ) );

        localDate = DateTimeUtil.asLocalDate( null );
        assertNull( localDate );
    }
    
    
    @Test
    @Suite( groups = "checkintest" )
    public void testAsLocalDateTime( )
    {
        String testDateTime = "12-Oct-2016 16:12:14";
        LocalDateTime expectedDateTime = LocalDateTime.parse( testDateTime, DateTimeFormatter.ofPattern( "d-MMM-yyyy HH:mm:ss" ) );

        LocalDateTime localDateTime = DateTimeUtil.asLocalDateTime( testDateTime, "d-MMM-yyyy HH:mm:ss" );
        assertTrue( expectedDateTime.isEqual( localDateTime ) );

        localDateTime = DateTimeUtil.asLocalDateTime( null, null );
        assertNull( localDateTime );

        localDateTime = DateTimeUtil.asLocalDateTime( null, "d-MMM-yyyy HH:mm:ss" );
        assertNull( localDateTime );

        localDateTime = DateTimeUtil.asLocalDateTime( "20161012161214-0600", null );
        assertNotNull( localDateTime );
        
        localDateTime = DateTimeUtil.asLocalDateTime( "12-Oct-2016", "d-MMM-yyyy HH:mm:ss" );
        assertNull( localDateTime );
    }
    
    
    @Test
    @Suite( groups = "checkintest" )
    public void testAsZonedDateTime( )
    {
        String testDateTime = "12-Oct-2016 16:12:14-0600";
        ZonedDateTime expectedDateTime = ZonedDateTime.parse( testDateTime, DateTimeFormatter.ofPattern( "d-MMM-yyyy HH:mm:ssZ" ) );

        ZonedDateTime zonedDateTime = DateTimeUtil.asZonedDateTime( testDateTime, "d-MMM-yyyy HH:mm:ssZ" );
        assertTrue( expectedDateTime.isEqual( zonedDateTime ) );

        zonedDateTime = DateTimeUtil.asZonedDateTime( null, null );
        assertNull( zonedDateTime );

        zonedDateTime = DateTimeUtil.asZonedDateTime( null, "d-MMM-yyyy HH:mm:ssZ" );
        assertNull( zonedDateTime );

        zonedDateTime = DateTimeUtil.asZonedDateTime( "20161012161214-0600", null );
        assertNotNull( zonedDateTime );
        
        zonedDateTime = DateTimeUtil.asZonedDateTime( "12-Oct-2016", "d-MMM-yyyy HH:mm:ssZ" );
        assertNull( zonedDateTime );
    }
    
    
    @Test
    @Suite( groups = "checkinttest" )
    public void testCurrentZonedDateTime( )
    {
        SimpleDateFormat sdf = new SimpleDateFormat( "yyyyMMddHHmmss" );
        sdf.setTimeZone( TimeZone.getTimeZone( "GMT" ) );
        String datePlain = sdf.format( new Date() );
        
        String newDatePlain = DateTimeUtil.getCurrentZonedDateTime( "yyyyMMddHHmmss", ZoneId.of( "GMT" ) );
        String newDatePlainNullTz = DateTimeUtil.getCurrentZonedDateTime( "yyyyMMddHHmmss", null );
        
        assertEquals( datePlain, newDatePlain );
        Assert.assertNotEquals( datePlain, newDatePlainNullTz );
    }
}
