

package gov.va.med.cds.clinicaldata.vhim400;


import gov.va.med.cds.junit.runners.Suite;
import gov.va.med.cds.junit.runners.SuiteAwareRunner;
import junit.framework.Assert;

import org.dom4j.Document;
import org.dom4j.Element;
import org.easymock.EasyMock;
import org.junit.Test;
import org.junit.runner.RunWith;


@RunWith( SuiteAwareRunner.class )
public class PathwaysDataResponseTest
{

    @Test
    @Suite( groups = "checkintest" )
    public void testBuildEmptyDocument( )
    {
        String startXml = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>";
        String templateId = "testTemplateId";
        String requestId = "testRequestId";
        PathwaysDataResponseInterface pathwaysDataResponse = new PathwaysDataResponse400();

        //first test specifying template and request
        Document pathwaysData = pathwaysDataResponse.buildEmptyClinicalDocument( templateId, requestId );
        Assert.assertEquals( pathwaysData.getRootElement().element( "templateId" ).getText(), templateId );
        Assert.assertEquals( pathwaysData.getRootElement().element( "requestId" ).getText(), requestId );
        Assert.assertTrue( pathwaysData.asXML().startsWith( startXml ) );

        //test null template and request
        pathwaysData = pathwaysDataResponse.buildEmptyClinicalDocument();
        Assert.assertTrue( pathwaysData.getRootElement().element( "templateId" ) == null );
        Assert.assertTrue( pathwaysData.getRootElement().element( "requestId" ) == null );
        Assert.assertTrue( pathwaysData.asXML().startsWith( startXml ) );

        //test patient element with template and request
        pathwaysData = pathwaysDataResponse.buildEmptyClinicalDocumentWithPatient( "templateId", "requestId" );
        Assert.assertTrue( pathwaysData.getRootElement().element( "patient" ) != null );
        Assert.assertTrue( pathwaysData.asXML().startsWith( startXml ) );

        //test patient element with null template and request
        pathwaysData = pathwaysDataResponse.buildEmptyClinicalDocumentWithPatient();
        Assert.assertTrue( pathwaysData.getRootElement().element( "patient" ) != null );
        Assert.assertTrue( pathwaysData.getRootElement().element( "templateId" ) == null );
        Assert.assertTrue( pathwaysData.getRootElement().element( "requestId" ) == null );
        Assert.assertTrue( pathwaysData.asXML().startsWith( startXml ) );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testAddToExistingRequestedNationalId( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element pathwaysDataElement = EasyMock.createMock( Element.class );
        Element patientElement = EasyMock.createMock( Element.class );
        Element requestedNationalIdElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( pathwaysDataElement );
        EasyMock.expect( pathwaysDataElement.element( "patient" ) ).andReturn( patientElement );
        EasyMock.expect( patientElement.element( "requestedNationalId" ) ).andReturn( requestedNationalIdElement );
        requestedNationalIdElement.setText( "theNationalId" );

        EasyMock.replay( document, pathwaysDataElement, patientElement, requestedNationalIdElement );

        PathwaysDataResponseInterface pathwaysDataResponse = new PathwaysDataResponse400();
        document = pathwaysDataResponse.addRequestedNationalId( document, "theNationalId" );

        EasyMock.verify( document, pathwaysDataElement, patientElement, requestedNationalIdElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testAddToNonExistingRequestedNationalId( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element pathwaysDataElement = EasyMock.createMock( Element.class );
        Element patientElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( pathwaysDataElement );
        EasyMock.expect( pathwaysDataElement.element( "patient" ) ).andReturn( patientElement );
        EasyMock.expect( patientElement.element( "requestedNationalId" ) ).andReturn( null );
        patientElement.add( EasyMock.isA( Element.class ) );

        EasyMock.replay( document, pathwaysDataElement, patientElement );

        PathwaysDataResponseInterface pathwaysDataResponse = new PathwaysDataResponse400();
        document = pathwaysDataResponse.addRequestedNationalId( document, "theNationalId" );

        EasyMock.verify( document, pathwaysDataElement, patientElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetChildElementNoCreateExists( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element pathwaysDataElement = EasyMock.createMock( Element.class );
        Element patientsElement = EasyMock.createMock( Element.class );
        Element patientElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( pathwaysDataElement );
        EasyMock.expect( pathwaysDataElement.element( "patients" ) ).andReturn( patientsElement );
        EasyMock.expect( patientsElement.element( "patient" ) ).andReturn( patientElement );

        EasyMock.replay( document, pathwaysDataElement, patientsElement, patientElement );

        PathwaysDataResponseInterface pathwaysDataResponse = new PathwaysDataResponse400();
        Element element = pathwaysDataResponse.getChildElement( document, "patients/patient" );

        Assert.assertEquals( patientElement, element );
        EasyMock.verify( document, pathwaysDataElement, patientsElement, patientElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetChildElementNoCreateDoesNotExist( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element pathwaysDataElement = EasyMock.createMock( Element.class );
        Element patientsElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( pathwaysDataElement );
        EasyMock.expect( pathwaysDataElement.element( "patients" ) ).andReturn( patientsElement );
        EasyMock.expect( patientsElement.element( "patient" ) ).andReturn( null );

        EasyMock.replay( document, pathwaysDataElement, patientsElement );

        PathwaysDataResponseInterface pathwaysDataResponse = new PathwaysDataResponse400();
        Element element = pathwaysDataResponse.getChildElement( document, "patients/patient" );

        Assert.assertNull( element );
        EasyMock.verify( document, pathwaysDataElement, patientsElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetErrorResponseTemplateId( )
    {
        PathwaysDataResponseInterface pathwaysDataResponse = new PathwaysDataResponse400();
        Assert.assertEquals( "PathwaysResponse400", pathwaysDataResponse.getErrorResponseTemplateId() );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetErrorSectionRoot( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element pathwaysDataElement = EasyMock.createMock( Element.class );
        Element errorSectionElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( pathwaysDataElement );
        EasyMock.expect( pathwaysDataElement.element( "errorSection" ) ).andReturn( errorSectionElement );

        EasyMock.replay( document, pathwaysDataElement, errorSectionElement );

        PathwaysDataResponseInterface pathwaysDataResponse = new PathwaysDataResponse400();
        Element element = pathwaysDataResponse.getErrorSectionRoot( document );

        Assert.assertEquals( errorSectionElement, element );
        EasyMock.verify( document, pathwaysDataElement, errorSectionElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetRequestedNationalId( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element pathwaysDataElement = EasyMock.createMock( Element.class );
        Element patientElement = EasyMock.createMock( Element.class );
        Element requestNationalIdElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( pathwaysDataElement );
        EasyMock.expect( pathwaysDataElement.element( "patient" ) ).andReturn( patientElement );
        EasyMock.expect( patientElement.element( "requestedNationalId" ) ).andReturn( requestNationalIdElement );

        EasyMock.replay( document, pathwaysDataElement, patientElement, requestNationalIdElement );

        PathwaysDataResponseInterface pathwaysDataResponse = new PathwaysDataResponse400();
        Element element = pathwaysDataResponse.getRequestedNationalId( document );

        Assert.assertEquals( requestNationalIdElement, element );
        EasyMock.verify( document, pathwaysDataElement, patientElement, requestNationalIdElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetRequestId( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element pathwaysDataElement = EasyMock.createMock( Element.class );
        Element requestIdElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( pathwaysDataElement );
        EasyMock.expect( pathwaysDataElement.element( "requestId" ) ).andReturn( requestIdElement );

        EasyMock.replay( document, pathwaysDataElement, requestIdElement );

        PathwaysDataResponseInterface pathwaysDataResponse = new PathwaysDataResponse400();
        Element element = pathwaysDataResponse.getRequestId( document );

        Assert.assertEquals( requestIdElement, element );
        EasyMock.verify( document, pathwaysDataElement, requestIdElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetTemplateId( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element pathwaysDataElement = EasyMock.createMock( Element.class );
        Element templateIdElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( pathwaysDataElement );
        EasyMock.expect( pathwaysDataElement.element( "templateId" ) ).andReturn( templateIdElement );

        EasyMock.replay( document, pathwaysDataElement, templateIdElement );

        PathwaysDataResponseInterface pathwaysDataResponse = new PathwaysDataResponse400();
        Element element = pathwaysDataResponse.getTemplateId( document );

        Assert.assertEquals( templateIdElement, element );
        EasyMock.verify( document, pathwaysDataElement, templateIdElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testSetTemplateIdText( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element pathwaysDataElement = EasyMock.createMock( Element.class );
        Element templateIdElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( pathwaysDataElement );
        EasyMock.expect( pathwaysDataElement.element( "templateId" ) ).andReturn( templateIdElement );
        templateIdElement.setText( "templateId100" );

        EasyMock.replay( document, pathwaysDataElement, templateIdElement );

        PathwaysDataResponseInterface pathwaysDataResponse = new PathwaysDataResponse400();
        Element element = pathwaysDataResponse.setTemplateIdText( document, "templateId100" );

        Assert.assertEquals( templateIdElement, element );
        EasyMock.verify( document, pathwaysDataElement, templateIdElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetPatientElementExists( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element pathwaysDataElement = EasyMock.createMock( Element.class );
        Element patientElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( pathwaysDataElement );
        EasyMock.expect( pathwaysDataElement.element( "patient" ) ).andReturn( patientElement );

        EasyMock.replay( document, pathwaysDataElement, patientElement );

        PathwaysDataResponseInterface pathwaysDataResponse = new PathwaysDataResponse400();
        Element element = pathwaysDataResponse.getPatientElement( document );

        Assert.assertEquals( patientElement, element );
        EasyMock.verify( document, pathwaysDataElement, patientElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetPatientElementDoesNotExist( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element pathwaysDataElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( pathwaysDataElement );
        EasyMock.expect( pathwaysDataElement.element( "patient" ) ).andReturn( null );
        pathwaysDataElement.add( EasyMock.isA( Element.class ) );

        EasyMock.replay( document, pathwaysDataElement );

        PathwaysDataResponseInterface pathwaysDataResponse = new PathwaysDataResponse400();
        Element element = pathwaysDataResponse.getPatientElement( document );

        Assert.assertNotNull( element );
        EasyMock.verify( document, pathwaysDataElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetClinicalRoot( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element pathwaysDataElement = EasyMock.createMock( Element.class );
        Element patientElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( pathwaysDataElement );
        EasyMock.expect( pathwaysDataElement.element( "patient" ) ).andReturn( patientElement );

        EasyMock.replay( document, pathwaysDataElement, patientElement );

        PathwaysDataResponseInterface pathwaysDataResponse = new PathwaysDataResponse400();
        Element element = pathwaysDataResponse.getClinicalRoot( document );

        Assert.assertEquals( patientElement, element );
        EasyMock.verify( document, pathwaysDataElement, patientElement );
    }

};
