

package gov.va.med.cds.clinicaldata.vhim400;


import gov.va.med.cds.junit.runners.Suite;
import gov.va.med.cds.junit.runners.SuiteAwareRunner;
import junit.framework.Assert;

import org.dom4j.Document;
import org.dom4j.Element;
import org.easymock.EasyMock;
import org.junit.Test;
import org.junit.runner.RunWith;


@RunWith( SuiteAwareRunner.class )
public class ClinicalDataResponseTest
{

    @Test
    @Suite( groups = "checkintest" )
    public void testBuildEmptyClinicalDocument( )
    {
        String startXml = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>";
        String templateId = "testTemplateId";
        String requestId = "testRequestId";
        ClinicalDataResponseInterface clinicalDataResponse = new ClinicalDataResponse400();

        //first test specifying template and request
        Document clinicalData = clinicalDataResponse.buildEmptyClinicalDocument( templateId, requestId );
        Assert.assertEquals( clinicalData.getRootElement().element( "templateId" ).getText(), templateId );
        Assert.assertEquals( clinicalData.getRootElement().element( "requestId" ).getText(), requestId );
        Assert.assertTrue( clinicalData.asXML().startsWith( startXml ) );

        //test null template and request
        clinicalData = clinicalDataResponse.buildEmptyClinicalDocument();
        Assert.assertTrue( clinicalData.getRootElement().element( "templateId" ) == null );
        Assert.assertTrue( clinicalData.getRootElement().element( "requestId" ) == null );
        Assert.assertTrue( clinicalData.asXML().startsWith( startXml ) );

        //test patient element with template and request
        clinicalData = clinicalDataResponse.buildEmptyClinicalDocumentWithPatient( "templateId", "requestId" );
        Assert.assertTrue( clinicalData.getRootElement().element( "patient" ) != null );
        Assert.assertTrue( clinicalData.asXML().startsWith( startXml ) );

        //test patient element with null template and request
        clinicalData = clinicalDataResponse.buildEmptyClinicalDocumentWithPatient();
        Assert.assertTrue( clinicalData.getRootElement().element( "patient" ) != null );
        Assert.assertTrue( clinicalData.getRootElement().element( "templateId" ) == null );
        Assert.assertTrue( clinicalData.getRootElement().element( "requestId" ) == null );
        Assert.assertTrue( clinicalData.asXML().startsWith( startXml ) );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testAddToExistingRequestedNationalId( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element clinicalDataElement = EasyMock.createMock( Element.class );
        Element patientElement = EasyMock.createMock( Element.class );
        Element requestedNationalIdElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( clinicalDataElement );
        EasyMock.expect( clinicalDataElement.element( "patient" ) ).andReturn( patientElement );
        EasyMock.expect( patientElement.element( "requestedNationalId" ) ).andReturn( requestedNationalIdElement );
        requestedNationalIdElement.setText( "theNationalId" );

        EasyMock.replay( document, clinicalDataElement, patientElement, requestedNationalIdElement );

        ClinicalDataResponseInterface clinicalDataResponse = new ClinicalDataResponse400();
        document = clinicalDataResponse.addRequestedNationalId( document, "theNationalId" );

        EasyMock.verify( document, clinicalDataElement, patientElement, requestedNationalIdElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testAddToNonExistingRequestedNationalId( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element clinicalDataElement = EasyMock.createMock( Element.class );
        Element patientElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( clinicalDataElement );
        EasyMock.expect( clinicalDataElement.element( "patient" ) ).andReturn( patientElement );
        EasyMock.expect( patientElement.element( "requestedNationalId" ) ).andReturn( null );
        patientElement.add( EasyMock.isA( Element.class ) );

        EasyMock.replay( document, clinicalDataElement, patientElement );

        ClinicalDataResponseInterface clinicalDataResponse = new ClinicalDataResponse400();
        document = clinicalDataResponse.addRequestedNationalId( document, "theNationalId" );

        EasyMock.verify( document, clinicalDataElement, patientElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetChildElementNoCreateExists( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element clinicalDataElement = EasyMock.createMock( Element.class );
        Element patientsElement = EasyMock.createMock( Element.class );
        Element patientElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( clinicalDataElement );
        EasyMock.expect( clinicalDataElement.element( "patients" ) ).andReturn( patientsElement );
        EasyMock.expect( patientsElement.element( "patient" ) ).andReturn( patientElement );

        EasyMock.replay( document, clinicalDataElement, patientsElement, patientElement );

        ClinicalDataResponseInterface clinicalDataResponse = new ClinicalDataResponse400();
        Element element = clinicalDataResponse.getChildElement( document, "patients/patient" );

        Assert.assertEquals( patientElement, element );
        EasyMock.verify( document, clinicalDataElement, patientsElement, patientElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetChildElementNoCreateDoesNotExist( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element clinicalDataElement = EasyMock.createMock( Element.class );
        Element patientsElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( clinicalDataElement );
        EasyMock.expect( clinicalDataElement.element( "patients" ) ).andReturn( patientsElement );
        EasyMock.expect( patientsElement.element( "patient" ) ).andReturn( null );

        EasyMock.replay( document, clinicalDataElement, patientsElement );

        ClinicalDataResponseInterface clinicalDataResponse = new ClinicalDataResponse400();
        Element element = clinicalDataResponse.getChildElement( document, "patients/patient" );

        Assert.assertNull( element );
        EasyMock.verify( document, clinicalDataElement, patientsElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetErrorResponseTemplateId( )
    {
        ClinicalDataResponseInterface clinicalDataResponse = new ClinicalDataResponse400();
        Assert.assertEquals( "CdsResponse400", clinicalDataResponse.getErrorResponseTemplateId() );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetErrorSectionRoot( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element clinicalDataElement = EasyMock.createMock( Element.class );
        Element errorSectionElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( clinicalDataElement );
        EasyMock.expect( clinicalDataElement.element( "errorSection" ) ).andReturn( errorSectionElement );

        EasyMock.replay( document, clinicalDataElement, errorSectionElement );

        ClinicalDataResponseInterface clinicalDataResponse = new ClinicalDataResponse400();
        Element element = clinicalDataResponse.getErrorSectionRoot( document );

        Assert.assertEquals( errorSectionElement, element );
        EasyMock.verify( document, clinicalDataElement, errorSectionElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetRequestedNationalId( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element clinicalDataElement = EasyMock.createMock( Element.class );
        Element patientElement = EasyMock.createMock( Element.class );
        Element requestNationalIdElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( clinicalDataElement );
        EasyMock.expect( clinicalDataElement.element( "patient" ) ).andReturn( patientElement );
        EasyMock.expect( patientElement.element( "requestedNationalId" ) ).andReturn( requestNationalIdElement );

        EasyMock.replay( document, clinicalDataElement, patientElement, requestNationalIdElement );

        ClinicalDataResponseInterface clinicalDataResponse = new ClinicalDataResponse400();
        Element element = clinicalDataResponse.getRequestedNationalId( document );

        Assert.assertEquals( requestNationalIdElement, element );
        EasyMock.verify( document, clinicalDataElement, patientElement, requestNationalIdElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetRequestId( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element clinicalDataElement = EasyMock.createMock( Element.class );
        Element requestIdElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( clinicalDataElement );
        EasyMock.expect( clinicalDataElement.element( "requestId" ) ).andReturn( requestIdElement );

        EasyMock.replay( document, clinicalDataElement, requestIdElement );

        ClinicalDataResponseInterface clinicalDataResponse = new ClinicalDataResponse400();
        Element element = clinicalDataResponse.getRequestId( document );

        Assert.assertEquals( requestIdElement, element );
        EasyMock.verify( document, clinicalDataElement, requestIdElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetTemplateId( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element clinicalDataElement = EasyMock.createMock( Element.class );
        Element templateIdElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( clinicalDataElement );
        EasyMock.expect( clinicalDataElement.element( "templateId" ) ).andReturn( templateIdElement );

        EasyMock.replay( document, clinicalDataElement, templateIdElement );

        ClinicalDataResponseInterface clinicalDataResponse = new ClinicalDataResponse400();
        Element element = clinicalDataResponse.getTemplateId( document );

        Assert.assertEquals( templateIdElement, element );
        EasyMock.verify( document, clinicalDataElement, templateIdElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testSetTemplateIdText( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element clinicalDataElement = EasyMock.createMock( Element.class );
        Element templateIdElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( clinicalDataElement );
        EasyMock.expect( clinicalDataElement.element( "templateId" ) ).andReturn( templateIdElement );
        templateIdElement.setText( "templateId100" );

        EasyMock.replay( document, clinicalDataElement, templateIdElement );

        ClinicalDataResponseInterface clinicalDataResponse = new ClinicalDataResponse400();
        Element element = clinicalDataResponse.setTemplateIdText( document, "templateId100" );

        Assert.assertEquals( templateIdElement, element );
        EasyMock.verify( document, clinicalDataElement, templateIdElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetPatientElementExists( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element clinicalDataElement = EasyMock.createMock( Element.class );
        Element patientElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( clinicalDataElement );
        EasyMock.expect( clinicalDataElement.element( "patient" ) ).andReturn( patientElement );

        EasyMock.replay( document, clinicalDataElement, patientElement );

        ClinicalDataResponseInterface clinicalDataResponse = new ClinicalDataResponse400();
        Element element = clinicalDataResponse.getPatientElement( document );

        Assert.assertEquals( patientElement, element );
        EasyMock.verify( document, clinicalDataElement, patientElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetPatientElementDoesNotExist( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element clinicalDataElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( clinicalDataElement );
        EasyMock.expect( clinicalDataElement.element( "patient" ) ).andReturn( null );
        clinicalDataElement.add( EasyMock.isA( Element.class ) );

        EasyMock.replay( document, clinicalDataElement );

        ClinicalDataResponseInterface clinicalDataResponse = new ClinicalDataResponse400();
        Element element = clinicalDataResponse.getPatientElement( document );

        Assert.assertNotNull( element );
        EasyMock.verify( document, clinicalDataElement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testGetClinicalRoot( )
    {
        Document document = EasyMock.createMock( Document.class );
        Element clinicalDataElement = EasyMock.createMock( Element.class );
        Element patientElement = EasyMock.createMock( Element.class );

        EasyMock.expect( document.getRootElement() ).andReturn( clinicalDataElement );
        EasyMock.expect( clinicalDataElement.element( "patient" ) ).andReturn( patientElement );

        EasyMock.replay( document, clinicalDataElement, patientElement );

        ClinicalDataResponseInterface clinicalDataResponse = new ClinicalDataResponse400();
        Element element = clinicalDataResponse.getClinicalRoot( document );

        Assert.assertEquals( patientElement, element );
        EasyMock.verify( document, clinicalDataElement, patientElement );
    }

};
