

package gov.va.med.cds.xml.transform;


import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.xml.transform.Result;
import javax.xml.transform.Templates;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.sax.SAXSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.io.DocumentResult;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.xml.sax.InputSource;

import gov.va.med.cds.exception.ErrorCodeEnum;
import net.sf.saxon.Configuration;
import net.sf.saxon.TransformerFactoryImpl;
import net.sf.saxon.lib.ExtensionFunctionDefinition;


/***********************************************************************************************************************
 * 
 * XsltHelper loader class compiles XSLTs and holds the Transformer objects in a
 * non-persistent data structure in
 * 
 * physical memory
 * 
 * 
 * 
 * XsltHelper can also be instantiated using a constructor that takes a list of
 * xlst template locations and ExtensionFunctionDefinition objects
 * 
 * if Saxon integrated extensions are needed.
 * 
 * 
 * 
 * @author DNS   BHAGAV
 * 
 * 
 * 
 */

public final class XsltHelper
    implements
        TransformerInterface
{
    public XsltHelper( )
    {
        super();
    }


    public XsltHelper( List<String> aXsltLocations, List<ExtensionFunctionDefinition> aExtFunctions )
        throws TransformerConfigurationException,
            IOException
    {
        super();

        loadXsltTransformsIntoMemory( aXsltLocations, aExtFunctions );
    }

    
    private static Log logger = LogFactory.getLog( XsltHelper.class );
    // this is the container for compiled XSLTs
    private Map<String, Templates> xsltTemplate = new HashMap<String, Templates>();


    /*******************************************************************************************************************
     * 
     * Compiles XSLT files and stores the compiled XSLT in a non persistent
     * store - does not register ExtensionFunctionDefinitions.
     * 
     * 
     * 
     * @throws IOException
     * 
     * @throws TransformerConfigurationException
     * 
     */

    private void loadXsltTransformsIntoMemory( List<String> aXsltLocations )
        throws IOException,
            TransformerConfigurationException
    {
        // empty ExtensionFunctionDefinitions - there aren't any to register

        loadXsltTransformsIntoMemory( aXsltLocations, new ArrayList<ExtensionFunctionDefinition>() );
    }


    /*******************************************************************************************************************
     * 
     * Compiles XSLT files and stores the compiled XSLT in a non persistent
     * store and registers any ExtensionFunctionDefinitions present.
     * 
     * 
     * 
     * @throws IOException
     * 
     * @throws TransformerConfigurationException
     * 
     */

    private void loadXsltTransformsIntoMemory( List<String> aXsltLocations, List<ExtensionFunctionDefinition> aExtFunctions )
        throws IOException,
            TransformerConfigurationException
    {
        StreamSource templateStreamSource = null;
        InputStream is = null;
        TransformerFactory transformerFactory = new TransformerFactoryImpl();

        // register saxonExtensionFunctions
        Configuration saxonConfig = ( ( TransformerFactoryImpl )transformerFactory ).getConfiguration();
        registerSaxonExtensionFunctions( saxonConfig, aExtFunctions );

        Resource[] resources = null;
        PathMatchingResourcePatternResolver resolver = new PathMatchingResourcePatternResolver();

        for ( String xsltPath : aXsltLocations )
        {
            resources = resolver.getResources( xsltPath.trim() );

            for ( Resource resource : resources )
            {
                try
                {
                    logger.debug( "XSLT resource: " + resource.getFilename() + " is being loaded." );
                    is = resource.getInputStream();
                    templateStreamSource = new StreamSource( is );
                    xsltTemplate.put( resource.getFilename(), transformerFactory.newTemplates( templateStreamSource ) );
                }
                finally
                {
                    if ( is != null )
                    {
                        is.close();
                    }
                }
            }
        }
    }


    /**
     * 
     * Method registers Saxon Extension Functions with the Saxon
     * framework/transform factory
     * 
     * 
     * 
     * @param aSaxonConfig
     *            - Transform factory configuration
     * 
     * @param aExtFunctions
     *            - List of ExtensionFunctionDefintion objects
     * 
     */

    private void registerSaxonExtensionFunctions( Configuration aSaxonConfig, List<ExtensionFunctionDefinition> aExtFunctions )
    {
        for ( ExtensionFunctionDefinition extFunction : aExtFunctions )
        {
            aSaxonConfig.registerExtensionFunction( extFunction );
        }
    }


    /**
     * 
     * Utility method intended to reduced duplicated logic in clients needing to
     * transfrom an xml file with an xslt
     * 
     * template - for ease this method returns a usable String value
     * 
     * 
     * 
     * @param aSource
     *            The XML data as a string.
     * 
     * @param xsltKey
     *            The key to look up the transformer by.
     * 
     * @return The transformed results.
     * 
     */

    public String transformMessageAsString( String aSource, String aXsltKey )
    {
        return transformMessageAsString( aSource, aXsltKey, ( Map<String, Object> )null );
    }


    /**
     * 
     * Utility method intended to reduced duplicated logic in clients needing to
     * transfrom an xml file with an xslt
     * 
     * template - for ease this method returns a usable String value
     * 
     * 
     * 
     * @param aSource
     *            The XML data as a string.
     * 
     * @param aXsltKey
     *            The key to look up the transformer by.
     * 
     * @param aParameters
     *            The parameters that will be passed to the Transformer.
     * 
     * @return The transformed results.
     * 
     */

    public String transformMessageAsString( String aSource, String aXsltKey, Map<String, Object> aParameters )
    {
        try
        {
            Document result = transformMessage( aSource, aXsltKey, aParameters );
            String tempStr = result.asXML();

            return tempStr;
        }
        catch ( Exception e )
        {
            throw new XsltTransformationException( ErrorCodeEnum.XSLT_HELPER_TRANSLATION_EXCEPTION, e, aXsltKey );
        }
    }


    public String transformMessageAsPlainText( String aSource, String aXsltKey )
    {
        return transformMessageAsPlainText( aSource, aXsltKey, ( Map<String, Object> )null );
    }


    public String transformMessageAsPlainText( String aSource, String aXsltKey, Map<String, Object> aParameters )
    {
        try
        {
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            Result result = new StreamResult( bos );
            Transformer transformer = getXsltTransformer( aXsltKey );
            setTransformerParameters( transformer, aParameters );
            transformer.transform( new SAXSource( new InputSource( new StringReader( aSource ) ) ), result );
            String tempStr = bos.toString();
            bos.flush();
            bos.close();

            return tempStr;
        }
        catch ( Exception e )
        {
            throw new XsltTransformationException( ErrorCodeEnum.XSLT_HELPER_TRANSLATION_EXCEPTION, e, aXsltKey );
        }
    }


    /**
     * 
     * 
     * 
     * @param aSource
     * 
     * @param aXsltKey
     * 
     * @param result
     * 
     */

    public Document transformMessage( String aSource, String aXsltKey )
    {
        return transformMessage( aSource, aXsltKey, null );
    }


    public Document transformMessage( String aSource, String aXsltKey, Map<String, Object> aParameters )
    {
        try
        {
            Transformer transformer = getXsltTransformer( aXsltKey );
            setTransformerParameters( transformer, aParameters );
            DocumentResult result = new DocumentResult();
            transformer.transform( new SAXSource( new InputSource( new StringReader( aSource ) ) ), result );

            return result.getDocument();
        }
        catch ( Exception e )
        {
            throw new XsltTransformationException( ErrorCodeEnum.XSLT_HELPER_TRANSLATION_EXCEPTION, e, aXsltKey );
        }
    }


    private void setTransformerParameters( Transformer transformer, Map<String, Object> parameters )
    {
        /*
         * JLA The following code is being commented out as per Fortify Quality Code Review: Dead Code: Expression is Always false.
         * this method is only called with parameters == null
         * 
        if ( parameters != null )
        {
            for ( Entry<String, Object> entry : parameters.entrySet() )
            {
                transformer.setParameter( entry.getKey(), entry.getValue() );
            }
        }
         */
    }


    /*******************************************************************************************************************
     * 
     * Look for a compiled XSLT Transformer in the store
     * 
     * 
     * 
     * @param xsltTransformerName
     *            is the name of XSLT file for which a compiled Transformer is
     *            sought
     * 
     * @return Transformer for a compiled XSLT
     * 
     * @throws NullPointerException
     *             if no transformer template exists for transform name.
     * 
     */

    public Transformer getXsltTransformer( String aXsltTransformerName )
        throws TransformerConfigurationException
    {
        return xsltTemplate.get( aXsltTransformerName ).newTransformer();
    }


    public void setXsltLocations( List<String> aXsltLocations )
        throws Exception
    {
        loadXsltTransformsIntoMemory( aXsltLocations );
    }

}
