

package gov.va.med.cds.util;


import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Date;
import java.util.TimeZone;


public class DateTimeUtil
{
    public final static String FORMAT_FULL = "yyyyMMddHHmmss.SSS";
    public final static String FORMAT_PARTIAL = "yyyMddHHmm";
    public final static String FORMAT_DATE_TIME = "yyyyMMdd.HHmmss";
    public final static String FORMAT_TZ = "Z";
    public final static String FORMAT_HL7 = "yyyyMMddHHmmssZ";
    public final static String FORMAT_DATE_STR = "MM/dd/yyyy";
    public final static String FORMAT_TIME_STR = "HH:mm:ss";
    public final static String FORMAT_FULL_TZ = FORMAT_FULL + FORMAT_TZ;


    /**
     *
     * This method gets the current date time in a "yyyyMMddHHmmss.SSSZ" formatted string.
     *
     * @return String. This is the current date time in "yyyyMMddHHmmss.SSSZ" format.
     *
     * @since jdk1.8
     *
     */
    public static String getCurrentISOTime( )
    {
        final String defaultFormat = FORMAT_FULL_TZ;
        ZonedDateTime zonedDateTime = ZonedDateTime.now();
        DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern( defaultFormat );

        return zonedDateTime.format( dateFormatter );
    }


    /**
     *
     * This method gets the current date time and formats it to the format passed in.
     * If the format passed in is null the format "yyyyMMddHHmmss.SSSZ" is used
     *
     * @param aFormat
     *            a format for the return string. ex: "yyyyMMddHHmmssZ"
     * @return String. This is the ISO date format that corresponds to the given
     *         date in milliseconds.
     *
     * @since jdk1.8
     *
     */
    public static String getCurrentISOTime( String aFormat )
    {
        final String defaultFormat = FORMAT_FULL_TZ;
        ZonedDateTime zonedDateTime = ZonedDateTime.now();
        DateTimeFormatter dateFormatter = null;

        if ( null != aFormat )
        {
            dateFormatter = DateTimeFormatter.ofPattern( aFormat );
        }
        else
        {
            dateFormatter = DateTimeFormatter.ofPattern( defaultFormat );
        }

        return zonedDateTime.format( dateFormatter );
    }


    /**
     *
     * This method gets the current date time for the ZoneId passed in and formats it to a "yyyyMMddHHmmss.SSSZ" formatted string.
     * If the ZoneId passed in is null, the default zone is used.
     *
     * @param aZoneId
     *            if null, use default.
     * @return String. This is the ISO date format that corresponds to the given
     *         date in milliseconds.
     *
     * @since jdk1.8
     *
     */
    public static String getCurrentISOTime( ZoneId aZoneId )
    {
        final String defaultFormat = FORMAT_FULL_TZ;
        ZonedDateTime zonedDateTime = null;
        DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern( defaultFormat );

        if ( null != aZoneId )
        {
            zonedDateTime = ZonedDateTime.now( aZoneId );
        }
        else
        {
            zonedDateTime = ZonedDateTime.now();
        }

        return zonedDateTime.format( dateFormatter );
    }


    /**
     *
     * This method converts the given Instant for the ZoneId passed in and formats it to a "yyyyMMddHHmmss.SSSZ" formatted string.
     * If the ZoneId passed in is null, the default zone is used.
     *
     * @param aInstant
     *            the time, in milliseconds
     * @param aZoneId
     *            if null, use default.
     * @return String. This is the ISO date format that corresponds to the given
     *         date in milliseconds.
     *
     * @since jdk1.8
     *
     */
    public static String getISOTime( Instant aInstant, ZoneId aZoneId )
    {
        final String defaultFormat = FORMAT_FULL_TZ;
        ZonedDateTime zonedDateTime = null;
        DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern( defaultFormat );

        if ( null != aZoneId )
        {
            zonedDateTime = aInstant.atZone( aZoneId );
//            zonedDateTime = ZonedDateTime.ofInstant( aInstant, aZoneId );
        }
        else
        {
            zonedDateTime = aInstant.atZone( ZoneId.systemDefault() );
//            zonedDateTime = ZonedDateTime.ofInstant( aInstant, ZoneId.systemDefault() );
        }

        return zonedDateTime.format( dateFormatter );
    }


    /**
     *
     * This method converts the given time from milliseconds for the ZoneId passed in and formats it to a "yyyyMMddHHmmss.SSSZ" formatted string.
     *
     * @param aTimeInMillis
     *            the time, in milliseconds
     * @param aZoneId
     *            if null, use default.
     * @return String. A "yyyyMMddHHmmss.SSSZ" formatted string that corresponds to the given
     *         date in milliseconds and the ZoneId passed in.
     *
     * @since jdk1.8
     *
     */
    public static String getISOTime( long aTimeInMillis, ZoneId aZoneId )
    {
        final String defaultFormat = FORMAT_FULL_TZ;
        ZonedDateTime zonedDateTime = null;
        DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern( defaultFormat );

        if ( null != aZoneId )
        {
            zonedDateTime = Instant.ofEpochMilli( aTimeInMillis ).atZone( aZoneId );
//            zonedDateTime = ZonedDateTime.ofInstant( Instant.ofEpochMilli( aTimeInMillis ), aZoneId );
        }
        else
        {
            zonedDateTime = Instant.ofEpochMilli( aTimeInMillis ).atZone( ZoneId.systemDefault() );
//            zonedDateTime = ZonedDateTime.ofInstant( Instant.ofEpochMilli( aTimeInMillis ), ZoneId.systemDefault() );
        }

        return zonedDateTime.format( dateFormatter );
    }


    /**
     *
     * This method converts the given time from milliseconds for the TimeZone passed in and formats it to a "yyyyMMddHHmmss.SSSZ" formatted string.
     *
     * @param aTimeInMillis
     *            the time, in milliseconds
     * @param aTimezone
     *            if null, use default.
     * @return String. A "yyyyMMddHHmmss.SSSZ" formatted string that corresponds to the given
     *         date in milliseconds and the TimeZone passed in.
     *
     * @since jdk1.8
     *
     */
    public static String getISOTime( long aTimeInMillis, TimeZone aTimezone )
    {
        final String defaultFormat = FORMAT_FULL_TZ;
        ZoneId zoneId = null;
        DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern( defaultFormat );

        if ( null != aTimezone )
        {
            zoneId = ZoneId.of( aTimezone.getID() );
        }
        else
        {
            zoneId = ZoneId.systemDefault();
        }

        ZonedDateTime zonedDateTime = Instant.ofEpochMilli( aTimeInMillis ).atZone( zoneId );
//        ZonedDateTime zonedDateTime = ZonedDateTime.ofInstant( Instant.ofEpochMilli( aTimeInMillis ), zoneId );

        return zonedDateTime.format( dateFormatter );
    }


    /**
     *
     * This method converts the given time from milliseconds to ISO format
     * string.
     *
     * @param milliseconds
     *            the time, in milliseconds
     * @param timezone
     *            if null, use default.
     * @return String. This is the ISO date format that corresponds to the
     *
     *         given date in milliseconds.
     *
     * @since jdk1.2
     *
     */
    public static String getISOTimeOld( long milliseconds, TimeZone timezone )
    {
        final String defaultFormat = FORMAT_FULL + FORMAT_TZ;
        DateFormat df = new SimpleDateFormat( defaultFormat );

        if ( null != timezone )
        {
            df.setTimeZone( timezone );
        }

        return df.format( new Date( milliseconds ) );
    }


    public static String getPartialISOTime( long milliseconds, TimeZone timezone )
    {
        final String defaultFormat = FORMAT_PARTIAL;
        DateFormat df = new SimpleDateFormat( defaultFormat );

        if ( null != timezone )
        {
            df.setTimeZone( timezone );
        }

        return df.format( new Date( milliseconds ) );
    }


    public static String getPartialISOTime( Instant aInstant, ZoneId aZoneId )
    {
        final String defaultFormat = FORMAT_PARTIAL;
        String returnValue = null;
        ZoneId zoneId = aZoneId;
        ZonedDateTime zonedDateTime = null;
        DateTimeFormatter dateFormatter = null;

        if ( null != aInstant )
        {
            dateFormatter = DateTimeFormatter.ofPattern( defaultFormat );

            if ( null == aZoneId )
            {
                zoneId = ZoneId.systemDefault();
            }

            zonedDateTime = ZonedDateTime.ofInstant( aInstant, zoneId );
            returnValue = zonedDateTime.format( dateFormatter );
        }

        return returnValue;
    }


    public static String getDateTimeDotSeparated( long milliseconds, TimeZone timezone )
    {
        final String defaultFormat = FORMAT_DATE_TIME;
        DateFormat df = new SimpleDateFormat( defaultFormat );

        if ( null != timezone )
        {
            df.setTimeZone( timezone );
        }

        return df.format( new Date( milliseconds ) );
    }


    public static String getDateTimeDotSeparated( Instant aInstant, ZoneId aZoneId )
    {
        final String defaultFormat = FORMAT_DATE_TIME;
        String returnValue = null;
        ZoneId zoneId = aZoneId;
        ZonedDateTime zonedDateTime = null;
        DateTimeFormatter dateFormatter = null;

        if ( null != aInstant )
        {
            dateFormatter = DateTimeFormatter.ofPattern( defaultFormat );
            
            if ( null == aZoneId )
            {
                zoneId = ZoneId.systemDefault();
            }

            zonedDateTime = ZonedDateTime.ofInstant( aInstant, zoneId );
            returnValue = zonedDateTime.format( dateFormatter );
        }

        return returnValue;
    }


    public static String stripTimeZoneOffset( String timeStamp )
    {
        return timeStamp.substring( 0, timeStamp.indexOf( "." ) );
    }


    public static String getCurrentDate( String aFormat )
    {
        final String defaultFormat = FORMAT_DATE_STR;
        LocalDate localDate = LocalDate.now();
        DateTimeFormatter dateFormatter = null;

        if ( null != aFormat )
        {
            dateFormatter = DateTimeFormatter.ofPattern( aFormat );
        }
        else
        {
            dateFormatter = DateTimeFormatter.ofPattern( defaultFormat );
        }

        return localDate.format( dateFormatter );
    }


    public static String getCurrentTime( String aFormat )
    {
        final String defaultFormat = FORMAT_TIME_STR;
        LocalTime localTime = LocalTime.now();
        DateTimeFormatter timeFormatter = null;

        if ( null != aFormat )
        {
            timeFormatter = DateTimeFormatter.ofPattern( aFormat );
        }
        else
        {
            timeFormatter = DateTimeFormatter.ofPattern( defaultFormat );
        }

        return localTime.format( timeFormatter );
    }


    public static String getCurrentDateTime( String aFormat )
    {
        final String defaultFormat = FORMAT_FULL;
        LocalDateTime localDateTime = LocalDateTime.now();
        DateTimeFormatter dateFormatter = null;

        if ( null != aFormat )
        {
            dateFormatter = DateTimeFormatter.ofPattern( aFormat );
        }
        else
        {
            dateFormatter = DateTimeFormatter.ofPattern( defaultFormat );
        }

        return localDateTime.format( dateFormatter );
    }


    public static String getCurrentZonedDateTime( String aFormat )
    {
        final String defaultFormat = FORMAT_HL7;
        ZonedDateTime zonedDateTime = ZonedDateTime.now();
        DateTimeFormatter zonedDateTimeFormatter = null;

        if ( null != aFormat )
        {
            zonedDateTimeFormatter = DateTimeFormatter.ofPattern( aFormat );
        }
        else
        {
            zonedDateTimeFormatter = DateTimeFormatter.ofPattern( defaultFormat );
        }

        return zonedDateTime.format( zonedDateTimeFormatter );
    }


    public static String getCurrentZonedDateTime( String aFormat, ZoneId aZoneId )
    {
        final String defaultFormat = FORMAT_FULL;
        ZonedDateTime zonedDateTime = null;
        DateTimeFormatter dateFormatter = null;

        if ( null != aZoneId )
        {
            zonedDateTime = ZonedDateTime.now( aZoneId );
        }
        else
        {
            zonedDateTime = ZonedDateTime.now();
        }
        
        if ( null != aFormat )
        {
            dateFormatter = DateTimeFormatter.ofPattern( aFormat );
        }
        else
        {
            dateFormatter = DateTimeFormatter.ofPattern( defaultFormat );
        }

        return zonedDateTime.format( dateFormatter );
    }


    public static boolean isDateValidHl7Format( String aMessageDateTime )
    {
        final String defaultFormat = FORMAT_HL7;
        boolean isValid = true;
        DateTimeFormatter dateFormatter = null;

        try
        {
            // first check that the string input date is complete - incomplete
            // input strings can create valid dates so the string must match
            // regex
            if ( aMessageDateTime.matches( "([0-9]{14})-([0-9]{4})" ) )
            {
                // secondary test check if a valid date can be constructed
                dateFormatter = DateTimeFormatter.ofPattern( defaultFormat );
                ZonedDateTime.parse( aMessageDateTime, dateFormatter );
            }
            else
            {
                isValid = false;
            }
        }
        catch ( Exception eD )
        {
            isValid = false;
        }

        return isValid;
    }


    public static String formatLocalDate( LocalDate aLocalDate )
    {
        final String defaultFormat = FORMAT_DATE_STR;
        String formattedLocalDate = null;
        DateTimeFormatter dateFormatter = null;

        if ( null != aLocalDate )
        {
            dateFormatter = DateTimeFormatter.ofPattern( defaultFormat );
            formattedLocalDate = aLocalDate.format( dateFormatter );
        }

        return formattedLocalDate;
    }


    public static String formatLocalTime( LocalTime aLocalTime )
    {
        final String defaultFormat = FORMAT_TIME_STR;
        String formattedLocalTime = null;
        DateTimeFormatter dateFormatter = null;

        if ( null != aLocalTime )
        {
            dateFormatter = DateTimeFormatter.ofPattern( defaultFormat );
            formattedLocalTime = aLocalTime.format( dateFormatter );
        }

        return formattedLocalTime;
    }


    public static String formatLocalDateTime( LocalDateTime aLocalDateTime )
    {
        final String defaultFormat = FORMAT_DATE_TIME;
        String formattedLocalDateTime = null;
        DateTimeFormatter dateFormatter = null;

        if ( null != aLocalDateTime )
        {
            dateFormatter = DateTimeFormatter.ofPattern( defaultFormat );
            formattedLocalDateTime = aLocalDateTime.format( dateFormatter );
        }

        return formattedLocalDateTime;
    }


    public static String formatZonedDateTime( ZonedDateTime aZonedDateTime )
    {
        final String defaultFormat = FORMAT_HL7;
        String formattedZonedDateTime = null;
        DateTimeFormatter dateFormatter = null;

        if ( null != aZonedDateTime )
        {
            dateFormatter = DateTimeFormatter.ofPattern( defaultFormat );
            formattedZonedDateTime = aZonedDateTime.format( dateFormatter );
        }

        return formattedZonedDateTime;
    }


    public static String convertDate2String( Date aDate )
    {
        final String defaultFormat = FORMAT_DATE_STR;
        String returnValue = null;
        DateFormat dateFormat = new SimpleDateFormat( defaultFormat );

        if ( null != aDate )
        {
            dateFormat = new SimpleDateFormat( defaultFormat );
            returnValue = dateFormat.format( aDate );
        }

        return returnValue;
    }


    public static LocalDate asLocalDate( Date aDate )
    {
        LocalDate returnLocalDate = null;

        if ( null != aDate )
        {
            Instant instant = aDate.toInstant();
            ZonedDateTime zonedDateTime = instant.atZone( ZoneId.systemDefault() );
            returnLocalDate = zonedDateTime.toLocalDate();
        }

        return returnLocalDate;
    }


    public static LocalDateTime asLocalDateTime( String aLocalDateTime, String aFormat )
    {
        final String defaultFormat = FORMAT_HL7;
        LocalDateTime returnLocalDateTime = null;
        DateTimeFormatter localDateTimeFormatter = null;

        if ( null != aLocalDateTime )
        {
            if ( null != aFormat )
            {
                localDateTimeFormatter = DateTimeFormatter.ofPattern( aFormat );
            }
            else
            {
                localDateTimeFormatter = DateTimeFormatter.ofPattern( defaultFormat );
            }

            try
            {
                returnLocalDateTime = LocalDateTime.parse( aLocalDateTime, localDateTimeFormatter );
            }
            catch ( DateTimeParseException dtpe )
            {
                // throw away exception
            }
        }

        return returnLocalDateTime;
    }


    public static ZonedDateTime asZonedDateTime( String aZonedDateTime, String aFormat )
    {
        final String defaultFormat = FORMAT_HL7;
        ZonedDateTime returnZonedDateTime = null;
        DateTimeFormatter dateTimeFormatter = null;

        if ( null != aZonedDateTime )
        {
            if ( null != aFormat )
            {
                dateTimeFormatter = DateTimeFormatter.ofPattern( aFormat );
            }
            else
            {
                dateTimeFormatter = DateTimeFormatter.ofPattern( defaultFormat );
            }

            try
            {
                returnZonedDateTime = ZonedDateTime.parse( aZonedDateTime, dateTimeFormatter );
            }
            catch ( DateTimeParseException dtpe )
            {
                // throw away exception
            }
        }

        return returnZonedDateTime;
    }
}
