

package gov.va.med.cds.audit;


import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.springframework.util.StringUtils;

import gov.va.med.cds.audit.persistence.AuditLogPersistenceManagerInterface;
import gov.va.med.cds.audit.persistence.CdsAuditClobStore;
import gov.va.med.cds.audit.persistence.CdsAuditClobStoreInterface;
import gov.va.med.cds.audit.persistence.PathwaysAuditClobStore;
import gov.va.med.cds.exception.AuditClobStoreException;
import gov.va.med.cds.exception.ErrorCodeEnum;


public class VhimRequestAuditLogger
    implements
        VhimRequestAuditLoggerInterface
{
    private static Log logger = LogFactory.getLog( VhimRequestAuditLogger.class );
    private final String CLIENT_NAME_ELEMENT = "clientName";
    private final String CLIENT_REQUEST_TM_ELEMENT = "clientRequestInitiationTime";
    private ReadAuditFactoryInteface readAuditFactory = null;
    private String cdsAppName;
    private AuditLogPersistenceManagerInterface auditLogPersistenceManager;
    private CdsAuditClobStoreInterface auditClobStore;
    private String sendingFacilityStartElement;
    private String sendingFacilityEndElement;
    private String messageDateTimeStartElement;
    private String messageDateTimeEndElement;
    private List<String> haTemplateIdList;
    private List<String> pathwaysCreateTemplateIdList;
    private List<String> vhimWriteClientSendingFacilityIdList;


    public void auditReadRequest( String aTemplateId, String aFilterRequest, String aFilterId, String aRequestId, long aStartTime,
            String aResponse, String aSamlToken )
    {
        this.auditReadRequest( null, aTemplateId, aFilterRequest, aFilterId, aRequestId, aStartTime, aResponse, aSamlToken );
    }


    public void auditReadRequest( String aClientId, String aTemplateId, String aFilterRequest, String aFilterId, String aRequestId,
            long aStartTime, String aResponse, String aSamlToken )
    {
        AbstractAuditLog auditLog = readAuditFactory.createAuditLog( aClientId, aTemplateId, aFilterRequest, aFilterId, aRequestId, aStartTime,
                aResponse, aSamlToken );

        try
        {
            auditLogPersistenceManager.storeAuditLog( auditLog );
        }
        catch ( Exception e )
        {
            StringBuilder sb = new StringBuilder();
            sb.append( ErrorCodeEnum.CDS_AUDIT_LOG_ERROR.getClientMessage() );
            sb.append( " " );
            sb.append( e.getMessage() );

            if ( logger.isErrorEnabled() )
            {
                logger.error( gov.va.med.cds.util.LogMessageUtil.buildMessage( null, aRequestId, cdsAppName, sb.toString() ) );
            }
            // As per REQ R7.7.2 - Results shall be returned to the client
            // regardless of success or failure of audit.
            // REQ R7.7.3 - In case of failure of audit, an exception shall be
            // logged to CDS_APPLICATION_LOG
            // Exception is swallowed here....
            // throw new CdsAuditLogException(
            // ErrorCodeEnum.CDS_AUDIT_LOG_ERROR, msgString );
        }
    }


    public void auditWriteRequest( String aTemplateId, String aWriteRequest, String aRequestId, String aSamlToken )
    {
        if ( !StringUtils.hasText( aTemplateId ) )
        {
            aTemplateId = "<<templateId>>";
        }

        if ( !StringUtils.hasText( aRequestId ) )
        {
            aRequestId = "<<requestId>>";
        }

        if ( !StringUtils.hasText( aWriteRequest ) )
        {
            aWriteRequest = "<<request>>";
        }

        String sendingFacility = getElementValue( aWriteRequest, sendingFacilityStartElement, sendingFacilityEndElement );
        if ( !StringUtils.hasText( sendingFacility ) )
        {
            sendingFacility = "<<StationNumber>>";
        }

        String messageType = "<<SendingApp>>";
        String updateDateTime = getElementValue( aWriteRequest, messageDateTimeStartElement, messageDateTimeEndElement );
        if ( !StringUtils.hasText( updateDateTime ) )
        {
            updateDateTime = "<<updateDateTime>>";
        }

        Document requestDoc = null;
        String clientName = null;
        String clientRequestInitiationTime = null;
        try
        {
            requestDoc = DocumentHelper.parseText( aWriteRequest );
            clientName = getElementValue( requestDoc, CLIENT_NAME_ELEMENT );
            clientRequestInitiationTime = getElementValue( requestDoc, CLIENT_REQUEST_TM_ELEMENT );
        }
        catch ( Exception e )
        {
            // We still want to persist an audit log even if the request is
            // invalid
        }

        try
        {
            if ( pathwaysCreateTemplateIdList.contains( aTemplateId ) )
            {
                PathwaysAuditClobStore pathwaysAuditClobStore = new PathwaysAuditClobStore( aRequestId, sendingFacility, aWriteRequest,
                        updateDateTime, messageType, aTemplateId, clientName, clientRequestInitiationTime, aSamlToken );
                auditClobStore.persistPathwaysAuditClob( pathwaysAuditClobStore );
            }
            else
            {
                CdsAuditClobStore auditClob = new CdsAuditClobStore( aRequestId, sendingFacility, aWriteRequest, updateDateTime, messageType,
                        aTemplateId, clientName, clientRequestInitiationTime );
                auditClobStore.persistCdsAuditClob( auditClob );
            }
        }
        catch ( Exception e )
        {
            StringBuilder sb = new StringBuilder();
            sb.append( ErrorCodeEnum.CDS_AUDIT_CLOB_STORE_ERROR.getClientMessage() );
            sb.append( " " );
            sb.append( e.getMessage() );

            String msgString = sb.toString();

            if ( logger.isErrorEnabled() )
            {
                logger.error( gov.va.med.cds.util.LogMessageUtil.buildMessage( null, aRequestId, cdsAppName, msgString ) );
            }

            throw new AuditClobStoreException( ErrorCodeEnum.CDS_AUDIT_CLOB_STORE_ERROR, msgString );
        }
    }


    private String getElementValue( String aRequest, String aStartReqEx, String aEndReqEx )
    {
        String returnString = null;

        Pattern pattern = Pattern.compile( aStartReqEx, Pattern.CASE_INSENSITIVE );
        Matcher matcher = pattern.matcher( aRequest );

        int searchPosition = 0;
        if ( matcher.find() )
        {
            searchPosition = matcher.end();
        }

        pattern = Pattern.compile( aEndReqEx, Pattern.CASE_INSENSITIVE );
        matcher = pattern.matcher( aRequest );

        if ( searchPosition > 0 && matcher.find( searchPosition ) )
        {
            returnString = aRequest.substring( searchPosition, matcher.start() );
        }

        return returnString;
    }


    private String getElementValue( Document aDocument, String aElement )
    {
        Element targetElement = ( Element )aDocument.selectSingleNode( String.format( "//%s", aElement ) );
        if ( targetElement != null )
        {
            return targetElement.getText();
        }

        return null;
    }


    public void setAuditLogPersistenceManager( AuditLogPersistenceManagerInterface anAuditLogPersistenceManager )
    {
        this.auditLogPersistenceManager = anAuditLogPersistenceManager;
    }


    public void setCdsAuditClobStore( CdsAuditClobStoreInterface aCdsAuditClobStore )
    {
        this.auditClobStore = aCdsAuditClobStore;
    }


    public void setSendingFacilityStartElement( String aSendingFacilityStartElement )
    {
        this.sendingFacilityStartElement = aSendingFacilityStartElement;
    }


    public void setSendingFacilityEndElement( String aSendingFacilityEndElement )
    {
        this.sendingFacilityEndElement = aSendingFacilityEndElement;
    }


    public void setMessageDateTimeStartElement( String aMessageDateTimeStartElement )
    {
        this.messageDateTimeStartElement = aMessageDateTimeStartElement;
    }


    public void setMessageDateTimeEndElement( String aMessageDateTimeEndElement )
    {
        this.messageDateTimeEndElement = aMessageDateTimeEndElement;
    }


    public void setCdsAppName( String aCdsAppName )
    {
        this.cdsAppName = aCdsAppName;
    }


    public List<String> getHaTemplateIdList( )
    {
        return haTemplateIdList;
    }


    public void setHaTemplateIdList( List<String> aHaTemplateIdList )
    {
        this.haTemplateIdList = aHaTemplateIdList;
    }


    public void setReadAuditFactory( ReadAuditFactoryInteface aReadAuditFactory )
    {
        this.readAuditFactory = aReadAuditFactory;
    }


    public void setVhimWriteClientSendingFacilityIdList( List<String> aVhimWriteClientSendingFacilityIdList )
    {
        this.vhimWriteClientSendingFacilityIdList = aVhimWriteClientSendingFacilityIdList;
    }


    public void setPathwaysCreateTemplateIdList( List<String> aPathwaysCreateTemplateIdList )
    {
        this.pathwaysCreateTemplateIdList = aPathwaysCreateTemplateIdList;
    }

}
