

package gov.va.med.cds.logger;


import gov.va.med.cds.audit.AbstractAuditLog;
import gov.va.med.cds.audit.persistence.AuditLogPersistenceManagerInterface;
import gov.va.med.cds.ejb.interceptor.CdsLoggerAuditAutowiredInterceptor;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.jndi.JndiTemplate;

import javax.annotation.Resource;
import javax.interceptor.Interceptors;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.ObjectMessage;
import javax.naming.NamingException;


/**
 * ReadAuditMessageDrivenBean receives JMS Messages on ReadAuditQueue, and
 * stores those messages in the database. The asynchronous handling of READ
 * Requests audit information is expected to improve overall performance of the
 * system.
 */
@Interceptors( CdsLoggerAuditAutowiredInterceptor.class )
public class ReadAuditMessageDrivenBean
    implements
        MessageListener
{

    protected final Log logger = LogFactory.getLog( getClass() );

    private JndiTemplate jndiTemplate = new JndiTemplate();

    private static final String DEFAULT_AUDIT_LOG_PM_BEAN_ID = "cdsHibernateAuditLogPersistenceManager";

    @Autowired
    protected ApplicationContext appContext;

    @Resource( )
    private String AuditLogPersistenceManagerBeanId;

    private AuditLogPersistenceManagerInterface auditLogPersistenceManager;


    public void ejbCreate( )
    {
        this.auditLogPersistenceManager = ( AuditLogPersistenceManagerInterface )appContext.getBean( AuditLogPersistenceManagerBeanId );

        if ( null == auditLogPersistenceManager )
        {
            this.auditLogPersistenceManager = ( AuditLogPersistenceManagerInterface )appContext.getBean( DEFAULT_AUDIT_LOG_PM_BEAN_ID );
        }
    }


    /***
     * onMessage receives JMS Messages from ReadAuditQueue, and writes audit
     * information to database
     * 
     * @param readRequestAuditInformation
     *            is the audit information created from a READ Request
     */
    public void onMessage( Message readRequestAuditInformation )
    {
        try
        {
            ObjectMessage objectMessage = ( ( ObjectMessage )readRequestAuditInformation );
            auditLogPersistenceManager.storeAuditLog( ( AbstractAuditLog )objectMessage.getObject() );
        }
        catch ( Exception e )
        {
            logger.error( gov.va.med.cds.util.LogMessageUtil.buildMessage( null, null, auditLogPersistenceManager.getAppName(),
                            "Error auditing read request message. " ), e );
        }
    }


    @SuppressWarnings( "unchecked" )
    protected Object safeGetEnvironmentEntry( String name, @SuppressWarnings( "rawtypes" ) Class type, Object defaultValue )
    {
        Object environmentEntry = defaultValue;
        try
        {
            environmentEntry = this.jndiTemplate.lookup( name, type );
        }
        catch ( NamingException ne )
        {
            logger.warn( String.format( "Error initializing CDS VIM MDB. Error Reason: %s", ne.getMessage() ), ne );
        }

        return environmentEntry;
    }
}