

package gov.va.med.cds.exceptionframework;


import gov.va.med.cds.clinicaldata.vhim400.ClinicalDataResponseInterface;
import gov.va.med.cds.clinicaldata.vhim400.PathwaysDataResponseInterface;
import gov.va.med.cds.exception.ErrorCodeEnum;
import gov.va.med.cds.exception.InitializationException;
import gov.va.med.cds.junit.runners.Suite;
import gov.va.med.cds.junit.runners.SuiteAwareRunner;
import gov.va.med.cds.request.ErrorSectionHelperInterface;
import gov.va.med.cds.request.ValidationException;
import gov.va.med.cds.template.TemplateHelperInterface;
import gov.va.med.cds.testharness.vhim400.TemplateIdHelper;

import org.dom4j.Document;
import org.easymock.EasyMock;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.runner.RunWith;

import java.io.IOException;
import java.util.Map;


@RunWith( SuiteAwareRunner.class )
public class ExceptionHandlerTest
{
    String EMPTY_CLINICAL_DATA = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><clinicaldata:ClinicalData xmlns:clinicaldata=\"Clinicaldata\"><templateId>AllergiesRead40010</templateId><requestId>aRequestId</requestId></clinicaldata:ClinicalData>";
    private static ClinicalDataResponseInterface dataResponse = null;
    private static ExceptionLoggerInterface guaranteedLoger = null;
    private static TemplateHelperInterface templateHelper = null;
    private static final String CDS_APP_NAME = "CDS 3.6";
    private static final String PATHWAYS_APP_NAME = "Pathways 1.0";


    // Static ExceptionHandler may be initialized in previous tests and stay initialized through the scope of running all suite of core tests.
    // Before test set up mocks on exception handler store objects that it was initialized with - whatever they may be.
    @BeforeClass
    static public void beforeExceptionHandlerTest( )
    {
        dataResponse = ExceptionHandler.getClinicalDataResponse();
        guaranteedLoger = ExceptionHandler.getGuaranteedLogger();
        templateHelper = ExceptionHandler.getTemplateHelper();
    }


    // After test run set exception handler original objects back.
    @AfterClass
    static public void afterExceptionHandlerTest( )
    {
        ExceptionHandler.setClinicalDataResponse( dataResponse );
        ExceptionHandler.setGuaranteedLogger( guaranteedLoger );
        ExceptionHandler.setTemplateHelper( templateHelper );
    }


    @SuppressWarnings( "unchecked" )
    @Test
    @Suite( groups = "checkintest" )
    public void testHandleExceptioWithFilterRequest( )
        throws Exception
    {
        //Create Mocks
        ExceptionLoggerInterface guaranteedLoggerMock = EasyMock.createNiceMock( ExceptionLoggerInterface.class );
        ErrorSectionHelperInterface errorSectionHelperMock = EasyMock.createMock( ErrorSectionHelperInterface.class );
        TemplateHelperInterface templateHelperMock = EasyMock.createMock( TemplateHelperInterface.class );
        ClinicalDataResponseInterface clinicalDataResponseMock = EasyMock.createMock( ClinicalDataResponseInterface.class );
        Document clinicalDataDocumentMock = EasyMock.createMock( Document.class );
        Map<String, PathwaysDataResponseInterface> configurableResponseMapMock = EasyMock.createNiceMock( Map.class );

        ExceptionHandler.setClinicalDataResponse( clinicalDataResponseMock );
        ExceptionHandler.setTemplateHelper( templateHelperMock );
        ExceptionHandler.setGuaranteedLogger( guaranteedLoggerMock );
        ExceptionHandler.setCdsAppName( CDS_APP_NAME );
        ExceptionHandler.setPathwaysConfigurableResponseMap( configurableResponseMapMock );

        Exception exception = new ValidationException( ErrorCodeEnum.CANNOT_FIND_FILTER_VALIDATOR );
        String templateId = TemplateIdHelper.ALLERGIES_READ_TEMPLATE_ID;

        // Set expectations
        EasyMock.expect( clinicalDataResponseMock.buildEmptyClinicalDocumentWithoutPatient( templateId, "requestId" ) ).andReturn(
                        clinicalDataDocumentMock );
//        EasyMock.expect(
//                        guaranteedLoggerMock.log( exception, LoggingSeverity.FATAL, ErrorCodeEnum.CANNOT_FIND_FILTER_VALIDATOR.getClientMessage(),
//                                        templateId, "requestId", CDS_APP_NAME ) ).andReturn( true );
        EasyMock.expect( templateHelperMock.getErrorSectionHelper( templateId ) ).andReturn( errorSectionHelperMock );
        EasyMock.expect(
                        errorSectionHelperMock.buildErrorSection( EasyMock.isA( Document.class ), EasyMock.isA( ExceptionInfo.class ),
                                        EasyMock.eq( "requestId" ) ) ).andReturn( clinicalDataDocumentMock );

        EasyMock.replay( guaranteedLoggerMock, errorSectionHelperMock, clinicalDataResponseMock, templateHelperMock, clinicalDataDocumentMock,
                        configurableResponseMapMock );

        // Call the method
        ExceptionHandler.handleException( exception, templateId, "filterRequest", "filterId", "requestId", CDS_APP_NAME );

        EasyMock.verify( guaranteedLoggerMock, errorSectionHelperMock, clinicalDataResponseMock, templateHelperMock, clinicalDataDocumentMock );
    }


    @SuppressWarnings( "unchecked" )
    @Test
    @Suite( groups = "checkintest" )
    public void testHandleExceptionWithEmptyRequest( )
        throws Exception
    {
        //Create Mocks
        ExceptionLoggerInterface guaranteedLoggerMock = EasyMock.createNiceMock( ExceptionLoggerInterface.class );
        ErrorSectionHelperInterface errorSectionHelperMock = EasyMock.createMock( ErrorSectionHelperInterface.class );
        TemplateHelperInterface templateHelperMock = EasyMock.createMock( TemplateHelperInterface.class );
        ClinicalDataResponseInterface clinicalDataResponseMock = EasyMock.createMock( ClinicalDataResponseInterface.class );
        Document clinicalDataDocumentMock = EasyMock.createMock( Document.class );
        Map<String, PathwaysDataResponseInterface> configurableResponseMapMock = EasyMock.createNiceMock( Map.class );

        ExceptionHandler.setClinicalDataResponse( clinicalDataResponseMock );
        ExceptionHandler.setTemplateHelper( templateHelperMock );
        ExceptionHandler.setGuaranteedLogger( guaranteedLoggerMock );
        ExceptionHandler.setCdsAppName( CDS_APP_NAME );
        ExceptionHandler.setPathwaysConfigurableResponseMap( configurableResponseMapMock );

        Exception exception = new ValidationException( ErrorCodeEnum.READ_REQUEST_INPUT_PARAMETERS_NULL );
        String templateId = "CdsResponse400";
        String requestId = "request_id_not_set";

        // Set expectations
        EasyMock.expect( clinicalDataResponseMock.buildEmptyClinicalDocumentWithoutPatient( templateId, requestId ) ).andReturn(
                        clinicalDataDocumentMock );
//        EasyMock.expect(
//                        guaranteedLoggerMock.log( exception, LoggingSeverity.FATAL,
//                                        ErrorCodeEnum.READ_REQUEST_INPUT_PARAMETERS_NULL.getClientMessage(), templateId, requestId, CDS_APP_NAME ) )
//                        .andReturn( true );
        EasyMock.expect( templateHelperMock.getErrorSectionHelper( templateId ) ).andReturn( errorSectionHelperMock );
        EasyMock.expect(
                        errorSectionHelperMock.buildErrorSection( EasyMock.isA( Document.class ), EasyMock.isA( ExceptionInfo.class ),
                                        EasyMock.eq( requestId ) ) ).andReturn( null );

        EasyMock.replay( guaranteedLoggerMock, errorSectionHelperMock, clinicalDataResponseMock, templateHelperMock, clinicalDataDocumentMock,
                        configurableResponseMapMock );

        // Call the method
        ExceptionHandler.handleException( exception, "", "", "", "", CDS_APP_NAME );

        EasyMock.verify( guaranteedLoggerMock, errorSectionHelperMock, clinicalDataResponseMock, templateHelperMock, clinicalDataDocumentMock );
    }


    @SuppressWarnings( "unchecked" )
    @Test
    @Suite( groups = "checkintest" )
    public void testHandleExceptioWithEmptyRequestPathways( )
        throws Exception
    {
        //Create Mocks
        ExceptionLoggerInterface guaranteedLoggerMock = EasyMock.createNiceMock( ExceptionLoggerInterface.class );
        ErrorSectionHelperInterface errorSectionHelperMock = EasyMock.createMock( ErrorSectionHelperInterface.class );
        TemplateHelperInterface templateHelperMock = EasyMock.createMock( TemplateHelperInterface.class );
        Document clinicalDataDocumentMock = EasyMock.createMock( Document.class );
        PathwaysDataResponseInterface pathwaysDataResponseMock = EasyMock.createNiceMock( PathwaysDataResponseInterface.class );
        Map<String, PathwaysDataResponseInterface> configurableResponseMapMock = EasyMock.createNiceMock( Map.class );

        ExceptionHandler.setClinicalDataResponse( pathwaysDataResponseMock );
        ExceptionHandler.setTemplateHelper( templateHelperMock );
        ExceptionHandler.setGuaranteedLogger( guaranteedLoggerMock );
        ExceptionHandler.setPathwaysAppName( PATHWAYS_APP_NAME );
        ExceptionHandler.setPathwaysConfigurableResponseMap( configurableResponseMapMock );
        ExceptionHandler.setPathwaysDataResponse( pathwaysDataResponseMock );

        Exception exception = new ValidationException( ErrorCodeEnum.READ_REQUEST_INPUT_PARAMETERS_NULL );
        String templateId = "PathwaysResponse400";
        String requestId = "request_id_not_set";

        // Set expectations
        EasyMock.expect( pathwaysDataResponseMock.buildEmptyPathwaysDocumentWithoutPatient( templateId, requestId ) ).andReturn(
                        clinicalDataDocumentMock );
//        EasyMock.expect(
//                        guaranteedLoggerMock.log( exception, LoggingSeverity.FATAL,
//                                        ErrorCodeEnum.READ_REQUEST_INPUT_PARAMETERS_NULL.getClientMessage(), templateId, requestId, PATHWAYS_APP_NAME ) )
//                        .andReturn( true );
        EasyMock.expect( templateHelperMock.getErrorSectionHelper( templateId ) ).andReturn( errorSectionHelperMock );
        EasyMock.expect(
                        errorSectionHelperMock.buildErrorSection( EasyMock.isA( Document.class ), EasyMock.isA( ExceptionInfo.class ),
                                        EasyMock.eq( requestId ) ) ).andReturn( null );

        EasyMock.replay( guaranteedLoggerMock, errorSectionHelperMock, pathwaysDataResponseMock, templateHelperMock, clinicalDataDocumentMock,
                        configurableResponseMapMock );

        // Call the method
        ExceptionHandler.handleException( exception, "", "", "", "", PATHWAYS_APP_NAME );

        EasyMock.verify( guaranteedLoggerMock, errorSectionHelperMock, pathwaysDataResponseMock, templateHelperMock, clinicalDataDocumentMock );
    }

    @SuppressWarnings( "unchecked" )
    @Test
    @Suite( groups = "checkintest" )
    public void testHandleExceptioWithEmptyFilterIdPathways( )
        throws Exception
    {
        //Create Mocks
        ExceptionLoggerInterface guaranteedLoggerMock = EasyMock.createNiceMock( ExceptionLoggerInterface.class );
        ErrorSectionHelperInterface errorSectionHelperMock = EasyMock.createMock( ErrorSectionHelperInterface.class );
        TemplateHelperInterface templateHelperMock = EasyMock.createMock( TemplateHelperInterface.class );
        Document clinicalDataDocumentMock = EasyMock.createMock( Document.class );
        PathwaysDataResponseInterface pathwaysDataResponseMock = EasyMock.createNiceMock( PathwaysDataResponseInterface.class );
        Map<String, PathwaysDataResponseInterface> configurableResponseMapMock = EasyMock.createNiceMock( Map.class );

        ExceptionHandler.setClinicalDataResponse( pathwaysDataResponseMock );
        ExceptionHandler.setTemplateHelper( templateHelperMock );
        ExceptionHandler.setGuaranteedLogger( guaranteedLoggerMock );
        ExceptionHandler.setPathwaysAppName( PATHWAYS_APP_NAME );
        ExceptionHandler.setPathwaysConfigurableResponseMap( configurableResponseMapMock );
        ExceptionHandler.setPathwaysDataResponse( pathwaysDataResponseMock );

        Exception exception = new ValidationException( ErrorCodeEnum.READ_REQUEST_INPUT_PARAMETERS_NULL );
        String templateId = "PathwaysResponse400";
        String requestId = "request_id_not_set";

        // Set expectations
        EasyMock.expect( pathwaysDataResponseMock.buildEmptyPathwaysDocumentWithoutPatient( templateId, requestId ) ).andReturn(
                        clinicalDataDocumentMock );
//        EasyMock.expect(
//                        guaranteedLoggerMock.log( exception, LoggingSeverity.FATAL,
//                                        ErrorCodeEnum.READ_REQUEST_INPUT_PARAMETERS_NULL.getClientMessage(), templateId, requestId, PATHWAYS_APP_NAME ) )
//                        .andReturn( true );
        EasyMock.expect( templateHelperMock.getErrorSectionHelper( templateId ) ).andReturn( errorSectionHelperMock );
        EasyMock.expect(
                        errorSectionHelperMock.buildErrorSection( EasyMock.isA( Document.class ), EasyMock.isA( ExceptionInfo.class ),
                                        EasyMock.eq( requestId ) ) ).andReturn( null );

        EasyMock.replay( guaranteedLoggerMock, errorSectionHelperMock, pathwaysDataResponseMock, templateHelperMock, clinicalDataDocumentMock,
                        configurableResponseMapMock );

        // Call the method
        ExceptionHandler.handleException( exception, "", "FilterXml", "", "", PATHWAYS_APP_NAME );

        EasyMock.verify( guaranteedLoggerMock, errorSectionHelperMock, pathwaysDataResponseMock, templateHelperMock, clinicalDataDocumentMock );
    }
    
    @SuppressWarnings( "unchecked" )
    @Test
    @Suite( groups = "brokentest" )
    public void testHandleException_AbstractCdsBaseException( )
        throws Exception
    {
        //Create Mocks
        ExceptionLoggerInterface guaranteedLoggerMock = EasyMock.createNiceMock( ExceptionLoggerInterface.class );
        ErrorSectionHelperInterface errorSectionHelperMock = EasyMock.createMock( ErrorSectionHelperInterface.class );
        TemplateHelperInterface templateHelperMock = EasyMock.createMock( TemplateHelperInterface.class );
        ClinicalDataResponseInterface clinicalDataResponseMock = EasyMock.createMock( ClinicalDataResponseInterface.class );
        Document clinicalDataDocumentMock = EasyMock.createMock( Document.class );
        Map<String, PathwaysDataResponseInterface> configurableResponseMapMock = EasyMock.createNiceMock( Map.class );

        ExceptionHandler.setClinicalDataResponse( clinicalDataResponseMock );
        ExceptionHandler.setTemplateHelper( templateHelperMock );
        ExceptionHandler.setGuaranteedLogger( guaranteedLoggerMock );
        ExceptionHandler.setCdsAppName( CDS_APP_NAME );
        ExceptionHandler.setPathwaysConfigurableResponseMap( configurableResponseMapMock );

        Exception exception = new ValidationException( ErrorCodeEnum.CANNOT_FIND_FILTER_VALIDATOR );
        String templateId = TemplateIdHelper.ALLERGIES_READ_TEMPLATE_ID;

        // Set expectations
        EasyMock.expect( clinicalDataResponseMock.buildEmptyClinicalDocumentWithoutPatient( templateId, "requestId" ) ).andReturn(
                        clinicalDataDocumentMock );
//        EasyMock.expect(
//                        guaranteedLoggerMock.log( exception, LoggingSeverity.FATAL, ErrorCodeEnum.CANNOT_FIND_FILTER_VALIDATOR.getClientMessage(),
//                                        templateId, "requestId", CDS_APP_NAME ) ).andReturn( true );
        EasyMock.expect( templateHelperMock.getErrorSectionHelper( templateId ) ).andReturn( errorSectionHelperMock );
        EasyMock.expect(
                        errorSectionHelperMock.buildErrorSection( EasyMock.isA( Document.class ), EasyMock.isA( ExceptionInfo.class ),
                                        EasyMock.eq( "requestId" ) ) ).andReturn( clinicalDataDocumentMock );
       
        EasyMock.replay( guaranteedLoggerMock, errorSectionHelperMock, clinicalDataResponseMock, templateHelperMock, clinicalDataDocumentMock,
                        configurableResponseMapMock );

        // Call the method
        ExceptionHandler.handleException( exception, templateId, "requestId", CDS_APP_NAME );

        EasyMock.verify( guaranteedLoggerMock, errorSectionHelperMock, clinicalDataResponseMock, templateHelperMock, clinicalDataDocumentMock,
                        configurableResponseMapMock );
    }


    @SuppressWarnings( "unchecked" )
    @Test
    @Suite( groups = "checkintest" )
    public void testHandleExceptionWithPathways_AbstractCdsBaseException( )
        throws Exception
    {
        //Create Mocks
        ExceptionLoggerInterface guaranteedLoggerMock = EasyMock.createNiceMock( ExceptionLoggerInterface.class );
        ErrorSectionHelperInterface errorSectionHelperMock = EasyMock.createMock( ErrorSectionHelperInterface.class );
        TemplateHelperInterface templateHelperMock = EasyMock.createMock( TemplateHelperInterface.class );
        PathwaysDataResponseInterface pathwaysDataResponseMock = EasyMock.createMock( PathwaysDataResponseInterface.class );
        Document clinicalDataDocumentMock = EasyMock.createMock( Document.class );
        Map<String, PathwaysDataResponseInterface> configurableResponseMapMock = EasyMock.createNiceMock( Map.class );

        ExceptionHandler.setClinicalDataResponse( pathwaysDataResponseMock );
        ExceptionHandler.setTemplateHelper( templateHelperMock );
        ExceptionHandler.setGuaranteedLogger( guaranteedLoggerMock );
        ExceptionHandler.setPathwaysAppName( PATHWAYS_APP_NAME );
        ExceptionHandler.setPathwaysConfigurableResponseMap( configurableResponseMapMock );
        ExceptionHandler.setPathwaysDataResponse( pathwaysDataResponseMock );
        

        Exception exception = new ValidationException( ErrorCodeEnum.CANNOT_FIND_FILTER_VALIDATOR );
        String templateId = TemplateIdHelper.APPOINTMENTS_READ_TEMPLATE_ID;

        // Set expectations
        EasyMock.expect( pathwaysDataResponseMock.buildEmptyPathwaysDocumentWithoutPatient( templateId, "requestId" ) ).andReturn(
                        clinicalDataDocumentMock );
//        EasyMock.expect(
//                        guaranteedLoggerMock.log( exception, LoggingSeverity.FATAL, ErrorCodeEnum.CANNOT_FIND_FILTER_VALIDATOR.getClientMessage(),
//                                        templateId, "requestId", PATHWAYS_APP_NAME ) ).andReturn( true );
        EasyMock.expect( templateHelperMock.getErrorSectionHelper( templateId ) ).andReturn( errorSectionHelperMock );
        EasyMock.expect(
                        errorSectionHelperMock.buildErrorSection( EasyMock.isA( Document.class ), EasyMock.isA( ExceptionInfo.class ),
                                        EasyMock.eq( "requestId" ) ) ).andReturn( clinicalDataDocumentMock );
        
        EasyMock.replay( guaranteedLoggerMock, errorSectionHelperMock, pathwaysDataResponseMock, templateHelperMock, clinicalDataDocumentMock,
                        configurableResponseMapMock );

        // Call the method
        ExceptionHandler.setPathwaysAppName( PATHWAYS_APP_NAME );
        ExceptionHandler.handleException( exception, templateId, "requestId", PATHWAYS_APP_NAME );

        EasyMock.verify( guaranteedLoggerMock, errorSectionHelperMock, pathwaysDataResponseMock, templateHelperMock, clinicalDataDocumentMock );
    }


    @SuppressWarnings( "unchecked" )
    @Test
    @Suite( groups = "checkintest" )
    public void testHandleException_Pathways( )
        throws Exception
    {
        //Create Mocks
        ExceptionLoggerInterface guaranteedLoggerMock = EasyMock.createNiceMock( ExceptionLoggerInterface.class );
        ErrorSectionHelperInterface errorSectionHelperMock = EasyMock.createMock( ErrorSectionHelperInterface.class );
        TemplateHelperInterface templateHelperMock = EasyMock.createMock( TemplateHelperInterface.class );
        PathwaysDataResponseInterface pathwaysDataResponseMock = EasyMock.createMock( PathwaysDataResponseInterface.class );
        Document clinicalDataDocumentMock = EasyMock.createMock( Document.class );
        Map<String, PathwaysDataResponseInterface> configurableResponseMapMock = EasyMock.createNiceMock( Map.class );

        ExceptionHandler.setClinicalDataResponse( pathwaysDataResponseMock );
        ExceptionHandler.setTemplateHelper( templateHelperMock );
        ExceptionHandler.setGuaranteedLogger( guaranteedLoggerMock );
        ExceptionHandler.setPathwaysAppName( PATHWAYS_APP_NAME );
        ExceptionHandler.setPathwaysConfigurableResponseMap( configurableResponseMapMock );
        ExceptionHandler.setPathwaysDataResponse( pathwaysDataResponseMock );

        Exception exception = new ValidationException( ErrorCodeEnum.CANNOT_FIND_FILTER_VALIDATOR );
        String templateId = TemplateIdHelper.ALLERGIES_READ_TEMPLATE_ID;

        // Set expectations
        EasyMock.expect( pathwaysDataResponseMock.buildEmptyPathwaysDocumentWithoutPatient( templateId, "requestId" ) ).andReturn(
                        clinicalDataDocumentMock );
//        EasyMock.expect(
//                        guaranteedLoggerMock.log( exception, LoggingSeverity.FATAL, ErrorCodeEnum.CANNOT_FIND_FILTER_VALIDATOR.getClientMessage(),
//                                        templateId, "requestId", PATHWAYS_APP_NAME ) ).andReturn( true );
        EasyMock.expect( templateHelperMock.getErrorSectionHelper( templateId ) ).andReturn( errorSectionHelperMock );
        EasyMock.expect(
                        errorSectionHelperMock.buildErrorSection( EasyMock.isA( Document.class ), EasyMock.isA( ExceptionInfo.class ),
                                        EasyMock.eq( "requestId" ) ) ).andReturn( clinicalDataDocumentMock );

        EasyMock.replay( guaranteedLoggerMock, errorSectionHelperMock, pathwaysDataResponseMock, templateHelperMock, clinicalDataDocumentMock,
                        configurableResponseMapMock );

        // Call the method
        ExceptionHandler.handleException( exception, templateId, "requestId", PATHWAYS_APP_NAME );

        EasyMock.verify( guaranteedLoggerMock, errorSectionHelperMock, pathwaysDataResponseMock, templateHelperMock, clinicalDataDocumentMock );
    }


    @SuppressWarnings( "unchecked" )
    @Test
    @Suite( groups = "checkintest" )
    public void testHandleException_AbstractCdsBaseException_InputParamsNull( )
        throws Exception
    {
        //Create Mocks
        ExceptionLoggerInterface guaranteedLoggerMock = EasyMock.createNiceMock( ExceptionLoggerInterface.class );
        ErrorSectionHelperInterface errorSectionHelperMock = EasyMock.createMock( ErrorSectionHelperInterface.class );
        TemplateHelperInterface templateHelperMock = EasyMock.createMock( TemplateHelperInterface.class );
        ClinicalDataResponseInterface clinicalDataResponseMock = EasyMock.createMock( ClinicalDataResponseInterface.class );
        Document clinicalDataDocumentMock = EasyMock.createMock( Document.class );
        Map<String, PathwaysDataResponseInterface> configurableResponseMapMock = EasyMock.createNiceMock( Map.class );

        ExceptionHandler.setClinicalDataResponse( clinicalDataResponseMock );
        ExceptionHandler.setTemplateHelper( templateHelperMock );
        ExceptionHandler.setGuaranteedLogger( guaranteedLoggerMock );
        ExceptionHandler.setCdsAppName( CDS_APP_NAME );
        ExceptionHandler.setPathwaysConfigurableResponseMap( configurableResponseMapMock );

        Exception exception = new ValidationException( ErrorCodeEnum.READ_REQUEST_INPUT_PARAMETERS_NULL );
        String templateId = TemplateIdHelper.ALLERGIES_READ_TEMPLATE_ID;

        // Set expectations
        EasyMock.expect( clinicalDataResponseMock.buildEmptyClinicalDocumentWithoutPatient( templateId, "requestId" ) ).andReturn(
                        clinicalDataDocumentMock );
//        EasyMock.expect(
//                        guaranteedLoggerMock.log( exception, LoggingSeverity.FATAL,
//                                        ErrorCodeEnum.READ_REQUEST_INPUT_PARAMETERS_NULL.getClientMessage(), templateId, "requestId", CDS_APP_NAME ) )
//                        .andReturn( true );
        EasyMock.expect( templateHelperMock.getErrorSectionHelper( templateId ) ).andReturn( errorSectionHelperMock );
        EasyMock.expect(
                        errorSectionHelperMock.buildErrorSection( EasyMock.isA( Document.class ), EasyMock.isA( ExceptionInfo.class ),
                                        EasyMock.eq( "requestId" ) ) ).andReturn( null );

        EasyMock.replay( guaranteedLoggerMock, errorSectionHelperMock, clinicalDataResponseMock, templateHelperMock, clinicalDataDocumentMock,
                        configurableResponseMapMock );

        // Call the method
        ExceptionHandler.handleException( exception, templateId, "requestId", CDS_APP_NAME );

        EasyMock.verify( guaranteedLoggerMock, errorSectionHelperMock, clinicalDataResponseMock, templateHelperMock, clinicalDataDocumentMock );
    }


    @SuppressWarnings( "unchecked" )
    @Test
    @Suite( groups = "brokentest" )
    public void testHandleExceptionPathways_AbstractCdsBaseException_InputParamsNull( )
        throws Exception
    {
        //Create Mocks
        ExceptionLoggerInterface guaranteedLoggerMock = EasyMock.createNiceMock( ExceptionLoggerInterface.class );
        ErrorSectionHelperInterface errorSectionHelperMock = EasyMock.createMock( ErrorSectionHelperInterface.class );
        TemplateHelperInterface templateHelperMock = EasyMock.createMock( TemplateHelperInterface.class );
        PathwaysDataResponseInterface pathwaysDataResponseMock = EasyMock.createMock( PathwaysDataResponseInterface.class );
        Document clinicalDataDocumentMock = EasyMock.createMock( Document.class );
        Map<String, PathwaysDataResponseInterface> configurableResponseMapMock = EasyMock.createNiceMock( Map.class );

        ExceptionHandler.setClinicalDataResponse( pathwaysDataResponseMock );
        ExceptionHandler.setTemplateHelper( templateHelperMock );
        ExceptionHandler.setGuaranteedLogger( guaranteedLoggerMock );
        ExceptionHandler.setPathwaysAppName( PATHWAYS_APP_NAME );
        ExceptionHandler.setPathwaysConfigurableResponseMap( configurableResponseMapMock );
        ExceptionHandler.setPathwaysDataResponse( pathwaysDataResponseMock );

        Exception exception = new ValidationException( ErrorCodeEnum.READ_REQUEST_INPUT_PARAMETERS_NULL );
        String templateId = TemplateIdHelper.APPOINTMENTS_READ_TEMPLATE_ID;

        // Set expectations
        EasyMock.expect( pathwaysDataResponseMock.buildEmptyPathwaysDocumentWithPatient( templateId, "requestId" ) ).andReturn( clinicalDataDocumentMock );
//        EasyMock.expect(
//                        guaranteedLoggerMock.log( exception, LoggingSeverity.FATAL,
//                                        ErrorCodeEnum.READ_REQUEST_INPUT_PARAMETERS_NULL.getClientMessage(), templateId, "requestId",
//                                        PATHWAYS_APP_NAME ) ).andReturn( true );
        EasyMock.expect( templateHelperMock.getErrorSectionHelper( templateId ) ).andReturn( errorSectionHelperMock );
        EasyMock.expect(
                        errorSectionHelperMock.buildErrorSection( EasyMock.isA( Document.class ), EasyMock.isA( ExceptionInfo.class ),
                                        EasyMock.eq( "requestId" ) ) ).andReturn( clinicalDataDocumentMock );

        EasyMock.replay( guaranteedLoggerMock, errorSectionHelperMock, pathwaysDataResponseMock, templateHelperMock, clinicalDataDocumentMock,
                        configurableResponseMapMock );

        // Call the method
        ExceptionHandler.setPathwaysAppName( PATHWAYS_APP_NAME );
        ExceptionHandler.handleException( exception, templateId, "requestXML", "filterId", "requestId", PATHWAYS_APP_NAME );

        EasyMock.verify( guaranteedLoggerMock, errorSectionHelperMock, pathwaysDataResponseMock, templateHelperMock, clinicalDataDocumentMock );
    }


    @SuppressWarnings( "unchecked" )
    @Test
    @Suite( groups = "brokentest" )
    public void testHandleExceptionPathways_AbstractCdsBaseException_InputParamsNull_DefaultTemplateId( )
        throws Exception
    {
        //Create Mocks
        ExceptionLoggerInterface guaranteedLoggerMock = EasyMock.createNiceMock( ExceptionLoggerInterface.class );
        ErrorSectionHelperInterface errorSectionHelperMock = EasyMock.createMock( ErrorSectionHelperInterface.class );
        TemplateHelperInterface templateHelperMock = EasyMock.createMock( TemplateHelperInterface.class );
        PathwaysDataResponseInterface pathwaysDataResponseMock = EasyMock.createMock( PathwaysDataResponseInterface.class );
        Document clinicalDataDocumentMock = EasyMock.createMock( Document.class );
        Map<String, PathwaysDataResponseInterface> configurableResponseMapMock = EasyMock.createNiceMock( Map.class );

        ExceptionHandler.setClinicalDataResponse( pathwaysDataResponseMock );
        ExceptionHandler.setTemplateHelper( templateHelperMock );
        ExceptionHandler.setGuaranteedLogger( guaranteedLoggerMock );
        ExceptionHandler.setPathwaysAppName( PATHWAYS_APP_NAME );
        ExceptionHandler.setPathwaysConfigurableResponseMap( configurableResponseMapMock );
        ExceptionHandler.setPathwaysDataResponse( pathwaysDataResponseMock );

        Exception exception = new ValidationException( ErrorCodeEnum.READ_REQUEST_INPUT_PARAMETERS_NULL );
        String templateId = TemplateIdHelper.PATHWAYS_RESPONSE_ID;

        // Set expectations
        EasyMock.expect( pathwaysDataResponseMock.buildEmptyPathwaysDocumentWithoutPatient( templateId, "requestId" ) ).andReturn(
                        clinicalDataDocumentMock );
//        EasyMock.expect(
//                        guaranteedLoggerMock.log( exception, LoggingSeverity.FATAL,
//                                        ErrorCodeEnum.READ_REQUEST_INPUT_PARAMETERS_NULL.getClientMessage(), templateId, "requestId",
//                                        PATHWAYS_APP_NAME ) ).andReturn( true );
        EasyMock.expect( templateHelperMock.getErrorSectionHelper( templateId ) ).andReturn( errorSectionHelperMock );
        EasyMock.expect(
                        errorSectionHelperMock.buildErrorSection( EasyMock.isA( Document.class ), EasyMock.isA( ExceptionInfo.class ),
                                        EasyMock.eq( "requestId" ) ) ).andReturn( clinicalDataDocumentMock );

        EasyMock.replay( guaranteedLoggerMock, errorSectionHelperMock, pathwaysDataResponseMock, templateHelperMock, clinicalDataDocumentMock,
                        configurableResponseMapMock );

        // Call the method
        ExceptionHandler.setPathwaysAppName( PATHWAYS_APP_NAME );
        ExceptionHandler.handleException( exception, templateId, "requestXML", "filterId", "requestId", PATHWAYS_APP_NAME );

        EasyMock.verify( guaranteedLoggerMock, errorSectionHelperMock, pathwaysDataResponseMock, templateHelperMock, clinicalDataDocumentMock );
    }


    @SuppressWarnings( "static-access" )
    @Test( expected = InitializationException.class )
    @Suite( groups = "checkintest" )
    public void testGuaranteedLogger_IsNull( )
        throws Exception
    {
        //Create Mocks
        TemplateHelperInterface templateHelperMock = EasyMock.createMock( TemplateHelperInterface.class );
        ClinicalDataResponseInterface clinicalDataResponseMock = EasyMock.createMock( ClinicalDataResponseInterface.class );

        ExceptionHandler exceptionHandler = new ExceptionHandler();
        exceptionHandler.setClinicalDataResponse( clinicalDataResponseMock );
        exceptionHandler.setTemplateHelper( templateHelperMock );
        exceptionHandler.setGuaranteedLogger( null );

        exceptionHandler.afterPropertiesSet();
    }


    @SuppressWarnings( "static-access" )
    @Test( expected = InitializationException.class )
    @Suite( groups = "checkintest" )
    public void testTemplateHelper_IsNull( )
        throws Exception
    {
        //Create Mocks
        ClinicalDataResponseInterface clinicalDataResponseMock = EasyMock.createMock( ClinicalDataResponseInterface.class );
        ExceptionLoggerInterface guaranteedLoggerMock = EasyMock.createMock( ExceptionLoggerInterface.class );

        ExceptionHandler exceptionHandler = new ExceptionHandler();
        exceptionHandler.setClinicalDataResponse( clinicalDataResponseMock );
        exceptionHandler.setTemplateHelper( null );
        exceptionHandler.setGuaranteedLogger( guaranteedLoggerMock );

        exceptionHandler.afterPropertiesSet();
    }


    @SuppressWarnings( "static-access" )
    @Test( expected = InitializationException.class )
    @Suite( groups = "checkintest" )
    public void testClinicalDataResponse_IsNull( )
        throws Exception
    {
        //Create Mocks
        TemplateHelperInterface templateHelperMock = EasyMock.createMock( TemplateHelperInterface.class );
        ExceptionLoggerInterface guaranteedLoggerMock = EasyMock.createMock( ExceptionLoggerInterface.class );
        Map<String, PathwaysDataResponseInterface> configurableResponseMapMock = EasyMock.createNiceMock( Map.class );

        ExceptionHandler exceptionHandler = new ExceptionHandler();
        exceptionHandler.setClinicalDataResponse( null );
        exceptionHandler.setTemplateHelper( templateHelperMock );
        exceptionHandler.setGuaranteedLogger( guaranteedLoggerMock );
        ExceptionHandler.setPathwaysConfigurableResponseMap( configurableResponseMapMock );

        exceptionHandler.afterPropertiesSet();
    }


    @Test
    @Suite( groups = "checkintest" )
    public void testHandleException_NonAbstractCdsBaseException( )
        throws Exception
    {
        //Create Mocks
        ExceptionLoggerInterface guaranteedLoggerMock = EasyMock.createNiceMock( ExceptionLoggerInterface.class );
        ErrorSectionHelperInterface errorSectionHelperMock = EasyMock.createMock( ErrorSectionHelperInterface.class );
        TemplateHelperInterface templateHelperMock = EasyMock.createMock( TemplateHelperInterface.class );
        ClinicalDataResponseInterface clinicalDataResponseMock = EasyMock.createMock( ClinicalDataResponseInterface.class );
        Document clinicalDataDocumentMock = EasyMock.createMock( Document.class );
        Map<String, PathwaysDataResponseInterface> configurableResponseMapMock = EasyMock.createNiceMock( Map.class );

        ExceptionHandler.setClinicalDataResponse( clinicalDataResponseMock );
        ExceptionHandler.setTemplateHelper( templateHelperMock );
        ExceptionHandler.setGuaranteedLogger( guaranteedLoggerMock );
        ExceptionHandler.setPathwaysConfigurableResponseMap( configurableResponseMapMock );

        Exception exception = new IOException();
        String templateId = TemplateIdHelper.ALLERGIES_READ_TEMPLATE_ID;

        // Set expectations
        EasyMock.expect( clinicalDataResponseMock.buildEmptyClinicalDocumentWithoutPatient( templateId, "requestId" ) ).andReturn(
                        clinicalDataDocumentMock );
//        EasyMock.expect(
//                        guaranteedLoggerMock.log( EasyMock.isA( SystemException.class ), EasyMock.eq( LoggingSeverity.FATAL ),
//                                        EasyMock.isA( String.class ), EasyMock.eq( templateId ), EasyMock.eq( "requestId" ),
//                                        EasyMock.eq( CDS_APP_NAME ) ) ).andReturn( true );
        EasyMock.expect( templateHelperMock.getErrorSectionHelper( templateId ) ).andReturn( errorSectionHelperMock );
        EasyMock.expect(
                        errorSectionHelperMock.buildErrorSection( EasyMock.isA( Document.class ), EasyMock.isA( ExceptionInfo.class ),
                                        EasyMock.eq( "requestId" ) ) ).andReturn( null ).anyTimes();

        EasyMock.replay( guaranteedLoggerMock, errorSectionHelperMock, clinicalDataResponseMock, templateHelperMock, clinicalDataDocumentMock );

        // Call the method
        ExceptionHandler.setCdsAppName( CDS_APP_NAME );
        ExceptionHandler.handleException( exception, templateId, "requestId", CDS_APP_NAME );

        EasyMock.verify( guaranteedLoggerMock, errorSectionHelperMock, clinicalDataResponseMock, templateHelperMock, clinicalDataDocumentMock );
    }


    @Test
    @Suite( groups = "checkintest" )
    public void testErrorCodeEnumLoggerThrowsRunTimeException( )
        throws Exception
    {
        //Create Mocks
        ExceptionLoggerInterface guaranteedLoggerMock = EasyMock.createNiceMock( ExceptionLoggerInterface.class );
        ErrorSectionHelperInterface errorSectionHelperMock = EasyMock.createMock( ErrorSectionHelperInterface.class );
        TemplateHelperInterface templateHelperMock = EasyMock.createMock( TemplateHelperInterface.class );
        ClinicalDataResponseInterface clinicalDataResponseMock = EasyMock.createMock( ClinicalDataResponseInterface.class );
        Document clinicalDataDocumentMock = EasyMock.createMock( Document.class );
        Map<String, PathwaysDataResponseInterface> configurableResponseMapMock = EasyMock.createNiceMock( Map.class );

        ExceptionHandler.setClinicalDataResponse( clinicalDataResponseMock );
        ExceptionHandler.setTemplateHelper( templateHelperMock );
        ExceptionHandler.setGuaranteedLogger( guaranteedLoggerMock );
        ExceptionHandler.setPathwaysConfigurableResponseMap( configurableResponseMapMock );

        Exception exception = new ValidationException( ErrorCodeEnum.CANNOT_FIND_FILTER_VALIDATOR );
        String templateId = TemplateIdHelper.ALLERGIES_READ_TEMPLATE_ID;

        // Set expectations
        EasyMock.expect( clinicalDataResponseMock.buildEmptyClinicalDocumentWithoutPatient( templateId, "requestId" ) ).andReturn(
                        clinicalDataDocumentMock );
//        EasyMock.expect(
//                        guaranteedLoggerMock.log( EasyMock.isA( RuntimeException.class ), EasyMock.eq( LoggingSeverity.FATAL ),
//                                        EasyMock.isA( String.class ), EasyMock.eq( templateId ), EasyMock.eq( "requestId" ),
//                                        EasyMock.eq( CDS_APP_NAME ) ) ).andReturn( true );
        EasyMock.expect( templateHelperMock.getErrorSectionHelper( templateId ) ).andReturn( errorSectionHelperMock );
        EasyMock.expect(
                        errorSectionHelperMock.buildErrorSection( EasyMock.isA( Document.class ), EasyMock.isA( ExceptionInfo.class ),
                                        EasyMock.eq( "requestId" ) ) ).andReturn( null ).anyTimes();

        EasyMock.replay( guaranteedLoggerMock, errorSectionHelperMock, clinicalDataResponseMock, templateHelperMock, clinicalDataDocumentMock );

        // Call the method
        ExceptionHandler.setCdsAppName( CDS_APP_NAME );
        ExceptionHandler.handleException( exception, templateId, "requestId", CDS_APP_NAME );

        EasyMock.verify( guaranteedLoggerMock, errorSectionHelperMock, clinicalDataResponseMock, templateHelperMock, clinicalDataDocumentMock );

    }


    @SuppressWarnings( "unchecked" )
    @Test
    @Suite( groups = "checkintest" )
    public void testHandleException_UseDefaults( )
        throws Exception
    {
        //Create Mocks
        ExceptionLoggerInterface guaranteedLoggerMock = EasyMock.createNiceMock( ExceptionLoggerInterface.class );
        ErrorSectionHelperInterface errorSectionHelperMock = EasyMock.createMock( ErrorSectionHelperInterface.class );
        TemplateHelperInterface templateHelperMock = EasyMock.createMock( TemplateHelperInterface.class );
        ClinicalDataResponseInterface clinicalDataResponseMock = EasyMock.createMock( ClinicalDataResponseInterface.class );
        Document clinicalDataDocumentMock = EasyMock.createMock( Document.class );
        Map<String, PathwaysDataResponseInterface> configurableResponseMapMock = EasyMock.createNiceMock( Map.class );

        ExceptionHandler.setClinicalDataResponse( clinicalDataResponseMock );
        ExceptionHandler.setTemplateHelper( templateHelperMock );
        ExceptionHandler.setGuaranteedLogger( guaranteedLoggerMock );
        ExceptionHandler.setPathwaysConfigurableResponseMap( configurableResponseMapMock );
        ExceptionHandler.setCdsAppName( CDS_APP_NAME );

        Exception exception = new RuntimeException();

        // Set expectations
//        EasyMock.expect(
//                        guaranteedLoggerMock.log( EasyMock.isA( SystemException.class ), EasyMock.eq( LoggingSeverity.FATAL ),
//                                        EasyMock.isA( String.class ), EasyMock.eq( "CdsResponse400" ), EasyMock.eq( "request_id_not_set" ),
//                                        EasyMock.eq( CDS_APP_NAME ) ) ).andReturn( true );
        EasyMock.expect( templateHelperMock.getErrorSectionHelper( "CdsResponse400" ) ).andReturn( errorSectionHelperMock );
        EasyMock.expect(
                        errorSectionHelperMock.buildErrorSection( EasyMock.isA( Document.class ), EasyMock.isA( ExceptionInfo.class ),
                                        EasyMock.eq( "request_id_not_set" ) ) ).andReturn( clinicalDataDocumentMock );

        EasyMock.replay( guaranteedLoggerMock, errorSectionHelperMock, clinicalDataResponseMock, templateHelperMock, clinicalDataDocumentMock,
                        configurableResponseMapMock );

        // Call the method
        ExceptionHandler.handleException( clinicalDataDocumentMock, exception, null, null, CDS_APP_NAME );

        EasyMock.verify( guaranteedLoggerMock, errorSectionHelperMock, clinicalDataResponseMock, templateHelperMock, clinicalDataDocumentMock );
    }


    @SuppressWarnings( "unchecked" )
    @Test( expected = NullPointerException.class )
    @Suite( groups = "checkintest" )
    public void testErrorSectionHelperNotConfiguredForTemplateId( )
    {
        //Create Mocks
        ExceptionLoggerInterface guaranteedLoggerMock = EasyMock.createNiceMock( ExceptionLoggerInterface.class );
        ErrorSectionHelperInterface errorSectionHelperMock = EasyMock.createMock( ErrorSectionHelperInterface.class );
        TemplateHelperInterface templateHelperMock = EasyMock.createMock( TemplateHelperInterface.class );
        ClinicalDataResponseInterface clinicalDataResponseMock = EasyMock.createMock( ClinicalDataResponseInterface.class );
        Document clinicalDataDocumentMock = EasyMock.createMock( Document.class );
        Map<String, PathwaysDataResponseInterface> configurableResponseMapMock = EasyMock.createNiceMock( Map.class );

        ExceptionHandler.setClinicalDataResponse( clinicalDataResponseMock );
        ExceptionHandler.setTemplateHelper( templateHelperMock );
        ExceptionHandler.setGuaranteedLogger( guaranteedLoggerMock );
        ExceptionHandler.setPathwaysConfigurableResponseMap( configurableResponseMapMock );

        Exception exception = new IOException();
        String templateId = "Unknown Template Id";
        String cdsResponseTemplateId = "CdsResponse400";

        // Set expectations
        EasyMock.expect( clinicalDataResponseMock.buildEmptyClinicalDocumentWithoutPatient( templateId, "requestId" ) ).andReturn(
                        clinicalDataDocumentMock );
        EasyMock.expect( clinicalDataResponseMock.buildEmptyClinicalDocumentWithoutPatient( cdsResponseTemplateId, "requestId" ) ).andReturn(
                        clinicalDataDocumentMock );
//        EasyMock.expect(
//                        guaranteedLoggerMock.log( EasyMock.isA( SystemException.class ), EasyMock.eq( LoggingSeverity.FATAL ),
//                                        EasyMock.isA( String.class ), EasyMock.eq( templateId ), EasyMock.eq( "requestId" ),
//                                        EasyMock.eq( CDS_APP_NAME ) ) ).andReturn( true );
        EasyMock.expect( templateHelperMock.getErrorSectionHelper( templateId ) ).andReturn( null );
        EasyMock.expect( templateHelperMock.getErrorSectionHelper( cdsResponseTemplateId ) ).andReturn( null );

        EasyMock.replay( guaranteedLoggerMock, errorSectionHelperMock, clinicalDataResponseMock, templateHelperMock, clinicalDataDocumentMock,
                        configurableResponseMapMock );

        // Call the method
        ExceptionHandler.handleException( exception, templateId, "requestId", CDS_APP_NAME );
    }


    @SuppressWarnings( "unchecked" )
    @Test
    @Suite( groups = "checkintest" )
    public void testLogRootException( )
    {
        //Create Mocks
        ExceptionLoggerInterface guaranteedLoggerMock = EasyMock.createNiceMock( ExceptionLoggerInterface.class );
        TemplateHelperInterface templateHelperMock = EasyMock.createMock( TemplateHelperInterface.class );
        ClinicalDataResponseInterface clinicalDataResponseMock = EasyMock.createMock( ClinicalDataResponseInterface.class );
        Map<String, PathwaysDataResponseInterface> configurableResponseMapMock = EasyMock.createNiceMock( Map.class );

        ExceptionHandler.setClinicalDataResponse( clinicalDataResponseMock );
        ExceptionHandler.setTemplateHelper( templateHelperMock );
        ExceptionHandler.setGuaranteedLogger( guaranteedLoggerMock );
        ExceptionHandler.setPathwaysConfigurableResponseMap( configurableResponseMapMock );

        Exception exception = new IOException();
        String templateId = "CdsResponse400";

//        EasyMock.expect(
//                        guaranteedLoggerMock.log( EasyMock.isA( SystemException.class ), EasyMock.eq( LoggingSeverity.FATAL ),
//                                        EasyMock.isA( String.class ), EasyMock.eq( templateId ), EasyMock.eq( "requestId" ),
//                                        EasyMock.eq( PATHWAYS_APP_NAME ) ) ).andReturn( true );

        EasyMock.replay( guaranteedLoggerMock );

        // Call the method
        ExceptionHandler.setPathwaysAppName( PATHWAYS_APP_NAME );
        ExceptionHandler.logRootException( exception, templateId, "requestId", PATHWAYS_APP_NAME );

        EasyMock.verify( guaranteedLoggerMock );
    }

}
