

package gov.va.med.cds.persistence.hibernate;


import static gov.va.med.cds.exception.ErrorCodeEnum.HDRII_OPERATION_FAILED;

import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.Period;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.dom4j.Element;
import org.hibernate.query.Query;
import org.springframework.context.annotation.EnableAspectJAutoProxy;
import org.springframework.transaction.annotation.EnableTransactionManagement;
import org.springframework.transaction.annotation.Isolation;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.hibernate.Session;

import gov.va.med.cds.clinicaldata.DomainEntryPoint;
import gov.va.med.cds.clinicaldata.Operation;
import gov.va.med.cds.exception.ErrorCodeEnum;
import gov.va.med.cds.exception.OperationNotSupportedException;
import gov.va.med.cds.persistence.PersistenceException;
import gov.va.med.cds.persistence.WritePersistenceManagerInterface;
import gov.va.med.cds.request.ValidationException;
import gov.va.med.cds.rules.BooleanRuleInterface;
import gov.va.med.cds.template.generated.clinicaldata.Patients;
import gov.va.med.cds.template.generated.clinicaldata.RoutingDataType;
import gov.va.med.cds.template.generated.common.ClinicalDataPatient;
import gov.va.med.cds.template.generated.surveys.CensusSurveyResponse;
import gov.va.med.cds.template.generated.surveyscommon.Vendor;
import gov.va.med.cds.util.DateTimeUtil;


@EnableTransactionManagement
@EnableAspectJAutoProxy
public class WriteableCensusSurveyPersistenceManager
    extends
        AbstractHibernatePersistenceManager
    implements
        WritePersistenceManagerInterface
{
    private List<BooleanRuleInterface> rules;
    private String cdsAppName;
    private CensusResponsePersistenceManagerInterface censusResponsesPersistenceManager;
    private String censusEarliestStartDate;
    private static final String DATE_FORMAT = "yyyy-MM-dd";
    private static LocalDate CENSUS_EARLIEST_START_DATE = LocalDate.of( 2005, 10, 2 );


    public boolean isApplicable( Object aCritera ) 
        throws PersistenceException
    {
        boolean b = false;
        for ( BooleanRuleInterface r : rules )
        {
            b = r.evaluate( aCritera );
            if ( b )
            {
                break;
            }
        }

        return b;
    }


    public String performCUADOnClinicalData( String templateId, Operation operation, Object aObject, DomainEntryPoint domainEntryPoint )
        throws PersistenceException
    {
        throw new OperationNotSupportedException( ErrorCodeEnum.METHOD_NOT_IMPLEMENTED );
        // return "";
    }


    /**
     * Handles the CUAD requests and applies the transaction to the data-source
     * as directed by the request type.
     * 
     * @param aTemplateId
     *            The identifier of the template being created, updated,
     *            appended, or deleted.
     * @param requestType
     *            - request type could be Create, Update, Append and Delete.
     * @param object
     *            - CDM objects to apply to the data-source.
     */
    @Override
    @Transactional( value = "hdr2TransactionManager", propagation = Propagation.REQUIRED, isolation = Isolation.READ_COMMITTED, readOnly = false )
    public String performCUADOnPatientsData( String aTemplateId, String aRequestId, Operation aOperation, Object aObject, DomainEntryPoint aDomainEntryPoint ) 
        throws PersistenceException
    {
        Session session = null;
        String resultId = "";

        try
        {
            session = getSessionFactory( aTemplateId ).getCurrentSession();
            Patients patients = ( Patients )unmarshalElementToObject( ( Element )aObject, aDomainEntryPoint );

            // Oracle Stored Proc -
            // census_master_record_mgt.census_master_records
            CensusIDs censusIDs = addCensusSegments( aTemplateId, aRequestId, aOperation, aObject, aDomainEntryPoint, session, patients );

            // check for Duplicate Census Activity Reports - do not persist
            // ingest data into CENSUS_RESPONSE_INGEST
            String errorMessage = censusIDs.geterrorMessage();
            if ( ( null != errorMessage ) && errorMessage.equalsIgnoreCase( "WRITE_REQUEST_DUPLICATE_SEGMENT" ) )
            {
                return resultId;
            }

            // insert data into CENSUS_RESPONSE_INGEST using hibernate
            resultId = censusResponsesPersistenceManager.persistCensusResponse( aTemplateId, aRequestId, aOperation, aDomainEntryPoint,
                    session, censusIDs, patients );

            double start = System.currentTimeMillis();

            // Oracle Stored Proc -
            // census_master_record_mgt.census_segment_complete
            updateCensusSegmentStatus( aOperation, aDomainEntryPoint, session, censusIDs );

            double end = ( System.currentTimeMillis() - start );
            if ( LOGGER.isDebugEnabled() )
            {
                LOGGER.debug( "===========================================================================\n" );
                LOGGER.debug( "Total time updating census records status: " + end + "\n" );
                LOGGER.debug( "===========================================================================\n" );
            }
        }
        catch ( Exception e )
        {
            throw e;
        }

        return resultId;
    }


    private CensusIDs addCensusSegments( String aTemplateId, String aRequestId, Operation aOperation, Object aObject, 
            DomainEntryPoint aDomainEntryPoint, Session aSession, Patients aPatients )
    {
        CensusIDs censusIDs = null;
        try
        {
            List<ClinicalDataPatient> patientsList = aPatients.getPatient();
            Integer censusExpectedCount = aPatients.getSegment().getLast();
            String segmentDatetime = aPatients.getSegment().getDatetime().toString();
            Integer currentSegment = aPatients.getSegment().getCurrent();

            // RoutingDataType is used to send census acknowledgement at the end
            // of the census transformation process
            RoutingDataType routingData = aPatients.getRoutingData().get( 0 );
            String sendingApplication = routingData.getSendingApplication();
            String stationNumber = routingData.getStationNumber();
            String sendingFacility = routingData.getSendingFacility();
            String receivingApplication = routingData.getReceivingApplication();
            String receivingFacility = routingData.getReceivingFacility();

            ClinicalDataPatient patient = patientsList.get( 0 );
            CensusSurveyResponse censusSurveyResponse = patient.getCensusSurveyResponse().get( 0 );
            Vendor vendor = censusSurveyResponse.getVendor();
            String vendorName = vendor.getVendorName();
            String vendorFacility = vendor.getVendorNumber();
            Date reportStartDate = censusSurveyResponse.getReportStartDate();
            Date reportEndDate = censusSurveyResponse.getReportEndDate();

            validateStartDateAndEndDate( reportStartDate, reportEndDate, aRequestId, stationNumber );

            Query query = aSession.getNamedQuery( "census_master_record_mgt.census_master_records" )
                    .setParameter( "reportStartDate", reportStartDate ).setParameter( "vendorName", vendorName )
                    .setParameter( "vendorFacility", vendorFacility ).setParameter( "segmentDatetime", segmentDatetime )
                    .setParameter( "censusExpectedCount", censusExpectedCount ).setParameter( "currentSegment", currentSegment )
                    .setParameter( "senderAddress", null ).setParameter( "reportEndDt", reportEndDate )
                    .setParameter( "requestId", aRequestId ).setParameter( "sendingApplication", sendingApplication )
                    .setParameter( "stationNo", stationNumber ).setParameter( "sendingFacility", sendingFacility )
                    .setParameter( "receivingApplication", receivingApplication ).setParameter( "receivingFacility", receivingFacility );

            List<CensusIDs> result = query.list();
            censusIDs = result.get( 0 );

            String errorMessage = censusIDs.geterrorMessage();
            String ackStatus = censusIDs.getAckStatus();

            if ( null != errorMessage )
            {
                // check for Duplicate Census Activity Reports - Story# 376951
                if ( errorMessage.equalsIgnoreCase( "WRITE_REQUEST_DUPLICATE_SEGMENT" ) )
                {
                    return censusIDs;
                }
                else if ( !errorMessage.equalsIgnoreCase( "SUCCESS" ) )
                {
                    handleErrors( errorMessage, segmentDatetime, reportStartDate, currentSegment, reportEndDate, aRequestId, stationNumber,
                            ackStatus );
                }
            }
        }
        catch ( ValidationException e )
        {
            throw e;
        }
        catch ( PersistenceException e )
        {
            throw e;
        }
        catch ( Exception e )
        {
            String rootCauseMessage = ExceptionUtils.getRootCause( e ) == null ? e.getMessage() : ExceptionUtils.getRootCause( e ).getMessage();
            throw new PersistenceException( HDRII_OPERATION_FAILED, e, aOperation.name(), rootCauseMessage, aDomainEntryPoint.getName() );
        }

        return censusIDs;
    }


    private void updateCensusSegmentStatus( Operation aOperation, DomainEntryPoint aDomainEntryPoint, Session aSession, CensusIDs aCensusIDs )
    {
        Long censusMasterID = aCensusIDs.getcensusMasterID();
        Long censusSegmentID = aCensusIDs.getcensusSegmentID();
        try
        {
            Query completeQuery = aSession.getNamedQuery( "census_master_record_mgt.census_segment_complete" )
                    .setParameter( "censusMasterID", censusMasterID ).setParameter( "censusSegmentID", censusSegmentID );
            /*
             * JLA Quality Code Scan - Poor Style: Value Never Read commenting
             * out resultSegment 
             * List resultSegment = completeQuery.list();
             */
            completeQuery.list();

        }
        catch ( Exception e )
        {
            String rootCauseMessage = ExceptionUtils.getRootCause( e ) == null ? e.getMessage()
                    : ExceptionUtils.getRootCause( e ).getMessage();
            throw new PersistenceException( HDRII_OPERATION_FAILED, e, aOperation.name(), rootCauseMessage, aDomainEntryPoint.getName() );
        }
    }


    private void validateStartDateAndEndDate( Date aStartDate, Date aEndDate, String aRequestId, String aVendorId ) 
        throws ValidationException
    {
        LocalDate today = LocalDate.now();
        LocalDate localStartDate = DateTimeUtil.asLocalDate( aStartDate );

        // this should never happen
        if ( null == localStartDate )
        {
            throw new ValidationException( ErrorCodeEnum.WRITE_REQUEST_CENSUS_INVALID_START_DATE,
                    DateTimeUtil.convertDate2String( aStartDate ), DateTimeUtil.convertDate2String( aEndDate ), aRequestId, aVendorId );
        }

        // start Date is not Sunday
        if ( DayOfWeek.SUNDAY != localStartDate.getDayOfWeek() ||
        // start Date is not earlier than 10/2/2005
                localStartDate.isBefore( CENSUS_EARLIEST_START_DATE ) ||
                // start Date is later than current date
                localStartDate.isAfter( today ) )
        {
            throw new ValidationException( ErrorCodeEnum.WRITE_REQUEST_CENSUS_INVALID_START_DATE,
                    DateTimeUtil.convertDate2String( aStartDate ), DateTimeUtil.convertDate2String( aEndDate ), aRequestId, aVendorId );
        }

        LocalDate localEndDate = DateTimeUtil.asLocalDate( aEndDate );

        // this should never happen
        if ( null == localEndDate )
        {
            throw new ValidationException( ErrorCodeEnum.WRITE_REQUEST_CENSUS_INVALID_START_DATE,
                    DateTimeUtil.convertDate2String( aStartDate ), DateTimeUtil.convertDate2String( aEndDate ), aRequestId, aVendorId );
        }

        // end date is not Saturday
        if ( DayOfWeek.SATURDAY != localEndDate.getDayOfWeek() ||
        // start Date is not earlier than 10/2/2005
                localEndDate.isAfter( today ) )
        {
            throw new ValidationException( ErrorCodeEnum.WRITE_REQUEST_CENSUS_INVALID_START_DATE,
                    DateTimeUtil.convertDate2String( aStartDate ), DateTimeUtil.convertDate2String( aEndDate ), aRequestId, aVendorId );
        }

        if ( 6 != Period.between( localStartDate, localEndDate ).getDays() )
        {
            // end day is not the following Saturday of start date
            throw new ValidationException( ErrorCodeEnum.WRITE_REQUEST_CENSUS_INVALID_START_DATE,
                    DateTimeUtil.convertDate2String( aStartDate ), DateTimeUtil.convertDate2String( aEndDate ), aRequestId, aVendorId );
        }
    }


    private void handleErrors( String anErrorMessage, String aSegmentDatetime, Date aReportStartDate, int aCurrentSegment, Date aReportEndDate,
            String aRequestId, String aVendorId, String anAckStatus ) throws PersistenceException
    {
        // census segment has an error, and ack_status is NOT INPROGRESS, ignore
        // the rest below
        // WRITE_REQUEST_CENSUS_ACK_ALREADY_SENT error messages is checked in
        // ApplicationAcknowledgementWork before sending AR to avoid sending AR
        // again
        if ( ( null != anAckStatus ) && !anAckStatus.equalsIgnoreCase( "INPROGRESS" ) )
        {
            throw new PersistenceException( ErrorCodeEnum.WRITE_REQUEST_CENSUS_ACK_ALREADY_SENT );
        }

        if ( anErrorMessage.contentEquals( "WRITE_REQUEST_CENSUS_TOO_MANY_SEGMENTS" ) )
        {
            throw new PersistenceException( ErrorCodeEnum.WRITE_REQUEST_CENSUS_NO_OF_SEGS_RECEPTION_TIME_EXPIRED,
                    DateTimeUtil.convertDate2String( aReportStartDate ), DateTimeUtil.convertDate2String( aReportEndDate ), aRequestId,
                    aVendorId );
        }
        else if ( anErrorMessage.contentEquals( "WRITE_REQUEST_CENSUS_SEGMENT_DUPLICATE" )
                || anErrorMessage.contentEquals( "WRITE_REQUEST_CENSUS_ALREADY_SUBMITTED" ) )
        {
            throw new PersistenceException( ErrorCodeEnum.WRITE_REQUEST_CENSUS_SEGMENT_DUPLICATE,
                    DateTimeUtil.convertDate2String( aReportStartDate ), aSegmentDatetime, String.valueOf( aCurrentSegment ) );
        }
        else if ( anErrorMessage.contentEquals( "WRITE_REQUEST_CENSUS_STATUS_ALREADY_COMPLETE" ) )
        {
            throw new PersistenceException( ErrorCodeEnum.WRITE_REQUEST_CENSUS_STATUS_COMPLETE,
                    DateTimeUtil.convertDate2String( aReportStartDate ), aSegmentDatetime );
        }
        else if ( anErrorMessage.contentEquals( "WRITE_REQUEST_CENSUS_INVALID_START_DATE" ) )
        {
            throw new PersistenceException( ErrorCodeEnum.WRITE_REQUEST_CENSUS_INVALID_START_DATE,
                    DateTimeUtil.convertDate2String( aReportStartDate ), DateTimeUtil.convertDate2String( aReportEndDate ), aRequestId,
                    aVendorId );
        }
        else
        {
            throw new PersistenceException( ErrorCodeEnum.WRITE_REQUEST_CENSUS_SEGMENT_FAILED,
                    DateTimeUtil.convertDate2String( aReportStartDate ), aSegmentDatetime, String.valueOf( aCurrentSegment ) );
        }
    }


    /**
     * Handles hibernate HDRII create processing.
     * 
     * @param aSession
     *            - active hibernate session.
     * @param aObject
     *            - create request.
     */
    protected String createData( Session aSession, CensusSurveyResponse aSurveyResponse )
    {
        return aSession.save( aSurveyResponse ).toString();
    }


    /**
     * Sets the rules to evaluate to determine if the persistence manager is
     * applicable for the request.
     * 
     * @param aRules
     *            The boolean rules to be evaluated.
     */
    public void setRules( List<BooleanRuleInterface> aRules )
    {
        this.rules = aRules;
    }


    public void setCdsAppName( String aCdsAppName )
    {
        this.cdsAppName = aCdsAppName;
    }


    public void setCensusResponsesPersistenceManager( CensusResponsePersistenceManagerInterface aCensusResponsesPersistenceManager )
    {
        this.censusResponsesPersistenceManager = aCensusResponsesPersistenceManager;
    }


    public void setCensusEarliestStartDate( String aCensusEarliestStartDate )
    {
        this.censusEarliestStartDate = aCensusEarliestStartDate;

        if ( null != this.censusEarliestStartDate )
        {
            DateTimeFormatter dateFormat = DateTimeFormatter.ofPattern( DATE_FORMAT );
            CENSUS_EARLIEST_START_DATE = LocalDate.parse( aCensusEarliestStartDate, dateFormat );
        }
    }
}
