/**
 * 
 */


package gov.va.med.cds.persistence.hibernate;


import gov.va.med.cds.exception.ErrorCodeEnum;
import gov.va.med.cds.exceptionframework.ExceptionHandler;
import gov.va.med.cds.filter.EntryFilterInterface;
import gov.va.med.cds.persistence.QueryAssociationInterface;
import gov.va.med.cds.persistence.QueryWorkInterface;
import gov.va.med.cds.template.generated.JaxBMarshallerUnmarshallerInterface;
import gov.va.med.cds.util.PerformanceLog;
import gov.va.med.cds.util.TimeoutUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.query.Query;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;


/**
 * @author vhaislegberb
 *
 *         The <code>DefaultQueryWork</code> class implements the
 *         convention-based named query name building mechanism defined
 *         originally in CDS v2.0. This convention based approach append to the
 *         entry point name optionally the 'pids' or patient identifiers, the
 *         start and end dates, and the record identifiers.
 */
public class DefaultQueryWork
    implements
        QueryWorkInterface
{
    protected static final Log LOGGER = LogFactory.getLog( DefaultQueryWork.class );
    protected static PerformanceLog performanceLog = PerformanceLog.getInstance();

    private List<Exception> exceptions = new ArrayList<Exception>();
    private Map<String, String> templateTimeoutMap = null;
    private QueryNameBuilderInterface queryNameBuilder = null;
    private FilterParameterExtractorInterface filterParameterExtractor = null;
    private Map<String, QueryParameterTransformerInterface> parameterTransformerMap = null;

    protected Session session;
    protected Element results = DocumentHelper.createElement( "results" );
    protected EntryFilterInterface entryFilter;
    protected List<String> personIdentifiers;
    protected QueryAssociationInterface queryAssociation;
    protected Class<?> pointInTimeUserType;
    protected String applicationName;
    protected String siteId;
    protected QueryParameterBinderInterface queryParameterBinder = null;
    protected JaxBMarshallerUnmarshallerInterface marshallerUnmarshaller;


    /**
     * Constructs a new instance of DefaultQueryWork for use in processing
     * filtering requests and querying information from the data store.
     * 
     * @param session
     *            The Hibernate Session instance.
     * @param queryAssociation
     *            Support for multi query domain reads.
     * @param entryFilter
     *            Entry filter defines what the client is asking for and how
     *            that data should be filtered.
     * @param queryNameBuilder
     *            Defines the algorithm for building or resolving a filter to a
     *            Hibernate named query name.
     * @param filterParameterExtractor
     *            Contains logic for how to extract parameter values form the
     *            filter.
     * @param parameterTransformerMap
     *            A map of parameter names to parameter transformer instances to
     *            use to transform parameters prior to use in the query.
     * @param personIdentifiers
     *            The list of person identifiers to query for for Patient
     *            Centric queries.
     * @param pointInTimeUserType
     *            The Point In Time user type to use in setting date parameters
     *            on the query.
     * @param applicationName
     *            The application name used for audit and logging purposes.
     * @param siteId
     *            The site identifier of the application issuing the query.
     */
    public DefaultQueryWork( Session session, QueryAssociationInterface queryAssociation, EntryFilterInterface entryFilter,
                    QueryNameBuilderInterface queryNameBuilder, FilterParameterExtractorInterface filterParameterExtractor,
                    Map<String, QueryParameterTransformerInterface> parameterTransformerMap, QueryParameterBinderInterface queryParameterBinder,
                    List<String> personIdentifiers, Class<?> pointInTimeUserType, String applicationName, String siteId )
    {
        this( session, queryAssociation, entryFilter, queryNameBuilder, filterParameterExtractor, parameterTransformerMap, queryParameterBinder, null,
                        personIdentifiers, pointInTimeUserType, applicationName, siteId, null );
    }


    /**
     * Constructs a new instance of DefaultQueryWork for use in processing
     * filtering requests and querying information from the data store.
     * 
     * @param session
     *            The Hibernate Session instance.
     * @param queryAssociation
     *            Support for multi query domain reads.
     * @param entryFilter
     *            Entry filter defines what the client is asking for and how
     *            that data should be filtered.
     * @param queryNameBuilder
     *            Defines the algorithm for building or resolving a filter to a
     *            Hibernate named query name.
     * @param filterParameterExtractor
     *            Contains logic for how to extract parameter values form the
     *            filter.
     * @param parameterTransformerMap
     *            A map of parameter names to parameter transformer instances to
     *            use to transform parameters prior to use in the query.
     * @param personIdentifiers
     *            The list of person identifiers to query for for Patient
     *            Centric queries.
     * @param pointInTimeUserType
     *            The Point In Time user type to use in setting date parameters
     *            on the query.
     * @param applicationName
     *            The application name used for audit and logging purposes.
     * @param siteId
     *            The site identifier of the application issuing the query.
     * @param marshallerUnmarshaller
     *            The JaxBMarshallerUnmarshaller object used in unmarshalling
     *            and marshalling XML to and from Java POJO objects.
     */
    public DefaultQueryWork( Session session, QueryAssociationInterface queryAssociation, EntryFilterInterface entryFilter,
                    QueryNameBuilderInterface queryNameBuilder, FilterParameterExtractorInterface filterParameterExtractor,
                    Map<String, QueryParameterTransformerInterface> parameterTransformerMap, QueryParameterBinderInterface queryParameterBinder,
                    List<String> personIdentifiers, Class<?> pointInTimeUserType, String applicationName, String siteId,
                    JaxBMarshallerUnmarshallerInterface marshallerUnmarshaller )
    {
        this( session, queryAssociation, entryFilter, queryNameBuilder, filterParameterExtractor, parameterTransformerMap, queryParameterBinder, null,
                        personIdentifiers, pointInTimeUserType, applicationName, siteId, marshallerUnmarshaller );
    }


    /**
     * Constructs a new instance of DefaultQueryWork for use in processing
     * filtering requests and querying information from the data store.
     * 
     * @param session
     *            The Hibernate Session instance.
     * @param queryAssociation
     *            Support for multi query domain reads.
     * @param entryFilter
     *            Entry filter defines what the client is asking for and how
     *            that data should be filtered.
     * @param queryNameBuilder
     *            Defines the algorithm for building or resolving a filter to a
     *            Hibernate named query name.
     * @param filterParameterExtractor
     *            Contains logic for how to extract parameter values form the
     *            filter.
     * @param parameterTransformerMap
     *            A map of parameter names to parameter transformer instances to
     *            use to transform parameters prior to use in the query.
     * @param templateTimeoutMap
     *            The map of timeout values for templates for use in
     *            constraining how long queries can run for particular domains.
     * @param personIdentifiers
     *            The list of person identifiers to query for for Patient
     *            Centric queries.
     * @param pointInTimeUserType
     *            The Point In Time user type to use in setting date parameters
     *            on the query.
     * @param applicationName
     *            The application name used for audit and logging purposes.
     * @param siteId
     *            The site identifier of the application issuing the query.
     */
    public DefaultQueryWork( Session session, QueryAssociationInterface queryAssociation, EntryFilterInterface entryFilter,
                    QueryNameBuilderInterface queryNameBuilder, FilterParameterExtractorInterface filterParameterExtractor,
                    Map<String, QueryParameterTransformerInterface> parameterTransformerMap, QueryParameterBinderInterface queryParameterBinder,
                    Map<String, String> templateTimeoutMap, List<String> personIdentifiers, Class<?> pointInTimeUserType, String applicationName,
                    String siteId, JaxBMarshallerUnmarshallerInterface marshallerUnmarshaller )
    {
        this.session = session;
        this.entryFilter = entryFilter;
        this.queryAssociation = queryAssociation;
        this.personIdentifiers = personIdentifiers;
        this.pointInTimeUserType = pointInTimeUserType;

        this.queryNameBuilder = queryNameBuilder;
        this.filterParameterExtractor = filterParameterExtractor;
        this.parameterTransformerMap = parameterTransformerMap;
        this.queryParameterBinder = queryParameterBinder;

        this.templateTimeoutMap = templateTimeoutMap;
        this.applicationName = applicationName;
        this.siteId = siteId;
        this.marshallerUnmarshaller = marshallerUnmarshaller;
    }


    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Runnable#run()
     */
    @Override
    public void run( )
    {
        long startTime = System.currentTimeMillis();

        try
        {
            Query query = prepareQuery( session, queryAssociation.getAssociationName(), entryFilter, personIdentifiers );
            if ( query != null )
            {
                if ( getTemplateTimeoutMap() != null )
                {
                    long queryTimeout = TimeoutUtil.processTimeout( getTemplateTimeoutMap(), entryFilter, this.applicationName );
                    query.setTimeout( ( int )queryTimeout );
                }

                /*
                 * JLA Fortify Quality Code Scan - Poor Style: Redundant Initialization
                 *   commenting out startTime
                startTime = System.currentTimeMillis();
                 */

                boolean noDataFoundOrInvalidSamlToken = true;

                for ( Object element : query.getResultList() )
                {
                    // The query executed successfully and returned data.
                    noDataFoundOrInvalidSamlToken = false;

                    Element marshalledElement = marshalObjectToElement( element );
                    this.results.add( marshalledElement );
                }

                // No data found or SAML token is invalid.
                if ( noDataFoundOrInvalidSamlToken && isCacheSession( session ) && !( entryFilter.getSamlTokenAsString().isEmpty() ) )
                {

                    if ( isSamlTokenValid( entryFilter.getSamlTokenAsString(), session ) )
                    {
                        // SAML token is valid, do nothing. The DFN has no data.
                    }
                    else
                    {
                        // SAML token is not valid, throw an exception
                        throw new SamlTokenValidationException( ErrorCodeEnum.SAML_TOKEN_FAILED_VALIDATION, siteId );
                    }
                }

                long finishTime = System.currentTimeMillis();
                if ( performanceLog.isInfoEnabled() )
                {
                    performanceLog.logFinishTime( startTime, finishTime, "DefaultQueryWork.run()", entryFilter.getDomainEntryPoint(),
                                    entryFilter.getTemplateId(), entryFilter.getRequestId(), siteId, true, "jdbc" );
                }
            }
        }
        catch ( Exception ex )
        {
            long finishTime = System.currentTimeMillis();
            if ( performanceLog.isInfoEnabled() )
            {
                performanceLog.logFinishTime( startTime, finishTime, "DefaultQueryWork.run()", entryFilter.getDomainEntryPoint(),
                                entryFilter.getTemplateId(), entryFilter.getRequestId(), siteId, false, "jdbc" );
            }

            // LOGGER.error( "************************************* TIMEOUT
            // EXCEPTION
            // *********************************************************" );
            addException( ex, applicationName );
        }
    }


    private boolean isSamlTokenValid( String aSamlToken, Session aSession )
    {
        boolean returnValue = true;
        
        List<String> samlValidationResult = aSession.createNativeQuery( "select VISTA.HDSREPSAML_SAMLtest ( " + "\'" + aSamlToken + "\'" + " )" )
                        .list();

        if ( samlValidationResult.get( 0 ).startsWith( "0" ) )
        {
            // Validation Failed
            returnValue =  false;
        }

        return returnValue;
    }


    /**
     * Handles completing the prepared hibernate statements.
     * 
     * @param aSession
     *            - hibernate session used to obtain named prepared statement.
     * @param aEntryFilter
     *            hibernate HDRII filters.
     * @param aPatient
     *            patient
     * @return Populated named prepared statement.
     */
    @SuppressWarnings( "rawtypes" )
    protected Query prepareQuery( Session aSession, String anAssociationName, EntryFilterInterface aEntryFilter, List<String> aPersonIdentifiersList )
    {
        // Dialect dialect =
        // aSession.getSessionFactory().getSessionFactory().getJdbcServices().getDialect();

        // Initialize the query parameter map with the filter parameter values.
        Map<String, Object> queryParameterMap = getFilterParameters( aEntryFilter );

        if ( aEntryFilter.isPatientCentricFilter() )
        {
            if ( ( aPersonIdentifiersList == null ) || ( aPersonIdentifiersList.size() == 0 ) )
            {
                // If this is a patient centric query but there are no person
                // identifier in the list, return a null query
                // which will short circuit the query work.
                return null;
            }
            else
            {
                // Patient identifiers come into the query work as a parameter.
                // Add the patient identifiers to the
                // query parameters map, transforming as is necessary, in
                // preparation for query execution.
                addPatientIdentifiers( aEntryFilter, aPersonIdentifiersList, queryParameterMap );
            }
        }

        // Use the Query Name Builder instance to build the query name based on
        // a pre-determined algorithm.
        String queryName = queryNameBuilder.buildQueryName( aEntryFilter, anAssociationName );

        if ( isCacheSession( aSession ) )
        {
            if ( !( aEntryFilter.getSamlTokenAsString().isEmpty() ) )
            {
                queryName = queryName + ".samlToken";
            }
        }

        Query query = null;
        try
        {
            // Lookup the named query from the Hibernate session.
            query = session.getNamedQuery( queryName );
        }
        catch ( Exception ie )
        {
            // some hbm files have defined a name query of 'templateId.default'
            // query based on several optional parameters - it would be
            // difficult to construct an exact named query for each
            if ( isCacheSession( aSession ) )
            {
                if ( !( aEntryFilter.getSamlTokenAsString().isEmpty() ) )
                {
                    queryName = aEntryFilter.getTemplateId() + ".default.samlToken";
                }
                else
                {
                    queryName = aEntryFilter.getTemplateId() + ".default";
                }
            }
            else
            {
                queryName = aEntryFilter.getTemplateId() + ".default";

            }

            query = session.getNamedQuery( queryName );

        }

        // Set SAML Token in Query
        if ( queryName.contains( "samlToken" ) )
        {
            query.setParameter( "samlTokenAsString", aEntryFilter.getSamlTokenAsString() );
        }

        // Bind the query parameters to the bind variables in the query using
        // the query binder configured in the query work.
        queryParameterBinder.bindQueryParameters( query, queryParameterMap, pointInTimeUserType );

        // return the query, ready for use.
        return query;
    }


    private boolean isCacheSession( Session aSession )
    {
        // SAML Token Processing
        String dialectAsString = ( String )aSession.getSessionFactory().getProperties().get( "hibernate.dialect" );

        if ( !( "org.hibernate.dialect.Oracle12cDialect".equals( dialectAsString ) ) )
        {
            return true;
        }

        return false;
    }


    protected Element marshalObjectToElement( Object aData )
        throws Exception
    {
        // discover if the instance Object (data) is a top level domain Class
        // (clazz) or if it is a subquery Class we need to retrieve
        Class<?> clazz = aData.getClass();

        return this.marshallerUnmarshaller.marshalToXmlElement( aData, clazz );
    }


    /*
     * (non-Javadoc)
     * 
     * @see commonj.work.Work#isDaemon()
     */
    @Override
    public boolean isDaemon( )
    {
        return false;
    }


    /*
     * (non-Javadoc)
     * 
     * @see commonj.work.Work#release()
     */
    @Override
    public void release( )
    {
        try
        {
            if ( this.session != null && this.session.isOpen() )
            {
                this.session.close();
            }
        }
        catch ( HibernateException e )
        {
            addException( e, applicationName );
        }
    }


    /*
     * @see gov.va.med.cds.persistence.QueryWork#getResults()
     */
    @Override
    public Element getResults( )
    {
        return results;
    }


    /*
     * (non-Javadoc)
     * 
     * @see gov.va.med.cds.persistence.QueryWork#getExceptions( )
     */
    @Override
    public List<Exception> getExceptions( )
    {
        return exceptions;
    }


    /*
     * (non-Javadoc)
     * 
     * @see
     * gov.va.med.cds.persistence.QueryWork#addException(java.lang.Exception)
     */
    @Override
    public void addException( Exception anException, String anApplicationName )
    {
        ExceptionHandler.logRootException( anException, entryFilter.getTemplateId(), entryFilter.getRequestId(), anApplicationName );
        this.exceptions.add( anException );
    }


    public void setTemplateTimeoutMap( Map<String, String> aTemplateTimeoutMap )
    {
        this.templateTimeoutMap = aTemplateTimeoutMap;
    }


    public Map<String, String> getTemplateTimeoutMap( )
    {
        return templateTimeoutMap;
    }


    public void setQueryNameBuilder( QueryNameBuilderInterface aQueryNameBuilder )
    {
        this.queryNameBuilder = aQueryNameBuilder;
    }


    /*
     * (non-Javadoc)
     * 
     * @see gov.va.med.cds.persistence.hibernate.QueryWork#getQueryAssociation()
     */
    public QueryAssociationInterface getQueryAssociation( )
    {
        return queryAssociation;
    }


    protected final void addPatientIdentifiers( EntryFilterInterface anEntryFilter, List<String> aPersonIdentifiersList,
                    Map<String, Object> queryParameterMap )
    {
        QueryParameterTransformerInterface personIdentifierParameterTransformer = parameterTransformerMap
                        .get( FilterParameterExtractorInterface.PN_PERSON_IDENTIFIERS );
        if ( personIdentifierParameterTransformer != null )
        {
            personIdentifierParameterTransformer.transformAndSetParameter( FilterParameterExtractorInterface.PN_PERSON_IDENTIFIERS, personIdentifiers,
                            queryParameterMap );
        }
        else
        {
            queryParameterMap.put( FilterParameterExtractorInterface.PN_PERSON_IDENTIFIERS, personIdentifiers );
        }
    }


    protected final Map<String, Object> getFilterParameters( EntryFilterInterface anEntryFilter )
    {
        return filterParameterExtractor.extractNamedParameters( parameterTransformerMap, anEntryFilter );
    }


    public JaxBMarshallerUnmarshallerInterface getMarshallerUnmarshaller( )
    {
        return marshallerUnmarshaller;
    }


    public void setMarshallerUnmarshaller( JaxBMarshallerUnmarshallerInterface aMarshallerUnmarshaller )
    {
        this.marshallerUnmarshaller = aMarshallerUnmarshaller;
    }
}