
package gov.va.med.cds.persistence.hibernate;

import static gov.va.med.cds.exception.ErrorCodeEnum.HDRII_OPERATION_FAILED;

import java.util.Date;
import java.util.List;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Session;
import org.hibernate.exception.ConstraintViolationException;

import gov.va.med.cds.clinicaldata.DomainEntryPoint;
import gov.va.med.cds.clinicaldata.Operation;
import gov.va.med.cds.exception.ErrorCodeEnum;
import gov.va.med.cds.persistence.PersistenceException;
import gov.va.med.cds.template.generated.clinicaldata.Patients;
import gov.va.med.cds.template.generated.common.ClinicalDataPatient;
import gov.va.med.cds.template.generated.surveys.CensusSurveyResponse;
import gov.va.med.cds.util.DateTimeUtil;

public class CensusResponsesPersistenceManager implements CensusResponsePersistenceManagerInterface {

	protected static final Log LOGGER = LogFactory.getLog(CensusResponsesPersistenceManager.class);

	private String cdsAppName;

	/**
	 * Handles the CUAD requests and applies the transaction to the data-source
	 * as directed by the request type.
	 * 
	 * @param templateId
	 *            The identifier of the template being created, updated,
	 *            appended, or deleted.
	 * @param requestType
	 *            - request type could be Create, Update, Append and Delete.
	 * @param object
	 *            - CDM objects to apply to the data-source.
	 */
	@Override
	public String persistCensusResponse(String templateId, String aRequestId, Operation aOperation,
			DomainEntryPoint domainEntryPoint, Session session, CensusIDs censusIDs, Patients patients)
			throws PersistenceException {

		String resultId = "";

		try {
			Long censusMasterID = censusIDs.getcensusMasterID();
			Long censusSegmentID = censusIDs.getcensusSegmentID();

			List<ClinicalDataPatient> patientsList = patients.getPatient();
			if (LOGGER.isDebugEnabled()) {
				LOGGER.debug(gov.va.med.cds.util.LogMessageUtil.buildMessage(templateId, aRequestId, cdsAppName,
						"Storing Census records for:" + patientsList.size() + " patients"));
			}

			int batchCount = 0;
			double start = System.currentTimeMillis();

			CensusSurveyResponse censusRecord = null;
			ClinicalDataPatient patient = patientsList.get(0);
			for (int i = 1; i <= patientsList.size(); i++) {
				batchCount++;
				patient = (ClinicalDataPatient) patientsList.get(i - 1);

				censusRecord = patient.getCensusSurveyResponse().get(0);
				censusRecord.setRequestId(aRequestId);
				censusRecord.setCensusSegmentID(censusSegmentID);
				censusRecord.setCensusMasterID(censusMasterID);

				resultId = createData(session, censusRecord);
				if (batchCount % 100 == 0) // 100, same as the JDBC batch size
				{
					if (LOGGER.isDebugEnabled()) {
						LOGGER.debug("Writing batch of 100 census records: " + i);
					}
					// flush a batch of inserts and release memory:
					session.flush();
					session.clear();
				}
			}

			/*
			 * flush a last uneven batch of inserts and release memory which
			 * does not fall under mod 100
			 */
			session.flush();
			session.clear();

			double end = (System.currentTimeMillis() - start);
			if (LOGGER.isDebugEnabled()) {
				LOGGER.debug("===========================================================================\n");
				LOGGER.debug("Total time storing census patient records: " + end + "\n");
				LOGGER.debug("Total item or record count: " + batchCount + "\n");
				LOGGER.debug("===========================================================================\n");
			}

		}
		// try to figure out what caused the violation
		catch (ConstraintViolationException e) {
			throw new HibernateConstraintViolationException(e);
		}
		// already handled and turned into one of ours so rethrow it
		catch (PersistenceException e) {
			throw e;
		}
		// any other problems are just big and will require someone to look at
		// so check log file
		catch (Exception e) {
			String rootCauseMessage = ExceptionUtils.getRootCause(e) == null ? e.getMessage()
					: ExceptionUtils.getRootCause(e).getMessage();
			throw new PersistenceException(HDRII_OPERATION_FAILED, e, aOperation.name(), rootCauseMessage,
					domainEntryPoint.getName());
		}

		return resultId;
	}

	protected void errorHanding(String errorMessage, String segmentDatetime, Date reportStartDate, int currentSegment,
			Date reportEndDate, String requestId, String vendorId, String ackStatus) throws PersistenceException {
		// census segment han an error, and ack_status is NOT INPROGRESS, ignore
		// the rest below
		if (null != ackStatus && !ackStatus.equalsIgnoreCase("INPROGRESS")) {
			throw new PersistenceException(ErrorCodeEnum.WRITE_REQUEST_CENSUS_ACK_ALREADY_SENT);
		}

		if (errorMessage.contentEquals("WRITE_REQUEST_CENSUS_TOO_MANY_SEGMENTS")) {
			throw new PersistenceException(ErrorCodeEnum.WRITE_REQUEST_CENSUS_NO_OF_SEGS_RECEPTION_TIME_EXPIRED,
					DateTimeUtil.convertDate2String(reportStartDate), DateTimeUtil.convertDate2String(reportEndDate),
					requestId, vendorId);
		} else if (errorMessage.contentEquals("WRITE_REQUEST_CENSUS_SEGMENT_DUPLICATE")
				|| errorMessage.contentEquals("WRITE_REQUEST_CENSUS_ALREADY_SUBMITTED")) {
			throw new PersistenceException(ErrorCodeEnum.WRITE_REQUEST_CENSUS_SEGMENT_DUPLICATE,
					DateTimeUtil.convertDate2String(reportStartDate), segmentDatetime, String.valueOf(currentSegment));
		} else if (errorMessage.contentEquals("WRITE_REQUEST_CENSUS_STATUS_ALREADY_COMPLETE")) {
			throw new PersistenceException(ErrorCodeEnum.WRITE_REQUEST_CENSUS_STATUS_COMPLETE,
					DateTimeUtil.convertDate2String(reportStartDate), segmentDatetime);
		} else if (errorMessage.contentEquals("WRITE_REQUEST_CENSUS_INVALID_START_DATE")) {
			throw new PersistenceException(ErrorCodeEnum.WRITE_REQUEST_CENSUS_INVALID_START_DATE,
					DateTimeUtil.convertDate2String(reportStartDate), DateTimeUtil.convertDate2String(reportEndDate),
					requestId, vendorId);
		} else {
			throw new PersistenceException(ErrorCodeEnum.WRITE_REQUEST_CENSUS_SEGMENT_FAILED,
					DateTimeUtil.convertDate2String(reportStartDate), segmentDatetime, String.valueOf(currentSegment));
		}
	}

	/**
	 * Handles hibernate HDRII create processing.
	 * 
	 * @param session
	 *            - active hibernate session.
	 * @param aObject
	 *            - create request.
	 */
	// @Override
	protected String createData(Session session, CensusSurveyResponse surveyResponse) {
		return session.save(surveyResponse).toString();
	}

	public void setCdsAppName(String cdsAppName) {
		this.cdsAppName = cdsAppName;
	}

}
