/**
 * 
 */


package gov.va.med.cds.internal;


import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;

import gov.va.med.cds.clinicaldata.Operation;
import gov.va.med.cds.clinicaldata.TemplateMetaDataInterface;
import gov.va.med.cds.exception.ErrorCodeEnum;
import gov.va.med.cds.filter.CdsFilterFactoryInterface;
import gov.va.med.cds.filter.CdsFilterInterface;
import gov.va.med.cds.filter.FilterPatientResolverInterface;
import gov.va.med.cds.request.ValidationException;
import gov.va.med.cds.request.WriteRequestInterface;
import gov.va.med.cds.response.WriteResponseGeneratorInterface;
import gov.va.med.cds.template.TemplateHelperInterface;
import gov.va.med.cds.template.TemplateManager;
import gov.va.med.cds.template.TemplateWriteRequest;
import gov.va.med.cds.tfs.util.TemplateMetaDataHelper;
import gov.va.med.cds.transaction.TransactionManagerInterface;
import gov.va.med.cds.util.NestedElementWrapperUtil;


/**
 * @author susarlan
 *
 */
public class ClinicalDataServiceSynchronousInternal
    implements
        ClinicalDataServiceSynchronousInternalInterface
{
    private Log logger = LogFactory.getLog( ClinicalDataServiceSynchronousInternal.class );
    private static final String START_TAG = "<";
    private static final String END_TAG = "</";

    private TransactionManagerInterface transactionManager;
    private CdsFilterFactoryInterface filterFactory;
    private FilterPatientResolverInterface filterPatientResolver;
    private TemplateManager templateManager;
    private TemplateHelperInterface templateHelper;
    private boolean addPatientIdentifiers = true;
    private String cdsAppName;
    private String pathwaysAppName;
    private Map<String, String> namespacesMap;
    private NestedElementWrapperUtil nestedElementWrapperUtil;
    private Map<String, List<String>> emptyElementListMap;


    public List<Document> readClinicalData( String aFilterRequest, String aFilterId, String aTemplateId, String aRequestId )
    {
        if ( logger.isDebugEnabled() )
        {
            logger.debug( "Processing read request - filterid: " + aFilterId + ", templateId: " + aTemplateId + ", requestId: " + aRequestId );
        }

        // Parse the input filter request XML
        Document filterRequestDocument = this.getDocument( aFilterRequest );
        if ( isPatientCentricRequest( filterRequestDocument ) )
        {
            filterRequestDocument = filterPatientResolver.resolveIdentifiersInFilter( aFilterId, filterRequestDocument );
        }

        CdsFilterInterface cdsFilter = filterFactory.create( aFilterId, filterRequestDocument, aTemplateId, aRequestId );
        if ( cdsFilter == null )
        {
            throw new ValidationException( ErrorCodeEnum.READ_REQUEST_FILTER_XML_NULL );
        }

        validateReadRequest( aTemplateId, aRequestId, cdsFilter );

        List<Document> results = transactionManager.performReadOnClinicalData( aTemplateId, aRequestId, cdsFilter );

        // adding filter patient identifier data into results to be aggregated into response 
        // if needed, siteCentric filter will have only one entryPointFilter
        if ( addPatientIdentifiers && !cdsFilter.getEntryPointFilters().get( 0 ).isSiteCentricFilter() )
        {
            Document patientIdentifierDocument = templateHelper.getPatientIdentifierDocumentBuilder( aTemplateId )
                    .createPatientWithIdentifiersDocument( cdsFilter, aTemplateId, aRequestId );
            results.add( 0, patientIdentifierDocument );
        }

        return results;
    }


    @SuppressWarnings("unused")
    private void validateReadRequest( String aTemplateId, String aRequestId, CdsFilterInterface aCdsFilter )
    {
        TemplateMetaDataInterface templateMetaData = templateManager.validateReadTemplateId( aTemplateId );
        /*
         * JLA Quality Code Scan updates - Poor Style: Value Never Read
         * commenting out vhimVersion String vhimVersion =
         * cdsFilter.getVhimVersion();
         */
        Collection<String> templateEntryPoints = TemplateMetaDataHelper.getEntryPoints( templateMetaData.getDomainEntryPoints() );
        Collection<String> domainEntryPoints = ( aCdsFilter != null ) ? aCdsFilter.getDomainEntryPoints() : null;

        // TODO: We may need additional validation of whether CDS can process the incoming filter request 
        // and return the specified template (templateId) [BJE]

        // Ensure templateEntryPoints is a subset of domainEntryPoints
        if ( null != domainEntryPoints )
        {
            for ( String domainEntryPoint : domainEntryPoints )
            {
                if ( !templateEntryPoints.contains( domainEntryPoint.trim() ) )
                {
                    throw new ValidationException( ErrorCodeEnum.REQUEST_TYPE_MISMATCH_TEMPLATE_ENTRYPOINT_FILTER_ENTRYPOINT,
                            new String[] { templateEntryPoints.toString(), domainEntryPoints.toString() } );
                }
            }
        }
    }


    public Document createClinicalData( String aCreateRequest, String aTemplateId, String aRequestId )
    {
        return doCUAD( Operation.Create, aCreateRequest, aTemplateId, aRequestId );
    }


    public Document updateClinicalData( String anUpdateRequest, String aTemplateId, String aRequestId )
    {
        return doCUAD( Operation.Update, anUpdateRequest, aTemplateId, aRequestId );
    }


    public Document deleteClinicalData( String aDeleteRequest, String aTemplateId, String aRequestId )
    {
        return doCUAD( Operation.Delete, aDeleteRequest, aTemplateId, aRequestId );
    }


    public Document appendClinicalData( String anAppendRequest, String aTemplateId, String aRequestId )
    {
        return doCUAD( Operation.Append, anAppendRequest, aTemplateId, aRequestId );
    }


    public List<Document> writeClinicalData( String aCreateRequest, String aTemplateId, String aRequestId )
    {
        return doVistaCreate( Operation.Create, aCreateRequest, aTemplateId, aRequestId );
    }


    public Document doCUAD( Operation anOperation, String aClinicalData, String aTemplateId, String aRequestId )
    {
        Document writeResultDocument = null;
        Document clinicalDataDocument = null;
        WriteResponseGeneratorInterface writeResponseGenerator = templateHelper.getWriteResponseGenerator( aTemplateId );

        try
        {
            if ( logger.isDebugEnabled() )
            {
                logger.debug( "Processing request " + anOperation + " for - templateId: " + aTemplateId + ", requestId: " + aRequestId
                        + ", request: " + aClinicalData );
            }

            validateCUADRequest( aClinicalData, aTemplateId, aRequestId, anOperation );

            aClinicalData = removeUnwantedNameSpaces( aClinicalData, aTemplateId );
            clinicalDataDocument = DocumentHelper.parseText( aClinicalData );
            clinicalDataDocument = nestedElementWrapperUtil.addNestedElements( clinicalDataDocument, aTemplateId );
            clinicalDataDocument = removeEmptyElements( clinicalDataDocument, aTemplateId );
            
            List<Element> createdRecordIdentifiers = transactionManager.performCUADOnClinicalData( anOperation, clinicalDataDocument,
                    aTemplateId, aRequestId );

            writeResultDocument = writeResponseGenerator.generateSuccessfulWriteResponse( aTemplateId, aRequestId, createdRecordIdentifiers );
        }
        catch ( Exception e )
        {
            // since persist Census master and census segment table will be done under same transaction as persist census data.
            // so if persist census data failed, then persist census master and census segment failed as well
            writeResultDocument = writeResponseGenerator.generateWriteErrorResponse( aTemplateId, aRequestId, e, clinicalDataDocument,
                    cdsAppName );
        }

        return writeResultDocument;
    }


    private List<Document> doVistaCreate( Operation anOperation, String aClinicalData, String aTemplateId, String aRequestId )
    {
        Document writeResultDocument = null;
        Document clinicalDataDocument = null;
        List<Document> results = null;

        WriteResponseGeneratorInterface writeResponseGenerator = templateHelper.getWriteResponseGenerator( aTemplateId );

        try
        {
            if ( logger.isDebugEnabled() )
            {
                logger.debug( "Processing request " + anOperation + " for - templateId: " + aTemplateId + ", requestId: " + aRequestId
                        + ", request: " + aClinicalData );
            }

            validateCUADRequest( aClinicalData, aTemplateId, aRequestId, anOperation );

            aClinicalData = removeUnwantedNameSpaces( aClinicalData, aTemplateId );
            clinicalDataDocument = DocumentHelper.parseText( aClinicalData );
            clinicalDataDocument = nestedElementWrapperUtil.addNestedElements( clinicalDataDocument, aTemplateId );
            clinicalDataDocument = removeEmptyElements( clinicalDataDocument, aTemplateId );

            results = transactionManager.performCUADVistAData( anOperation, clinicalDataDocument, aTemplateId, aRequestId );
        }
        catch ( Exception e )
        {
            writeResultDocument = writeResponseGenerator.generateWriteErrorResponse( aTemplateId, aRequestId, e, clinicalDataDocument,
                    pathwaysAppName );
            results = new ArrayList<Document>();
            results.add( writeResultDocument );
        }

        return results;
    }


    private void validateCUADRequest( String aClinicalData, String aTemplateId, String aRequestId, Operation anOperation )
    {
        WriteRequestInterface cuadRequest = new TemplateWriteRequest( aClinicalData, anOperation );
        cuadRequest.setRequestId( aRequestId );
        cuadRequest.setTemplateId( aTemplateId );

        templateManager.validateWriteRequest( cuadRequest );
    }


    private Document getDocument( String aFilterRequest )
    {
        // Parse the input filter request XML
        try
        {
            return DocumentHelper.parseText( aFilterRequest );
        }
        catch ( DocumentException e )
        {
            throw new ValidationException( ErrorCodeEnum.FILTER_PARSER_DOM_EXCEPTION, e, aFilterRequest, e.getMessage() );
        }
    }


    private boolean isPatientCentricRequest( Document aFilterRequestDocument )
    {
        boolean returnValue = true;
        Element rootElement = aFilterRequestDocument.getRootElement();
        Element entryPointFilterElement = rootElement.element( "entryPointFilter" );
        String isPatientCentricStr = entryPointFilterElement.attributeValue( "isPatientCentric" );
        if ( ( isPatientCentricStr != null ) && ( isPatientCentricStr.length() > 0 ) )
        {
            returnValue =  Boolean.parseBoolean( isPatientCentricStr ) == Boolean.TRUE;
        }

        return returnValue;
    }


    private String removeUnwantedNameSpaces( String aClinicalData, String aTemplateId )
    {
        String namespaceValue = null;
        if ( namespacesMap.containsKey( aTemplateId ) )
        {
            namespaceValue = namespacesMap.get( aTemplateId );
            aClinicalData = aClinicalData.replace( START_TAG + namespaceValue, START_TAG );
            aClinicalData = aClinicalData.replace( END_TAG + namespaceValue, END_TAG );
        }

        return aClinicalData;
    }


    @SuppressWarnings("unchecked")
    private Document removeEmptyElements( Document aClinicalDataDocument, String aTemplateId ) 
        throws DocumentException
    {
        List<String> emptyElementList = emptyElementListMap.get( aTemplateId );

        if ( emptyElementList != null )
        {
            for ( String xpath : emptyElementList )
            {
                List<Element> elements = aClinicalDataDocument.selectNodes( xpath );
                if ( ( elements != null ) && ( elements.size() > 0 ) )
                {
                    for ( Element element : elements )
                    {
                        if ( ( element.getText() == null ) || ( element.getText().equals( "" ) ) )
                        {
                            element.detach();
                        }
                    }
                }
            }
        }

        return aClinicalDataDocument;
    }


    public void setTransactionManager( TransactionManagerInterface aTransactionManager )
    {
        this.transactionManager = aTransactionManager;
    }


    public void setFilterFactory( CdsFilterFactoryInterface aFilterFactory )
    {
        this.filterFactory = aFilterFactory;
    }


    public void setFilterPatientResolver( FilterPatientResolverInterface aFilterPatientResolver )
    {
        this.filterPatientResolver = aFilterPatientResolver;
    }


    public void setTemplateManager( TemplateManager aTemplateManager )
    {
        this.templateManager = aTemplateManager;
    }


    public void setTemplateHelper( TemplateHelperInterface aTemplateHelper )
    {
        this.templateHelper = aTemplateHelper;
    }


    public void setAddPatientIdentifiers( boolean anAddPatientIdentifiers )
    {
        this.addPatientIdentifiers = anAddPatientIdentifiers;
    }


    public void setCdsAppName( String aCdsAppName )
    {
        this.cdsAppName = aCdsAppName;
    }


    public void setNamespacesMap( Map<String, String> aNamespacesMap )
    {
        this.namespacesMap = aNamespacesMap;
    }


    public void setNestedElementWrapperUtil( NestedElementWrapperUtil aNestedElementWrapperUtil )
    {
        this.nestedElementWrapperUtil = aNestedElementWrapperUtil;
    }


    public void setEmptyElementListMap( Map<String, List<String>> anEmptyElementListMap )
    {
        this.emptyElementListMap = anEmptyElementListMap;
    }


    public void setPathwaysAppName( String aPathwaysAppName )
    {
        this.pathwaysAppName = aPathwaysAppName;
    }

}
