

package gov.va.med.cds.ejb;


import gov.va.med.mediator.client.messagemediator.MessageMediatorInterface;
import gov.va.med.cds.exception.RuntimeMDBOnMessageException;
import gov.va.med.cds.junit.runners.Suite;
import gov.va.med.cds.junit.runners.SuiteAwareSpringRunner;
import gov.va.med.cds.response.util.ResponseErrorSectionExtractor;

import javax.jms.BytesMessage;
import javax.jms.ObjectMessage;
import javax.jms.TextMessage;

import org.easymock.EasyMock;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;

import weblogic.jms.common.BytesMessageImpl;
import weblogic.jms.common.ObjectMessageImpl;
import weblogic.jms.common.TextMessageImpl;


@RunWith( SuiteAwareSpringRunner.class )
@ContextConfiguration( locations = { "classpath:gov/va/med/cds/ejb/CDSClinicalMdbTest-context.xml" } )
public class CDSClinicalMdbTest
{
    @Autowired
    private ResponseErrorSectionExtractor responseErrorSectionExtractor;

    private static String RESPONSE_WITH_NO_ERROR = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><clinicaldata:ClinicalData xmlns:clinicaldata=\"Clinicaldata\"><templateId>AllergyAssessmentCreate40020</templateId><requestId>E851B78E-0807-1A28-E86C-E1B362624068</requestId><errorSection/></clinicaldata:ClinicalData>";
    private static String RESPONSE_WITH_UNIQUE_CONSTRAINT = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><clinicaldata:ClinicalData xmlns:clinicaldata=\"Clinicaldata\"><templateId>AllergyAssessmentCreate40020</templateId><requestId>E851B78E-0807-1A28-E86C-E1B362624068</requestId><errorSection><fatalErrors><errorId>55253592077</errorId><exception>gov.va.med.cds.persistence.hibernate.HibernateConstraintViolationException</exception><exceptionMessage>HDRII_CONSTRAINT_VIOLATED</exceptionMessage><errorCode>HDRII_CONSTRAINT_VIOLATED</errorCode><displayMessage>HDR2 Integrity constraint violated:BaseBatchUpdateException: [BEA][Oracle JDBCDriver][Oracle]ORA-00001: unique constraint(ALLERGIES.ALLERGY_RECORDID_UNQ_CK) violated</displayMessage></fatalErrors></errorSection></clinicaldata:ClinicalData>";
    private static String RESPONSE_WITH_SCHEMA_VALIDATION_FAILED = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><clinicaldata:ClinicalData xmlns:clinicaldata=\"Clinicaldata\"><templateId>VitalsignsCreateOrUpdate40060</templateId><requestId>5A2B3C920D5F3</requestId><errorSection><fatalErrors><errorId>5A2B3C920D5F3</errorId><exception>gov.va.med.cds.xml.schema.SchemaValidationException</exception><exceptionMessage>SCHEMA_VALIDATION_FAILED</exceptionMessage><errorCode>SCHEMA_VALIDATION_FAILED</errorCode><displayMessage>Schema validation failed because of cvc-complex-type.2.4.a: Invalid content was found starting with element 'patient'. One of '{&quot;&quot;:recordIdentifier}' is expected.</displayMessage></fatalErrors></errorSection></clinicaldata:ClinicalData>";
    private static String STR_MESSAGE = "Hello this is a new message";


    @Test
    @Suite( groups = "checkintest", order = 5 )
    public void testOnMessageWithBytesMessage( )
        throws Exception
    {
        CDSClinicalMdb cdsClinicalMdb = new CDSClinicalMdb();

        BytesMessage message = new BytesMessageImpl();
        message.setBooleanProperty( MessageMediatorInterface.PROP_PERSIST_ORIGINAL, true );
        message.writeBytes( STR_MESSAGE.getBytes() );
        message.reset();

        MessageMediatorInterface messageMediatorMock = EasyMock.createMock( MessageMediatorInterface.class );
        EasyMock.expect( messageMediatorMock.processMessage( STR_MESSAGE, true ) ).andReturn( RESPONSE_WITH_NO_ERROR );

        cdsClinicalMdb.setMessageMediator( messageMediatorMock );
        cdsClinicalMdb.setResponseErrorSectionExtractor( responseErrorSectionExtractor );

        EasyMock.replay( messageMediatorMock );

        cdsClinicalMdb.onMessage( message );
    }


    @Test
    @Suite( groups = "checkintest", order = 1 )
    public void testOnMessageWithTextMessage( )
        throws Exception
    {
        CDSClinicalMdb cdsClinicalMdb = new CDSClinicalMdb();

        TextMessage message = new TextMessageImpl();
        message.setBooleanProperty( MessageMediatorInterface.PROP_PERSIST_ORIGINAL, true );
        message.setText( STR_MESSAGE );

        MessageMediatorInterface messageMediatorMock = EasyMock.createMock( MessageMediatorInterface.class );
        EasyMock.expect( messageMediatorMock.processMessage( STR_MESSAGE, true ) ).andReturn( RESPONSE_WITH_NO_ERROR );

        cdsClinicalMdb.setMessageMediator( messageMediatorMock );
        cdsClinicalMdb.setResponseErrorSectionExtractor( responseErrorSectionExtractor );

        EasyMock.replay( messageMediatorMock );

        cdsClinicalMdb.onMessage( message );
    }


    @Test( expected = NullPointerException.class )
    @Suite( groups = "checkintest", order = 2 )
    public void testOnMessageWithObjectMessage( )
        throws Exception
    {
        CDSClinicalMdb cdsClinicalMdb = new CDSClinicalMdb();

        ObjectMessage message = new ObjectMessageImpl();
        message.setBooleanProperty( MessageMediatorInterface.PROP_PERSIST_ORIGINAL, true );
        cdsClinicalMdb.onMessage( message );
    }


    @Test
    @Suite( groups = "checkintest", order = 3 )
    public void testOnMessageWithTextMessageWithDuplicateException( )
        throws Exception
    {
        CDSClinicalMdb cdsClinicalMdb = new CDSClinicalMdb();

        TextMessage message = new TextMessageImpl();
        message.setBooleanProperty( MessageMediatorInterface.PROP_PERSIST_ORIGINAL, true );
        message.setText( STR_MESSAGE );

        MessageMediatorInterface messageMediatorMock = EasyMock.createMock( MessageMediatorInterface.class );
        EasyMock.expect( messageMediatorMock.processMessage( STR_MESSAGE, true ) ).andReturn( RESPONSE_WITH_UNIQUE_CONSTRAINT );

        cdsClinicalMdb.setMessageMediator( messageMediatorMock );
        cdsClinicalMdb.setResponseErrorSectionExtractor( responseErrorSectionExtractor );

        EasyMock.replay( messageMediatorMock );

        cdsClinicalMdb.onMessage( message );
    }


    @Test( expected = RuntimeMDBOnMessageException.class )
    @Suite( groups = "checkintest", order = 4 )
    public void testOnMessageWithTextMessageWithNonDuplicateException( )
        throws Exception
    {
        CDSClinicalMdb cdsClinicalMdb = new CDSClinicalMdb();

        TextMessage message = new TextMessageImpl();
        message.setBooleanProperty( MessageMediatorInterface.PROP_PERSIST_ORIGINAL, true );
        message.setText( STR_MESSAGE );

        MessageMediatorInterface messageMediatorMock = EasyMock.createMock( MessageMediatorInterface.class );
        EasyMock.expect( messageMediatorMock.processMessage( STR_MESSAGE, true ) ).andReturn( RESPONSE_WITH_SCHEMA_VALIDATION_FAILED );

        cdsClinicalMdb.setMessageMediator( messageMediatorMock );
        cdsClinicalMdb.setResponseErrorSectionExtractor( responseErrorSectionExtractor );

        EasyMock.replay( messageMediatorMock );

        cdsClinicalMdb.onMessage( message );
    }


    @Test( expected = NullPointerException.class )
    @Suite( groups = "checkintest", order = 6 )
    public void testOnMessageWithTextMessageWithoutPersistanceProperty( )
        throws Exception
    {
        CDSClinicalMdb cdsClinicalMdb = new CDSClinicalMdb();

        TextMessage message = new TextMessageImpl();
        message.setText( STR_MESSAGE );

        MessageMediatorInterface messageMediatorMock = EasyMock.createMock( MessageMediatorInterface.class );
        EasyMock.expect( messageMediatorMock.processMessage( STR_MESSAGE, true ) ).andReturn( RESPONSE_WITH_NO_ERROR );

        cdsClinicalMdb.setMessageMediator( messageMediatorMock );
        cdsClinicalMdb.setResponseErrorSectionExtractor( responseErrorSectionExtractor );

        EasyMock.replay( messageMediatorMock );

        cdsClinicalMdb.onMessage( message );
    }

}
