

package gov.va.med.cds.persistence.hibernate.common;


import static org.easymock.EasyMock.expect;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import gov.va.med.cds.junit.runners.Suite;
import gov.va.med.cds.junit.runners.SuiteAwareSpringRunner;
import gov.va.med.cds.util.DateTimeUtil;

import java.io.Serializable;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Properties;
import java.util.TimeZone;

import org.easymock.EasyMock;
import org.hibernate.HibernateException;
import org.hibernate.engine.spi.SessionImplementor;
import org.hibernate.type.StandardBasicTypes;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.test.context.ContextConfiguration;


@RunWith( SuiteAwareSpringRunner.class )
@ContextConfiguration( locations = { "classpath:gov/va/med/cds/config/timeZoneOffsetContext.xml" } )
public class VistaPointInTimeUserTypeTest
{
    private static final String TEST_NO_CONVERTION = "20";
    private static final String TEST_DATE_PLUS_OFFSET = "20071231235959.999+0600";
    private static final String TEST_DATE_MINUS_OFFSET = "20071231235959.999-0600";
    private static final String TEST_DATE_MINUS_OFFSET_COMPARE = "20071231235959-0600";
    private static final String TEST_NON_CONFORMANT_DATE = "2007/23/1235959.999-0600";
    private static final double CONVERTED_VALUE = 3071231.235959;
    private static final double NO_CONVERTION_VALUE = 0.0;
    private static VistaPointInTimeUserType vistaPointInTimeUserType = null;
    private static gov.va.med.cds.template.generated.basedatatypes.PointInTime pit = null;
    private static String isoTime = null;
    private static long time = 1281045978450l;


    @BeforeClass
    @Suite( groups = { "checkintest" } )
    public static void beforeVistaPointInTimeUserTypeTestClassSetUp( )
        throws Exception
    {
        vistaPointInTimeUserType = new VistaPointInTimeUserType();
        pit = new gov.va.med.cds.template.generated.basedatatypes.PointInTime();
        isoTime = DateTimeUtil.getISOTime( time, TimeZone.getDefault() );
        pit.setLiteral( isoTime );
       
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testNullSafeGetWithNullTimestamp( )
        throws HibernateException,
            SQLException
    {

        ResultSet rs = EasyMock.createMock( ResultSet.class );
        String[] names = { "timestamp" };
        SessionImplementor sessionImplementor = EasyMock.createMock( SessionImplementor.class );

        expect( rs.getString( names[0] ) ).andReturn( null );

        EasyMock.replay( rs );
        PointInTime result = ( PointInTime )vistaPointInTimeUserType.nullSafeGet( rs, names, sessionImplementor, null );

        assertNull( result );
        EasyMock.verify( rs );

    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testNullSafeGet( )
        throws HibernateException,
            SQLException
    {

        ResultSet rs = EasyMock.createMock( ResultSet.class );
        String[] names = { "timestamp" };
        SessionImplementor sessionImplementor = EasyMock.createMock( SessionImplementor.class );

        String dateTime = Double.toString( CONVERTED_VALUE ) + "^520";
        expect( rs.getString( names[0] ) ).andReturn( dateTime );
        
        expect( rs.wasNull() ).andReturn( false );

        EasyMock.replay( rs, sessionImplementor );
        gov.va.med.cds.template.generated.basedatatypes.PointInTime result = ( gov.va.med.cds.template.generated.basedatatypes.PointInTime )vistaPointInTimeUserType.nullSafeGet( rs, names, sessionImplementor, null );

       
        assertEquals( result.getLiteral(), TEST_DATE_MINUS_OFFSET_COMPARE );
        EasyMock.verify(sessionImplementor );
    }



    @Test
    @Suite( groups = { "checkintest" } )
    public void testNullSafeSetWithNull( )
        throws HibernateException,
            SQLException
    {
        int index = 1;
        PreparedStatement statement = EasyMock.createMock( PreparedStatement.class );
        SessionImplementor sessionImplementor = EasyMock.createMock( SessionImplementor.class );

        statement.setNull( index, StandardBasicTypes.DATE.sqlType() );

        EasyMock.replay( statement );
        vistaPointInTimeUserType.nullSafeSet( statement, null, 1, sessionImplementor );
        EasyMock.verify( statement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testNullSafeSetWithTimestamp( )
        throws HibernateException,
            SQLException
    {
        int index = 1;
        PreparedStatement statement = EasyMock.createMock( PreparedStatement.class );
        SessionImplementor sessionImplementor = EasyMock.createMock( SessionImplementor.class );

        String timest = "20101231235959.450-0600";
        PointInTime timestamp = new PointInTime();
        timestamp.setTimestamp( timest );

        statement.setDate( index, vistaPointInTimeUserType.isoTime2Date( timest ) );

        EasyMock.replay( statement );
        vistaPointInTimeUserType.nullSafeSet( statement, timestamp, 1, sessionImplementor );
        EasyMock.verify( statement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testNullSafeSetWithNullTimestamp( )
        throws HibernateException,
            SQLException
    {
        int index = 1;
        PreparedStatement statement = EasyMock.createMock( PreparedStatement.class );
        SessionImplementor sessionImplementor = EasyMock.createMock( SessionImplementor.class );

        PointInTime timestamp = new PointInTime();
        timestamp.setTimestamp( null );

        statement.setNull( index, StandardBasicTypes.DATE.sqlType() );

        EasyMock.replay( statement );
        vistaPointInTimeUserType.nullSafeSet( statement, timestamp, 1, sessionImplementor );
        EasyMock.verify( statement );
    }


    @Test( expected = HibernateException.class )
    @Suite( groups = { "checkintest" } )
    public void testNullSafeSetNonPointInTimeValue( )
        throws HibernateException,
            SQLException
    {
        int index = 1;
        PreparedStatement statement = EasyMock.createMock( PreparedStatement.class );
        SessionImplementor sessionImplementor = EasyMock.createMock( SessionImplementor.class );
        Object object = new String( "NotAnElementInstance" );
        //this should throw exception ...
        vistaPointInTimeUserType.nullSafeSet( statement, object, index, sessionImplementor );
    }


    @Test( expected = UnsupportedOperationException.class )
    @Suite( groups = { "checkintest" } )
    public void testSetPropertyValue( )
        throws HibernateException,
            SQLException
    {
        //this should throw exception ...
        vistaPointInTimeUserType.setPropertyValue( null, 1, pit );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testIsoTime2DoubleConversion( )
    {
        Double isoDouble = vistaPointInTimeUserType.isoTime2Double( TEST_DATE_PLUS_OFFSET );
        assertEquals( isoDouble.doubleValue(), CONVERTED_VALUE, 0 );
        isoDouble = vistaPointInTimeUserType.isoTime2Double( TEST_DATE_MINUS_OFFSET );
        assertEquals( isoDouble.doubleValue(), CONVERTED_VALUE, 0 );
        isoDouble = vistaPointInTimeUserType.isoTime2Double( TEST_NO_CONVERTION );
        assertEquals( isoDouble.doubleValue(), NO_CONVERTION_VALUE, 0 );
        isoDouble = vistaPointInTimeUserType.isoTime2Double( null );
        assertEquals( isoDouble.doubleValue(), NO_CONVERTION_VALUE, 0 );
        isoDouble = vistaPointInTimeUserType.isoTime2Double( TEST_NON_CONFORMANT_DATE );
        assertEquals( isoDouble, null );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testIsoTime2DateConversion( )
    {
        //TODO - determine if this needs to be kept - isoTime2Date - not used
        Date date = vistaPointInTimeUserType.isoTime2Date( pit.getLiteral() );
        String dateSt = DateTimeUtil.getISOTime( date.getTime(), TimeZone.getDefault() );

        //now compare without nano/millis and timezone offset - isoTime2Date removes millis
        assertEquals( pit.getLiteral().split( "\\." )[0], dateSt.split( "\\." )[0] );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testBasicGetValuesTest( )
    {

        assertEquals( vistaPointInTimeUserType.getPropertyNames()[0], "timestamp" );
        assertEquals( vistaPointInTimeUserType.getPropertyTypes()[0], StandardBasicTypes.STRING );

        Serializable cache = new String( "cache" );
        assertEquals( vistaPointInTimeUserType.assemble( cache, null ), cache );
        assertEquals( vistaPointInTimeUserType.assemble( cache, null, null ), cache );

        Object original = new String( "original" );
        Object almostOriginal = new String( "original" );
        assertEquals( vistaPointInTimeUserType.replace( original, null, null, null ), original );
        assertEquals( vistaPointInTimeUserType.deepCopy( original ), almostOriginal );
        assertEquals( vistaPointInTimeUserType.disassemble( original, null ), ( Serializable )original );
        assertFalse( vistaPointInTimeUserType.equals( original, almostOriginal ) );
        assertEquals( vistaPointInTimeUserType.returnedClass(), PointInTime.class );
        assertFalse( vistaPointInTimeUserType.isMutable() );

        String object = "object";
        assertEquals( vistaPointInTimeUserType.hashCode( "object" ), object.hashCode() );
        assertEquals( vistaPointInTimeUserType.hashCode( null ), 0 );

        //91 is the int value for Hibernate.DATE.sqlType() 
        assertEquals( vistaPointInTimeUserType.sqlTypes()[0], 91 );

        String timestamp = pit.getLiteral();
        assertEquals( isoTime, timestamp );

        assertEquals( vistaPointInTimeUserType.getPropertyValue( pit, 2 ), timestamp );
        Double time = vistaPointInTimeUserType.isoTime2Double( timestamp );
        assertEquals( vistaPointInTimeUserType.getPropertyValue( pit, 1 ), time );
        assertNull( vistaPointInTimeUserType.getPropertyValue( "someObject", 1 ) );

        Properties props = new Properties();
        props.put( VistaPointInTimeUserType.SET_VALUES, VistaPointInTimeUserType.SV_NUMERIC_ONLY );
        vistaPointInTimeUserType.setParameterValues( props );
        assertEquals( vistaPointInTimeUserType.setValues, VistaPointInTimeUserType.SV_NUMERIC_ONLY );
    }
}
