

package gov.va.med.cds.rpc.ejb;


import javax.annotation.Resource;
import javax.ejb.Stateless;
import javax.interceptor.Interceptors;
import javax.jws.WebService;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import gov.va.med.cds.audit.VhimRequestAuditLoggerInterface;
import gov.va.med.cds.client.ClinicalDataServiceInterface;
import gov.va.med.cds.config.BeanConstantsInterface;
import gov.va.med.cds.ejb.interceptor.CdsAutowiredInterceptor;
import gov.va.med.repositories.Pathways;


@WebService(endpointInterface = "gov.va.med.repositories.Pathways", 
targetNamespace="http://repositories.med.DNS   /pathways",   
portName="PathwaysHttpPort", 
serviceName="Pathways")
@Stateless(name="ClinicalDataServiceRpcSynchronous", mappedName="ejb/remote/gov/va/med/cds/ClinicalDataServiceRpcSynchronousSession")
@Interceptors(CdsAutowiredInterceptor.class)
public class ClinicalDataServiceRpcSynchronousSession implements Pathways
{
    
    protected final Log logger = LogFactory.getLog(getClass());
    
    @Autowired
    private ClinicalDataServiceInterface clinicalDataServiceSynchronous;
    @Autowired
    private String pathwaysAppName;
    
    @Autowired
    @Qualifier(BeanConstantsInterface.VHIM_REQUEST_PATHWAYS_AUDIT_LOGGER)
    private VhimRequestAuditLoggerInterface vhimRequestPathwaysAuditLogger;
    
    @Autowired
    @Qualifier( BeanConstantsInterface.VHIM_REQUEST_AUDIT_LOGGER )
    private VhimRequestAuditLoggerInterface vhimRequestAuditLogger;
    
    /**
     * 
     * @see gov.va.med.cds.client.ClinicalDataServiceInterface#isAlive()
     */
    public boolean isAlive( )
    {
        if ( logger.isDebugEnabled() )
            logger.debug( gov.va.med.cds.util.LogMessageUtil.buildMessage( null, null, pathwaysAppName,
                            "About to call the POMO isAlive() method from inside the SLSB" ) );

        boolean result = clinicalDataServiceSynchronous.isAlive();

        if ( logger.isDebugEnabled() )
            logger.debug( gov.va.med.cds.util.LogMessageUtil.buildMessage( null, null, pathwaysAppName,
                            "Finished calling the POMO isAlive() method from inside the SLSB" ) );
     
        return result;
    }


    /**
     * Synchronous version of a clinical data read
     *
     * @param templateId ID of the template for which the filter request applies
     * @param filterRequest Filter request specified in xml format according to the CDS filter schema
     * @param filterId ID of the named (canned) filter. The filterId may be null if no canned filter is referenced
     * @param requestId Client specified unique request Id
     * @return Results of the read returned in XML according to the CDS read results schema
     */
    public String readData( String templateId, String filterRequest, String filterId, String requestId )
    {
    	long startTime = System.currentTimeMillis();
        
        if( filterRequest != null)
        {
            filterRequest = filterRequest.trim();
        }

        String response =  clinicalDataServiceSynchronous.readClinicalData( templateId, filterRequest, filterId, requestId );
        
        vhimRequestPathwaysAuditLogger.auditReadRequest( templateId, filterRequest, filterId, requestId, startTime, response );
        
        return response;
    }
    
    
    public String createData(String createXml, String templateId, String requestId)
    {
    	long startTime = System.currentTimeMillis();
        
        if( createXml != null)
        {
        	createXml = createXml.trim();
        }

        vhimRequestAuditLogger.auditWriteRequest( templateId, createXml, requestId );
        
        String response = clinicalDataServiceSynchronous.createClinicalData(createXml, templateId, requestId, false);
        
    	return response;
    }
    
  
}
