

package gov.va.med.cds.request;

import org.dom4j.Document;


/**
 *
 * Performs the CRUAD operations based on the client request. See gov.va.med.cds.config.requestProcessorContext.xml for
 * Spring DI details.
 *
 */
public interface RequestProcessorInterface
{

    /**
     * Returns the client requested clinical data from the database by applying the database/cdm/vdm filters as
     * specified in the filter xml by the client. This involves marshalling and validating the filter xml to get the
     * requested data, vdm/cdm/database filters; reading data from the database by applying database filters; creation
     * of cdm objects and applying cdm filters; mapping cdm to vdm objects and applying vdm filters and marshalling and
     * validating the data and return the data to the client in xml format.
     *
     * @param templateId Read template id aganist which the response data is validated before sending to the client.
     * @param filterXmlRequest Contains what data is needed and the different filter conditions to be applied to the
     *            data returned to the client.
     * @param filterId ID of the named (canned) filter. The filterId may be null if no canned filter is referenced
     * @param requestId Client specified unique request Id
     * @return Response data in xml format validated aganist its template.
     */
    Document readClinicalData( String templateId, String filterXmlRequest, String filterId, String requestId );


    /**
     * Creates the clinical data in the database based on the request xml data. This involves validating the request
     * xml; creation of vdm objects and mapping vdm object data to cdm object data; Applying data processing rules and
     * persisting in the database in a Spring initiated transaction context.
     *
     * @param request Request xml.
     * @param templateId ID of the template for which the create request applies
     * @param requestId Client specified unique request Id
     * @return Response data in validated xml format.
     *
     */
    Document createClinicalData( String request, String templateId, String requestId );


    /**
     * Updates the clinical data in the database based on the request xml data. This involves validating and
     * unmarshalling the request xml; creation of vdm objects and mapping vdm object data to cdm object data; Applying
     * data processing rules and persisting in the database in a Spring initiated transaction context.
     *
     * @param request Request xml.
     * @param templateId ID of the template for which the update request applies
     * @param requestId Client specified unique request Id
     * @return Response data in validated xml format.
     */
    Document updateClinicalData( String request, String templateId, String requestId );


    /**
     * Creates the clinical data if it doesn't exist, updates if it does, based on the request xml data. This involves validating and
     * unmarshalling the request xml; creation of vdm objects and mapping vdm object data to cdm object data; Applying
     * data processing rules and persisting in the database in a Spring initiated transaction context.
     *
     * @param request Request xml.
     * @param templateId ID of the template for which the update request applies
     * @param requestId Client specified unique request Id
     * @return Response data in validated xml format.
     */
    Document createOrUpdateClinicalData( String request, String templateId, String requestId );


    /**
     * Appends the clinical data in the database based on the request xml data. This involves validating and
     * unmarshalling the request xml; creation of vdm objects and mapping vdm object data to cdm object data; Applying
     * data processing rules and persisting in the database in a Spring initiated transaction context.
     *
     * @param request Request xml.
     * @param templateId ID of the template for which the append request applies
     * @param requestId Client specified unique request Id
     * @return Response data in validated xml format.
     */
    Document appendClinicalData( String request, String templateId, String requestId );


    /**
     * Deletes the clinical data in the database based on the request xml data. This involves validating and
     * unmarshalling the request xml; creation of vdm objects and mapping vdm object data to cdm object data; Applying
     * data processing rules and persisting in the database in a Spring initiated transaction context.
     *
     * @param request Request xml.
     * @param templateId ID of the template for which the append request applies
     * @param requestId Client specified unique request Id
     * @return Response data in validated xml format.
     */
    Document deleteClinicalData( String request, String templateId, String requestId );

}
