'use strict';
var _ = require('lodash');
var moment = require('moment');
var lzma = require('lzma');
var referenceRangeConfig = require('./reference-range-config');

module.exports.addNormalizedName = addNormalizedName;
module.exports.addCalculatedBMI = addCalculatedBMI;
module.exports.addReferenceRanges = addReferenceRanges;
module.exports.filterVlerData = filterVlerData;
module.exports.setExpirationLabel = setExpirationLabel;
module.exports.setStandardizedDescription = setStandardizedDescription;
module.exports.decompressFullHtml = decompressFullHtml;
module.exports.setTimeSince = setTimeSince;
module.exports._calculateBmi = calculateBmi;
module.exports._findStartDateString = findStartDateString;
module.exports._setTimeSinceItem = setTimeSinceItem;


function addNormalizedName(jdsData) {
    _.each(jdsData.data.items, function(item) {
        var normalizedName = null;
        if (hasCodesDisplay(item)) {
            normalizedName = getCodesDisplay(item);
        } else if (item.qualifiedName !== undefined) {
            normalizedName = item.qualifiedName;
        } else if (item.productFormName !== undefined) {
            normalizedName = item.productFormName;
        } else {
            normalizedName = item.name;
        }

        item.normalizedName = normalizedName;
    });
    return jdsData;
}

function hasCodesDisplay(item) {
    var hasDisplay = false;
    if (item.codes !== undefined) {

        for (var code = 0; code < item.codes.length; code++) {
            if (item.codes[code].display !== undefined) {
                hasDisplay = true;

            }
        }
    }

    return hasDisplay;
}

function getCodesDisplay(item) {

    for (var code = 0; code < item.codes.length; code++) {
        if (item.codes[code].display !== undefined) {
            return item.codes[code].display;
        }
    }
}

function setExpirationLabel(jdsData) {
    _.each(jdsData.data.items, function(item) {
        var overallStop = '';
        if (item.stopped) {
            item.stopped = item.stopped.toString();
        }

        if (item.overallStop) {
            item.overallStop = item.overallStop.toString();
        }

        if (item.overallStop) {
            overallStop = moment(item.overallStop.toString(), 'YYYYMMDDHHmmssSSS');
        } else {
            overallStop = moment(item.stopped, 'YYYYMMDDHHmmssSSS');
        }
        var currentDate = moment();
        //console.log('stop expiration label: ------- ' + overallStop);

        if (!item.vaStatus) {
            item.calculatedStatus = '';
            return;
        }
        switch (item.vaStatus.toUpperCase()) {
            case 'PENDING':
                item.calculatedStatus = 'PENDING'; //is this what we should put here? FYI, 9E7A;164 has Pending meds.
                break;
            case 'ACTIVE':
                if (item.vaType === 'N') {
                    item.calculatedStatus = 'Start';
                }
                if (overallStop < currentDate) {
                    item.calculatedStatus = 'Expired';
                } else {
                    item.calculatedStatus = 'Active';
                }
                break;
            case 'SUSPEND':
                item.calculatedStatus = 'Ordered';
                break;
            case 'HOLD':
                item.calculatedStatus = 'Ordered';
                break;
            case 'EXPIRED':
                if (item.vaType === 'N') {
                    item.calculatedStatus = 'Start';
                } else {
                    item.calculatedStatus = 'Expired';
                }
                break;
            case 'UNRELEASED':
                item.calculatedStatus = '';
                break;
            case 'DISCONTINUED':
                item.calculatedStatus = 'Discontinued';
                break;
            case 'DISCONTINUED/EDIT':
                item.calculatedStatus = 'Discontinued';
                break;
            default:
                // return 'Add a new case to expirationLabel';
                item.calculatedStatus = item.vaStatus;
        }
    });

    return jdsData;
}


var DATE_FORMAT = 'YYYYMMDDHHmmssSSS';


/**
 * @param {*} item JDS response item
 * @param {moment} today
 * @returns {String} Start Date in DATE_FORMAT
 */
function findStartDateString(item, today) {
    if (!_.isUndefined(item.lastFilled) && !_.isNull(item.lastFilled)) {
        var startDateToUse = item.lastFilled.toString();

        var lastFilledMoment = moment(startDateToUse, DATE_FORMAT);

        if (!lastFilledMoment.isValid()) {
            item.lastFilled = null;
            return findStartDateString(item, today);
        }

        var lastFilledTime = lastFilledMoment.toISOString();
        var lastFilledLocalTime = moment.utc(lastFilledTime).toDate();

        lastFilledTime = moment(lastFilledLocalTime);


        var secondToLastFill = item.fills[item.fills.length - 2];
        var advancedFillForReleaseAfterCurrentFillExpiresExists = (
            lastFilledTime > today &&
            item.fills.length > 1 &&
            secondToLastFill !== undefined &&
            secondToLastFill.dispenseDate !== undefined
        );

        if (advancedFillForReleaseAfterCurrentFillExpiresExists) {
            //use the date of the fill the patient is currently on
            startDateToUse = secondToLastFill.dispenseDate.toString();
        }
        return startDateToUse;

    }

    var fills = _.get(item, 'fills[0].dispenseDate');
    if (!_.isUndefined(fills)) {
        return fills;
    }

    var lastAdmin = _.get(item, 'lastAdmin');
    if (!_.isUndefined(lastAdmin)) {
        return lastAdmin;
    }

    return item.overallStart;
}




function setTimeSinceItem(today, item) {
    var ordered = _.get(item, 'orders[0].ordered');

    if (item.overallStart) {
        item.overallStart = item.overallStart.toString();
    } else if (ordered) {
        item.overallStart = ordered.toString();
    }

    if (item.stopped) {
        item.stopped = item.stopped.toString();
    }

    if (!item.fills) {
        item.fills = [];
    }

    var startDateToUse = findStartDateString(item, today);
    var startDate = moment(startDateToUse, DATE_FORMAT);

    var overallStopStr = item.overallStop || item.stopped || '';
    var overallStop = moment(overallStopStr.toString(), DATE_FORMAT);

    if (startDate < overallStop && overallStop < today) {
        startDate = overallStop;
    }

    // ensure the start time is no sooner than the medication order time and no later than right now.
    if (!_.isUndefined(ordered)) {
        var orderedDate = moment(ordered.toString(), DATE_FORMAT);
        if (startDate <= orderedDate || startDate > today) {
            startDate = moment(ordered.toString(), DATE_FORMAT).add(1, 'second');
        }
    }

    var duration = moment.duration(today.diff(startDate));
    var months = parseFloat(duration.asMonths());

    item.lastAction = startDate.format(DATE_FORMAT);
    item.expirationDate = overallStop.format(DATE_FORMAT);
    item.recent = months <= 6;
}

function setTimeSince(jdsData) {
    if(!_.get(jdsData, 'data.items')) {
        return jdsData;
    }

    var setTimeWithToday = _.partial(setTimeSinceItem, moment());
    _.each(jdsData.data.items, setTimeWithToday);

    return jdsData;
}

function setStandardizedDescription(jdsData) {
    _.each(jdsData.data.items, function(item) {
        _.each(item.codes, function(code) {
            if (code.system.indexOf('http://snomed.info/sct') !== -1) {
                item.standardizedDescription = code.display;
                item.snomedCode = code.code;
            }
        });

    });

    return jdsData;
}

function decompressFullHtml(jdsData, callback) {
    var done = _.after(jdsData.data.items.length, function(err) {
        if (err) {
            return callback(err, null);
        }
        return callback(null, jdsData);
    });

    _.each(jdsData.data.items, function(item) {
        //check for fullHtml as call type could be 'vler_list'
        if (item.compressed && item.fullHtml) {
            decompress(item.fullHtml, function(err, result) {
                if (err) {
                    done(err);
                }
                item.fullHtml = result;
                done(null);
            });
        }else {
            done(null);
        }
    });
}

function decompress(fullHtml, callback) {
    var fullHtmlBytes =  new Uint8Array(
        new Buffer(fullHtml, 'base64'));
    lzma.decompress(fullHtmlBytes, function(result) {
        if (result === false) {
            callback('patientRecordAnnotator: Failed to decompress VLER HTML', null);
        }
        callback(null, result);
    });
}

/*
    Add calculated BMI to the domain data
    Each facility data is used to calculate BMI.  No crossover between different facilities.
    For each facility, use last height of all days and last measured weight on a day are used to calculate BMI
 */


function addCalculatedBMI(jdsData) {
    var domainData = jdsData;

    var items = _.filter(jdsData.data.items, function(item) {
        if (item.typeName &&
            item.result &&
            ((item.typeName.toLowerCase().indexOf('height') >= 0) ||
                (item.typeName.toLowerCase().indexOf('weight') >= 0)) &&
            isValidVitalMeasurement(item.result)) {
            return item;
        }
    });
    if (items.length <= 0) {
        return domainData;
    }
    items = _.groupBy(items, function(item) {
        return item.facilityCode;
    });
    var facilities = _.keys(items);
    var lastHeight;
    var heights;
    var weights;
    var weightGroups;
    var days;
    var weightGroupWeights;
    var bmi;
    _.each(facilities, function(facility) {
        heights = _.filter(items[facility], function(item) {
            if ((item.typeName.toLowerCase().indexOf('height') >= 0) &&
                isValidVitalMeasurement(item.result)) {
                return item;
            }
        });
        if (heights.length <= 0) {
            return;
        }
        heights = _(heights).chain().sortBy('observed').reverse().value();

        lastHeight = heights[0];
        weights = _.filter(items[facility], function(item) {
            if ((item.typeName.toLowerCase().indexOf('weight') >= 0) &&
                isValidVitalMeasurement(item.result)) {
                return item;
            }
        });
        if (weights.length <= 0) {
            return;
        }
        weightGroups = _.groupBy(weights, function(item) {
            return item.observed.toString().substring(0, 8);
        });
        days = _.keys(weightGroups);
        _.each(days, function(day) {
            weightGroupWeights = _(weightGroups[day]).chain().sortBy('observed').reverse().value();
            bmi = calculateBmi(lastHeight.result, lastHeight.units, weightGroupWeights[0].result,
                weightGroupWeights[0].units);
            domainData.data.items.push({
                facilityCode: weightGroupWeights[0].facilityCode,
                facilityName: weightGroupWeights[0].facilityName,
                observed: weightGroupWeights[0].observed,
                result: bmi,
                summary: 'BMI ' + bmi,
                typeName: 'BMI',
                facilityDisplay: weightGroupWeights[0].facilityDisplay,
                facilityMoniker: weightGroupWeights[0].facilityMoniker
            });
        });
    });
    var count = domainData.data.items.length;
    if (count > domainData.data.totalItems) {
        domainData.data.totalItems = count;
        domainData.data.currentItemCount = count;
        domainData.data.items =  _.sortBy(domainData.data.items, 'observed').reverse();
    }

    return domainData;
}

function addReferenceRanges(jdsData) {
    _.each(jdsData.data.items, function(item) {
        var refRange = referenceRangeConfig.getReferenceRanges()[item.typeName.toLowerCase()];
        if (refRange) {
            item.low = (refRange.override && refRange.low) || item.low || refRange.low;
            item.high = (refRange.override && refRange.high) || item.high || refRange.high;
        }
    });
    return jdsData;
}

/*
    Filter VLER data to only send individual records when a request is sent from frontend
    and not send all records at once.
 */


function filterVlerData(vlerCallType, vler_uid, name, jdsData) {

    var domainData = jdsData;

    if(domainData.data.totalItems < 1) {
        return domainData;
    }

    if ((vlerCallType === 'vler_list') && (name === 'vlerdocument')) { // && domainData.sections) {
        var itemsLength = domainData.data.items.length; // && domainData.sections) {
        var itemsCount;

        var sectionsCount;
        var sectionsLength;
        for (itemsCount = 0; itemsCount < itemsLength; itemsCount++) {

            //remove sections if they exist
            if (domainData.data.items[itemsCount].sections) {
                sectionsLength = domainData.data.items[itemsCount].sections.length;
                for (sectionsCount = 0; sectionsCount < sectionsLength; sectionsCount++) {
                    delete domainData.data.items[itemsCount].sections;
                }
            }

            //remove fullHtml if it exists
            if (domainData.data.items[itemsCount].fullHtml) {
                delete domainData.data.items[itemsCount].fullHtml;
            }
        }
    } else if ((vlerCallType === 'vler_modal') && (name === 'vlerdocument')) {

        var count;
        for (count = 0; count < domainData.data.items.length; count++) {
            if (domainData.data.items[count].uid !== vler_uid) {
                domainData.data.items.splice(count, 1);
                count--;
            }
        }
    }

    return domainData;
}

function isValidVitalMeasurement(measurement) {
    return measurement !== 'Pass' &&
        measurement !== 'Refused';
}

function calculateBmi(height, heightUnit, weight, weightUnit) {
    if (heightUnit === 'cm') {
        height = height * 0.393701;
    }
    if (weightUnit === 'kg') {
        weight = weight * 2.20462;
    }

    var bmi = (weight / Math.pow(height, 2) * 703).toFixed(1);

    if (isNaN(bmi)) {
        return 'No Record';
    }

    return bmi;
}
