'use strict';
var aep = require('./activity-event-process-resource');
var httpstatus = require('../../../../core/httpstatus');

describe('Activity Event Processor', function() {

    describe('Templating', function() {

        it('correctly applies templates when given a valid empty template', function() {
            var inputTemplate = '{"patientName":"Andromeda"}';
            var inputContent = {
                'test': 'test1'
            };
            var output = aep._applyTemplate(inputTemplate, inputContent);

            expect(output).to.eql(inputTemplate);
        });

        it('correctly ignores templates when given an invalid template', function() {
            var inputTemplate = '{"patientName":"{{bad value}"}';
            var inputContent = {
                'bad value': 'dont insert me'
            };
            var output = aep._applyTemplate(inputTemplate, inputContent);

            expect(output).to.eql(inputTemplate);
        });

        it('correctly applies templates when given a valid object', function() {
            var inputTemplate = '{"patientName":"{{content}}"}';
            var inputContent = {
                'content': 'hello'
            };
            var expectedOutput = '{"patientName":"hello"}';
            var output = aep._applyTemplate(inputTemplate, inputContent);

            expect(output).to.eql(expectedOutput);
        });

        it('correctly ignores templates when given an invalid object', function() {
            var inputTemplate = '{"patientName":"{{content}}"}';
            var inputContent = {
                'content2': 'hello2'
            };
            var expectedOutput = '{"patientName":""}';
            var output = aep._applyTemplate(inputTemplate, inputContent);

            expect(output).to.eql(expectedOutput);
        });

        it('correctly applies nested templates when given a valid object', function() {
            var inputTemplate = '{"patientName":"{{test.content}}"}';
            var inputContent = {
                'test': {
                    'content': 'success'
                }
            };
            var expectedOutput = '{"patientName":"success"}';
            var output = aep._applyTemplate(inputTemplate, inputContent);

            expect(output).to.eql(expectedOutput);
        });

        it('correctly applies template value when given special key', function() {
            var inputTemplate = '{"patientName":"stuff","fullBody": {{{obj RAW_REQUEST}}} }';
            var inputContent = {
                'test': {
                    'content': 'success'
                }
            };
            var expectedOutput = '{"patientName":"stuff","fullBody": {"test":{"content":"success"}} }';
            var output = aep._applyTemplate(inputTemplate, inputContent);

            expect(output).to.eql(expectedOutput);
        });

        it('correctly applies full template value when given special key', function() {
            var inputTemplate = '{{{obj RAW_REQUEST}}}';
            var inputContent = {
                'test': {
                    'content': 'success'
                }
            };
            var expectedOutput = '{"test":{"content":"success"}}';
            var output = aep._applyTemplate(inputTemplate, inputContent);

            expect(output).to.eql(expectedOutput);
        });
    });

    describe('startActivityEvent', function() {
        it('requires a request body object', function(done) {
            var req = {
                body: 7,
                audit: {},
                logger: {
                    debug: function() {}
                }
            };
            var res = {
                status: function(status) {
                    expect(status).to.equal(httpstatus.bad_request);
                    return this;
                },
                rdkSend: function(value) {
                    expect(value).to.equal('201 - Invalid request body');
                    done();
                }
            };

            aep.startActivityEvent(req, res);
        });

        it('requires a well formed database configuration', function(done) {
            var req = {
                body: {
                    fake: 'body'
                },
                audit: {},
                logger: {
                    debug: function() {}
                },
                app: {
                    config: {
                        oracledb: {
                            activityDatabase: {
                                userBadKey: '',
                                password: '',
                                connectString: ''
                            }
                        }
                    }
                }
            };
            var res = {
                status: function(status) {
                    expect(status).to.equal(httpstatus.bad_request);
                    return this;
                },
                rdkSend: function(value) {
                    expect(value).to.equal('210 - Invalid request configuration');
                    done();
                }
            };

            aep.startActivityEvent(req, res);
        });
    });

});
