package vistacore.order.lab;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.drools.core.util.KieFunctions;
import org.jboss.logging.Logger;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;
import org.kie.api.runtime.process.ProcessInstance;
import org.kie.api.runtime.process.WorkflowProcessInstance;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

import vistacore.order.kie.utils.WorkflowProcessInstanceUtil;

/**
 * This class was automatically generated by the data modeler tool.
 */

public class LabHelperUtil implements java.io.Serializable {

	static final long serialVersionUID = 1L;

	private static final Logger LOGGER = Logger.getLogger(LabHelperUtil.class);
	private static final Map<String, String> statusCodeMap;
	static {
		Map<String, String> tempMap = new java.util.HashMap<String, String>();
		tempMap.put("urn:va:order-status:dc", "Discontinued");
		tempMap.put("urn:va:order-status:comp", "Complete");
		tempMap.put("urn:va:order-status:hold", "Hold");
		tempMap.put("urn:va:order-status:flag", "Flagged");
		tempMap.put("urn:va:order-status:pend", "Pending");
		tempMap.put("urn:va:order-status:actv", "Active");
		tempMap.put("urn:va:order-status:exp", "Expired");
		tempMap.put("urn:va:order-status:schd", "Scheduled");
		tempMap.put("urn:va:order-status:part", "Partial Results");
		tempMap.put("urn:va:order-status:dlay", "Delayed");
		tempMap.put("urn:va:order-status:unr", "Unreleased");
		tempMap.put("urn:va:order-status:dc/e", "Discontinued/Edit");
		tempMap.put("urn:va:order-status:canc", "Canceled");
		tempMap.put("urn:va:order-status:laps", "Lapsed");
		tempMap.put("urn:va:order-status:rnew", "Renewed");
		tempMap.put("urn:va:order-status:none", "No Status");
		statusCodeMap = Collections.unmodifiableMap(tempMap);
	}

	// Used to validate formAction in ORDER.UPDATED in Monitor for Result subprocess;
	private static final List<String> LEGAL_FORM_ACTIONS = Collections.unmodifiableList(
			Arrays.asList(new String[] {"Remind Me Later", "Reassign Reminder", "Cancel Reminder"}));
	
	// The joda-time implementation of DateTimeFormatter is thread-safe - Java DateTimeFormatter is NOT!
	private static final DateTimeFormatter DATETIME_FORMATTER = ISODateTimeFormat.basicDate();

	
	public LabHelperUtil() {
	}

	/**
	 * Validates order statusCode received
	 * Called from a gateway expression
	 * @param statusCode
	 * @return true or false
	 */
	public static boolean isValidStatusCode(String statusCode) {
		if (statusCode == null) {
			return false;
		}
		if (statusCode.equals("urn:va:order-status:actv") || statusCode.equals("urn:va:order-status:pend")
				|| statusCode.equals("urn:va:order-status:unr") || statusCode.equals("urn:va:order-status:canc")
				|| statusCode.equals("urn:va:order-status:comp") || statusCode.equals("urn:va:order-status:dc")
				|| statusCode.equals("urn:va:order-status:exp") || statusCode.equals("urn:va:order-status:laps")
				|| statusCode.equals("urn:va:order-status:part") || statusCode.equals("urn:va:order-status:schd")) {
			return true;
		}
		return false;
	}

	/**
	 * Transforms statusCode into readable form
	 * @param statusCode
	 * @return string that contains status value
	 */
	public static String transformStatusCode(String statusCode) {
		if (statusCode == null) {
			return "";
		}

		if (statusCodeMap.containsKey(statusCode)) {
			return "Status : " + statusCodeMap.get(statusCode);
		}
		return "Status : Unknown (" + statusCode + ")";
	}

	/**
	 * Processes the initial signal ORDER.INITIATED
	 * 
	 * @param processInstance
	 *            a reference to the running process instance
	 */
	public static void processReceivedClinicalObject(ProcessInstance processInstance) {
		WorkflowProcessInstance workflowProcessInstance = (WorkflowProcessInstance) processInstance;
		try {
			SignalData signalData = WorkflowProcessInstanceUtil.getRequiredSignalData(workflowProcessInstance,
					"signalData");
			LOGGER.debug(String.format("LabHelperUtil.processReceivedClinicalObject: %s", signalData.toString()));
			workflowProcessInstance.setVariable("orderClinicalObject", signalData.getMessage());
			workflowProcessInstance.setVariable("clinicalObjectUid", signalData.getClinicalObjectUid());
			workflowProcessInstance.setVariable("listenerId", signalData.getListenerId());
			workflowProcessInstance.setVariable("noResultNotificationDate", signalData.getNoResultNotificationDate());
			workflowProcessInstance.setVariable("notificationDate", signalData.getNoResultNotificationDate());
			workflowProcessInstance.setVariable("orderStatusCode", signalData.getOrderStatusCode());
			workflowProcessInstance.setVariable("icn", signalData.getPid());
			workflowProcessInstance.setVariable("facility", signalData.getFacilityCode());
			workflowProcessInstance.setVariable("orderingProviderId", workflowProcessInstance.getVariable("initiator"));
			String ehmpState = signalData.getEhmpState();
			if (ehmpState.equals("draft")
					|| (ehmpState.equals("active") && KieFunctions.isEmpty(signalData.getOrderStatusCode()))) {
				workflowProcessInstance.setVariable("state", "draft");
			}
			workflowProcessInstance.setVariable("type", "Order");
			workflowProcessInstance.setVariable("subDomain", "Lab");
			workflowProcessInstance.setVariable("description", "Lab order entry and tracking");
			if (!KieFunctions.isEmpty(signalData.getLabTestText())) {
				workflowProcessInstance.setVariable("instanceName", signalData.getLabTestText());
			} else if (!KieFunctions.isEmpty(signalData.getName())) {
				workflowProcessInstance.setVariable("instanceName", signalData.getName());
			}
			workflowProcessInstance.setVariable("priority", 9);
			if (!KieFunctions.isEmpty(signalData.getUrgency())) {
				workflowProcessInstance.setVariable("urgency", signalData.getUrgency());
				int urgencyNumber = Integer.parseInt(signalData.getUrgency());
				workflowProcessInstance.setVariable("priority", urgencyNumber == 2 ? 1 : urgencyNumber == 4 ? 5 : 9);
			}
			if (!KieFunctions.isEmpty(signalData.getProviderUid())) {
				String[] arr = signalData.getProviderUid().split(":");
				if (arr.length > 4) {
					workflowProcessInstance.setVariable("orderingProviderId", arr[3] + ";" + arr[4]);
				}
			}
			workflowProcessInstance.setVariable("isValidStatusCode",
					LabHelperUtil.isValidStatusCode(signalData.getOrderStatusCode()));
			if (!KieFunctions.isEmpty(signalData.getAuthorUid())) {
				String[] arr = signalData.getAuthorUid().split(":");
				if (arr.length > 4) {
					workflowProcessInstance.setVariable("authorId", arr[3] + ";" + arr[4]);
					workflowProcessInstance.setVariable("initiator", arr[3] + ";" + arr[4]);
				}
			}
			workflowProcessInstance.setVariable("referenceId", signalData.getReferenceId());
		} 
		catch (NumberFormatException e) {
			LOGGER.error("Lab Business Process: An unexpected condition has happened with an Integer: " +
					e.getMessage(), e);
		} 
		catch (Exception e) {
			LOGGER.error("Lab Business Process: An unexpected condition has happened: " + e.getMessage(), e);
		}
	}

	/**
	 * Prepares orderClinicalObject JSON object
	 * Called from Store Clinical Object On Entry Action script
	 * @param processInstance a reference to the running process instance
	 */
	public static void enterStoreClinicalObject(ProcessInstance processInstance) {
		WorkflowProcessInstance workflowProcessInstance = (WorkflowProcessInstance) processInstance;
		try {
			SignalData signalData = WorkflowProcessInstanceUtil.getRequiredSignalData(workflowProcessInstance,
					"signalData");
			if (!KieFunctions.isEmpty(signalData.getReferenceId())) {
				String orderClinicalObject = WorkflowProcessInstanceUtil.getRequiredString(workflowProcessInstance,
						"orderClinicalObject");
				JsonParser parser = new JsonParser();
				JsonObject obj = parser.parse(orderClinicalObject).getAsJsonObject();
				if (obj.has("data")) {
					obj.remove("data");
				}
				obj.add("data", new JsonObject());
				Gson gson = new GsonBuilder().create();
				workflowProcessInstance.setVariable("orderClinicalObject", gson.toJson(obj));
			}
		}
		catch (Exception e) {
			LOGGER.error("Lab Business Process: An unexpected condition has happened: " + e.getMessage(), e);
		}
	}

	/**
	 * Updates the clinicalObjectUid from the WIH serviceResponse
	 * 
	 * Called from Store Clinical Object On Exit Action script
	 * @param processInstance a reference to the running process instance
	 */
	public static void exitStoreClinicalObject(ProcessInstance processInstance) {
		try {
			JsonObject jsonObject = validateServiceResponse(processInstance, "ClinicalObjectWriteService");
			if(jsonObject == null) {
				throw new Exception("ClinicalObjectWriteService: validateServiceResponse returned a NULL JsonObject");
			}
			JsonElement dataElement = jsonObject.get("uid");
			if (dataElement == null) {
				throw new Exception("Invalid response received(uid is missing) from ClinicalObjectWriteService");
			}
			
			WorkflowProcessInstance workflowProcessInstance = (WorkflowProcessInstance) processInstance;
			workflowProcessInstance.setVariable("clinicalObjectUid", dataElement.getAsString());
			
		} 
		catch (Exception e) {
			LOGGER.error("Lab Business Process: An unexpected condition has happened: " + e.getMessage(), e);
		}
	}

	/**
	 * Checks if the EventWriteService called from Update Processed Event State task
	 * returns a valid status.
	 * 
	 * Called from Store Clinical Object On Exit Action script
	 * @param processInstance a reference to the running process instance
	 */
	public static void exitUpdateProcessedEventState(ProcessInstance processInstance) {
		try {
			JsonObject jsonObject = validateServiceResponse(processInstance, "EventStateWriteService");
			if(jsonObject == null) {
				throw new Exception("EventStateWriteService: validateServiceResponse returned a NULL JsonObject");
			}			
		} 
		catch (Exception e) {
			LOGGER.error("Lab Business Process: An unexpected condition has happened: " + e.getMessage(), e);
		}
	}


	/**
	 * Checks if the SignalRegistrationService (aka Register for Signal) returns a valid status.
	 * 
	 * Called from Register for Signal On Exit Action script
	 * @param processInstance a reference to the running process instance
	 */
	public static void exitSignalRegistrationService(ProcessInstance processInstance) {
		try {
			JsonObject jsonObject = validateServiceResponse(processInstance, "SignalRegistrationService");
			if(jsonObject == null) {
				throw new Exception("EventStateWriteService: validateServiceResponse returned a NULL JsonObject");
			}			
		} 
		catch (Exception e) {
			LOGGER.error("Lab Business Process: An unexpected condition has happened: " + e.getMessage(), e);
		}
	}

	/**
	 * Checks if the SignalWriteService (e.g Write signal history) returns a valid status.
	 * 
	 * Called from Write Signal History - Exit Action script in BOTH Monitor for Collection and Signing subprocess
	 * @param processInstance a reference to the running process instance
	 */
	public static void exitSignalWriteService(ProcessInstance processInstance) {
		try {
			JsonObject jsonObject = validateServiceResponse(processInstance, "SignalWriteService");
			if(jsonObject == null) {
				throw new Exception("SignalWriteService: validateServiceResponse returned a NULL JsonObject");
			}			
		} 
		catch (Exception e) {
			LOGGER.error("Lab Business Process: An unexpected condition has happened: " + e.getMessage(), e);
		}
	}
	

	/**
	 * Processes the received signal from ORDER.UPDATED
	 * 
	 * @param processInstance
	 *            a reference to the running process instance
	 */
	public static void processOrderUpdated(ProcessInstance processInstance) {
		WorkflowProcessInstance workflowProcessInstance = (WorkflowProcessInstance) processInstance;
		try {
			SignalData signalData = WorkflowProcessInstanceUtil.getRequiredSignalData(workflowProcessInstance,
					"signalData");
			LOGGER.debug(String.format("LabHelperUtil.processOrderUpdated: %s", signalData.toString()));
			
			workflowProcessInstance.setVariable("orderStatusCode", signalData.getOrderStatusCode());
			workflowProcessInstance.setVariable("icn", signalData.getPid());
			workflowProcessInstance.setVariable("facility", signalData.getFacilityCode());
			workflowProcessInstance.setVariable("isValidStatusCode",
					LabHelperUtil.isValidStatusCode(signalData.getOrderStatusCode()));
			LOGGER.debug(String.format("kcontext.getVariable(isValidStatusCode): %s", workflowProcessInstance.getVariable("isValidStatusCode")));

			workflowProcessInstance.setVariable("signalHistory_internal",
					LabHelperUtil.transformStatusCode(signalData.getOrderStatusCode()));
			
			if (!KieFunctions.isEmpty(signalData.getUrgency())) {
				workflowProcessInstance.setVariable("urgency", signalData.getUrgency());
				int urgencyNumber = Integer.parseInt(signalData.getUrgency());
				workflowProcessInstance.setVariable("priority", urgencyNumber == 2 ? 1 : urgencyNumber == 4 ? 5 : 9);
			}
			if (!KieFunctions.isEmpty(signalData.getProviderUid())) {
				String[] arr = signalData.getProviderUid().split(":");
				if (arr.length > 4) {
					workflowProcessInstance.setVariable("orderingProviderId", arr[3] + ";" + arr[4]);
				}
			}
		}
		catch(Exception e) {
			LOGGER.error("Lab Business Process: An unexpected condition has happened with an integer: " + e.getMessage(), e);
		}
	}

	/**
	 * Processes the received signal from ORDER.UPDATED in Signing SubProcess
	 * 
	 * @param processInstance
	 *            a reference to the running process instance
	 */
	public static void processSigningOrderUpdated(ProcessInstance processInstance) {
		WorkflowProcessInstance workflowProcessInstance = (WorkflowProcessInstance) processInstance;
		try {
			SignalData signalData = WorkflowProcessInstanceUtil.getRequiredSignalData(workflowProcessInstance,
					"signalData");
			LOGGER.debug(String.format("LabHelperUtil.processSigningOrderUpdated: %s", signalData.toString()));

			workflowProcessInstance.setVariable("orderStatusCode", signalData.getOrderStatusCode());
			workflowProcessInstance.setVariable("isValidStatusCode",
					LabHelperUtil.isValidStatusCode(signalData.getOrderStatusCode()));

			if (!KieFunctions.isEmpty(signalData.getClinicalObjectUid())) {
				if (!KieFunctions.equalsTo(signalData.getClinicalObjectUid(), workflowProcessInstance.getVariable("clinicalObjectUid")+"")) {
					workflowProcessInstance.setVariable("clinicalObjectUid", signalData.getClinicalObjectUid());
				}
			}
			workflowProcessInstance.setVariable("signalHistory_internal",
					LabHelperUtil.transformStatusCode(signalData.getOrderStatusCode()));
			if (!KieFunctions.isEmpty(signalData.getLabTestText())) {
				workflowProcessInstance.setVariable("instanceName", signalData.getLabTestText());
			}
			if (!KieFunctions.isEmpty(signalData.getNoResultNotificationDate())) {
				workflowProcessInstance.setVariable("noResultNotificationDate", signalData.getNoResultNotificationDate());
				workflowProcessInstance.setVariable("notificationDate", signalData.getNoResultNotificationDate());
			}
			if (!KieFunctions.isEmpty(signalData.getUrgency())) {
				workflowProcessInstance.setVariable("urgency", signalData.getUrgency());
				int urgencyNumber = Integer.parseInt(signalData.getUrgency());
				workflowProcessInstance.setVariable("priority", urgencyNumber == 2 ? 1 : urgencyNumber == 4 ? 5 : 9);
			}
			if (!KieFunctions.isEmpty(signalData.getProviderUid())) {
				String[] arr = signalData.getProviderUid().split(":");
				if (arr.length > 4) {
					workflowProcessInstance.setVariable("signalOwner_internal", arr[3] + ";" + arr[4]);
				}
			}
		}
		catch(Exception e) {
			LOGGER.error("Lab Business Process: An unexpected condition has happened: " + e.getMessage(), e);
		}
	}
	

	/**
	 * Processes the received signal from ORDER.UPDATED in Monitor for Collection SubProcess
	 * 
	 * @param processInstance
	 *            a reference to the running process instance
	 */
	public static void processCollectionOrderUpdated(ProcessInstance processInstance) {
		WorkflowProcessInstance workflowProcessInstance = (WorkflowProcessInstance) processInstance;
		try {
			SignalData signalData = WorkflowProcessInstanceUtil.getRequiredSignalData(workflowProcessInstance,
					"signalData");
			LOGGER.debug(String.format("LabHelperUtil.processSigningOrderUpdated: %s", signalData.toString()));

			workflowProcessInstance.setVariable("orderStatusCode", signalData.getOrderStatusCode());
			workflowProcessInstance.setVariable("isValidStatusCode", LabHelperUtil.isValidStatusCode(signalData.getOrderStatusCode()));
			
			if (!KieFunctions.isEmpty(signalData.getClinicalObjectUid())) {
				if (!KieFunctions.equalsTo(signalData.getClinicalObjectUid(), (String)workflowProcessInstance.getVariable("clinicalObjectUid"))) {
					workflowProcessInstance.setVariable("clinicalObjectUid", signalData.getClinicalObjectUid());
				}
			}
			workflowProcessInstance.setVariable("signalHistory_internal", LabHelperUtil.transformStatusCode(signalData.getOrderStatusCode()));
			
			if (!KieFunctions.isEmpty(signalData.getProviderUid())) {
				String[] arr = signalData.getProviderUid().split(":");
				if (arr.length > 4) {
					workflowProcessInstance.setVariable("signalOwner_internal", arr[3] + ";" + arr[4]);
				}
			}
		}
		catch(Exception e) {
			LOGGER.error("Lab Business Process: An unexpected condition has happened: " + e.getMessage(), e);
		}
	}
		
		
	/**
	 * Processes the received signal from ORDER.UPDATED in Signing SubProcess
	 * 
	 * @param processInstance
	 *            a reference to the running process instance
	 */
	public static void checkNotificationDateForCollection(ProcessInstance processInstance) {
		WorkflowProcessInstance workflowProcessInstance = (WorkflowProcessInstance) processInstance;
		try	{
			DateTime notificationDateTime = DATETIME_FORMATTER.parseDateTime(workflowProcessInstance.getVariable("notificationDate")+"");
			DateTime dtNow = new DateTime(DateTimeZone.UTC);
			long durationInMins = (notificationDateTime.getMillis() - dtNow.getMillis()) / 60000;
			if (durationInMins <= 0) {
				workflowProcessInstance.setVariable("waitTime", "0");
			} else {
				workflowProcessInstance.setVariable("waitTime", durationInMins + "m");
			}
			LOGGER.debug(String.format("waitTime:%s", workflowProcessInstance.getVariable("waitTime")));
			LOGGER.debug(String.format("notificationDate:", workflowProcessInstance.getVariable("notificationDate")));
			LOGGER.debug(String.format("actor:", workflowProcessInstance.getVariable("actor")));
			LOGGER.debug(String.format("groups:%s", workflowProcessInstance.getVariable("groups")));
		}
		catch(Exception e) {
			LOGGER.error("Lab Business Process: An unexpected condition has happened: " + e.getMessage(), e);
		}
	}
	
	
	
	/**
	 * Processes the received signal from ORDER.UPDATED in Signing SubProcess
	 * 
	 * @param processInstance
	 *            a reference to the running process instance
	 */
	public static void exitLabSampleMissing(ProcessInstance processInstance) {
		WorkflowProcessInstance workflowProcessInstance = (WorkflowProcessInstance) processInstance;
		try {
			String formAction = workflowProcessInstance.getVariable("formAction")+"";
			if (!KieFunctions.equalsTo(formAction, "Remind Me Later")
					&& !KieFunctions.equalsTo(formAction, "Reassign Reminder")
					&& !KieFunctions.equalsTo(formAction, "Cancel Reminder")) {
				throw new Exception("Invalid formAction");
			}
			if (KieFunctions.equalsTo(formAction, "Remind Me Later")
					&& !KieFunctions.isEmpty((String)workflowProcessInstance.getVariable("newNotificationDate_internal"))) {
				workflowProcessInstance.setVariable("notificationDate", (String)workflowProcessInstance.getVariable("newNotificationDate_internal"));
			}
			if (KieFunctions.equalsTo(formAction, "Reassign Reminder")) {
				workflowProcessInstance.setVariable("actor", (String)workflowProcessInstance.getVariable("newActor_internal"));
				workflowProcessInstance.setVariable("groups", (String)workflowProcessInstance.getVariable("newGroups_internal"));
			}
			workflowProcessInstance.setVariable("taskHistory", (String)workflowProcessInstance.getVariable("userComment"));
			workflowProcessInstance.setVariable("taskHistoryAction", (String)workflowProcessInstance.getVariable("formAction"));
		}
		catch(Exception e) {
			System.out.println("Lab Business Process: An unexpected condition has happened: " + e.getMessage());
		}
	}
	
		
	/**
	 * Check Notification Date in Monitor for Results subprocess
	 * 
	 * @param processInstance
	 *            a reference to the running process instance
	 */
	public static void checkNotificationDate(ProcessInstance processInstance) {
		WorkflowProcessInstance workflowProcessInstance = (WorkflowProcessInstance) processInstance;
		try {
			DateTime notificationDateTime = DATETIME_FORMATTER.parseDateTime((String) (String)workflowProcessInstance.getVariable("notificationDate"));
			DateTime dtNow = new DateTime(DateTimeZone.UTC);
			long durationInMins = (notificationDateTime.getMillis() - dtNow.getMillis()) / 60000;
			if (durationInMins <= 0) {
				workflowProcessInstance.setVariable("waitTime", "0");
			} else {
				workflowProcessInstance.setVariable("waitTime", durationInMins + "m");
			}
			LOGGER.debug(String.format("waitTime:%s", (String)workflowProcessInstance.getVariable("waitTime")));
		}
		catch(Exception e) {
			LOGGER.error("Lab Business Process: An unexpected condition has happened: " + e.getMessage(), e);
		}		
	}
		
	
	/**
	 * Set Lab Results Missing HumanTask properties before invoking the task in
	 * Monitor for Results subprocess
	 * 
	 * @param processInstance
	 *            a reference to the running process instance
	 */
	public static void entryLabResultsMissing(ProcessInstance processInstance) {
		WorkflowProcessInstance workflowProcessInstance = (WorkflowProcessInstance) processInstance;
		DateTime earlyDate = new DateTime(DateTimeZone.UTC); 
		String earlyDateString = earlyDate.toString(ISODateTimeFormat.dateTimeNoMillis()); 
		workflowProcessInstance.setVariable("earliestDate", earlyDateString);
		DateTime dueDate = new DateTime(DateTimeZone.UTC).plusDays(30); 
		String dueDateString = dueDate.toString(ISODateTimeFormat.dateTimeNoMillis()); 
		workflowProcessInstance.setVariable("p_dueDate", dueDateString); 
	}


	/**
	 * Set Lab Results Missing HumanTask properties before invoking the task in
	 * Monitor for Results subprocess
	 * 
	 * @param processInstance
	 *            a reference to the running process instance
	 */
	public static void exitLabResultsMissing(ProcessInstance processInstance) {
		WorkflowProcessInstance workflowProcessInstance = (WorkflowProcessInstance) processInstance;
		try	{
			String formAction = (String)workflowProcessInstance.getVariable("formAction");
			if ( !LEGAL_FORM_ACTIONS.contains(formAction)) {
				throw new Exception("Invalid formAction");
			}

			if (KieFunctions.equalsTo(formAction, "Remind Me Later")
					&& !KieFunctions.isEmpty((String)workflowProcessInstance.getVariable("newNotificationDate_internal"))) {
				workflowProcessInstance.setVariable("notificationDate", (String)workflowProcessInstance.getVariable("newNotificationDate_internal"));
			} else if (KieFunctions.equalsTo(formAction, "Reassign Reminder")) {
				workflowProcessInstance.setVariable("actor", (String)workflowProcessInstance.getVariable("newActor_internal"));
				workflowProcessInstance.setVariable("groups", (String)workflowProcessInstance.getVariable("newGroups_internal"));
			}
			workflowProcessInstance.setVariable("taskHistory", (String)workflowProcessInstance.getVariable("userComment"));
			workflowProcessInstance.setVariable("taskHistoryAction", formAction);
		}
		catch(Exception e) {
			LOGGER.error("Lab Business Process: An unexpected condition has happened: " + e.getMessage(), e);
		}
	}

	

	/**
	 * Processes the received signal from ORDER.UPDATED in Monitor for Collection SubProcess
	 * 
	 * @param processInstance
	 *            a reference to the running process instance
	 */
	public static void processMonitorResultsOrderUpdated(ProcessInstance processInstance) {
		WorkflowProcessInstance workflowProcessInstance = (WorkflowProcessInstance) processInstance;
		try {
			SignalData signalData = WorkflowProcessInstanceUtil.getRequiredSignalData(workflowProcessInstance,
					"signalData");
			LOGGER.debug(String.format("LabHelperUtil.processSigningOrderUpdated: %s", signalData.toString()));
			
			workflowProcessInstance.setVariable("orderStatusCode", signalData.getOrderStatusCode());
			workflowProcessInstance.setVariable("isValidStatusCode", LabHelperUtil.isValidStatusCode(signalData.getOrderStatusCode()));
			
			if (!KieFunctions.isEmpty(signalData.getClinicalObjectUid())) {
				if (!KieFunctions.equalsTo(signalData.getClinicalObjectUid(), 
						(String)workflowProcessInstance.getVariable("clinicalObjectUid"))) {
					
					workflowProcessInstance.setVariable("clinicalObjectUid", signalData.getClinicalObjectUid());
				}
			}
			workflowProcessInstance.setVariable("signalHistory_internal", LabHelperUtil.transformStatusCode(signalData.getOrderStatusCode()));
			
			if (!KieFunctions.isEmpty(signalData.getProviderUid())) {
				String[] arr = signalData.getProviderUid().split(":");
				if (arr.length > 4) {
					workflowProcessInstance.setVariable("signalOwner_internal", arr[3] + ";" + arr[4]);
				}
			}
		}
		catch(Exception e) {
			LOGGER.error("Lab Business Process: An unexpected condition has happened: " + e.getMessage(), e);
		}
	}
	
	
	//-------------------------------------------------------------------------------------------------
	//         ------------------------ COMMON UTILITY FUNCTIONS ----------------------------
	//-------------------------------------------------------------------------------------------------
	/**
	 * Processes ServiceResponse received from calling a service Called from
	 * service On Exit Action script
	 * 
	 * @param processInstance
	 *            a reference to the running process instance
	 * @param taskName
	 *            the service name being called
	 *            
	 *  Returns the validated WIH serviceResponse as a JsonObject or NULL if an exception is thrown/caught
	 */
	public static JsonObject validateServiceResponse(ProcessInstance processInstance, String taskName) {
		WorkflowProcessInstance workflowProcessInstance = (WorkflowProcessInstance) processInstance;
		try {
			String serviceResponse = WorkflowProcessInstanceUtil.getRequiredString(workflowProcessInstance,
					"serviceResponse");
			LOGGER.debug("serviceResponse: " + serviceResponse);
			JsonParser parser = new JsonParser();
			JsonObject jsonObject = parser.parse(serviceResponse).getAsJsonObject();
			JsonElement dataElement = jsonObject.get("status");
			if (dataElement == null) {
				throw new Exception("Invalid response received(status is missing) from Task " + taskName);
			}
			if (dataElement.getAsInt() != 200) {
				dataElement = jsonObject.get("error");
				if (dataElement != null && !dataElement.getAsString().isEmpty()) {
					throw new Exception(dataElement.getAsString());
				} else {
					throw new Exception("UNKNOWN ERROR");
				}
			}
			return jsonObject;
		} catch (Exception e) {
			LOGGER.error("Lab Business Process: An unexpected condition has happened: " + e.getMessage(), e);
		}
		// There has been an error so return null;
		return null;
	}
}