package org.opencds.config.migrate.builder;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.opencds.common.cache.OpencdsCache;
import org.opencds.common.exceptions.OpenCDSRuntimeException;
import org.opencds.common.terminology.CodeSystems;
import org.opencds.common.xml.XmlConverter;
import org.opencds.common.xml.XmlEntity;
import org.opencds.config.api.model.Concept;
import org.opencds.config.api.model.ConceptDeterminationMethod;
import org.opencds.config.api.model.ConceptMapping;
import org.opencds.config.api.model.impl.CDMIdImpl;
import org.opencds.config.api.model.impl.ConceptDeterminationMethodImpl;
import org.opencds.config.api.model.impl.ConceptImpl;
import org.opencds.config.api.model.impl.ConceptMappingImpl;
import org.opencds.config.migrate.ConfigResourceType;
import org.opencds.config.migrate.OpencdsBaseConfig;
import org.opencds.config.migrate.cache.ConfigCacheRegion;
import org.opencds.config.migrate.terminology.CD;
import org.opencds.config.migrate.utilities.ConfigResourceUtility;
import org.xml.sax.SAXParseException;

public class CDMConfigBuilder {
    private static Log log = LogFactory.getLog(CDMConfigBuilder.class);
    
    private static final String CODE = "code";
    private static final String CODE_SYSTEM = "codeSystem";
    private static final String CODE_SYSTEM_NAME = "codeSystemName";
    private static final String CODE_SYSTEM_OID = "codeSystemOID";
    private static final String CODE_SYSTEM_DISPLAY_NAME = "codeSystemDisplayName";
    private static final String CONCEPT_DETERMINATION_METHOD = "conceptDeterminationMethod";
    private static final String DISPLAY_NAME = "displayName";
    private static final String MEMBERS_FOR_CODE_SYSTEM = "membersForCodeSystem";
    private static final String OPENCDS_CONCEPT = "openCdsConcept";

    private Map<String, String> myCodeSystemOIDtoCodeSystemDisplayNameMap = new HashMap<>(); 
    
    private final OpencdsBaseConfig opencdsBaseConfig;
    private final ConfigResourceUtility configResourceUtility;

    public CDMConfigBuilder(OpencdsBaseConfig opencdsBaseConfig, ConfigResourceUtility configResourceUtility) {
        this.opencdsBaseConfig = opencdsBaseConfig;
        this.configResourceUtility = configResourceUtility;
    }

    public void loadCDMs(OpencdsCache cache) {
        loadCodeSystems(opencdsBaseConfig, cache);
        loadConceptFiles(opencdsBaseConfig, cache);
    }
    
    private void loadCodeSystems(OpencdsBaseConfig opencdsBaseConfig, OpencdsCache cache) {
        // load code system OID to display name map and displayName to OID map      
        String openCDSCodeSystems = configResourceUtility.getResourceAsString(opencdsBaseConfig, ConfigResourceType.CODE_SYSTEMS);
        XmlEntity codeSystemsRootEntity = null;
        try 
        {
            codeSystemsRootEntity = XmlConverter.getInstance().unmarshalXml(openCDSCodeSystems, false, null);
        } catch (SAXParseException e) {
            e.printStackTrace();
            throw new OpenCDSRuntimeException("SimpleTerminologyService.initialize() failed to unmarshall '" + openCDSCodeSystems + "' " + e.getMessage());
        }

        // get target OpenCDS resource element
        ArrayList<XmlEntity> codeSystemEntitites = codeSystemsRootEntity.getChildrenWithLabel(CODE_SYSTEM); 
        for (XmlEntity codeSystemEntity : codeSystemEntitites)
        {
            String  codeSystemOid           = codeSystemEntity.getAttributeValue(CODE_SYSTEM_OID);
            String  codeSystemDisplayName   = codeSystemEntity.getAttributeValue(CODE_SYSTEM_DISPLAY_NAME);
            myCodeSystemOIDtoCodeSystemDisplayNameMap.put(codeSystemOid, codeSystemDisplayName);
            log.trace("codeSystem: " + codeSystemOid + ", name: " + codeSystemDisplayName);
        }   
    }
    
    private void loadConceptFiles(OpencdsBaseConfig config, OpencdsCache cache) {
        // load concept specification files
        // autogenerated concept mappings
        Map<CD, Map<Concept, Set<Concept>>> conceptDeterminationMethods = new HashMap<>();
        List<String> fileNames = configResourceUtility.listResourceNamesByType(config, ConfigResourceType.AUTOGEN_CONCEPT_MAPPINGS_SPEC);
        loadEachFile(conceptDeterminationMethods, fileNames);

        // manual concept mappings
        List<String> fileNames2 = configResourceUtility.listResourceNamesByType(config, ConfigResourceType.MANUAL_CONCEPT_MAPPINGS_SPEC);
        loadEachFile(conceptDeterminationMethods, fileNames2);

        // now loop through and build the CDMs...
        List<ConceptDeterminationMethod> cdms = new ArrayList<>();
        for (Entry<CD, Map<Concept, Set<Concept>>> cdmEntry : conceptDeterminationMethods.entrySet()) {
            List<ConceptMapping> conceptMappings = new ArrayList<>();
            for (Entry<Concept, Set<Concept>> toEntry : cdmEntry.getValue().entrySet()) {
                conceptMappings.add(ConceptMappingImpl.create(toEntry.getKey(), new ArrayList<>(toEntry.getValue())));
            }
            cdms.add(ConceptDeterminationMethodImpl.create(
                    CDMIdImpl.create(cdmEntry.getKey().getCodeSystem(), cdmEntry.getKey().getCode(), "1.0"),
                    cdmEntry.getKey().getDisplayName(),
                    cdmEntry.getKey().getDisplayName(),
                    new Date(),
                    System.getenv("USER"),
                    conceptMappings));
        }
        cache.put(ConfigCacheRegion.METADATA, ConfigResourceType.CONCEPT_DETERMINATION_METHOD, cdms);
    }
    
    private void loadEachFile(Map<CD, Map<Concept, Set<Concept>>> conceptDeterminationMethods, List<String> fileNames) {
    
        // CDM per file
        /*
         * cdm ->
         *   to ->
         *     from[]
         */
        // cdm, to, from(list)
        for (String fileName : fileNames)
        {
            String fileNameLowerCase = fileName.toLowerCase();  //so we catch .xml, .XML, .XmL, etc.  des 20121112
            if (fileNameLowerCase.endsWith(".xml")) // without this check, when packaged into jars, seems to find a filename of "" that gets processed
            {
                log.debug("attempting to unmarshal file: " + fileName);
                String fileXmlAsString = configResourceUtility.getResourceAsString(opencdsBaseConfig, fileName);

                XmlEntity fileNamesRootEntity = null;
                try {
                    fileNamesRootEntity = XmlConverter.getInstance().unmarshalXml(fileXmlAsString, false, null);
                } catch (SAXParseException e) {
                    e.printStackTrace();
                    throw new OpenCDSRuntimeException("SimpleTerminologyService.loadEachFile() failed to unmarshall '" + fileXmlAsString + "' " + e.getMessage());
                }

                // get concept determination method
                XmlEntity conceptDeterminationMethodEntity = fileNamesRootEntity.getFirstChildWithLabel(CONCEPT_DETERMINATION_METHOD);
                CD conceptDeterminationMethod  = new CD(CodeSystems.CODE_SYSTEM_OID_OPENCDS_CONCEPTS,
                        getCodeSystemFromOID(CodeSystems.CODE_SYSTEM_OID_OPENCDS_CONCEPTS),
                        conceptDeterminationMethodEntity.getAttributeValue(CODE), 
                        conceptDeterminationMethodEntity.getAttributeValue(DISPLAY_NAME));
                
                // ------------------------------------
                // update map for cdm if we haven't already...
                if (!conceptDeterminationMethods.containsKey(conceptDeterminationMethod)) {
                    conceptDeterminationMethods.put(conceptDeterminationMethod, new HashMap<Concept, Set<Concept>>());
                }
                
                // get target OpenCDS concept
                XmlEntity openCdsConceptEntity = fileNamesRootEntity.getFirstChildWithLabel(OPENCDS_CONCEPT);
                CD openCdsConcept = new CD( CodeSystems.CODE_SYSTEM_OID_OPENCDS_CONCEPTS,
                        getCodeSystemFromOID(CodeSystems.CODE_SYSTEM_OID_OPENCDS_CONCEPTS),
                                            openCdsConceptEntity.getAttributeValue(CODE), 
                                            openCdsConceptEntity.getAttributeValue(DISPLAY_NAME));
                
                // create the toConcept
                Concept toConcept = ConceptImpl.create(
                        openCdsConcept.getCode(),
                        openCdsConcept.getCodeSystem(),
                        openCdsConcept.getCodeSystemName(),
                        openCdsConcept.getDisplayName(),
                        null,
                        null);
                
                // ------------------------------------
                // update map for toConcept, if we already haven't
                if (!conceptDeterminationMethods.get(conceptDeterminationMethod).containsKey(toConcept)) {
                    conceptDeterminationMethods.get(conceptDeterminationMethod).put(toConcept, new LinkedHashSet<Concept>());
                }

                // get members for code system
                XmlEntity membersForCodeSystemEntity = fileNamesRootEntity.getFirstChildWithLabel(MEMBERS_FOR_CODE_SYSTEM);
                String codeSystem = membersForCodeSystemEntity.getAttributeValue(CODE_SYSTEM);
                String codeSystemName = membersForCodeSystemEntity.getAttributeValue(CODE_SYSTEM_NAME);

                for (XmlEntity cdEntity : (ArrayList<XmlEntity>) membersForCodeSystemEntity.getChildren())
                {
                    CD cd = new CD(codeSystem, "", cdEntity.getAttributeValue(CODE), cdEntity.getAttributeValue(DISPLAY_NAME));
                    conceptDeterminationMethods.get(conceptDeterminationMethod).get(toConcept).add(
                            ConceptImpl.create(cd.getCode(), codeSystem, codeSystemName, cd.getDisplayName(), null, null));
                }
            }
        }
    }

    /**
     * Returns a String containing the OpenCDS CodeSystem corresponding to the submitted OID
     * key = CodeSystem OID; 
     * target = CodeSystem Display name.
     * 
     * @param codeSystemOID
     * @return
     */
    private String getCodeSystemFromOID(String codeSystemOID) {
        if ((codeSystemOID == null) || ("".equals(codeSystemOID))) {
            return null;
        } else {
            return myCodeSystemOIDtoCodeSystemDisplayNameMap.get(codeSystemOID);
        }
    }

}
