'use strict';

//-----------------------------------------------------------------------------------
//  This module contains utilities for manipulating the worker-config.json structure.
//
// Author: Les Westberg
//------------------------------------------------------------------------------------

var _ = require('underscore');
var queueConfig = require(global.VX_JOBFRAMEWORK).QueueConfig;

//------------------------------------------------------------------------------------
// This function creates a node containing the vistA sites set up so they are keyed
// by stationNumber rather than by siteHash.
//
// config: The worker config file.
// returns The worker config file that has been changed.
//-----------------------------------------------------------------------------------
function createVistaSitesByStationCombined(config) {
    if (!_.isObject(config)) {
        return config;
    }

    var vistaSitesByStationCombined = {};

    // First pick up the vistaSites items.
    //------------------------------------
    if (_.isObject(config.vistaSites)) {
        _.each(config.vistaSites, function(vistaSite, siteHash) {
            if ((_.isObject(vistaSite)) && (vistaSite.stationNumber)) {
                vistaSite.siteHash = siteHash;
                vistaSitesByStationCombined[String(vistaSite.stationNumber)] = vistaSite;
            }
        });
    }

    // Next pick up the HDR sites items
    //----------------------------------
    if ((_.isObject(config.hdr)) && (_.isObject(config.hdr.hdrSites))) {
        _.each(config.hdr.hdrSites, function(hdrSite, siteHash) {
            if ((_.isObject(hdrSite)) && (hdrSite.stationNumber)) {
                hdrSite.siteHash = siteHash;
                vistaSitesByStationCombined[String(hdrSite.stationNumber)] = hdrSite;
            }
        });
    }

    config.vistaSitesByStationCombined = vistaSitesByStationCombined;
    return config;
}

//-----------------------------------------------------------------------------------
// Populates beanstalk section for main config node and each "vxsyncEnvironments" nodes
//
// Parameters:
// config: The worker config file.
//-----------------------------------------------------------------------------------
function populateBeanstalkConfigs(config) {
    if(!_.isObject(config)){
        return;
    }

    if(config.beanstalk){
        config.beanstalk = queueConfig.createFullBeanstalkConfig(config.beanstalk);
    }
    if(config.osync && config.osync.beanstalk) {
        config.osync.beanstalk = queueConfig.createFullBeanstalkConfig(config.osync.beanstalk);
    }

    _.each(config.vxsyncEnvironments, function(environment) {
        if (environment.vxsync && environment.vxsync.beanstalk){
            environment.vxsync.beanstalk = queueConfig.createFullBeanstalkConfig(environment.vxsync.beanstalk);
        }

        if (environment.osync && environment.osync.beanstalk) {
            environment.osync.beanstalk = queueConfig.createFullBeanstalkConfig(environment.osync.beanstalk);
        }
    });
}

module.exports.createVistaSitesByStationCombined = createVistaSitesByStationCombined;
module.exports.populateBeanstalkConfigs = populateBeanstalkConfigs;