'use strict';

require('../../../../env-setup');

var _ = require('underscore');

var log = require(global.VX_DUMMIES + 'dummy-logger');

// NOTE: be sure next lines are commented out before pushing
// log = require('bunyan').createLogger({
//     name: 'job-middleware-spec',
//     level: 'debug'
// });

var DummyRequest = require('../../../frames/dummy-request');
var DummyResponse = require('../../../frames/dummy-response');

var mockJobData = require(global.VX_ROOT + 'mocks/jds/jds-mock-job-data');
var mockSyncData = require(global.VX_ROOT + 'mocks/jds/jds-mock-sync-data');

var JobAPI = require(global.VX_UTILS + 'middleware/job-middleware');

var options = { 'log': log, 'config': {}, 'jdsClient': {}, 'jobStatusUpdater': {} };

var JdsClientDummy = require(global.VX_DUMMIES + 'jds-client-dummy');

var buildJob = JobAPI._test._buildJob;
var getJobHistory = JobAPI._test._getJobHistory;
var getSyncStatus = JobAPI._test._getSyncStatus;
var jobVerification = JobAPI._test._jobVerification;
var interceptUnforcedJobs = JobAPI._test._interceptUnforcedJobs;
var publishJob = JobAPI._test._publishJob;

var jobUtil = require(global.VX_UTILS + 'job-utils');

describe('middleware/job.js', function() {
    describe('buildJob()', function() {
        it('Builds an enterprise-sync-request job from a valid request', function() {
            var request = new DummyRequest({
                'pid': 'ABCD;0'
            });
            request.jpid = '21EC2020-3AEA-4069-A2DD-08002B30309D';
            request.patientIdentifier = {
                'type': 'pid',
                'value': 'ABCD;0'
            };
            request.referenceInfo = {
                'requestId': 'job-middleware-requestId',
                'sessionId': 'job-middleware-sessionId'
            };
            var response = new DummyResponse();
            var jobFactory = function(r) {
                return jobUtil.createEnterpriseSyncRequest(r.patientIdentifier, r.jpid, r.force, null, null, r.referenceInfo);
            };
            buildJob.call(options, jobFactory, request, response, function() {});
            expect(response.job).toBeDefined();
            expect(response.job.type).toEqual('enterprise-sync-request');
            expect(response.job.jpid).toEqual('21EC2020-3AEA-4069-A2DD-08002B30309D');
            expect(response.job.patientIdentifier.type).toEqual('pid');
            expect(response.job.patientIdentifier.value).toEqual('ABCD;0');
            expect(response.job.referenceInfo).toBeDefined();
            expect(response.job.referenceInfo.requestId).toBe('job-middleware-requestId');
            expect(response.job.referenceInfo.sessionId).toBe('job-middleware-sessionId');
        });
    });

    describe('getJobHistory()', function() {
        var opts = _.clone(options);
        beforeEach(function() {
            opts.jdsClient.getJobStatus = jasmine.createSpy().andCallFake(function(job, filter, callback) {
                callback(null, {statusCode: 200}, { 'items': mockJobData(1)[job.jpid] });
            });
        });

        it('Retrieves the job history for a known JPID', function() {
            var request = new DummyRequest({
                'pid': 'ABCD;3'
            });
            request.jpid = '21EC2020-3AEA-4069-A2DD-CCCCCCCCCCCC';
            request.patientIdentifier = {
                'type': 'pid',
                'value': 'ABCD;3'
            };
            var response = new DummyResponse();
            response.job = {
                'type': 'enterprise-sync-request',
                'patientIdentifier': request.patientIdentifier,
                'jpid': request.jpid
            };
            var next = jasmine.createSpy().andCallFake(function() {
                expect(response.jobStates).toBeDefined();
                expect(response.jobStates[0]).toBeDefined();
                expect(response.jobStates[0].jpid).toEqual(request.jpid);
                expect(opts.jdsClient.getJobStatus).toHaveBeenCalled();
            });

            runs(function() {
                getJobHistory.call(opts, request, response, next);
            });

            waitsFor(function() {
                return next.callCount > 0;
            });
        });

        it('Retrieves the filtered job history for a known JPID', function() {
            var request = new DummyRequest({
                'pid': 'ABCD;3'
            });
            request.jpid = '21EC2020-3AEA-4069-A2DD-CCCCCCCCCCCC';
            request.filter = {
                'filter': '?filter=eq(status,\"created\")'
            };
            request.patientIdentifier = {
                'type': 'pid',
                'value': 'ABCD;3'
            };
            var response = new DummyResponse();
            response.job = {
                'type': 'enterprise-sync-request',
                'patientIdentifier': request.patientIdentifier,
                'jpid': request.jpid
            };
            var next = jasmine.createSpy().andCallFake(function() {
                expect(response.jobStates).toBeDefined();
                expect(response.jobStates[0]).toBeDefined();
                expect(response.jobStates[0].jpid).toEqual(request.jpid);
                expect(opts.jdsClient.getJobStatus).toHaveBeenCalled();
            });

            runs(function() {
                getJobHistory.call(opts, request, response, next);
            });

            waitsFor(function() {
                return next.callCount > 0;
            });
        });

        it('Error path: get error from JDS (no filter)', function(){
            var request = new DummyRequest({
                'pid': 'ABCD;3'
            });
            request.jpid = '21EC2020-3AEA-4069-A2DD-CCCCCCCCCCCC';
            request.filter = {
                'filter': '?filter=eq(status,\"created\")'
            };
            request.patientIdentifier = {
                'type': 'pid',
                'value': 'ABCD;3'
            };
            var response = new DummyResponse();

            var options = { 'log': log, 'config': {}, 'jdsClient': {}, 'jobStatusUpdater': {} };

            var jdsClientDummy = new JdsClientDummy(log, options.config);
            jdsClientDummy._setResponseData('ERROR', null, null);
            options.jdsClient = jdsClientDummy;

            response.job = {
                'type': 'enterprise-sync-request',
                // job does not have patientIdentifier when getJobHistory is called within sync status retrieval process
                //'patientIdentifier': request.patientIdentifier,
                'jpid': request.jpid
            };

            response.status = jasmine.createSpy().andCallFake(function() {
                expect(response.jobStates).not.toBeDefined();
                expect(response.status.calls).toBeDefined();
                expect(response.status.calls[0]).toBeDefined();
                expect(response.status.calls[0].args).toBeDefined();
                expect(response.status.calls[0].args[0]).toEqual(500);
                return this;
            });

            runs(function() {
                getJobHistory.call(options, request, response, function(){});
            });

            waitsFor(function() {
                return response.status.callCount > 0;
            });
        });

        it('Error path: get error from JDS (with filter)', function(){
            var request = new DummyRequest({
                'pid': 'ABCD;3'
            });
            request.jpid = '21EC2020-3AEA-4069-A2DD-CCCCCCCCCCCC';
            request.patientIdentifier = {
                'type': 'pid',
                'value': 'ABCD;3'
            };
            var response = new DummyResponse();

            var options = { 'log': log, 'config': {}, 'jdsClient': {}, 'jobStatusUpdater': {} };

            var jdsClientDummy = new JdsClientDummy(log, options.config);
            jdsClientDummy._setResponseData('ERROR', null, null);
            options.jdsClient = jdsClientDummy;

            response.job = {
                'type': 'enterprise-sync-request',
                // job does not have patientIdentifier when getJobHistory is called within sync status retrieval process
                //'patientIdentifier': request.patientIdentifier,
                'jpid': request.jpid
            };

            response.status = jasmine.createSpy().andCallFake(function() {
                expect(response.jobStates).not.toBeDefined();
                expect(response.status.calls).toBeDefined();
                expect(response.status.calls[0]).toBeDefined();
                expect(response.status.calls[0].args).toBeDefined();
                expect(response.status.calls[0].args[0]).toEqual(500);
                return this;
            });

            runs(function() {
                getJobHistory.call(options, request, response, function(){});
            });

            waitsFor(function() {
                return response.status.callCount > 0;
            });
        });

        it('Error path: no response from JDS', function(){
            var request = new DummyRequest({
                'pid': 'ABCD;3'
            });
            request.jpid = '21EC2020-3AEA-4069-A2DD-CCCCCCCCCCCC';
            request.patientIdentifier = {
                'type': 'pid',
                'value': 'ABCD;3'
            };
            var response = new DummyResponse();

            var options = { 'log': log, 'config': {}, 'jdsClient': {}, 'jobStatusUpdater': {} };

            var jdsClientDummy = new JdsClientDummy(log, options.config);
            jdsClientDummy._setResponseData(null, null, null);
            options.jdsClient = jdsClientDummy;

            response.job = {
                'type': 'enterprise-sync-request',
                // job does not have patientIdentifier when getJobHistory is called within sync status retrieval process
                //'patientIdentifier': request.patientIdentifier,
                'jpid': request.jpid
            };

            response.status = jasmine.createSpy().andCallFake(function() {
                expect(response.jobStates).not.toBeDefined();
                expect(response.status.calls).toBeDefined();
                expect(response.status.calls[0]).toBeDefined();
                expect(response.status.calls[0].args).toBeDefined();
                expect(response.status.calls[0].args[0]).toEqual(500);
                return this;
            });

            runs(function() {
                getJobHistory.call(options, request, response, function(){});
            });

            waitsFor(function() {
                return response.status.callCount > 0;
            });
        });

        it('Error path: non-200 response from JDS', function(){
            var request = new DummyRequest({
                'pid': 'ABCD;3'
            });
            request.jpid = '21EC2020-3AEA-4069-A2DD-CCCCCCCCCCCC';
            request.patientIdentifier = {
                'type': 'pid',
                'value': 'ABCD;3'
            };
            var response = new DummyResponse();

            var options = { 'log': log, 'config': {}, 'jdsClient': {}, 'jobStatusUpdater': {} };

            var jdsClientDummy = new JdsClientDummy(log, options.config);
            jdsClientDummy._setResponseData(null, {statusCode: 404, body: 'JPID is a required field'}, null);
            options.jdsClient = jdsClientDummy;

            response.job = {
                'type': 'enterprise-sync-request',
                // job does not have patientIdentifier when getJobHistory is called within sync status retrieval process
                //'patientIdentifier': request.patientIdentifier,
                'jpid': request.jpid
            };

            response.status = jasmine.createSpy().andCallFake(function() {
                expect(response.jobStates).not.toBeDefined();
                expect(response.status.calls).toBeDefined();
                expect(response.status.calls[0]).toBeDefined();
                expect(response.status.calls[0].args).toBeDefined();
                expect(response.status.calls[0].args[0]).toEqual(500);
                return this;
            });

            runs(function() {
                getJobHistory.call(options, request, response, function(){});
            });

            waitsFor(function() {
                return response.status.callCount > 0;
            });
        });
    });

    describe('getSyncStatus()', function() {
        var opts = _.clone(options);
        beforeEach(function() {
            opts.jdsClient.getSyncStatus = jasmine.createSpy().andCallFake(function(job, callback) {
                callback(null, {}, mockSyncData('21EC2020-3AEA-4069-A2DD-08002B30309D', 4));
            });
        });

        it('Retrieves the sync status for a known JPID', function() {
            var request = new DummyRequest({
                'pid': 'ABCD;0'
            });
            request.jpid = '21EC2020-3AEA-4069-A2DD-08002B30309D';
            request.patientIdentifier = {
                'type': 'pid',
                'value': 'ABCD;0'
            };
            var response = new DummyResponse();
            response.job = {
                'jpid': request.jpid
            };
            var next = jasmine.createSpy().andCallFake(function() {
                expect(response.syncStatus).toBeDefined();
                expect(response.syncStatus.data.items[0]).toBeDefined();
                var syncStatus = response.syncStatus.data.items[0];
                expect(syncStatus.jpid).toEqual('21EC2020-3AEA-4069-A2DD-08002B30309D');
                expect(syncStatus.completedStamp).toBeDefined();
                expect(syncStatus.completedStamp.syncCompleted).toBe(true);
                expect(syncStatus.inProgress).toBeDefined();
                expect(syncStatus.inProgress.syncCompleted).toBe(false);
                expect(opts.jdsClient.getSyncStatus).toHaveBeenCalled();
            });

            runs(function() {
                getSyncStatus.call(opts, request, response, next);
            });

            waitsFor(function() {
                return next.callCount > 0;
            });
        });
    });

    describe('publishJob()', function() {
        var dummyRouter;
        var opts = _.clone(options);
        beforeEach(function() {
            dummyRouter = {
                'publish': jasmine.createSpy().andCallFake(function(job, callback) {
                    callback(null, [ '1' ]);
                })
            };

            opts.jobStatusUpdater.writeStatus = jasmine.createSpy().andCallFake(function(job, callback) {
                callback(null, {}, job);
            });
            opts.jobStatusUpdater.createJobStatus = function(job, callback) {
                job.status = 'created';
                opts.jobStatusUpdater.writeStatus(job, callback);
            };
        });

        it('Publishes a job and posts a status update for an unforced job', function() {
            var request = new DummyRequest({
                'pid': 'ABCD;0'
            });
            request.jpid = '21EC2020-3AEA-4069-A2DD-08002B30309D';
            request.patientIdentifier = {
                'type': 'pid',
                'value': 'ABCD;0'
            };
            var response = new DummyResponse();
            response.job = {
                'type': 'enterprise-sync-request',
                'patientIdentifier': request.patientIdentifier,
                'jpid': request.jpid
            };
            var next = jasmine.createSpy();

            runs(function() {
                publishJob.call(opts, dummyRouter, request, response, next);
            });

            waitsFor(function() {
                return next.callCount > 0;
            });

            runs(function() {
                expect(dummyRouter.publish).toHaveBeenCalled();
            });
        });

        it('Publishes a job and posts two status updates for a forced job', function() {
            var request = new DummyRequest({
                'pid': 'ABCD;0'
            });
            request.jpid = '21EC2020-3AEA-4069-A2DD-08002B30309D';
            request.patientIdentifier = {
                'type': 'pid',
                'value': 'ABCD;0'
            };
            var response = new DummyResponse();
            response.job = {
                'type': 'enterprise-sync-request',
                'patientIdentifier': request.patientIdentifier,
                'jpid': request.jpid,
                'force': true
            };
            response.currentJob = {
                'jobType': 'enterprise-sync-request',
                'patientIdentifier': request.patientIdentifier,
                'jpid': request.jpid,
                'rootJobId': '0',
                'jobId': '0',
                'status': 'started'
            };
            var next = jasmine.createSpy();

            runs(function() {
                publishJob.call(opts, dummyRouter, request, response, next);
            });

            waitsFor(function() {
                return next.callCount > 0;
            });

            runs(function() {
                expect(dummyRouter.publish).toHaveBeenCalled();
            });
        });
    });
});
