'use strict';

require('../../env-setup');

var _ = require('underscore');
var format = require('util').format;
var request = require('request');
var inspect = require(global.VX_UTILS + 'inspect');
var errorUtil = require(global.VX_UTILS + 'error');
var objUtil = require(global.VX_UTILS + 'object-utils');
var uuid = require('node-uuid');
var async = require('async');
var querystring = require('querystring');
var sizeof = require('object-sizeof');

/*
config = {
    maxMetastampSize: 20000,
    handlerMaxSockets: 5
    jds: {
        protocol: 'http' // JDS protocol https|http
        host: 'IP        ' // JDS IP
        port: PORT // JDS port
        timeout: 60000
    }
}
*/
function JdsClient(log, metrics, config) {
    if (!(this instanceof JdsClient)) {
        return new JdsClient(log, metrics, config);
    }

    this.log = log;
    this.metrics = metrics;
    this.config = config;
}

JdsClient.prototype.childInstance = function(childLog) {
    var self = this;
    var newInstance = new JdsClient(childLog, self.metrics, self.config);

    return newInstance;
};

// JdsClient.prototype.clearJdsData = function(callback) {
//     this.clearPatientIdentifiers(this.clearSyncStatus(this.clearJobStates(callback)));
// };

JdsClient.prototype.saveSyncStatus = function(metastamp, patientIdentifier, callback) {
    this.log.debug('Jds-client.saveSyncStatus()');
    this.log.debug(inspect(metastamp));
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'saveSyncStatus',
        'pid': patientIdentifier.value,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Save Sync Status', metricsObj);
    var self = this;

    if (!patientIdentifier) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS Save Sync Status in Error', metricsObj);
        return setTimeout(callback, 0, errorUtil.createFatal('No patientIdentifier object passed in'));
    }

    var identifierValue = patientIdentifier.value;
    if (patientIdentifier.type === 'edipi') {
        identifierValue = 'DOD;' + identifierValue;
    }

    var path = '/status/' + identifierValue;
    var metastampJobs = self._ensureMetastampSize(metastamp, self.config.maxMetastampSize || 20000);
    if (metastampJobs.length > 1) {
        self.log.warn('JdsClient.saveSyncStatus() metastamp broken into %s parts', metastampJobs.length);
    }
    async.each(metastampJobs, function(stamp, cb) {
        self.execute(path, stamp, 'POST', metricsObj, cb);
    }, function(err) {
        var response = {}; //hardcoded to account for inability to merge multiple responses
        if (!err) {
            response.statusCode = 200;
        }
        callback(err, response);
    });
};



//----------------------------------------------------------------------------
// Variadic Function
// getSyncStatus(patientIdentifier, filter, callback)
// getSyncStatus(patientIdentifier, callback)
//
// This method fetches a sync status for a patient with the filter (if any)
// appended to the url.
//
// patientIdentifier: A patient identifier in standard VxSync format. e.g.:
//          {
//              type: 'pid',
//              value: '9E7A;3'
//          }
//
// filter: An object containing a 'filter' property with the string to append
//          to the url of the sync status endpoint. Note that any escaping, etc.
//          must be performed prior to passing it to this method as the value in
//          the 'filter' attribute will be appended as-is. An example value for
//          this parameter could be:
//          {
//              filter: '?detailed=true&filter=lt("stampTime",20170101000000)'
//          }
//          Alternatively, the 'filter' parameter can be a string, which will
//          used as the filter. e.g.:
//          '?detailed=true&filter=lt("stampTime",20170101000000)'
//
// callback: The function to invoke upon error or completion of the fetch.
//----------------------------------------------------------------------------
JdsClient.prototype.getSyncStatus = function(patientIdentifier, filter, callback) {
    this.log.debug('Jds-client.getSyncStatus()');
    this.log.debug(inspect(patientIdentifier));
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getSyncStatus',
        'pid': patientIdentifier.value,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get Sync Status', metricsObj);

    var args = _.toArray(arguments);
    callback = args.pop();
    filter = args.length > 1 ? args.pop() : null;

    var path = '/status/' + patientIdentifier.value;

    if (!_.isEmpty(filter)) {
        if (_.isString(filter)) {
            path += filter;
        } else if (!_.isEmpty(filter.filter)) {
            path += filter.filter;
        }
    }

    this.execute(path, null, 'GET', metricsObj, callback);
};

//----------------------------------------------------------------------------
// Variadic Function
// getSimpleSyncStatus(patientIdentifier, filter, callback)
// getSimpleSyncStatus(patientIdentifier, callback)
//----------------------------------------------------------------------------
JdsClient.prototype.getSimpleSyncStatus = function(patientIdentifier, callback) {
    this.log.debug('Jds-client.getSimpleSyncStatus()');
    this.log.debug(inspect(patientIdentifier));
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getSimpleSyncStatus',
        'pid': patientIdentifier.value,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get Simple Sync Status', metricsObj);

    var args = _.toArray(arguments);
    callback = args.pop();


    var path = '/sync/combinedstat/' + patientIdentifier.value;

    if (arguments.length > 2) {
        path += arguments[1].filter;
    }

    this.execute(path, null, 'GET', metricsObj, callback);
};

JdsClient.prototype.clearSyncStatus = function(callback) {
    this.log.debug('Jds-client.clearSyncStatus()');
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'clearSyncStatus',
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Clear Sync Status', metricsObj);

    var path = '/status';
    var method = 'DELETE';
    this.execute(path, null, method, metricsObj, callback);
};

JdsClient.prototype.saveJobState = function(jobState, callback) {
    this.log.debug('Jds-client.saveJobState()');
    this.log.debug(inspect(jobState));
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'saveJobState',
        'jpid': jobState.jpid,
        'jobId': jobState.jobId,
        'rootJobId': jobState.rootJobId,
        'jobType': jobState.type,
        'process': uuid.v4(),
        'timer': 'start'
    };
    if (jobState.record) {
        metricsObj.pid = jobState.record.pid;
        metricsObj.uid = jobState.record.uid;
    }
    this.metrics.debug('JDS Save Job State', metricsObj);

    var path = '/job';
    this.execute(path, jobState, 'POST', metricsObj, callback);
};


//----------------------------------------------------------------------------
// Variadic Function:
// getJobStatus(job, callback)
// getJobStatus(job, filter, callback)
//
// job: An object with a "jpid" attribute or a "patientIdentifier" attribute (see below).
//
// filter: An object with a "filter" attribute. If this value is empty, null, undefined, or
//         does not have a "filter" attribute, then it is ignored and the call will be made
//         without a filter. The filter attribute should be of the form: '?filter=<filter>'
//         where <filter> is a filter for the JDS store: ?filter=eq(type,"enterprise-sync-request")
//
// The "job" parameter can have two formats:
// {
//     jpid: 6c2d9589-7554-469a-b480-d71f9d2a5d64
// }
//
// or
//
// {
//     patientIdentifier: {
//         value: '6c2d9589-7554-469a-b480-d71f9d2a5d64'
//     }
// }
//
// If the "job" parameter has both the "jpid" AND "patientIdentifier" attributes, the "jpid"
// attribute will take precedence.
//----------------------------------------------------------------------------
JdsClient.prototype.getJobStatus = function(job, callback) {
    this.log.debug('Jds-client.getJobStatus() %j', job);
    this.log.debug(inspect(job));
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getJobStatus',
        'jpid': job.jpid,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get Job Status', metricsObj);

    if (arguments.length < 2) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS Get Job Status in Error', metricsObj);
        return setTimeout(callback, 0, 'Invalid number of arguments');
    }

    var args = _.toArray(arguments);
    callback = args.pop();

    // Figure out what identifier we are going to use... jpid or patient identifier.
    //-------------------------------------------------------------------------------
    var params = objUtil.getProperty(arguments[0], 'jpid') || objUtil.getProperty(arguments[0], 'patientIdentifier', 'value');
    if (arguments.length > 2 && _.has(arguments[1], 'filter')) {
        params += arguments[1].filter;
    }

    var path = '/job/' + params;
    this.execute(path, job, 'GET', metricsObj, callback);
};

JdsClient.prototype.clearJobStatesByPatientIdentifier = function(patientIdentifier, callback) {
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'clearJobStatesByPatientIdentifier',
        'pid': patientIdentifier.value,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Clear Job States by PID', metricsObj);
    var path = '/job/' + patientIdentifier.value;
    var method = 'DELETE';
    this.execute(path, null, method, metricsObj, callback);
};

//----------------------------------------------------------------------------
// This method retrieves the patient Identifier list from JDS for the
// patientIdentifier in the given job.
//
// job:  The job for the patient.
// callback: The handler to call when this request is completed.
//----------------------------------------------------------------------------
JdsClient.prototype.getPatientIdentifier = function(job, callback) {
    this.log.debug('jds-client.getPatientIdentifier() %j', job);

    this.getPatientIdentifierByPid(job.patientIdentifier.value, callback);
};

//----------------------------------------------------------------------------
// This method retrieves the patient Identifier list from JDS for the given
// pid.
//
// pid:  The pid for the patient.
// callback: The handler to call when this request is completed.
//----------------------------------------------------------------------------
JdsClient.prototype.getPatientIdentifierByPid = function(pid, callback) {
    this.log.debug('jds-client.getPatientIdentifierByPid() %j', pid);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getPatientIdentifierByPid',
        'pid': pid,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get Corresponding IDs by PID', metricsObj);

    var path = '/vpr/jpid/' + pid;
    this.execute(path, null, 'GET', metricsObj, callback);
};

//----------------------------------------------------------------------------
// This method retrieves the patient Identifier list from JDS for the given
// icn.
//
// icn:  The icn for the patient.
// callback: The handler to call when this request is completed.
//----------------------------------------------------------------------------
JdsClient.prototype.getPatientIdentifierByIcn = function(icn, callback) {
    this.log.debug('jds-client.getPatientIdentifierByIcn() %j', icn);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getPatientIdentifierByIcn',
        'pid': icn,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get Corresponding IDs by PID', metricsObj);

    var path = '/vpr/jpid/' + icn;
    this.execute(path, null, 'GET', metricsObj, callback);
};

JdsClient.prototype.storePatientIdentifier = function(jdsPatientIdentificationRequest, callback) {
    this.log.debug('jds-client.storePatientIdentifier() %j', jdsPatientIdentificationRequest);
    this.log.debug(jdsPatientIdentificationRequest.jpid || 'No JPID provided.');
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'storePatientIdentifier',
        'jpid': jdsPatientIdentificationRequest.jpid,
        'pid': jdsPatientIdentificationRequest.patientIdentifiers,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Store Patient Identifier', metricsObj);

    var path = '/vpr/jpid/';
    if (typeof jdsPatientIdentificationRequest.jpid !== 'undefined') {
        path += jdsPatientIdentificationRequest.jpid;
    }

    this.execute(path, jdsPatientIdentificationRequest, 'POST', metricsObj, callback);
};

JdsClient.prototype.removePatientIdentifier = function(jpid, callback) {
    this.log.debug('jds-client.removePatientIdentifier() %j', jpid);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'removePatientIdentifier',
        'jpid': jpid,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Remove Patient Identifier', metricsObj);

    var path = '/vpr/jpid/' + jpid;
    this.execute(path, null, 'DELETE', metricsObj, callback);
};

JdsClient.prototype.clearPatientIdentifiers = function(callback) {
    this.log.debug('jds-client.clearPatientIdentifiers()');
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'clearPatientIdentifiers',
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Clear Patient Identifiers', metricsObj);

    var path = '/vpr/jpid/clear';
    this.execute(path, null, 'DELETE', metricsObj, callback);
};

//-----------------------------------------------------------------------------------------
// Store the object in the operational data store area of the JDS.
//
// operationalData: The object that represents the operational data to be stored.
// callback: The handler to call when this request is completed.
//-----------------------------------------------------------------------------------------
JdsClient.prototype.storeOperationalData = function(operationalData, callback) {
    this.log.debug('jds-client.storeOperationalData()');
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'storeOperationalData',
        'site': operationalData.systemId,
        'uid': operationalData.uid,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Store OPD', metricsObj);

    if (_.isEmpty(operationalData)) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS Store OPD in Error', metricsObj);
        return setTimeout(callback, 0, errorUtil.createFatal('No record passed in'));
    }

    var path = '/data';
    this.execute(path, operationalData, 'POST', metricsObj, callback);
};

//------------------------------------------------------------------------------------------
// Retrieve pt-select data using the patient's pid.
//
// pid: The pid to use to retrieve the pt-select data.
// callback: The handler to call when this request is completed.
//-------------------------------------------------------------------------------------------
JdsClient.prototype.getOperationalDataPtSelectByPid = function(pid, callback) {
    this.log.debug('jds-client.getOperationalDataPtSelectByPid() %j', pid);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getOperationalDataPtSelectByPid',
        'pid': pid,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get PT Select by PID', metricsObj);

    if (_.isEmpty(pid)) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS Get PT Select by PID in Error', metricsObj);
        return setTimeout(callback, 0, errorUtil.createFatal('No pid passed in'));
    }

    var path = '/data/index/pt-select-pid?range="' + pid + '"';
    this.execute(path, null, 'GET', metricsObj, callback);
};

//------------------------------------------------------------------------------------------
// Retrieve pt-select data using the patient's icn.
//
// icn: The icn to use to retrieve the pt-select data.
// callback: The handler to call when this request is completed.
//-------------------------------------------------------------------------------------------
JdsClient.prototype.getOperationalDataPtSelectByIcn = function(icn, callback) {
    this.log.debug('jds-client.getOperationalDataPtSelectByIcn() %j', icn);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getOperationalDataPtSelectByIcn',
        'pid': icn,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get PT Select by ICN', metricsObj);

    if (_.isEmpty(icn)) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS Get PT Select by ICN in Error', metricsObj);
        return setTimeout(callback, 0, errorUtil.createFatal('No icn passed in'));
    }

    var path = '/data/index/pt-select-icn?range="' + icn + '"';
    this.execute(path, null, 'GET', metricsObj, callback);
};

//-----------------------------------------------------------------------------------------
// Retrieve the operational data object by its UID.
//
// uid: The UID that identifies the operational data item being retrieved.
// callback: The handler to call when this request is completed.
// Returns  (through call back): the Operational data item.
//-----------------------------------------------------------------------------------------
JdsClient.prototype.getOperationalDataByUid = function(uid, callback) {
    this.log.debug('jds-client.getOperationalDataByUid() %j', uid);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getOperationalDataByUid',
        'uid': uid,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get PT Select by UID', metricsObj);

    if (_.isEmpty(uid)) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS Get PT Select by UID in Error', metricsObj);
        return setTimeout(callback, 0, errorUtil.createFatal('No uid passed in'));
    }

    var path = '/data/' + uid;
    this.execute(path, null, 'GET', metricsObj, callback);
};

//-----------------------------------------------------------------------------------------
// Delete the operational data object by its UID.
//
// uid: The UID that identifies the operational data item being deleted.
// callback: The handler to call when this request is completed.
//-----------------------------------------------------------------------------------------
JdsClient.prototype.deleteOperationalDataByUid = function(uid, callback) {
    this.log.debug('jds-client.deleteOperationalDataByUid() %j', uid);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'deleteOperationalDataByUid',
        'uid': uid,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Delete OPD by UID', metricsObj);

    if (_.isEmpty(uid)) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS Delete OPD by UID in Error', metricsObj);
        return setTimeout(callback, 0, errorUtil.createFatal('No uid passed in'));
    }

    var path = '/data/' + uid;
    this.execute(path, null, 'DELETE', metricsObj, callback);
};

//-----------------------------------------------------------------------------------------
// Retrieve the operational data objects for clinics by site from the location collection
//
// site: The site that filters the operational clinic data item being retrieved.
// clinic: The clinic that further filters the operational clinic data being retrieved.
// callback: The handler to call when this request is completed.
// Returns  (through call back): the Operational data item.
//-----------------------------------------------------------------------------------------
JdsClient.prototype.getOperationalDataBySiteAndClinic = function(site, clinic, callback) {
    this.log.debug('jds-client.getOperationalDataBySiteAndClinic() %j: %j', site, clinic);

    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getOperationalDataBySiteAndClinic',
        'site': site,
        'clinic': clinic,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get OPD Data By Site and Clinic', metricsObj);

    if (!site) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS Get OPD by site and clinic in Error', metricsObj);
        return setTimeout(callback, 0, errorUtil.createFatal('No site passed in'));
    } else if (!clinic) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS Get OPD by site and clinic in Error', metricsObj);
        return setTimeout(callback, 0, errorUtil.createFatal('No clinic passed in'));
    }

    var path = '/data/find/location?filter=ilike("uid","%25:' + site + ':%25"),ilike("name","' + clinic + '")';
    this.execute(path, null, 'GET', metricsObj, callback);
};

//-----------------------------------------------------------------------------------------
// Store the object in the mutable operational data store area of the JDS.
//
// operationalData: The object that represents the operational data to be stored.
// callback: The handler to call when this request is completed.
//-----------------------------------------------------------------------------------------
JdsClient.prototype.storeOperationalDataMutable = function(operationalData, callback) {
    this.log.debug('jds-client.storeOperationalDataMutable()');
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'storeOperationalDataMutable',
        'uid': operationalData.uid,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Store OPD Mutable', metricsObj);

    if (_.isEmpty(operationalData)) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS Store OPD Mutable in Error', metricsObj);
        return setTimeout(callback, 0, errorUtil.createFatal('No record passed in'));
    }

    var path = '/odmutable/set/this';
    this.execute(path, operationalData, 'POST', metricsObj, callback);
};

//-----------------------------------------------------------------------------------------
// Retrieve mutable operational data object by site id.
//
// siteId: The site id that identifies the operational data item being retrieved.
// callback: The handler to call when this request is completed.
// Returns  (through call back): the Operational data item.
//-----------------------------------------------------------------------------------------
JdsClient.prototype.getOperationalDataMutable = function(siteId, callback) {
    this.log.debug('jds-client.getOperationalDataMutable() %j', siteId);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getOperationalDataMutable',
        'site': siteId,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get OPD Mutable', metricsObj);

    if (_.isEmpty(siteId)) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS Get OPD Mutable in Error', metricsObj);
        return setTimeout(callback, 0, errorUtil.createFatal('No uid passed in'));
    }

    var path = '/odmutable/get/' + siteId;
    this.execute(path, null, 'GET', metricsObj, callback);
};

//-----------------------------------------------------------------------------------------
// Retrieve multiple mutable operational data objects by a filter
//
// filter: A JDS filter string (required!)
// callback: The handler to call when this request is completed.
// Returns  (through call back): an array of items.
//-----------------------------------------------------------------------------------------
JdsClient.prototype.getOperationalDataMutableByFilter = function(filterString, callback) {
    this.log.debug('jds-client.getOperationalDataMutableByFilter');
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getOperationalDataMutableByFilter',
        'filterString': filterString,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get All OPD Mutable', metricsObj);

    if (_.isEmpty(filterString)) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS Get OPD Mutable By Filter in Error', metricsObj);
        return setTimeout(callback, 0, errorUtil.createFatal('No filterString passed in'));
    }

    var path = '/odmutable/get/' + filterString;

    this.execute(path, null, 'GET', metricsObj, callback);
};

//-----------------------------------------------------------------------------------------
// Get the number of objects in the mutable operational data store
//
// Returns  (through call back): the Operational data item.
//-----------------------------------------------------------------------------------------
JdsClient.prototype.getOperationalDataMutableCount = function(callback) {
    this.log.debug('jds-client.getOperationalDataMutableCount()');
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getOperationalDataMutableCount',
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get OPD Mutable Count', metricsObj);

    var path = '/odmutable/length/this';
    this.execute(path, null, 'GET', metricsObj, callback);
};

//-----------------------------------------------------------------------------------------
// Delete the mutable operational data object by the site id.
//
// siteId: The site id that identifies the site for which site specific operational data should be deleted.
// callback: The handler to call when this request is completed.
//-----------------------------------------------------------------------------------------
JdsClient.prototype.deleteOperationalDataMutable = function(siteId, callback) {
    this.log.debug('jds-client.deleteOperationalDataMutable() %j', siteId);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'deleteOperationalDataMutable',
        'site': siteId,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get OPD Mutable Count', metricsObj);

    if (_.isEmpty(siteId)) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS Get OPD Mutable Count in Error', metricsObj);
        return setTimeout(callback, 0, errorUtil.createFatal('No site id passed in'));
    }

    var path = '/odmutable/destroy/' + siteId;
    this.execute(path, null, 'DELETE', metricsObj, callback);
};

//-----------------------------------------------------------------------------------------
// Delete all mutable operational data.
//
// callback: The handler to call when this request is completed.
//-----------------------------------------------------------------------------------------
JdsClient.prototype.clearOperationalDataMutableStorage = function(callback) {
    this.log.debug('jds-client.clearOperationalDataMutableStorage()');
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'clearOperationalDataMutableStorage',
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Clear OPD Mutable Storage', metricsObj);

    var path = '/odmutable/clear/this';
    this.execute(path, null, 'DELETE', metricsObj, callback);
};

JdsClient.prototype.saveOperationalSyncStatus = function(metastamp, siteId, callback) {
    this.log.debug('JdsClient.saveOperationalSyncStatus() %j', siteId);
    this.log.debug(inspect(metastamp));
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'saveOperationalSyncStatus',
        'site': siteId,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Save OPD Sync Status', metricsObj);
    var self = this;

    var path = '/statusod/' + siteId;
    // self.log.info('JdsClient.saveOperationalSyncStatus() %j',metastamp);
    var metastampJobs = self._ensureMetastampSize(metastamp, self.config.maxMetastampSize || 20000);
    if (metastampJobs.length > 1) {
        self.log.warn('JdsClient.saveOperationalSyncStatus() metastamp broken into %s parts', metastampJobs.length);
        // self.log.info('JdsClient.saveOperationalSyncStatus() %j',metastampJobs[0]);
    }
    async.each(metastampJobs, function(stamp, cb) {
        self.log.info('JdsClient.saveOperationalSyncStatus() %j', stamp);
        self.execute(path, stamp, 'POST', metricsObj, cb);
    }, function(err) {
        var response = {}; //hardcoded to account for inability to merge multiple responses
        if (!err) {
            response.statusCode = 200;
        }
        callback(err, response);
    });
};

JdsClient.prototype.getOperationalSyncStatus = function(siteId, callback) {
    this.log.debug('JdsClient.getOperationalSyncStatus() %j', siteId);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getOperationalSyncStatus',
        'site': siteId,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get OPD Sync Status', metricsObj);

    var path = '/statusod/' + siteId;
    this.execute(path, null, 'GET', metricsObj, callback);
};

JdsClient.prototype.getOperationalSyncStatusWithParams = function(siteId, params, callback) {
    this.log.debug('JdsClient.getOperationalSyncStatusWithParams() %j, %j', siteId, params);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getOperationalSyncStatusWithParams',
        'site': siteId,
        'process': uuid.v4(),
        'params': params,
        'timer': 'start'
    };
    this.metrics.debug('JDS Get OPD Sync Status With Params', metricsObj);

    if (_.isEmpty(params)) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS Get Operational Sync Status With Params in Error', metricsObj);
        return setTimeout(callback, 0, errorUtil.createFatal('No params passed in'));
    }

    var path = '/statusod/' + siteId;
    if (_.isObject(params)) {
        path += '?' + querystring.stringify(params);
    }
    this.execute(path, null, 'GET', metricsObj, callback);
};

JdsClient.prototype.deleteOperationalSyncStatus = function(siteId, callback) {
    this.log.debug('JdsClient.deleteOperationalSyncStatus() %j', siteId);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'deleteOperationalSyncStatus',
        'site': siteId,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Delete OPD Sync Status', metricsObj);

    var path = '/statusod/' + siteId;
    this.execute(path, null, 'DELETE', metricsObj, callback);
};

JdsClient.prototype.clearAllOperationalSyncStatus = function(callback) {
    this.log.debug('JdsClient.clearAllOperationalSyncStatus()');
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'clearAllOperationalSyncStatus',
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Clear All OPD Sync Status', metricsObj);

    var path = '/statusod/';
    this.execute(path, null, 'DELETE', metricsObj, callback);
};

//------------------------------------------------------------------------------------------
// Retrieve patientDemographics data using the patient's pid.  Note this is demographics
// that comes from patient domain (NOT operational pt-select data)
//
// pid: The pid to use to retrieve the data.
// callback: The handler to call when this request is completed.
//-------------------------------------------------------------------------------------------
JdsClient.prototype.getPtDemographicsByPid = function(pid, callback) {
    this.log.debug('jds-client.getPtDemographicsByPid() %j', pid);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getPtDemographicsByPid',
        'pid': pid,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS get PT Demographics by PID', metricsObj);

    if (_.isEmpty(pid)) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS get PT Demographics by PID in Error', metricsObj);
        this.log.error('jds-client.getPtDemographicsByPid: No pid passed in');
        return setTimeout(callback, 0, errorUtil.createFatal('No pid passed in'));
    }

    var path = '/vpr/' + pid;
    this.execute(path, null, 'GET', metricsObj, callback);
};

//------------------------------------------------------------------------------------------
// Retrieve patientDemographics data using the patient's icn.  Note this is demographics
// that comes from patient domain (NOT operational pt-select data)
//
// icn: The icn to use to retrieve the data.
// callback: The handler to call when this request is completed.
//-------------------------------------------------------------------------------------------
JdsClient.prototype.getPtDemographicsByIcn = function(icn, callback) {
    this.log.debug('jds-client.getPtDemographicsByIcn: Entered method: icn: %s', icn);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getPtDemographicsByIcn',
        'pid': icn,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS get PT Demographics by ICN', metricsObj);

    if (_.isEmpty(icn)) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS get PT Demographics by ICN in Error', metricsObj);
        this.log.error('jds-client.getPtDemographicsByIcn: No icn passed in');
        return setTimeout(callback, 0, errorUtil.createFatal('No icn passed in'));
    }

    var path = '/vpr/' + icn;
    this.execute(path, null, 'GET', metricsObj, callback);
};

//------------------------------------------------------------------------------------------
// This method marks an item on an operational data sync status as stored.
// It is only used for integration testing.
//------------------------------------------------------------------------------------------
JdsClient.prototype._markOperationalItemAsStored = function(metadata, callback) {
    this.log.debug('jds-client._markOperationalItemAsStored()');
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'markOperationalItemAsStored',
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get Patient Domain Data', metricsObj);

    var path = '/recordod';
    this.execute(path, metadata, 'POST', metricsObj, callback);
};

JdsClient.prototype.getPatientDomainData = function(identifier, domain, callback) {
    this.log.debug('jds-client.getPatientDomainData() %j %j', identifier, domain);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getPatientDomainData',
        'domain': domain,
        'pid': identifier,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get Patient Domain Data', metricsObj);

    var path = '/vpr/' + identifier + '/find/' + domain;
    this.execute(path, null, 'GET', metricsObj, callback);
};

JdsClient.prototype.storePatientDataFromJob = function(job, callback) {
    this.log.debug('jds-client.storePatientDataFromJob() %j', job);

    if (_.isEmpty(job.record)) {
        this.log.debug('jds-client.storePatientDataFromJob: Failed to store patient data.  job.record was empty.');
        return setTimeout(callback, 0, errorUtil.createFatal('No record passed in job'));
    }

    // var path = '/vpr/' + job.patientIdentifier.value;
    // this.execute(path, job.record, 'POST', callback);

    return this.storePatientData(job.record, callback);

};

//----------------------------------------------------------------------------------
// Store the patient data event to JDS.
//
// patientDataEvent: The patient data event to be stored.
// callback: The handler to call when the data is stored.   Signature is:
//           function(error, response) where:
//               error: Is the error that occurs.
//               response: Is the response from JDS.
//-----------------------------------------------------------------------------------
JdsClient.prototype.storePatientData = function(patientDataEvent, callback) {
    this.log.debug('jds-client.storePatientData() %j', patientDataEvent);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'storePatientData',
        'pid': patientDataEvent.pid,
        'uid': patientDataEvent.uid,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Store Patient Data', metricsObj);

    if (_.isEmpty(patientDataEvent)) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS Store Patient Data in Error', metricsObj);
        this.log.debug('jds-client.storePatientData: Failed to store patient data.  patientDataEvent was empty.');
        return setTimeout(callback, 0, errorUtil.createFatal('No patient data event passed in'));
    }

    var path = '/vpr';
    this.execute(path, patientDataEvent, 'POST', metricsObj, callback);
};

JdsClient.prototype.getPatientDataByUid = function(uid, callback) {
    this.log.debug('jds-client.getPatientDataByUid() %j', uid);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getPatientDataByUid',
        'uid': uid,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get Patient Data By UID', metricsObj);

    if (_.isEmpty(uid)) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS Get Patient Data By UID in Error', metricsObj);
        return setTimeout(callback, 0, errorUtil.createFatal('No uid passed in'));
    }

    var path = '/vpr/uid/' + uid;
    this.execute(path, null, 'GET', metricsObj, callback);
};

//-----------------------------------------------------------------------------------------
// Delete the patient data object by its UID.
//
// uid: The UID that identifies the patient data item being deleted.
// callback: The handler to call when this request is completed.
//-----------------------------------------------------------------------------------------
JdsClient.prototype.deletePatientDataByUid = function(uid, callback) {
    this.log.debug('jds-client.deletePatientDataByUid() %j', uid);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'deletePatientDataByUid',
        'uid': uid,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Delete Patient Data By UID', metricsObj);

    if (_.isEmpty(uid)) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS Delete Patient Data By UID in Error', metricsObj);
        return setTimeout(callback, 0, errorUtil.createFatal('No uid passed in'));
    }

    var path = '/vpr/uid/' + uid;
    this.execute(path, null, 'DELETE', metricsObj, callback);
};

//-----------------------------------------------------------------------------------------
// Delete all patient data for all identifiers tied to this pid.
//
// pid: The pid that identifies the patient.
// callback: The handler to call when this request is completed.
//-----------------------------------------------------------------------------------------
JdsClient.prototype.deletePatientByPid = function(pid, callback) {
    this.log.debug('jds-client.deletePatientByPid() %j', pid);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'deletePatientByPid',
        'pid': pid,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Delete Patient Data By PID', metricsObj);

    if (_.isEmpty(pid)) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS Delete Patient Data By PID in Error', metricsObj);
        this.log.error('jds-client.deletePatientByPid() No pid passed in');
        return setTimeout(callback, 0, errorUtil.createFatal('No pid passed in'));
    }

    var path = '/vpr/' + pid;
    this.execute(path, null, 'DELETE', metricsObj, callback);
};

//----------------------------------------------------------------------------------
// Store the notification that an event was sent to SOLR.
//
// pid: The pid for this patient.
// storeEventInfo: The required event identification to correctly notify JDS
//                 of the event status.  The event should look as follows:
//                 {
//                   "uid": "urn:va:vital:9E7A:3:23",
//                   "eventStamp": 20040330215452,
//                   "type": "solr"
//                 }
// callback: The handler to call when the data is stored.   Signature is:
//           function(error, response) where:
//               error: Is the error that occurs.
//               response: Is the response from JDS.
//-----------------------------------------------------------------------------------
JdsClient.prototype.setEventStoreStatus = function(pid, storeEventInfo, callback) {
    this.log.debug('jds-client.setEventStoreStatus() pid: %s, storeEventInfo: %j', pid, storeEventInfo);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'setEventStoreStatus',
        'pid': pid,
        'uid': (_.isEmpty(storeEventInfo)) ? null : storeEventInfo.uid,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Event Store Status', metricsObj);

    if (_.isEmpty(storeEventInfo)) {
        metricsObj.timer = 'stop';
        this.metrics.debug('JDS Event Store Status in Error', metricsObj);
        this.log.debug('jds-client.setEventStoreStatus: Failed to store event status.  storeEventInfo was empty.');
        return setTimeout(callback, 0, errorUtil.createFatal('No store event info passed in'));
    }

    var path = '/status/' + pid + '/store';
    this.execute(path, storeEventInfo, 'POST', metricsObj, callback);
};



//--------------------------------------------------------------------------------------------------
// This function executes the JDS command.
//
// path: The JDS REST URL  (Without that http://<ip>:port)
// dataToPost:  If this is a put or post, then this is the data that is going to be put or posted.
// method: The type of http method (i.e. GET, PUT, POST, etc)
// callback: The call back function that should be called when the execute is completed.
//--------------------------------------------------------------------------------------------------
JdsClient.prototype.execute = function(path, dataToPost, method, metricsObj, callback) {
    this.log.debug(path);
    this.log.debug(inspect(dataToPost));
    metricsObj.timer = 'stop';
    if (_.isEmpty(this.config.jds)) {
        this.metrics.debug('JDS Execute in Error', metricsObj);
        return setTimeout(callback, 0, errorUtil.createFatal('No value passed for jds configuration'));
    }

    var url = format('%s://%s:%s%s', this.config.jds.protocol, this.config.jds.host, this.config.jds.port, path);

    if (method === 'POST' || method === 'PUT') {
        if (_.isEmpty(dataToPost)) {
            this.log.debug('jds-client.execute(): Sending a POST or PUT without dataToPost. url: %s', url);
            dataToPost = undefined;
        } else {
            var dataToPostWithoutPwd = objUtil.removeProperty(objUtil.removeProperty(dataToPost,'accessCode'),'verifyCode');
            this.log.debug('jds-client.execute(): Sending message to JDS. %s -> dataToPost: %j', url, dataToPostWithoutPwd);
        }
    } else {
        this.log.debug('jds-client.execute(): Sending message to JDS. url: %s', url);
        dataToPost = undefined;
    }

    var self = this;
    request({
        url: url,
        method: method || 'GET',
        json: dataToPost,
        timeout: this.config.jds.timeout || 60000,
        forever: true,
        agentOptions: {maxSockets: self.config.handlerMaxSockets || 5}
    }, function(error, response, body) {
        self.log.debug('jds-client.execute(): posted data to JDS %s', url);

        if (error || response.statusCode === 500) {
            self.log.error('jds-client.execute(): Unable to access JDS endpoint: %s %s', method, url);
            self.log.error('%j %j', error, body);

            self.metrics.debug('JDS Execute in Error', metricsObj);
            return callback(errorUtil.createTransient((error || body || 'Unknown Error')), response);
        }

        var json;
        if (_.isEmpty(body)) {
            self.log.debug('jds-client.execute(): Response body is empty.  Status code:', response.statusCode);
            self.metrics.debug('JDS Execute complete', metricsObj);
            return callback(null, response);
        }

        try {
            json = _.isObject(body) ? body : JSON.parse(body);
        } catch (parseError) {
            self.log.error('jds-client.execute(): Unable to parse JSON response:', body);
            self.log.error('jds-client.execute(): Unable to parse JSON response, and response is defined.  this is actually bad');
            self.log.error(inspect(parseError));
            self.log.error('::' + body + '::');
            json = body;
        }

        var responseWithoutPwd = objUtil.removeProperty(objUtil.removeProperty(json, 'accessCode'), 'verifyCode');
        self.log.debug('jds-client.execute(): JDS response is for the caller to handle %j', responseWithoutPwd);
        self.metrics.debug('JDS Execute complete', metricsObj);
        callback(null, response, json);
    });
};

JdsClient.prototype._validateMetastampSize = function(metastamp, maxSize) {
    var size = sizeof(metastamp);
    if (size > maxSize) {
        return false;
    } else {
        return true;
    }
};

/**
Takes a metastamp and splits it by data source. This is a first attempt
to break down a metastamp into a size that can be processed by JDS.
**/
JdsClient.prototype._splitMetastampBySource = function(metastamp) {
    var metastampShell;
    var sources = metastamp.sourceMetaStamp;
    var metastampBySource = [];
    if (_.keys(sources).length <= 1) { //This metastamp only has one source already
        return [metastamp];
    }
    metastamp.sourceMetaStamp = {};
    metastampShell = JSON.parse(JSON.stringify(metastamp));
    _.each(sources, function(value, key) {
        var siteStamp = JSON.parse(JSON.stringify(metastampShell));
        siteStamp.sourceMetaStamp[key] = value;
        metastampBySource.push(siteStamp);
    });
    return metastampBySource;
};

/**
Breaks up a metastamp by domain in hopes that single domains will be small enough
to store in JDS. It assumes that only one source exists in this metastamp.
**/
JdsClient.prototype._splitMetastampByDomain = function(metastamp) {
    var metastampShell;
    var source = _.keys(metastamp.sourceMetaStamp)[0];
    var domains = metastamp.sourceMetaStamp[source].domainMetaStamp;
    var metastampByDomain = [];
    if (_.keys(domains).length <= 1) { //this metastamp only has one domain already
        return [metastamp];
    }
    metastamp.sourceMetaStamp[source].domainMetaStamp = {};
    metastampShell = JSON.parse(JSON.stringify(metastamp));
    _.each(domains, function(value, key) {
        var domainStamp = JSON.parse(JSON.stringify(metastampShell));
        domainStamp.sourceMetaStamp[source].domainMetaStamp[key] = value;
        metastampByDomain.push(domainStamp);
    });
    return metastampByDomain;
};

/**
Breaks up a metastamp domain into appropriate sized metastamps that can be stored in JDS.
This assumes that the metastamp has already been broken up by site and domain such that
this metastamp only has one source with one domain.
**/
JdsClient.prototype._splitMetastampDomain = function(metastamp, maxSize) {
    var metastampSize = sizeof(metastamp);
    var source = _.keys(metastamp.sourceMetaStamp)[0];
    var domain = _.keys(metastamp.sourceMetaStamp[source].domainMetaStamp)[0];
    var eventListName;
    if (metastamp.sourceMetaStamp[source].domainMetaStamp[domain].eventMetaStamp) {
        eventListName = 'eventMetaStamp';
    } else {
        eventListName = 'itemMetaStamp';
    }
    var uids = metastamp.sourceMetaStamp[source].domainMetaStamp[domain][eventListName];
    metastamp.sourceMetaStamp[source].domainMetaStamp[domain][eventListName] = {};
    var metastampShell = JSON.parse(JSON.stringify(metastamp));
    var numberOfEvents = _.size(uids);
    var divisor = maxSize * numberOfEvents;
    var quotient = divisor / metastampSize;
    var eventsPerStamp = Math.floor(quotient);
    if (eventsPerStamp < 1) {
        eventsPerStamp = 1;
    }
    var metastampByEvent = [];
    var index = 0;
    _.each(uids, function(value, key) {
        var metastampIndex = Math.floor(index / eventsPerStamp);
        if (!metastampByEvent[metastampIndex]) {
            metastampByEvent[metastampIndex] = JSON.parse(JSON.stringify(metastampShell));
        }
        metastampByEvent[metastampIndex].sourceMetaStamp[source].domainMetaStamp[domain][eventListName][key] = value;
        index++;
    });
    return metastampByEvent;
};

/**
Ensures that a metastamp isn't so large that JDS cannot store it in memory.
**/
JdsClient.prototype._ensureMetastampSize = function(metastamp, size) {
    var self = this;
    if (self._validateMetastampSize(metastamp, size)) {
        return [metastamp];
    }
    var metastampJobs = [];
    var sourceJobs = self._splitMetastampBySource(metastamp);
    _.each(sourceJobs, function(sourceMetastamp) {
        if (self._validateMetastampSize(sourceMetastamp, size)) {
            metastampJobs.push(sourceMetastamp);
        } else {
            var domainJobs = self._splitMetastampByDomain(sourceMetastamp);
            _.each(domainJobs, function(domainMetastamp) {
                if (self._validateMetastampSize(domainMetastamp, size)) {
                    metastampJobs.push(domainMetastamp);
                } else {
                    var eventJobs = self._splitMetastampDomain(domainMetastamp, size);
                    metastampJobs = metastampJobs.concat(eventJobs);
                }
            });
        }
    });
    return metastampJobs;
};


JdsClient.prototype.addErrorRecord = function(errorRecord, callback) {
    this.log.debug('jds-client.addErrorRecord() %j', errorRecord);
    var metricsObj = {
        subsystem: 'JDS',
        action: 'addErrorRecord',
        process: uuid.v4(),
        timer: 'start'
    };
    this.metrics.debug('JDS addErrorRecord %j', metricsObj);
    this.execute('/vxsyncerr/', errorRecord, 'PUT', metricsObj, callback);
};

JdsClient.prototype.findErrorRecordByUid = function(uid, callback) {
    this.log.debug('jds-client.findErrorRecordByUid(%s)', uid);
    var metricsObj = {
        subsystem: 'JDS',
        action: 'findErrorRecordByUid',
        uid: uid,
        process: uuid.v4(),
        timer: 'start'
    };
    this.metrics.debug('JDS findErrorRecordByUid %j', metricsObj);
    this.execute('/vxsyncerr/' + uid, null, 'GET', metricsObj, callback);
};

JdsClient.prototype.findErrorRecordsByFilter = function(filter, callback) {
    this.log.debug('jds-client.findErrorRecordsByFilter() %s', filter);
    this.findErrorRecords({filter: filter}, callback);
};

JdsClient.prototype.findErrorRecords = function(query, callback) {
    this.log.debug('jds-client.findErrorRecords() query: %s', query);
    var metricsObj = {
        subsystem: 'JDS',
        action: 'findErrorRecords',
        query: query,
        process: uuid.v4(),
        timer: 'start'
    };
    this.metrics.debug('JDS findErrorRecords %j', metricsObj);

    var path = '/vxsyncerr/';

    if(_.has(query,'index')){
        path += (query.index)?'index/'+query.index:'';
        query = _.omit(query, 'index');
    }

    var queryString = querystring.stringify(query);
    if(queryString){
        path += '?' + queryString;
    }

    this.execute(path, null, 'GET', metricsObj, callback);
};

JdsClient.prototype.findErrorRecordsByRange = function(index, range, callback){
    this.log.debug('jds-client.findErrorRecordsByRange() index: %s, range: %s', index, range);
    this.findErrorRecords({index: index, range: range}, callback);
};

JdsClient.prototype.getErrorRecordCount = function(callback) {
    this.log.debug('jds-client.getErrorRecordCount()');
    var metricsObj = {
        subsystem: 'JDS',
        action: 'getErrorRecordCount',
        process: uuid.v4(),
        timer: 'start'
    };
    this.metrics.debug('JDS getErrorRecordCount %j', metricsObj);
    this.execute('/vxsyncerr', null, 'GET', metricsObj, function(error, response, result) {
        var doc_count = (result && result.doc_count)?(result.doc_count):null;
        return callback(null, response, doc_count);
    });
};

JdsClient.prototype.deleteErrorRecordByUid = function(uid, callback) {
    this.log.debug('jds-client.deleteErrorRecordByUid(%s)', uid);
    var metricsObj = {
        subsystem: 'JDS',
        action: 'deleteErrorRecordByUid',
        uid: uid,
        process: uuid.v4(),
        timer: 'start'
    };
    this.metrics.debug('JDS deleteErrorRecordByUid %j', metricsObj);
    this.execute('/vxsyncerr/' + uid, null, 'DELETE', metricsObj, callback);
};

JdsClient.prototype.deleteAllErrorRecords = function(callback) {
    this.log.debug('jds-client.deleteAllErrorRecords()');
    var metricsObj = {
        subsystem: 'JDS',
        action: 'deleteAllErrorRecords',
        process: uuid.v4(),
        timer: 'start'
    };
    this.metrics.debug('JDS deleteAllErrorRecords %j', metricsObj);
    this.execute('/vxsyncerr/?confirm=true', null, 'DELETE', metricsObj, callback);
};

JdsClient.prototype.deleteErrorRecordsByFilter = function(filter, callback){
    this.log.debug('jds-client.deleteErrorRecordsByFilter() %s', filter);
    var metricsObj = {
        subsystem: 'JDS',
        action: 'deleteErrorRecordsByFilter',
        filter: filter,
        process: uuid.v4(),
        timer: 'start'
    };
    this.metrics.debug('JDS deleteErrorRecordsByFilter %j', metricsObj);
    this.execute('/vxsyncerr/?filter=' + filter, null, 'DELETE', metricsObj, callback);
};

JdsClient.prototype.getLockOnErrorRecord = function(uid, callback) {
    this.log.debug('jds-client.getLockOnErrorRecord(%s)', uid);
    var metricsObj = {
        subsystem: 'JDS',
        action: 'getLockOnErrorRecord',
        uid: uid,
        process: uuid.v4(),
        timer: 'start'
    };
    this.metrics.debug('JDS getLockOnErrorRecord %j', metricsObj);
    this.execute('/vxsyncerr/lock/' + uid, null, 'GET', metricsObj, callback);
};

JdsClient.prototype.lockErrorRecord = function(uid, callback) {
    this.log.debug('jds-client.lockErrorRecord(%s)', uid);
    var metricsObj = {
        subsystem: 'JDS',
        action: 'lockErrorRecord',
        uid: uid,
        process: uuid.v4(),
        timer: 'start'
    };
    this.metrics.debug('JDS lockErrorRecord %j', metricsObj);
    this.execute('/vxsyncerr/lock/' + uid, null, 'PUT', metricsObj, callback);
};

JdsClient.prototype.unlockErrorRecord = function(uid, callback){
    this.log.debug('jds-client.unlockErrorRecord() %s', uid);
    var metricsObj = {
        subsystem: 'JDS',
        action: 'unlockErrorRecord',
        uid: uid,
        process: uuid.v4(),
        timer: 'start'
    };
    this.metrics.debug('JDS unlockErrorRecord %j', metricsObj);
    this.execute('/vxsyncerr/lock/' + uid, null, 'DELETE', metricsObj, callback);
};

JdsClient.prototype.getPatientList = function(lastAccessTime, callback) {
    this.log.debug('jds-client.getPatientList() %j', lastAccessTime);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getPatientList',
        'lastAccessTime': lastAccessTime,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get Patient Data By lastAccessTime', metricsObj);

    var path = '/vpr/all/patientlist';

    if (!_.isEmpty(lastAccessTime)) {
        path = '/vpr/all/patientlist?filter=lt(lastAccessTime,' + lastAccessTime + ')';
    }
    this.execute(path, null, 'GET', metricsObj, callback);
};

JdsClient.prototype.getPatientListBySite = function(site, callback) {
    this.log.debug('jds-client.getPatientListBySite() %j', site);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getPatientList',
        'site': site,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get Patient List By Site', metricsObj);

    var path = '/vpr/all/pid/pid';

    if (!_.isEmpty(site)) {
        path = '/vpr/all/index/pid/pid?filter=eq(site,' + site + ')';
    }
    this.execute(path, null, 'GET', metricsObj, callback);
};

JdsClient.prototype.getJpidFromQuery = function(patientIdentifiers, callback) {
    this.log.debug('jds-client.getJpidFromQuery() %j', patientIdentifiers);
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getJpidFromQuery',
        'patientIdentifiers': patientIdentifiers,
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get Jpid via query', metricsObj);

    var postBody = {
        'patientIdentifiers': patientIdentifiers
    };

    var path = '/vpr/jpid/query/';
    this.execute(path, postBody, 'POST', metricsObj, callback);
};

JdsClient.prototype.saveActiveUsers = function(activeUsers, callback) {
    this.log.debug('jds-client.saveActiveUsers()');
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'saveActiveUsers',
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Save Active Users', metricsObj);

    var postBody = {_id: 'osyncusers', users : activeUsers};

    var path = '/user/set/this';
    this.execute(path, postBody, 'POST', metricsObj, callback);
};

JdsClient.prototype.getActiveUsers = function(callback) {
    this.log.debug('jds-client.getActiveUsers()');
    var metricsObj = {
        'subsystem': 'JDS',
        'action': 'getActiveUsers',
        'process': uuid.v4(),
        'timer': 'start'
    };
    this.metrics.debug('JDS Get Active Users', metricsObj);

    var path = '/user/get/osyncusers';
    this.execute(path, null, 'GET', metricsObj, callback);
};

module.exports = JdsClient;
