#
# Cookbook Name:: jbpm
# Recipe:: default
#

####################################
# NOTES:
# This script does 3 things:
# 1) Patch EAP 6.4.0 to 6.4.11
# 2) Install jBPM 6.3.0 and apply patch 6.3.4 if jBPM is not installed
# 3) Patch jBPM 6.1.0 to 6.3.4 if jBPM 6.1.0 is installed
####################################

include_recipe "jboss-eap_wrapper"
jbpm_username = data_bag_item("credentials", "oracle_user_jbpm", node[:data_bag_string])["username"]
jbpm_password = data_bag_item("credentials", "oracle_user_jbpm", node[:data_bag_string])["password"]

oracle_node = find_optional_node_by_role("ehmp_oracle", node[:stack])
oracle_ip = oracle_node[:ipaddress] if !oracle_node.nil?
oracle_port = oracle_node[:ehmp_oracle][:oracle_config][:port] if !oracle_node.nil?
oracle_sid = oracle_node[:ehmp_oracle][:oracle_sid] if !oracle_node.nil?

template "#{node['jboss-eap']['jboss_home']}/bin/standalone.conf" do
  source 'standalone.conf.erb'
  owner 'jboss'
  group 'jboss'
  mode '0755'
  variables(
      :java_xmx => node[:jbpm][:install][:jboss_config][:java_xmx],
      :java_xms => node[:jbpm][:install][:jboss_config][:java_xms]
  )
  notifies :restart, "service[jboss]"
end

admin_password = Chef::EncryptedDataBagItem.load("credentials", "jbpm_admin_password", node[:data_bag_string])["password"]

directory "#{node[:jbpm][:workdir]}" do
  owner  'root'
  group  'root'
  mode '1777'
  action :create
  not_if { ::Dir.exist?("#{node[:jbpm][:workdir]}") }
end

ruby_block "get current EAP version" do
  block do
    begin
      node.default[:jbpm][:cur_eap_version] = /.*(\d+\.\d+\.\d+).*/.match(File.read(node[:jbpm][:eap_version_file]))[1]
      Chef::Log.warn("Current EAP Version is #{node[:jbpm][:cur_eap_version]}")
    rescue
      raise "Could not get current EAP version! Does #{node[:jbpm][:eap_version_file]} exist and contain a valid version?"
    end
  end
end

ruby_block "get current JBPM version" do
  block do
    begin
      node.default[:jbpm][:cur_jbpm_version] = `find /opt/jboss/standalone/deployments/business-central.war/WEB-INF/lib/ -name "jbpm-runtime-manager-*"|  sed 's^.*/^^' | sed 's/[^0-9]*//g'`
      Chef::Log.warn("Current JBPM Version is #{node[:jbpm][:cur_jbpm_version]}")
    rescue
      raise "Could not get current jBPM version!"
    end
  end
end

template "#{node[:jbpm][:workdir]}/jbpm.auto.xml" do
  source "jbpm.auto.xml.erb"
  mode "0755"
  variables(:admin_password => admin_password)
end

remote_file "#{Chef::Config['file_cache_path']}/jboss-eap-6.4.9-patch.zip" do
  source node[:jbpm][:install][:patch9]
  mode   "0755"
  use_conditional_get true
end

remote_file "#{Chef::Config['file_cache_path']}/jboss-eap-6.4.13-patch.zip" do
  source node[:jbpm][:install][:patch13]
  mode   "0755"
  use_conditional_get true
end

remote_file "#{Chef::Config['file_cache_path']}/jboss-bpmsuite-6.3.0.GA-installer.jar" do
  source node[:jbpm][:install][:source]
  mode   "0755"
  use_conditional_get true
end

remote_file "#{Chef::Config['file_cache_path']}/jboss-bpmsuite-6.1.5-patch.zip" do
  source node[:jbpm][:install][:bpmpatch615]
  mode   "0777"
  use_conditional_get true
  notifies :delete, "directory[#{node[:jbpm][:workdir]}/jboss-bpmsuite-6.1.5-patch]", :immediately
  only_if { node[:jbpm][:cur_jbpm_version].to_i == 6204 }
end

directory "#{node[:jbpm][:workdir]}/jboss-bpmsuite-6.1.5-patch" do
  owner node['jboss-eap'][:jboss_user]
  mode '755'
  recursive true
  action :create
  only_if { node[:jbpm][:cur_jbpm_version].to_i == 6204 }
end

execute "Extract bpmsuite-6.1.5-patch" do
  cwd "#{node[:jbpm][:workdir]}"
  command "sudo unzip -o #{Chef::Config.file_cache_path}/jboss-bpmsuite-6.1.5-patch.zip"
  only_if { node[:jbpm][:cur_jbpm_version].to_i == 6204}
end

remote_file "#{Chef::Config['file_cache_path']}/jboss-bpmsuite-6.1.5-to-6.3.0-patch.zip" do
  source node[:jbpm][:install][:bpmpatch615630]
  mode   "0777"
  use_conditional_get true
  notifies :delete, "directory[#{node[:jbpm][:workdir]}/jboss-bpmsuite-6.1.5-to-6.3.0.GA-patch]", :immediately
  only_if { node[:jbpm][:cur_jbpm_version].include? "62"}
end

directory "#{node[:jbpm][:workdir]}/jboss-bpmsuite-6.1.5-to-6.3.0.GA-patch" do
  owner node['jboss-eap'][:jboss_user]
  mode '755'
  recursive true
  action :create
  only_if { node[:jbpm][:cur_jbpm_version].include? "62"}
end

execute "Extract bpmsuite-6.1.5-to-6.3.0-patch" do
  cwd "#{node[:jbpm][:workdir]}"
  command "sudo unzip -o #{Chef::Config.file_cache_path}/jboss-bpmsuite-6.1.5-to-6.3.0-patch.zip"
  only_if { node[:jbpm][:cur_jbpm_version].include?("62") }
end

# the following bash blocks set the newly created setting files generated by jbpm patches as the working files
# the configure.rb recipe will then apply our custom settings to them
bash "Copy new xml files1" do
  code <<-EOH
    cd "#{node[:jbpm][:home]}/deployments/business-central.war/WEB-INF/classes"
    mv workbench-policy.properties.new workbench-policy.properties
    mv ErraiApp.properties.new ErraiApp.properties
    cd "#{node[:jbpm][:home]}/deployments/business-central.war/WEB-INF"
    mv web-ui-server.xml.new web-ui-server.xml
    mv web.xml.new web.xml
  EOH
  action :nothing
end

bash "Copy new xml files2" do
  code <<-EOH
    cd "#{node[:jbpm][:home]}/deployments/business-central.war/WEB-INF/classes/META-INF"
    mv persistence.xml.new persistence.xml
    mv kie-wb-deployment-descriptor.xml.new kie-wb-deployment-descriptor.xml
    cd "#{node[:jbpm][:home]}/deployments/dashbuilder.war/WEB-INF/"
    mv jboss-deployment-structure.xml.new jboss-deployment-structure.xml
    mv jboss-web.xml.new jboss-web.xml
    cd "#{node[:jbpm][:home]}/configuration/"
    mv standalone-full-ha.xml.new standalone-full-ha.xml
    mv standalone-full.xml.new standalone-full.xml
    mv standalone-ha.xml.new standalone-ha.xml
    mv standalone.xml.new standalone.xml
    mv standalone-osgi.xml.new standalone-osgi.xml
    cd "#{node['jboss-eap']['install_path']}/#{node['jboss-eap']['symlink']}/domain/configuration/"
    mv domain.xml.new domain.xml
  EOH
  action :nothing
end

remote_file "#{Chef::Config['file_cache_path']}/jboss-bpmsuite-6.3.4-patch.zip" do
  source node[:jbpm][:install][:bpmpatch634]
  mode   "0777"
  use_conditional_get true
  notifies :delete, "directory[#{node[:jbpm][:workdir]}/jboss-bpmsuite-6.3.4-patch]", :immediately
end

directory "#{node[:jbpm][:workdir]}/jboss-bpmsuite-6.3.4-patch" do
  owner node['jboss-eap'][:jboss_user]
  mode '755'
  recursive true
  action :create
end

execute "Extract bpmsuite-6.3.4-patch" do
  cwd "#{node[:jbpm][:workdir]}"
  command "sudo unzip -o #{Chef::Config.file_cache_path}/jboss-bpmsuite-6.3.4-patch.zip"
  only_if { (Dir.entries("#{node[:jbpm][:workdir]}/jboss-bpmsuite-6.3.4-patch/") - %w{ . .. }).empty? }
end

template "#{node[:jbpm][:workdir]}/niogit_config.xml"

# Patch EAP
execute "Install EAP-6.4.9" do
  command "/opt/jboss/bin/jboss-cli.sh --command=\"patch apply #{Chef::Config['file_cache_path']}/jboss-eap-6.4.9-patch.zip\""
  user node['jboss-eap'][:jboss_user]
  action :nothing
  only_if { node[:jbpm][:cur_eap_version].gsub(/\D/, '').to_i < 649 }
end

execute "Install EAP-6.4.13" do
  command "/opt/jboss/bin/jboss-cli.sh --command=\"patch apply #{Chef::Config['file_cache_path']}/jboss-eap-6.4.13-patch.zip\""
  user node['jboss-eap'][:jboss_user]
  notifies :run, "execute[Install EAP-6.4.9]", :before
  notifies :start, "service[jboss]", :immediately
  only_if { node[:jbpm][:cur_eap_version].gsub(/\D/, '').to_i < 6413 }
end

# install patch 6.3.4 if the current jBPM version is 6.3.0 which uses 6.4.0-Final-redhat-xx jars
execute "Install bpmsuite-6.3.4-patch" do
  cwd "#{node[:jbpm][:workdir]}/jboss-bpmsuite-6.3.4-patch"
  command "./apply-updates.sh /opt/jboss-6.4.0/ eap6.x"
  user node['jboss-eap'][:jboss_user]
  action :nothing
  notifies :stop, "service[jboss]", :before
  only_if { node[:jbpm][:cur_jbpm_version].to_i != 64013 }
end

# clear jbpm data from the database upon clean install
cookbook_file "#{node[:jbpm][:workdir]}/clear_jbpm_data.sql"

execute "Clear jBPM data upon clean install" do
  cwd node[:jbpm][:workdir]
  command "sqlplus -s /nolog <<-EOF>> #{node[:jbpm][:workdir]}/clear_jbpm_data.log
    WHENEVER OSERROR EXIT 9;
    WHENEVER SQLERROR EXIT SQL.SQLCODE;
    connect #{jbpm_username}/#{jbpm_password}@(DESCRIPTION=(ADDRESS=(PROTOCOL=TCP)(HOST=#{oracle_ip})(PORT=#{oracle_port}))(CONNECT_DATA=(SERVICE_NAME=#{oracle_sid})))
    @clear_jbpm_data.sql
    EXIT
  EOF"
  sensitive true
  action :nothing
  only_if { node[:jbpm][:allow_clear_data] }
end

execute "Check if jBPM is installed" do
  command 'echo "jBPM is not installed"'
  notifies :stop, "service[jboss]", :immediately
  notifies :run, "execute[Install jBPM]", :immediately
  not_if { ::File.exist?("#{node[:jbpm][:home]}/deployments/kie-server.war") }
end

# install jBPM 6.3.0 if this is a clean server and apply patch 6.3.4
execute "Install jBPM" do
  command "java -jar #{Chef::Config[:file_cache_path]}/jboss-bpmsuite-6.3.0.GA-installer.jar #{node[:jbpm][:workdir]}/jbpm.auto.xml"
  user node['jboss-eap'][:jboss_user]
  action :nothing
  notifies :run, "execute[Install bpmsuite-6.3.4-patch]", :immediately
  notifies :run, "execute[Clear jBPM data upon clean install]", :immediately
  notifies :start, "service[jboss]", :immediately
end

execute "Chmod Oracle driver" do
  command "chmod 755 -R /opt/jboss-6.4.0/modules/system/layers/base/com/oracle/"
  action :nothing
end

# install patch 6.1.5 if the current jBPM version is 6.1.0 which uses 6.2.0-Final-redhat-4 jars
execute "Install bpmsuite-6.1.5-patch" do
  cwd "#{node[:jbpm][:workdir]}/jboss-bpmsuite-6.1.5-patch"
  command "./apply-updates.sh /opt/jboss-6.4.0/ eap6.x"
  user node['jboss-eap'][:jboss_user]
  action :nothing
  notifies :stop, "service[jboss]", :before
  notifies :run, "execute[Chmod Oracle driver]", :before
  notifies :run, "bash[Copy new xml files1]", :immediately
  only_if { node[:jbpm][:cur_jbpm_version].to_i == 6204 }
end

# install patch 6.1.5-to-6.3.0 if the current jBPM version is 6.1.x which uses 6.2.0-Final-redhat-xx jars
execute "Install bpmsuite-6.3.0-patch" do
  cwd "#{node[:jbpm][:workdir]}/jboss-bpmsuite-6.1.5-to-6.3.0.GA-patch"
  command "./apply-updates.sh /opt/jboss-6.4.0/ eap6.x"
  user node['jboss-eap'][:jboss_user]
  notifies :stop, "service[jboss]", :before
  notifies :run, "execute[Install bpmsuite-6.1.5-patch]", :before
  notifies :run, "bash[Copy new xml files2]", :immediately
  notifies :run, "execute[Install bpmsuite-6.3.4-patch]", :immediately
  only_if { node[:jbpm][:cur_jbpm_version].include? "62"}
end

# The following is only needed when upgrading jbpm6.1 to 6.3
cookbook_file "#{node[:jbpm][:workdir]}/update_jbpm_30_schema.sql"
execute "Update jbpm oracle schema" do
  cwd node[:jbpm][:workdir]
  command "sqlplus -s /nolog <<-EOF> #{node[:jbpm][:workdir]}/update_jbpm_30_schema.log
    SET FEEDBACK ON SERVEROUTPUT ON
    WHENEVER OSERROR EXIT 9;
    WHENEVER SQLERROR EXIT SQL.SQLCODE;
    connect #{jbpm_username}/#{jbpm_password}@(DESCRIPTION=(ADDRESS=(PROTOCOL=TCP)(HOST=#{oracle_ip})(PORT=#{oracle_port}))(CONNECT_DATA=(SERVICE_NAME=#{oracle_sid})))
    @update_jbpm_30_schema.sql
    EXIT
  EOF"
  sensitive true
  only_if { node[:jbpm][:cur_jbpm_version].include? "62"}
end

common_directory "#{node['jbpm']['gitdir']}/.niogit" do
  owner node['jboss-eap'][:jboss_user]
  mode "0777"
  recursive true
end

common_directory "#{node[:jbpm][:m2_home]}" do
  owner node['jboss-eap'][:jboss_user]
  mode "0777"
  recursive true
end

# Using sed to insert rest-all in the application-roles.properties
execute "Add rest-all to #{node[:jbpm][:install][:admin_user]} user role" do
  command "sed -i 's/#{node[:jbpm][:install][:admin_user]}=admin,/#{node[:jbpm][:install][:admin_user]}=admin,rest-all/' #{node[:jbpm][:home]}/configuration/application-roles.properties"
  user node['jboss-eap'][:jboss_user]
  notifies :stop, "service[jboss]", :immediately
  not_if "grep rest-all #{node[:jbpm][:home]}/configuration/application-roles.properties"
end

# Using sed to insert lines in the configuration file, since the file is modified by the jbpm installation and therefore can't be a template
execute "Set configuration for niogit in standalone.xml" do
  cwd "#{node[:jbpm][:workdir]}"
  command "sed -i '/<system-properties>/r niogit_config.xml' #{node[:jbpm][:home]}/configuration/standalone.xml"
  notifies :stop, "service[jboss]", :immediately
  not_if "grep org.uberfire.nio.git.dir #{node[:jbpm][:home]}/configuration/standalone.xml"
end

# The next 3 execute blocks change the binding from 127.0.0.1 to 0.0.0.0 to allow access to the management console
execute "Set bind address.management in standalone.xml" do
  command "sed -i 's/jboss\.bind\.address\.management:127\.0\.0\.1/jboss.bind.address.management:0.0.0.0/' #{node['jboss-eap']['install_path']}/#{node['jboss-eap']['symlink']}/standalone/configuration/standalone.xml"
  notifies :stop, "service[jboss]", :immediately
  not_if "grep jboss.bind.address.management:0.0.0.0 #{node['jboss-eap']['install_path']}/#{node['jboss-eap']['symlink']}/standalone/configuration/standalone.xml"
end

execute "Set bind address in standalone.xml" do
  command "sed -i 's/jboss\.bind\.address\:127\.0\.0\.1/jboss.bind.address:0.0.0.0/' #{node['jboss-eap']['install_path']}/#{node['jboss-eap']['symlink']}/standalone/configuration/standalone.xml"
  notifies :stop, "service[jboss]", :immediately
  not_if "grep jboss.bind.address:0.0.0.0 #{node['jboss-eap']['install_path']}/#{node['jboss-eap']['symlink']}/standalone/configuration/standalone.xml"
end

execute "Set bind address.unsecure in standalone.xml" do
  command "sed -i 's/jboss\.bind\.address\.unsecure:127\.0\.0\.1/jboss.bind.address.unsecure:0.0.0.0/' #{node['jboss-eap']['install_path']}/#{node['jboss-eap']['symlink']}/standalone/configuration/standalone.xml"
  notifies :stop, "service[jboss]", :immediately
  not_if "grep jboss.bind.address.unsecure:0.0.0.0 #{node['jboss-eap']['install_path']}/#{node['jboss-eap']['symlink']}/standalone/configuration/standalone.xml"
end

logrotate_app "jboss_server_logs" do
  path "#{node['jboss-eap']['log_dir']}/server.log.????-??-??"
  enable true
  frequency "daily"
  lastaction [" # Remove rotated files older than 7 days
    find #{node['jboss-eap']['log_dir']} -name 'server.log.????-??-??' -mtime +7 -exec rm {} \;"]
end

logrotate_app "jboss_console_logs" do
  path "#{node['jboss-eap']['log_dir']}/console.log"
  enable true
  rotate "7"
  frequency "daily"
  dateformat ".%Y-%m-%d"
end
