--------------------------------------------------------
--  DDL for Package PCMM_API
--------------------------------------------------------

CREATE OR REPLACE PACKAGE "PCMM"."PCMM_API" AS

  FUNCTION pipe_teams_by_facility(i_facility_id IN VARCHAR2) RETURN TEAMS_TAB PIPELINED;
  FUNCTION pipe_teams_by_facility_patient(i_facility_id IN varchar2, i_icn IN varchar2) RETURN TEAMS_TAB PIPELINED;
  FUNCTION pipe_teams_by_patient(i_icn IN varchar2) RETURN TEAMS_TAB PIPELINED;
  FUNCTION pipe_teams_by_staff_ien(i_staff_ien IN varchar2) RETURN TEAMS_TAB PIPELINED;
  FUNCTION pipe_roles_by_team(i_team_id IN number) RETURN ROLES_TAB PIPELINED;

  PROCEDURE get_roles_for_team(
      i_team_id IN number,
      o_cursor  OUT SYS_REFCURSOR);

  PROCEDURE get_teams_for_facility(
      i_facility_id IN varchar2,
      o_cursor      OUT SYS_REFCURSOR);

  PROCEDURE get_teams_for_facility_patient(
      i_facility_id IN varchar2,
      i_icn         IN varchar2,
      o_cursor      OUT SYS_REFCURSOR);

  PROCEDURE get_teams_for_patient(
      i_icn    IN varchar2,
      o_cursor OUT SYS_REFCURSOR);

  PROCEDURE get_teams_for_user(
      i_staff_ien  IN varchar2,
      i_staff_site IN varchar2,
      o_cursor OUT SYS_REFCURSOR);

  PROCEDURE get_teams_for_user(
      i_staff_ien  IN varchar2,
      i_staff_division IN varchar2,
      o_cursor OUT SYS_REFCURSOR);

END PCMM_API;

/

--------------------------------------------------------
--  DDL for Package Body PCMM_API
--------------------------------------------------------

CREATE OR REPLACE PACKAGE BODY "PCMM"."PCMM_API"
AS
  
  FUNCTION pipe_teams_by_facility(i_facility_id IN varchar2) RETURN TEAMS_TAB PIPELINED AS
  BEGIN
    FOR rec IN (
      SELECT DISTINCT T.TEAM_ID, T.TEAM_NAME, I.STATIONNUMBER, S.STAFF_IEN
      FROM PCMM.STAFF S
      JOIN PCMM.TEAM_MEMBERSHIP TM ON S.STAFF_ID = TM.STAFF_ID
      JOIN PCMM.TEAM T ON T.TEAM_ID = TM.TEAM_ID
      JOIN SDSADM.STD_INSTITUTION I ON I.ID = T.VA_INSTITUTION_ID
      WHERE I.STATIONNUMBER=i_facility_id)
    LOOP
      PIPE ROW(TEAMS_ROW(rec.team_id, rec.team_name, rec.stationnumber, rec.staff_ien));
    END LOOP;
    RETURN;
  END;
  
  FUNCTION pipe_teams_by_facility_patient(i_facility_id IN varchar2, i_icn IN varchar2) RETURN TEAMS_TAB PIPELINED AS
  BEGIN
    FOR rec IN (
      SELECT DISTINCT T.TEAM_ID, T.TEAM_NAME, I.STATIONNUMBER, S.STAFF_IEN
      FROM PCMM.STAFF S
      JOIN PCMM.TEAM_MEMBERSHIP TM ON S.STAFF_ID = TM.STAFF_ID
      JOIN PCMM.TEAM T ON T.TEAM_ID = TM.TEAM_ID
      JOIN SDSADM.STD_INSTITUTION I ON I.ID = T.VA_INSTITUTION_ID
      JOIN PCMM.TEAM_PATIENT_ASSIGN TPA ON TPA.TEAM_ID = T.TEAM_ID
      JOIN PCMM.PCMM_PATIENT P ON P.PCMM_PATIENT_ID = TPA.PCMM_PATIENT_ID
      WHERE I.STATIONNUMBER=i_facility_id AND P.ICN=i_icn)
    LOOP
      PIPE ROW(TEAMS_ROW(rec.team_id, rec.team_name, rec.stationnumber, rec.staff_ien));
    END LOOP;
    RETURN;
  END;
  
  FUNCTION pipe_teams_by_patient(i_icn IN varchar2) RETURN TEAMS_TAB PIPELINED AS
  BEGIN
    FOR rec IN (
      SELECT DISTINCT T.TEAM_ID, T.TEAM_NAME, I.STATIONNUMBER, S.STAFF_IEN
      FROM PCMM.STAFF S
      JOIN PCMM.TEAM_MEMBERSHIP TM ON S.STAFF_ID = TM.STAFF_ID
      JOIN PCMM.TEAM T ON T.TEAM_ID = TM.TEAM_ID
      JOIN PCMM.TEAM_PATIENT_ASSIGN TPA ON T.TEAM_ID = TPA.TEAM_ID
      JOIN PCMM.PCMM_PATIENT P ON P.PCMM_PATIENT_ID = TPA.PCMM_PATIENT_ID
      JOIN SDSADM.STD_INSTITUTION I ON I.ID = T.VA_INSTITUTION_ID
      WHERE P.ICN=i_icn)
    LOOP
      PIPE ROW(TEAMS_ROW(rec.team_id, rec.team_name, rec.stationnumber, rec.staff_ien));
    END LOOP;
    RETURN;
  END;
  
  FUNCTION pipe_teams_by_staff_ien(i_staff_ien IN varchar2) RETURN TEAMS_TAB PIPELINED AS
  BEGIN
    FOR rec IN (
      SELECT DISTINCT T.TEAM_ID, T.TEAM_NAME, I.STATIONNUMBER, S.STAFF_IEN
      FROM PCMM.STAFF S
      JOIN PCMM.TEAM_MEMBERSHIP TM ON S.STAFF_ID = TM.STAFF_ID
      JOIN PCMM.TEAM T ON T.TEAM_ID = TM.TEAM_ID
      JOIN SDSADM.STD_INSTITUTION I ON I.ID = T.VA_INSTITUTION_ID
      WHERE S.STAFF_IEN=i_staff_ien)
    LOOP
      PIPE ROW(TEAMS_ROW(rec.team_id, rec.team_name, rec.stationnumber, rec.staff_ien));
    END LOOP;
    RETURN;
  END;
  
  FUNCTION pipe_roles_by_team(i_team_id IN number) RETURN ROLES_TAB PIPELINED AS
  BEGIN
    FOR rec IN (
      SELECT DISTINCT TR.PCM_STD_TEAM_ROLE_ID, TR.NAME, S.STAFF_IEN, I.STATIONNUMBER
      FROM PCMM.STAFF S
      JOIN PCMM.TEAM_MEMBERSHIP TM ON S.STAFF_ID = TM.STAFF_ID
      JOIN PCMM.TEAM T ON T.TEAM_ID = TM.TEAM_ID
      JOIN PCMM.PCM_STD_TEAM_ROLE TR ON TM.PCM_STD_TEAM_ROLE_ID = TR.PCM_STD_TEAM_ROLE_ID
      JOIN SDSADM.STD_INSTITUTION I ON I.ID = T.VA_INSTITUTION_ID
      WHERE T.TEAM_ID=i_team_id)
    LOOP
      PIPE ROW(ROLES_ROW(rec.pcm_std_team_role_id, rec.name, rec.staff_ien, rec.stationnumber));
    END LOOP;
    RETURN;
  END;
  
  PROCEDURE get_roles_for_team(
      i_team_id IN number,
      o_cursor  OUT SYS_REFCURSOR)
  AS
  BEGIN
    OPEN o_cursor FOR
    SELECT DISTINCT PCM_STD_TEAM_ROLE_ID, NAME
    FROM TABLE(pipe_roles_by_team(i_team_id)) TEAM_ROLES
    JOIN EHMP.EHMP_DIVISIONS D ON D.DIVISION_ID = TEAM_ROLES.STATIONNUMBER
    WHERE EXISTS (SELECT null
                  FROM EHMP.EHMP_ROUTING_USERS U
                  WHERE U.USER_IEN = TEAM_ROLES.STAFF_IEN AND
                        U.USER_SITE = D.SITE_CODE);
  END;

  PROCEDURE get_teams_for_facility(
      i_facility_id IN varchar2,
      o_cursor      OUT SYS_REFCURSOR)
  AS
  BEGIN
  
    OPEN o_cursor FOR
    SELECT DISTINCT TEAM_ID, TEAM_NAME, STATIONNUMBER
    FROM TABLE(pipe_teams_by_facility(i_facility_id)) TEAMS
    WHERE EXISTS (SELECT null
                  FROM EHMP.EHMP_ROUTING_USERS U
                  WHERE U.USER_IEN = TEAMS.STAFF_IEN AND
                        U.USER_SITE = (SELECT SITE_CODE
                                       FROM EHMP.EHMP_DIVISIONS
                                       WHERE DIVISION_ID = i_facility_id));

  END;

  PROCEDURE get_teams_for_facility_patient(
      i_facility_id IN varchar2,
      i_icn         IN varchar2,
      o_cursor      OUT SYS_REFCURSOR)
  AS
  BEGIN
    OPEN o_cursor FOR
    SELECT DISTINCT TEAM_ID, TEAM_NAME, STATIONNUMBER
    FROM TABLE(pipe_teams_by_facility_patient(i_facility_id, i_icn)) TEAMS
    JOIN EHMP.EHMP_DIVISIONS D ON D.DIVISION_ID = TEAMS.STATIONNUMBER
    WHERE EXISTS (SELECT null 
                  FROM EHMP.EHMP_ROUTING_USERS U 
                  WHERE U.USER_IEN = TEAMS.STAFF_IEN AND
                        U.USER_SITE = D.SITE_CODE);
  END;

  PROCEDURE get_teams_for_patient(
      i_icn    IN varchar2,
      o_cursor OUT SYS_REFCURSOR)
  AS
  BEGIN
    OPEN o_cursor FOR
    SELECT DISTINCT TEAM_ID, TEAM_NAME, STATIONNUMBER
    FROM TABLE(pipe_teams_by_patient(i_icn)) TEAMS
    JOIN EHMP.EHMP_DIVISIONS D ON D.DIVISION_ID = TEAMS.STATIONNUMBER
    WHERE EXISTS (SELECT null 
                  FROM EHMP.EHMP_ROUTING_USERS U 
                  WHERE U.USER_IEN = TEAMS.STAFF_IEN AND
                        U.USER_SITE = D.SITE_CODE);
      
  END;

  PROCEDURE get_teams_for_user(
      i_staff_ien  IN varchar2,
      i_staff_site IN varchar2,
      o_cursor OUT SYS_REFCURSOR)
  AS
  BEGIN
    OPEN o_cursor FOR
    SELECT DISTINCT TEAM_ID, TEAM_NAME, STATIONNUMBER
    FROM TABLE(pipe_teams_by_staff_ien(i_staff_ien)) TEAMS
    WHERE STATIONNUMBER IN (SELECT D.DIVISION_ID
                            FROM EHMP.EHMP_DIVISIONS D
                            WHERE D.SITE_CODE = i_staff_site);
  END;

  PROCEDURE get_teams_for_user(
      i_staff_ien  IN varchar2,
      i_staff_division IN varchar2,
      o_cursor OUT SYS_REFCURSOR)
  AS
  BEGIN
    OPEN o_cursor FOR
    SELECT DISTINCT T.TEAM_ID, T.TEAM_NAME, I.STATIONNUMBER, S.STAFF_IEN
    FROM PCMM.STAFF S
    JOIN PCMM.TEAM_MEMBERSHIP TM ON S.STAFF_ID = TM.STAFF_ID
    JOIN PCMM.TEAM T ON T.TEAM_ID = TM.TEAM_ID
    JOIN SDSADM.STD_INSTITUTION I ON I.ID = T.VA_INSTITUTION_ID
    WHERE S.STAFF_IEN=i_staff_ien AND
          I.STATIONNUMBER = i_staff_division;
  END;

END PCMM_API;
/
