package org.opencds.plugin;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Pattern;

import org.opencds.plugin.PluginContext.PostProcessPluginContext;
import org.opencds.plugin.util.VmrUtil;
import org.opencds.vmr.v1_0.internal.EvalTime;
import org.opencds.vmr.v1_0.internal.EvaluatedPerson;
import org.opencds.vmr.v1_0.internal.FocalPersonId;
import org.opencds.vmr.v1_0.internal.ObservationResult;
import org.opencds.vmr.v1_0.internal.datatypes.IVLDate;

/**
 * The {@link PrintObjectsPlugin} will find data conforming to a specific
 * pattern (see below) in the namedObjects map, in order to concatenate them
 * into a single pipe-delimited list of values that are originally intended to
 * provide context for decisions made by the given KnowledgeModule deployed
 * within OpenCDS.
 * <p>
 * In order to leverage this plugin, the KnowledgeModule must use the
 * appropriate DSLs to add data to the namedObjects map. The data are associated
 * to triplet of OpenCDS focusCode, OpenCDS attributeCode, and a
 * valueContextCode or name, which will be the key in the namedObjects map. The
 * values associated with that key may be any value (including a null value).
 * <p>
 * The data are inserted output vMR as an <tt>ObservationResult</tt>, where the
 * id is <tt>2.16.840.1.113883.3.795.5.1.1^PRINTNAMEDOBJECTS</tt>, and the value
 * is a pipe-delimited list in the following format:
 * 
 * <pre>
 * &lt;focusCode&gt;:&lt;attributeCode&gt;:&lt;valueContextCode&gt;:&lt;value&gt;|&lt;focusCode&gt;:&lt;attributeCode&gt;:&lt;valueContextode&gt;:&lt;value&gt;
 * </pre>
 * 
 * and so forth.
 * <p>
 * The focusCode must match the pattern of an OpenCDS concept, namely the
 * regular expression:
 * 
 * <pre>
 * ^C[0-9]{1,}:C[0-9]{1,}:.*$
 * </pre>.
 * 
 * @author phillip
 *
 *         <value> -> FC:AC:VCC:[List<Date>]1, 2, 3, 4, 5#mcg
 *
 */
public class PrintObjectsPlugin implements PostProcessPlugin {

    private static final Pattern PRINT_OBJECT_PATTERN = Pattern.compile("^C[0-9]{1,}:C[0-9]{1,}:.*$");
    private static final String COLON = ":";

    private static final String PRINT_NAMED_OBJECTS = "PRINTNAMEDOBJECTS";
    private static final String PRINT_OBJECTS_CD = "printObjects";
    private static final String PRINT_OBJECTS_DN = "printObjects found in global memory";
    private static final String PIPE = "|";

    @Override
    public void execute(PostProcessPluginContext context) {
        Map<String, Object> namedObjects = context.getNamedObjects();

        Set<String> namedObjectsSet = new TreeSet<>();
        System.err.println("Finding candiates for Print Objects");
        for (Map.Entry<String, Object> entry : namedObjects.entrySet()) {
            System.err.println("Candidate: " + entry.getKey());
            // key contains the focus concept Code, ending in a colon
            if (PRINT_OBJECT_PATTERN.matcher(entry.getKey()).find()) {
                // process from here... see VMRUtil for example usage of the
                // data generated by this code.
                namedObjectsSet.add(entry.getKey() + COLON
                        + (entry.getValue() != null ? (String) entry.getValue() : ""));
                System.err.println("Added entry to Print Objects: " + entry.getKey() + COLON
                        + (entry.getValue() != null ? (String) entry.getValue() : ""));
            }
        }
        StringBuilder sb = new StringBuilder();
        int length = namedObjectsSet.size();
        int counter = 0;
        for (String namedObject : namedObjectsSet) {
            counter++;
            sb.append(namedObject);
            if (counter < length) {
                sb.append(PIPE);
            }
        }

        // add to the vMR as an ObservationResult
        IVLDate obsTime = VmrUtil.createObsTime((EvalTime) context.getAllFactLists().get(EvalTime.class).get(0));

        ObservationResult obsResult = VmrUtil.createObservationResult(
                ((EvaluatedPerson) context.getAllFactLists().get(EvaluatedPerson.class).get(0)).getId(), obsTime,
                ((FocalPersonId) context.getAllFactLists().get(FocalPersonId.class).get(0)).getId(),
                PRINT_NAMED_OBJECTS, PRINT_OBJECTS_CD, PRINT_OBJECTS_DN, sb.toString());

        Map<String, List<?>> resultFactLists = context.getResultFactLists();
        List<ObservationResult> obsResults = (List<ObservationResult>) resultFactLists.get(ObservationResult.class
                .getSimpleName());
        if (obsResults == null) {
            obsResults = new ArrayList<>();
            resultFactLists.put(ObservationResult.class.getSimpleName(), obsResults);
        }
        obsResults.add(obsResult);
    }
}
