/**
 * *****************************************************************************
 *
 * Copyright 2015 Cognitive Medical Systems
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing permissions and limitations under the License.
 *
 *
 *
 ******************************************************************************
 */
package com.cognitive.cds.services.cdsinteractive;

import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.Consumes;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;

import org.apache.cxf.jaxrs.ext.MessageContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.cognitive.cds.invocation.CDSInvocationIFace;
import com.cognitive.cds.invocation.CDSInvoker;
import com.cognitive.cds.invocation.InvocationType;
import com.cognitive.cds.invocation.model.Context;
import com.cognitive.cds.invocation.model.FaultInfo;
import com.cognitive.cds.invocation.model.InvocationTarget;
import com.cognitive.cds.invocation.model.ResultBundle;
import com.cognitive.cds.invocation.model.StatusCode;

/**
 *
 * Generic Raw Service for interactive invocation of reasoning
 *
 * @author Jerry Goodnough
 * @version 1.0
 * @created 4/27/2015
 */
public class InvokeService implements InvokeIface {

	private static final Logger logger = LoggerFactory.getLogger(InvokeService.class);

	CDSInvocationIFace cdsInvoker;

	public InvokeService() {

	}

	public InvokeService(CDSInvoker cdsInvoker) {
		this.cdsInvoker = cdsInvoker;

	}

	public CDSInvocationIFace getCdsInvoker() {
		return cdsInvoker;
	}

	@Override
	@POST
	@Consumes("application/json")
	@Produces("application/json")
	@Path("/invokeRules")
	public ResultBundle invokeRules(InvokeServiceReq req, @javax.ws.rs.core.Context MessageContext context) {
		// How do we make sure that FHIR get serialized correctly?
		ResultBundle invokerResult;
		ErrorState errroState = checkRequestForErrors(req);
		if (errroState.ok) {
			logger.debug("Invoking " + req.getTarget().toString() + ", for " + req.getContext().toString());
			invokerResult = cdsInvoker.invoke(req.getTarget(), req.getContext(), req.getParameters(),
					req.getDataModel());
			logger.debug("Invoked " + req.getTarget().toString() + ", for " + req.getContext().toString());
		} else {
			invokerResult = errroState.bundle;
		}

		// Set the status code based on the
		HttpServletResponse response = context.getHttpServletResponse();
		response.setStatus(invokerResult.getStatus().getHttpStatus().getStatusCode());
		try {
			response.flushBuffer();
		} catch (Exception e) {
		}

		return invokerResult;
	}

	public void setCdsInvoker(CDSInvocationIFace cdsInvoker) {
		this.cdsInvoker = cdsInvoker;
	}

	/**
	 * Check if the request is valid
	 * 
	 * @param req
	 *            The invoke service request
	 * @return
	 */
	private ErrorState checkRequestForErrors(InvokeServiceReq req) {
		ErrorState error = new ErrorState();
		if (req.getTarget() == null || req.getContext() == null) {
			error.addError(StatusCode.INVALID_INPUT_DATA, "Target and Context are required paramters");
		} else {
			InvocationTarget target = req.getTarget();
			if (target.getIntentsSet().isEmpty()) {
				error.addError(StatusCode.INVALID_INPUT_DATA, "No requested intents");
			}
			if (target.getType() == null) {
				error.addError(StatusCode.INVALID_INPUT_DATA, "Missing request Type, Use Direct");
			} else {
				if (target.getType() != InvocationType.Direct) {
					error.addError(StatusCode.INVALID_INPUT_DATA,
							"Only Direct mode supported at this time, please use Direct");
				}
			}
			if (target.getMode() == null) {
				error.addError(StatusCode.INVALID_INPUT_DATA, "Missing request model - use Normal or Raw");
			}
			Context ctx = req.getContext();
			if (ctx.getSubject() == null) {
				error.addError(StatusCode.INVALID_INPUT_DATA, "A Subject is required");
			}
			if (ctx.getUser() == null) {
				error.addError(StatusCode.INVALID_INPUT_DATA, "A User is required");
			}
		}

		return error;
	}

	private class ErrorState {

		public boolean ok = true;
		public ResultBundle bundle;

		public void addError(StatusCode code, String detail) {
			FaultInfo info = new FaultInfo(detail);

			if (bundle == null) {
				bundle = new ResultBundle();
				bundle.setStatus(code);
			} else {
				if (!bundle.getStatus().equals(code)) {
					bundle.setStatus(StatusCode.MULTIPLE_FAULTS);
				}

			}
			bundle.getFaultInfo().add(info);
			ok = false;
		}
	}
}
