/*******************************************************************************
 *
 *   Copyright 2016 Cognitive Medical Systems
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *     
 *******************************************************************************/
package com.cognitive.cds.invocation;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.annotation.PostConstruct;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import com.cognitive.cds.invocation.exceptions.DataRetrievalException;
import com.cognitive.cds.invocation.exceptions.DataValidationException;
import com.cognitive.cds.invocation.model.CallMetrics;
import com.cognitive.cds.invocation.model.Context;
import com.cognitive.cds.invocation.model.EngineInfo;
import com.cognitive.cds.invocation.model.FaultInfo;
import com.cognitive.cds.invocation.model.Future;
import com.cognitive.cds.invocation.model.IntentMapping;
import com.cognitive.cds.invocation.model.InvocationConstants;
import com.cognitive.cds.invocation.model.InvocationConstants.StatusCodes;
import com.cognitive.cds.invocation.model.InvocationMapping;
import com.cognitive.cds.invocation.model.InvocationTarget;
import com.cognitive.cds.invocation.model.ResultBundle;
import com.cognitive.cds.invocation.model.Rule;
import com.cognitive.cds.invocation.model.StatusCode;

/**
 * Core implementation of the CDSInvocationIFace.
 *
 * @see CDSInvocationIFace
 *
 * @author Jerry Goodnough
 * @version 1.0
 * @created 11-Dec-2014 9:10:40 AM
 */
public class CDSInvoker implements CDSInvocationIFace {

	private static final Logger logger = LoggerFactory.getLogger(CDSInvoker.class);

    // Configuration Variables
	// Defined in the beans
	private Map<String, EngineInfo> enginesMap;
	private Map<String, IntentMapping> intentsMap;

	private List<CDSMetricsIFace> metricCollectors;
	private RepositoryLookupIFace repositoryLookupAgent;

	private DataModelHandlerIFace defaultDataModelHandler;

	private String dataModelHandlerBeanName;

	@Autowired
	private ApplicationContext springCtx;

	/**
	 * This is technically still optional since we could provide the engine url via the setter. If the setter is not
	 * used, then it's assumed that we should try to use the EngineInstanceStateManager.
	 */
	@Autowired(required = false)
	private EngineInstanceStateManagementIFace eism;

	public CDSInvoker() {

	}

	@PostConstruct
	protected void initalize() {
		if (defaultDataModelHandler == null && dataModelHandlerBeanName != null && !dataModelHandlerBeanName.isEmpty()) {
			defaultDataModelHandler = (DataModelHandlerIFace) springCtx.getBean(dataModelHandlerBeanName);
		}
	}

	/**
	 * Get a unique call Id for this request. This Id should be used to tied together results and metrics.
	 *
	 * @return A unique call Id
	 */
	protected String generateCallId() {
		java.util.UUID uuid = java.util.UUID.randomUUID();

		return uuid.toString();
	}

	/**
	 * Synchronous invocation of rules processing.
	 *
	 * @param invocationTarget Structure describing what should be invoked
	 * @param context The Context in which the execution occurs
	 * @param parameters Supplemental data parameters specific to the call
	 * @param dataModel The Data model to pass in to the reasoning model.
	 *
	 * Todo:
	 *
	 * 1) Investigate the changing the dataModel to Object or Serializable
	 *
	 */
	@Override
	public ResultBundle invoke(InvocationTarget invocationTarget, Context context, Properties parameters, Object inputDataModel) {

		String serializedDataModel = "";

		// Initialize the results bundle
		ResultBundle resultBundle = new ResultBundle();
		StatusCode status = StatusCode.SUCCESS;
		List<FaultInfo> faultList = new ArrayList<FaultInfo>();
		resultBundle.setFaultInfo(faultList);
		resultBundle.setStatus(status);

		List<String> intentList = invocationTarget.getIntentsSet();

		// Require an intent
		if (intentList == null || intentList.isEmpty()) {
			populateInvocationTargetIntentError(resultBundle, faultList);
			return resultBundle;
		}

        // ===================================
		// Create Call Id .. seed per name
		// ===================================
		String callId = generateCallId();

        // ===================================
		// Create starting metrics list to send it down chain - Ideally non
		// blocking
		// ===================================
		List<CallMetrics> metrics = new ArrayList<>();

        // =========================================
		// START TIME
		// PREP and ADD METRIC TO LIST TO SEND LATER
		// =========================================
		// Create a common metric name when multiple intents are used
		String metricInvokeId = StringUtils.join(intentList, ',');

		addMetric(metrics, context, "invoke", "", "begin", metricInvokeId, callId, invocationTarget);

		// FUTURE Fetch context specific behaviors here
		for (Iterator<String> iterator = intentList.iterator(); iterator.hasNext();) {
			String intent = iterator.next();

			// FUTURE Add logic for remote stored intents
			IntentMapping intentMapping = intentsMap.get(intent);

			// retrieve intent information from the repository
			if (intentMapping == null && repositoryLookupAgent != null) {
				intentMapping = repositoryLookupAgent.lookupIntent(intent);
			}

			logger.debug("Loaded intent {}", intent);

			// Any intent that is missing is a fault
			if (intentMapping == null) {
                // We should log the fault to the bundle and update the status
				// code as required. We should then try the next intent

				// Log the event
				logger.error("Undefined CDS Invocation on intent " + intent);
				status = StatusCode.USE_NOT_RECOGNIZED;
				resultBundle.setStatus(status);
				addIntentMappingError(resultBundle);
				continue;
			}

			List<String> intents = invocationTarget.getIntentsSet();
			if (intents != null && intents.contains(intent)) {
				logger.debug("Processing intent {}", intent);

                // FUTURE Split this into a multi-threaded process
				// Logic to update an intent -
				List<InvocationMapping> invocationMappingList;

				if (intentMapping.getGovernance() != null) {
					List<InvocationMapping> mappings = Collections.unmodifiableList(intentMapping.getInvocations());
					// Supplement The mapping based on the
					invocationMappingList = intentMapping.getGovernance().apply(mappings, invocationTarget, context, parameters, intent, intentMapping);
				} else {
                    // Ok no special governace - Check if there are supplemental
					// external calls - If so they are applied to every quested
					// intent.

					if (invocationTarget.getSupplementalMappings() != null) {
						invocationMappingList = new ArrayList<InvocationMapping>(invocationTarget.getSupplementalMappings().size() + intentMapping.getInvocations().size());
						invocationMappingList.addAll(invocationTarget.getSupplementalMappings());
						invocationMappingList.addAll(intentMapping.getInvocations());
					} else {
						// Nothing special - Use the stock mapping
						invocationMappingList = intentMapping.getInvocations();
					}
				}

				for (Iterator<InvocationMapping> iterator2 = invocationMappingList.iterator(); iterator2.hasNext();) {
					InvocationMapping invocationMapping = iterator2.next();

					String engine = invocationMapping.getEngineName();
					List<Rule> ruleList = invocationMapping.getRules();
					EngineInfo engineInfo = enginesMap.get(engine);

					// retrieve engine information from the repository
					if (engineInfo == null && repositoryLookupAgent != null) {
						engineInfo = repositoryLookupAgent.lookupEngine(engine);
					}

					if (engineInfo == null) {
						// Add an engine fault and continue to the next intent
						addConfigurationError(resultBundle, engine, intent);
						break;
					}

					logger.debug("Use Engine {}", engine);

					// Grab the default data model builder
					DataModelHandlerIFace dmHandler = this.defaultDataModelHandler;
                    // FUTURE: Add Data model switch logic

					// FUTURE: Added Context and Subject Specific Rules
					if (invocationTarget.getMode() == InvocationMode.Normal) {
						if (dmHandler == null) {
							logger.error("Attempt to use normal mode without a Data Handler");
						}

						// Determine if data validation is on for this mapping
						boolean validate = false;
						if (invocationTarget.isDataModelValidationEnabled() || invocationMapping.isValidateDataModel()) {
							validate = true;
						}

                        // TOOO Insert Data Fetch mapping and Call model
						// reconciliation here
						// For Each intent we should fetch data that is required
						// for that
						// Intent.For a specific intent there should be a data
						// map with the
						// Query strings required for that Intent.
						try {

							serializedDataModel = dmHandler.buildDataModel(invocationTarget.getMode(), invocationMapping.getDataQueries(), context, parameters, inputDataModel, validate);
						} catch (DataRetrievalException e) {

							// Handle failure to load data model
							this.addDataFetchError(resultBundle, e);
							// Ok try the next intent
							continue;
						} catch (DataValidationException e) {
							this.addDataValidationError(resultBundle, e);
							// Ok try the next intent
							continue;
						}
					}

                    // =========================================
					// RULE START TIME
					// PREP and ADD METRIC TO LIST TO SEND LATER
					// =========================================
					addMetric(metrics, context, "engine", engineInfo.getName(), "begin", metricInvokeId, callId, invocationTarget);

					try {
						logger.debug("Calling Engine");

						CDSEnginePlugInIFace cdsEngine = engineInfo.getEngine();

						//get a physical instance either from config or mongodb...
						engineInfo.getName();

						// Handle normal vs Raw Invoke
						ResultBundle engineBundle;
						if (invocationTarget.getMode() != InvocationMode.Raw) {
							engineBundle = cdsEngine.invoke(ruleList, serializedDataModel, callId, eism);
						} else {
							engineBundle = cdsEngine.invokeRaw(ruleList, inputDataModel, callId, eism);
						}
                        // We always copy the result bundle since it the main
						// one may already have errors in it
						logger.debug("Engine Done");

						// Copy any faults
						if (engineBundle.getFaultInfo() != null) {
							resultBundle.getFaultInfo().addAll(engineBundle.getFaultInfo());
						}

						// Copy any results
						if (engineBundle.getResults() != null) {
							logger.debug("Handling results");

							if (invocationTarget.getMode() != InvocationMode.Raw) {
								logger.debug("Normal mode Result handling");

								// Pick apart the return data and normalize
								dmHandler.translateResults(engineBundle.getResults(), resultBundle);
							} else {
								logger.debug("Raw Mode Result Han1dling");

								resultBundle.getResults().addAll(engineBundle.getResults());
							}
						}
						// Deal with status
						StatusCode cd = resultBundle.getStatus();
						if (cd != null) {
							if (cd.getCode().compareTo(StatusCodes.SUCCESS.getCode()) != 0) {
								logger.debug("Status updated to indicate a fault");

                                // Ok the engine failed
								// First we check if the existing bundle is
								// ok...
								if (resultBundle.getStatus().getCode().compareTo(StatusCodes.SUCCESS.getCode()) == 0) {
                                    // If so we switch the status to the current
									// state
									resultBundle.setStatus(cd);
								} else {
                                    // If not then we switch the status to
									// multiple faults
									resultBundle.setStatus(StatusCode.MULTIPLE_FAULTS);
								}

							}
						}

					} catch (Exception ex) {
						logger.error("Error processing engine " + engineInfo.getName(), ex);
					}

                    // =========================================
					// RULE END TIME
					// PREP and ADD METRIC TO LIST TO SEND LATER
					// =========================================
					// time = use the default current time in prepMetric.
					addMetric(metrics, context, "engine", engineInfo.getName(), "end", metricInvokeId, callId, invocationTarget);
				}
			}
		}

        // =========================================
		// END TIME
		// PREP and ADD METRIC TO LIST TO SEND LATER
		// =========================================
		addMetric(metrics, context, "invoke", "", "end", metricInvokeId, callId, invocationTarget);

		reportMetrics(metrics, resultBundle, invocationTarget);

		return resultBundle;
	}

	/*
	 * Private Function to log an Invalid Input Data.
	 */
	private void populateInvocationTargetIntentError(ResultBundle resultBundle, List<FaultInfo> faultList) {
		FaultInfo fault = new FaultInfo();
		String errorMessage = InvocationConstants.StatusCodes.INVALID_INPUT_DATA.getMessage() + " , " + InvocationConstants.INTENT_NOT_PROVIDED;
		fault.setFault(errorMessage);
		faultList.add(fault);
		resultBundle.setFaultInfo(faultList);

		resultBundle.setStatus(StatusCode.INVALID_INPUT_DATA);
		logger.error(errorMessage);
	}

	/*
	 * Private Function to log an Invalid Use
	 */
	/**
	 * Update a bundle status - If there is already a fault the change the overall status to multiple faults
	 *
	 * @param status
	 */
	private void updateBundleStatus(ResultBundle resultBundle, StatusCode status) {
		if (resultBundle.getStatus().getCode() != StatusCode.SUCCESS.getCode()) {
			resultBundle.setStatus(status);

		} else if (resultBundle.getStatus().getCode() != status.getCode()) {
			resultBundle.setStatus(StatusCode.MULTIPLE_FAULTS);
		}
	}

	private void addIntentMappingError(ResultBundle resultBundle) {
		FaultInfo fault = new FaultInfo();
		String errorMessage = InvocationConstants.StatusCodes.USE_NOT_RECOGNIZED.getMessage() + " , " + InvocationConstants.INTENT_NOT_CONFIGURED;
		fault.setFault(errorMessage);
		resultBundle.getFaultInfo().add(fault);
	}

	private void addConfigurationError(ResultBundle resultBundle, String engineName, String intent) {
		FaultInfo fault = new FaultInfo();
		String errorMessage = InvocationConstants.StatusCodes.CONFIGURATION_ERROR + ", Engine = " + engineName + ", Intent = " + intent;
		fault.setFault(errorMessage);
		resultBundle.getFaultInfo().add(fault);
		updateBundleStatus(resultBundle, StatusCode.CONFIGURATION_ERROR);
		logger.error(errorMessage);
	}

	private void addDataFetchError(ResultBundle resultBundle, DataRetrievalException exp) {
		FaultInfo fault = new FaultInfo();
		String errorMessage = null;
		if (exp.getMessage().contains(InvocationConstants.HTTP_AUTHORIZATION_ERROR_CODE)) {
			errorMessage = InvocationConstants.StatusCodes.AUTHENICATION_ERROR.getMessage() + " , " + exp.getMessage();
			resultBundle.setStatus(StatusCode.AUTHENICATION_ERROR);
		} else {
			errorMessage = InvocationConstants.StatusCodes.DATA_SERVER_NOT_AVAILABLE.getMessage() + " , Fetch Error: " + exp.getMessage();
			resultBundle.setStatus(StatusCode.DATA_SERVER_NOT_AVAILABLE);
		}

		fault.setFault(errorMessage);
		resultBundle.getFaultInfo().add(fault);
		logger.error(errorMessage, exp);
	}

	private void addDataValidationError(ResultBundle resultBundle, DataValidationException exp) {
		FaultInfo fault = new FaultInfo();
		String errorMessage = InvocationConstants.StatusCodes.INVALID_INPUT_DATA.getMessage() + " , Validation Error: " + exp.getMessage();
		resultBundle.setStatus(StatusCode.INVALID_INPUT_DATA);

		fault.setFault(errorMessage);
		resultBundle.getFaultInfo().add(fault);
		logger.error(errorMessage, exp);
	}

	private void reportMetrics(List<CallMetrics> metrics, ResultBundle bundle, InvocationTarget invocationTarget) {

        // First we generate a Summary metric
		// It is assumed that the metrics list follow the invocation order
		if (metrics != null) {
			int size = metrics.size();
			CallMetrics start = metrics.get(0);
			CallMetrics end = metrics.get(size - 1);
			CallMetrics summary = new CallMetrics();
			summary.setCallId(start.getCallId());
			summary.setContext(start.getContext());
			summary.setOrigin("");
			summary.setInvocation(start.getInvocation());
			summary.setTime(new Timestamp(Calendar.getInstance().getTime().getTime()));
			summary.setType("invoke");
			summary.setEvent("summary");
			summary.setTotalResults(bundle.getResults().size());
			summary.setInvocationType(invocationTarget.getType());
			long totalDuration = end.getTime().getTime() - start.getTime().getTime();
			summary.getTimings().put("total", new Long(totalDuration));

			// Calculate total call duration
			if (size > 3) {
                // Here we can assume one or more engines are called
				// The time in engines is the is the total duration from the
				// time the first engine is called until
				// The last engine completes.
				CallMetrics engineStart = metrics.get(1);
				CallMetrics engineEnd = metrics.get(size - 2);
				long setupTime = engineStart.getTime().getTime() - start.getTime().getTime();
				summary.getTimings().put("callSetup", new Long(setupTime));

                // There is a small possibility of two or more engines
				// starting but none of them generating an end event.
				// The following code guards against that condition
				if (engineEnd.getEvent().compareTo("end") == 0) {
					long timeInEngines = engineEnd.getTime().getTime() - engineStart.getTime().getTime();
					summary.getTimings().put("inEngines", new Long(timeInEngines));
					long resultTime = end.getTime().getTime() - engineEnd.getTime().getTime();
					summary.getTimings().put("handlingResults", new Long(resultTime));
				}
			}
			if (size == 3) {
                // We have a fault in an engine and thus missing data for engine
				// timing
				CallMetrics engineStart = metrics.get(1);
				long setupTime = engineStart.getTime().getTime() - start.getTime().getTime();
				summary.getTimings().put("callSetup", new Long(setupTime));

			} else {
                // We have either one or two boundary points
				// For Now we have no special metric to place here.
			}
			metrics.add(summary);
		}

		sendMetricsToCollectors(metrics);

	}

	/**
	 * Helper function to send metrics to the collection chain - Exposed so that the calling context may report other
	 * metrics to the same sources
	 *
	 * @param metrics
	 */
	public void sendMetricsToCollectors(List<CallMetrics> metrics) {
		// FUTURE make this a non-blocking task.
		if (metricCollectors != null) {
			Iterator<CDSMetricsIFace> iter = metricCollectors.iterator();
			while (iter.hasNext()) {
				CDSMetricsIFace metric = iter.next();
				metric.updateMetrics(metrics);

			}
		}
	}

	private void addMetric(List<CallMetrics> metrics, Context ctx, String type, String origin, String event, String invocation, String callId, InvocationTarget invocationTarget) {
		metrics.add(new CallMetrics(type, origin, event, invocation, ctx, callId, new Timestamp(Calendar.getInstance().getTime().getTime()), invocationTarget.type));
	}

	/**
	 * ************** Future Operation Stubs ********************
	 */
	@Override
	public void getAvailableRules() {

	}

	@Override
	public void introduceData() {

	}

	/**
	 * The following is a stub for the deferred invocation mechanism. TDB Support optional
	 */
	@Override
	public Future invokeDeferred() {
		return null;
	}

	/**
	 * The following is a stub for the async results retrieval
	 */
	@Override
	public ResultBundle retrieveResults() {
		return null;
	}

	/**
	 * Properties for Operational Configuration 
	 */
	/**
	 * CDS Engine Map used for String configuration
	 *
	 * @return A map if CDS engines
	 */
	public Map<String, EngineInfo> getEnginesMap() {
		return enginesMap;
	}

	public void setEnginesMap(Map<String, EngineInfo> enginesMap) {
		this.enginesMap = enginesMap;
	}

	/**
	 * The Usage intents for the invoker to be aware of. Used by Spring Configuration
	 *
	 * @return The intent map
	 */
	public Map<String, IntentMapping> getIntentsMap() {
		return intentsMap;
	}

	public void setIntentsMap(Map<String, IntentMapping> intentsMap) {
		this.intentsMap = intentsMap;
	}

	/**
	 * The mapping of metrics collection engines used. Used by Spring Configuration.
	 *
	 * @return
	 */
	public List<CDSMetricsIFace> getMetricCollectors() {
		return metricCollectors;
	}

	public void setMetricCollectors(List<CDSMetricsIFace> metricCollectors) {
		this.metricCollectors = metricCollectors;
	}

	public String getDataModelHandlerBeanName() {
		return dataModelHandlerBeanName;
	}

	public void setDataModelHandlerBeanName(String dataModelHandlerBeanName) {
		this.dataModelHandlerBeanName = dataModelHandlerBeanName;
	}

	/**
	 * @return the repositoryLookupAgent
	 */
	public RepositoryLookupIFace getRepositoryLookupAgent() {
		return repositoryLookupAgent;
	}

	/**
	 * @param repositoryLookupAgent
	 */
	public void setRepositoryLookupAgent(RepositoryLookupIFace repositoryLookupAgent) {
		this.repositoryLookupAgent = repositoryLookupAgent;
	}

}
