/*global sinon, describe, it */
'use strict';

var parse = require('./immunization-data-parser').parse;

var log = sinon.stub(require('bunyan').createLogger({ name: 'immunization-data-parser' }));
//var log = require('bunyan').createLogger({ name: 'immunization-data-parser' }); //Uncomment this line (and comment above) to see output in IntelliJ console

describe('unit test to validate immunization-data', function() {
    it('can parse the RPC data correctly', function () {
    	/* jshint -W109 */
        var result = parse(log, '143' + '\r\n' +
                                '1^VACCINIA (SMALLPOX)^SMALLPOX^75^^0^^^^Y^Vaccinia (smallpox) vaccine^^^ACAM2000|DRYVAX^' + '\r\n' +
                                '2^TD(ADULT) UNSPECIFIED FORMULATION^TD-ADULT^139^^1^^^^Y^Td(adult) unspecified formulation^^^^' + '\r\n' +
                                '3^DTP^DPT^01^^1^^^^Y^Diphtheria, tetanus toxoids and pertussis vaccine^^^^' + '\r\n' +
                                '4^TETANUS TOXOID, UNSPECIFIED FORMULATION^TET TOX^112^^1^^^TT^Y^Tetanus toxoid, unspecified formulation^^^^' + '\r\n' +
                                '5^TYPHOID, UNSPECIFIED FORMULATION^TYPHOID^91^^1^^^^Y^Typhoid vaccine, unspecified formulation^^^^' + '\r\n' +
                                '6^OPV^SABIN/OPV^02^^1^^^^Y^Poliovirus vaccine, live, oral^^^ORIMUNE^' + '\r\n' +
                                '7^POLIO, UNSPECIFIED FORMULATION^SALK^89^^1^^^^Y^Poliovirus vaccine, unspecified formulation^^^^' + '\r\n' +
                                '8^SWINE FLU MONO (HISTORICAL)^SWINE MONO^^^1^^^^N^^^^^' + '\r\n' +
                                '9^SWINE FLU BIVAL (HISTORICAL)^SWINE BIVA^^^1^^^^N^^^^^' + '\r\n' +
                                '10^HEP B, ADULT^HEP B^43^^0^^^HepB^Y^Hepatitis B vaccine, adult dosage^^5~HEPATITIS B VIS   02-02-12   CURRENT   1^ENGERIX-B-ADULT|RECOMBIVAX-ADULT^' + '\r\n' +
                                '11^MEASLES^MEASLES^05^^1^^^^Y^Measles virus vaccine^^^ATTENUVAX^' + '\r\n' +
                                '12^INFLUENZA, UNSPECIFIED FORMULATION^INFLUENZA^88^^1^^^^Y^Influenza virus vaccine, unspecified formulation^^^^' + '\r\n' +
                                '13^CHOLERA^CHOLERA^26^^1^^^^Y^Cholera vaccine^^^^' + '\r\n' +
                                '14^RUBELLA^RUBELLA^06^^0^^^^Y^Rubella virus vaccine^^^MERUVAX II^' + '\r\n' +
                                '15^MUMPS^MUMPS^07^^0^^^^Y^Mumps virus vaccine^^^MUMPSVAX^' + '\r\n' +
                                '16^BCG (HISTORICAL)^BCG^19^^1^^^^N^^^^^' + '\r\n' +
                                '17^MMR^MMR^03^^0^^^MMR^Y^Measles, mumps and rubella virus vaccine^^17~MEASLES/MUMPS/RUBELLA VIS   04-20-12   CURRENT   1^M-M-R II^' + '\r\n' +
                                '18^M/R^MR^04^^1^^^^Y^Measles and rubella virus vaccine^^^^' + '\r\n' +
                                '19^PNEUMOCOCCAL, UNSPECIFIED FORMULATION^PNEUMO-VAC^109^^1^^^^Y^Pneumococcal vaccine, unspecified formulation^^^^' + '\r\n' +
                                '20^YELLOW FEVER^YELLOW FEV^37^^0^^^^Y^Yellow fever vaccine^^31~YELLOW FEVER VIS   03-30-11   CURRENT   1^YF-VAX^' + '\r\n' +
                                '21^TYPHUS, HISTORICAL^TYPHUS^131^^1^^^^Y^Historical record of a typhus vaccination^^^^' + '\r\n' +
                                '22^RABIES, UNSPECIFIED FORMULATION^RABIES^90^^1^^^^Y^Rabies vaccine, unspecified formulation^^^^' + '\r\n' +
                                '23^DT (PEDIATRIC)^DT-PEDS^28^^0^^^DT^Y^Diphtheria and tetanus toxoids, adsorbed for pediatric use^^^DT(GENERIC)^' + '\r\n' +
                                '24^INFLUENZA B (HISTORICAL)^HIB^^^1^^^^N^^^^^' + '\r\n' +
                                '25^HEP A, UNSPECIFIED FORMULATION^HEP A^85^^1^^^HepA^Y^Hepatitis A vaccine, unspecified formulation^^^^' + '\r\n' +
                                '26^MENINGOCOCCAL, UNSPECIFIED FORMULATION^MENING.^108^^1^^^^Y^Meningococcal vaccine, unspecified formulation^^^^' + '\r\n' +
                                '27^ENCEPHALITIS (HISTORICAL)^ENCEPH.^^^1^^^^N^^^^^' + '\r\n' +
                                '28^ZOSTER^VARCELLA^121^^0^^^HZV^Y^Zoster vaccine, live^^30~VARICELLA (CHICKENPOX) VIS   03-13-08   CURRENT   1|27~SHINGLES VIS   10-06-09   CURRENT   1^VARIVAX|ZOSTAVAX^' + '\r\n' +
                                '29^DTAP, UNSPECIFIED FORMULATION^DTaP^107^^1^^^DTaP^Y^Diphtheria, tetanus toxoids and acellular pertussis vaccine, unspecified formulation^^^^' + '\r\n' +
                                '30^RUBELLA/MUMPS^RUB-MUMPS^38^^1^^^^Y^Rubella and mumps virus vaccine^^^BIAVAX II^' + '\r\n' +
                                '31^DIPHTHERIA (HISTORICAL)^DIPHTHERIA^^^1^^^^N^^^^^' + '\r\n' +
                                '32^DTB/HIB (HISTORICAL)^DTB/HIB^^^1^^^^N^^^^^' + '\r\n' +
                                '33^DTP, POLIO (HISTORICAL)^DTP POLIO^^^1^^^^N^^^^^' + '\r\n' +
                                '34^MMRV^MMR&V^94^^0^^^MMRV^Y^Measles, mumps, rubella, and varicella virus vaccine^^18~MEASLES/MUMPS/RUBELLA/VARICELLA VIS   05-21-10   CURRENT   1^PROQUAD^' + '\r\n' +
                                '35^PLAGUE^PLAGUE^23^^0^^^^Y^Plague vaccine^^^^' + '\r\n' +
                                '36^GAMMA GLOBULIN (HISTORICAL)^ISG^^^1^^^^N^^^^^' + '\r\n' +
                                '37^UNLISTED (HISTORICAL)^UNLISTED^^^1^^^^N^^^^^' + '\r\n' +
                                '38^SHERI (HISTORICAL)^SLB^^^1^^SLB^^N^^^^^' + '\r\n' +
                                '39^ADENOVIRUS, TYPE 4^ADEN TYP4^54^^1^^^^Y^Adenovirus vaccine, type 4, live, oral^^^^' + '\r\n' +
                                '40^ADENOVIRUS, TYPE 7^ADEN TYP7^55^^1^^^^Y^Adenovirus vaccine, type 7, live, oral^^^^' + '\r\n' +
                                '41^ANTHRAX^ANT SC^24^^0^^^^Y^Anthrax vaccine^^^BIOTHRAX^' + '\r\n' +
                                '42^BCG^BCG P^19^^0^^^^Y^Bacillus Calmette-Guerin vaccine^^^MYCOBAX|TICE BCG^' + '\r\n' +
                                '43^BCG,INTRAVESICAL (HISTORICAL)^BCG I^^^1^^^^N^^^^^' + '\r\n' +
                                '44^CHOLERA, ORAL (HISTORICAL)^CHOL ORAL^26^^1^^^^N^^^^^' + '\r\n' +
                                '45^HEP A, ADULT^HEPA AD^52^^0^^^HepA^Y^Hepatitis A vaccine, adult dosage^^4~HEPATITIS A VIS   10-25-11   CURRENT   1^HAVRIX-ADULT|VAQTA-ADULT^' + '\r\n' +
                                '46^HEP A, PED/ADOL, 2 DOSE^HEPA PED/ADOL-2^83^^0^^^HepA^Y^Hepatitis A vaccine, pediatric/adolescent dosage, 2 dose schedule^^4~HEPATITIS A VIS   10-25-11   CURRENT   1^HAVRIX-PEDS|VAQTA-PEDS^' + '\r\n' +
                                '47^HEP A, PED/ADOL, 3 DOSE^HEPA PED/ADOL-3^84^^1^^^HepA^Y^Hepatitis A vaccine, pediatric/adolescent dosage, 3 dose schedule^^^^' + '\r\n' +
                                '48^HEP A-HEP B^HEPA/HEPB AD^104^^0^^^HepA-HepB^Y^Hepatitis A and hepatitis B vaccine^^4~HEPATITIS A VIS   10-25-11   CURRENT   1|5~HEPATITIS B VIS   02-02-12   CURRENT   1^TWINRIX^' + '\r\n' +
                                '49^HIB (HBOC)^HIB,HBOC^47^^1^^^^Y^Haemophilus influenzae type b vaccine, HbOC conjugate^^^HIBTITER^' + '\r\n' +
                                '50^HIB (PRP-D)^HIB PRP-D^46^^1^^^^Y^Haemophilus influenzae type b vaccine, PRP-D conjugate^^^PROHIBIT^' + '\r\n' +
                                '51^HIB (PRP-OMP)^HIB PRP-OMP^49^^0^^^PRP-OMP^Y^Haemophilus influenzae type b vaccine, PRP-OMP conjugate^^6~HAEMOPHILUS INFLUENZAE TYPE B VIS   12-16-98   HISTORIC   1|7~HAEMOPHILUS INFLUENZAE TYPE B VIS   02-04-14   CURRENT   1^PEDVAXHIB^' + '\r\n' +
                                '52^HIB (PRP-T)^HIB PRP-T^48^^0^^^PRP-T^Y^Haemophilus influenzae type b vaccine, PRP-T conjugate^^6~HAEMOPHILUS INFLUENZAE TYPE B VIS   12-16-98   HISTORIC   1|7~HAEMOPHILUS INFLUENZAE TYPE B VIS   02-04-14   CURRENT   1^ACTHIB|HIBERIX|OMNIHIB^' + '\r\n' +
                                '53^INFLUENZA, UNSPECIFIED FORMULATION (HISTORICAL)^INFLUENZA^88^^1^^^^N^Influenza virus vaccine, unspecified formulation^^^^' + '\r\n' +
                                '54^INFLUENZA, WHOLE^FLU WHOLE^16^^1^^^^Y^Influenza virus vaccine, whole virus^^^^' + '\r\n' +
                                '55^INFLUENZA, LIVE, INTRANASAL^FLU NAS^111^^0^^^LAIV3^Y^Influenza virus vaccine, live, attenuated, for intranasal use^^11~INFLUENZA VACCINE - LIVE, INTRANASAL VIS   07-02-12   HISTORIC   1|12~INFLUENZA VACCINE - LIVE, INTRANASAL VIS   07-26-13   HISTORIC   1|37~INFLUENZA VACCINE - LIVE, INTRANASAL VIS   08-19-14   CURRENT   1^FLUMIST^' + '\r\n' +
                                '56^LYME DISEASE^LYME^66^^1^^^^Y^Lyme disease vaccine^^^^' + '\r\n' +
                                '57^PNEUMOCOCCAL,PED (HISTORICAL)^PNEUMO-PED^109^^1^^^^N^^^^^' + '\r\n' +
                                '58^RABIES, INTRAMUSCULAR INJECTION^RAB^18^^0^^^^Y^Rabies vaccine, for intramuscular injection^^24~RABIES VIS   10-06-09   CURRENT   1^IMOVAX|RABAVERT^' + '\r\n' +
                                '59^RABIES, INTRADERMAL INJECTION^RAB ID^40^^0^^^^Y^Rabies vaccine, for intradermal injection^^^IMOVAX ID|RABAVERT^' + '\r\n' +
                                '60^ROTAVIRUS, UNSPECIFIED FORMULATION^ROTO ORAL^122^^1^^^^Y^Rotavirus vaccine, unspecified formulation^^^^' + '\r\n' +
                                '61^TYPHOID, ORAL^TYP ORAL^25^^0^^^^Y^Typhoid vaccine, live, oral^^29~TYPHOID VIS   05-29-12   CURRENT   1^VIVOTIF BERNA^' + '\r\n' +
                                '62^TYPHOID, UNSPECIFIED FORMULATION (HISTORICAL)^TYP^91^^1^^^^N^Typhoid vaccine, unspecified formulation^^^^' + '\r\n' +
                                '63^TYPHOID, PARENTERAL^TYP H-P-SC/ID^41^^0^^^^Y^Typhoid vaccine, parenteral, other than acetone-killed, dried^^29~TYPHOID VIS   05-29-12   CURRENT   1^^' + '\r\n' +
                                '64^TYPHOID, PARENTERAL, AKD (U.S. MILITARY)^TYP AKD-SC^53^^0^^^^Y^Typhoid vaccine, parenteral, acetone-killed, dried (U.S. military)^^^TYPHOID-AKD^' + '\r\n' +
                                '65^HEP B, UNSPECIFIED FORMULATION^HEPB ILL^45^^1^^^HepB^Y^Hepatitis B vaccine, unspecified formulation^^^^' + '\r\n' +
                                '66^HIB-HEP B^HEPB/HIB^51^^0^^^Hib-HepB^Y^Haemophilus influenzae type b conjugate and Hepatitis B vaccine^^6~HAEMOPHILUS INFLUENZAE TYPE B VIS   12-16-98   HISTORIC   1|7~HAEMOPHILUS INFLUENZAE TYPE B VIS   02-04-14   CURRENT   1|5~HEPATITIS B VIS   02-02-12   CURRENT   1^COMVAX^' + '\r\n' +
                                '67^PNEUMOCOCCAL CONJUGATE PCV 13^PNEU PCV13^133^^0^^^PCV13^Y^Pneumococcal conjugate vaccine, 13 valent^^20~PNEUMOCOCCAL CONJUGATE (PCV13) VIS   04-16-10   HISTORIC   1|21~PNEUMOCOCCAL CONJUGATE (PCV13) VIS   02-27-13   CURRENT   1^PREVNAR 13^' + '\r\n' +
                                '1008^HEP B, ADOLESCENT OR PEDIATRIC^^08^^0^^^HepB^Y^Hepatitis B vaccine, pediatric or pediatric/adolescent dosage^^5~HEPATITIS B VIS   02-02-12   CURRENT   1^ENGERIX B-PEDS|RECOMBIVAX-PEDS^' + '\r\n' +
                                '1009^TD (ADULT), ADSORBED^^09^^0^^^Td^Y^Tetanus and diphtheria toxoids, adsorbed, for adult use^^34~TETANUS/DIPHTHERIA (TD) VIS   02-04-14   CURRENT   1^TD(GENERIC)|TD, (ADULT)^' + '\r\n' +
                                '1010^IPV^^10^^0^^^IPV^Y^Poliovirus vaccine, inactivated^^23~POLIO VIS   11-08-11   CURRENT   1^IPOL^' + '\r\n' +
                                '1011^PERTUSSIS^^11^^1^^^^Y^Pertussis vaccine^^^^' + '\r\n' +
                                '1015^INFLUENZA, SPLIT (INCL. PURIFIED SURFACE ANTIGEN)^^15^^1^^^^Y^Influenza virus vaccine, split virus (incl. purified surface antigen)-retired CODE^^^^' + '\r\n' +
                                '1017^HIB, UNSPECIFIED FORMULATION^^17^^1^^^^Y^Haemophilus influenzae type b vaccine, conjugate unspecified formulation^^^^' + '\r\n' +
                                '1020^DTAP^^20^^0^^^DTaP^Y^Diphtheria, tetanus toxoids and acellular pertussis vaccine^^3~DIPHTHERIA/TETANUS/PERTUSSIS (DTAP) VIS   05-17-07   CURRENT   1^ACEL-IMUNE|CERTIVA|INFANRIX|TRIPEDIA^' + '\r\n' +
                                '1022^DTP-HIB^^22^^1^^^^Y^DTP-Haemophilus influenzae type b conjugate vaccine^^^TETRAMUNE^' + '\r\n' +
                                '1031^HEP A, PEDIATRIC, UNSPECIFIED FORMULATION^^31^^1^^^HepA^Y^Hepatitis A vaccine, pediatric dosage, unspecified formulation^^^^' + '\r\n' +
                                '1032^MENINGOCOCCAL MPSV4^^32^^0^^^MPSV4^Y^Meningococcal polysaccharide vaccine (MPSV4)^^19~MENINGOCOCCAL VIS   10-14-11   CURRENT   1^MENOMUNE^' + '\r\n' +
                                '1035^TETANUS TOXOID, ADSORBED^^35^^0^^^^Y^Tetanus toxoid, adsorbed^^34~TETANUS/DIPHTHERIA (TD) VIS   02-04-14   CURRENT   1^TETANUS TOXOID (GENERIC)^' + '\r\n' +
                                '1039^JAPANESE ENCEPHALITIS SC^^39^^0^^^^Y^Japanese Encephalitis Vaccine SC^^^JE-VAX^' + '\r\n' +
                                '1042^HEP B, ADOLESCENT/HIGH RISK INFANT^^42^^1^^^HepB^Y^Hepatitis B vaccine, adolescent/high risk infant dosage^^5~HEPATITIS B VIS   02-02-12   CURRENT   1^^' + '\r\n' +
                                '1044^HEP B, DIALYSIS^^44^^0^^^HepB^Y^Hepatitis B vaccine, dialysis patient dosage^^5~HEPATITIS B VIS   02-02-12   CURRENT   1^RECOMBIVAX-DIALYSIS^' + '\r\n' +
                                '1050^DTAP-HIB^^50^^0^^^^Y^DTaP-Haemophilus influenzae type b conjugate vaccine^^^TRIHIBIT^' + '\r\n' +
                                '1062^HPV, QUADRIVALENT^^62^^0^^^HPV4^Y^Human papilloma virus vaccine, quadrivalent^^9~HUMAN PAPILLOMAVIRUS VACCINE (GARDASIL) VIS   02-22-12   HISTORIC   1|10~HUMAN PAPILLOMAVIRUS VACCINE (GARDASIL) VIS   05-17-13   CURRENT   1^GARDASIL^' + '\r\n' +
                                '1069^PARAINFLUENZA-3^^69^^1^^^^Y^Parainfluenza-3 virus vaccine^^^^' + '\r\n' +
                                '1074^ROTAVIRUS, TETRAVALENT^^74^^1^^^^Y^Rotavirus, live, tetravalent vaccine^^^^' + '\r\n' +
                                '1076^STAPHYLOCOCCUS BACTERIO LYSATE^^76^^1^^^^Y^Staphylococcus bacteriophage lysate^^^^' + '\r\n' +
                                '1077^TICK-BORNE ENCEPHALITIS^^77^^1^^^^Y^Tick-borne encephalitis vaccine^^^^' + '\r\n' +
                                '1078^TULAREMIA VACCINE^^78^^1^^^^Y^Tularemia vaccine^^^^' + '\r\n' +
                                '1080^VEE, LIVE^^80^^1^^^^Y^Venezuelan equine encephalitis, live, attenuated^^^^' + '\r\n' +
                                '1081^VEE, INACTIVATED^^81^^1^^^^Y^Venezuelan equine encephalitis, inactivated^^^^' + '\r\n' +
                                '1082^ADENOVIRUS, UNSPECIFIED FORMULATION^^82^^1^^^^Y^Adenovirus vaccine, unspecified formulation^^^^' + '\r\n' +
                                '1092^VEE, UNSPECIFIED FORMULATION^^92^^1^^^^Y^Venezuelan equine encephalitis vaccine, unspecified formulation^^^^' + '\r\n' +
                                '1100^PNEUMOCOCCAL CONJUGATE PCV 7^^100^^0^^^PCV7^Y^Pneumococcal conjugate vaccine, 7 valent^^^PREVNAR 7^' + '\r\n' +
                                '1101^TYPHOID, VICPS^^101^^0^^^^Y^Typhoid Vi capsular polysaccharide vaccine^^29~TYPHOID VIS   05-29-12   CURRENT   1^TYPHIM VI^' + '\r\n' +
                                '1102^DTP-HIB-HEP B^^102^^1^^^^Y^DTP- Haemophilus influenzae type b conjugate and hepatitis b vaccine^^^^' + '\r\n' +
                                '1103^MENINGOCOCCAL C CONJUGATE^^103^^1^^^^Y^Meningococcal C conjugate vaccine^^^^' + '\r\n' +
                                '1105^VACCINIA (SMALLPOX) DILUTED^^105^^1^^^^Y^Vaccinia (smallpox) vaccine, diluted^^^^' + '\r\n' +
                                '1106^DTAP, 5 PERTUSSIS ANTIGENS^^106^^0^^^DTaP^Y^Diphtheria, tetanus toxoids and acellular pertussis vaccine, 5 pertussis antigens^^3~DIPHTHERIA/TETANUS/PERTUSSIS (DTAP) VIS   05-17-07   CURRENT   1^DAPTACEL^' + '\r\n' +
                                '1110^DTAP-HEP B-IPV^^110^^0^^^DTaP-HepB-IPV^Y^DTaP-hepatitis B and poliovirus vaccine^^3~DIPHTHERIA/TETANUS/PERTUSSIS (DTAP) VIS   05-17-07   CURRENT   1|5~HEPATITIS B VIS   02-02-12   CURRENT   1|23~POLIO VIS   11-08-11   CURRENT   1^PEDIARIX^' + '\r\n' +
                                '1113^TD (ADULT) PRESERVATIVE FREE^^113^^0^^^Td^Y^Tetanus and diphtheria toxoids, adsorbed, preservative free, for adult use^^34~TETANUS/DIPHTHERIA (TD) VIS   02-04-14   CURRENT   1^DECAVAC|TENIVAC^' + '\r\n' +
                                '1114^MENINGOCOCCAL MCV4P^^114^^0^^^MCV4P^Y^Meningococcal polysaccharide (groups A, C, Y and W-135) diphtheria toxoid conjugate vaccine (MCV4P)^^19~MENINGOCOCCAL VIS   10-14-11   CURRENT   1^MENACTRA^' + '\r\n' +
                                '1115^TDAP^^115^^0^^^Tdap^Y^Tetanus toxoid, reduced diphtheria toxoid, and acellular pertussis vaccine, adsorbed^^33~TETANUS/DIPHTHERIA/PERTUSSIS (TDAP) VIS   05-09-13   CURRENT   1^ADACEL|BOOSTRIX^' + '\r\n' +
                                '1116^ROTAVIRUS, PENTAVALENT^^116^^0^^^RV5^Y^Rotavirus, live, pentavalent vaccine^^25~ROTAVIRUS VIS   12-06-10   HISTORIC   1|26~ROTAVIRUS VIS   08-26-13   CURRENT   1^ROTATEQ^' + '\r\n' +
                                '1118^HPV, BIVALENT^^118^^0^^^HPV2^Y^Human papilloma virus vaccine, bivalent^^8~HUMAN PAPILLOMAVIRUS VACCINE (CERVARIX) VIS   05-03-11   CURRENT   1^CERVARIX^' + '\r\n' +
                                '1119^ROTAVIRUS, MONOVALENT^^119^^0^^^RV1^Y^Rotavirus, live, monovalent vaccine^^25~ROTAVIRUS VIS   12-06-10   HISTORIC   1|26~ROTAVIRUS VIS   08-26-13   CURRENT   1^ROTARIX^' + '\r\n' +
                                '1120^DTAP-HIB-IPV^^120^^0^^^DTaP-IPV/Hib^Y^Diphtheria, tetanus toxoids and acellular pertussis vaccine, Haemophilus influenzae type b conjugate, and poliovirus vaccine, inactivated (DTaP-Hib-IPV)^^3~DIPHTHERIA/TETANUS/PERTUSSIS (DTAP) VIS   05-17-07   CURRENT   1|6~HAEMOPHILUS INFLUENZAE TYPE B VIS   12-16-98   HISTORIC   1|7~HAEMOPHILUS INFLUENZAE TYPE B VIS   02-04-14   CURRENT   1|23~POLIO VIS   11-08-11   CURRENT   1^PENTACEL^' + '\r\n' +
                                '1121^ZOSTER (HISTORICAL)^^121^^1^^^HZV^N^Zoster vaccine, live^^27~SHINGLES VIS   10-06-09   CURRENT   1^ZOSTAVAX^' + '\r\n' +
                                '1123^INFLUENZA, H5N1-1203^^123^^1^^^^Y^Influenza virus vaccine, H5N1, A/Vietnam/1203/2004 (national stockpile)^^^^' + '\r\n' +
                                '1125^NOVEL INFLUENZA-H1N1-09, NASAL^^125^^1^^^^Y^Novel Influenza-H1N1-09, live virus for nasal administration^^^^' + '\r\n' +
                                '1126^NOVEL INFLUENZA-H1N1-09, PRESERVATIVE-FREE^^126^^1^^^^Y^Novel influenza-H1N1-09, preservative-free, injectable^^^^' + '\r\n' +
                                '1127^NOVEL INFLUENZA-H1N1-09^^127^^1^^^^Y^Novel influenza-H1N1-09, injectable^^^^' + '\r\n' +
                                '1129^JAPANESE ENCEPHALITIS, UNSPECIFIED FORMULATION^^129^^1^^^^Y^Japanese Encephalitis vaccine, unspecified formulation^^^^' + '\r\n' +
                                '1130^DTAP-IPV^^130^^0^^^DTaP-IPV^Y^Diphtheria, tetanus toxoids and acellular pertussis vaccine, and poliovirus vaccine, inactivated^^3~DIPHTHERIA/TETANUS/PERTUSSIS (DTAP) VIS   05-17-07   CURRENT   1|23~POLIO VIS   11-08-11   CURRENT   1^KINRIX^' + '\r\n' +
                                '1132^DTAP-IPV-HIB-HEP B, HISTORICAL^^132^^1^^^DTaP-HepB-IPV^Y^Historical record of vaccine containing: * diphtheria, tetanus toxoids and acellular pertussis, * poliovirus, inactivated, * Haemophilus influenzae type b conjugate, * Hepatitis B (DTaP-Hib-IPV)^^^^' + '\r\n' +
                                '1134^JAPANESE ENCEPHALITIS IM^^134^^0^^^^Y^Japanese Encephalitis vaccine for intramuscular administration^^15~JAPANESE ENCEPHALITIS VIS   12-07-11   HISTORIC   1|16~JAPANESE ENCEPHALITIS VIS   01-24-14   CURRENT   1^IXIARO^' + '\r\n' +
                                '1136^MENINGOCOCCAL MCV4O^^136^^0^^^MCV4O^Y^Meningococcal oligosaccharide (groups A, C, Y and W-135) diphtheria toxoid conjugate vaccine (MCV4O)^^19~MENINGOCOCCAL VIS   10-14-11   CURRENT   1^MENVEO^' + '\r\n' +
                                '1137^HPV, UNSPECIFIED FORMULATION^^137^^1^^^^Y^HPV, unspecified formulation^^^^' + '\r\n' +
                                '1138^TD (ADULT)^^138^^0^^^^Y^Tetanus and diphtheria toxoids, not adsorbed, for adult use^^34~TETANUS/DIPHTHERIA (TD) VIS   02-04-14   CURRENT   1^^' + '\r\n' +
                                '1140^INFLUENZA, SEASONAL, INJECTABLE, PRESERVATIVE FREE^^140^^0^^^IIV3^Y^Influenza, seasonal, injectable, preservative free^^13~INFLUENZA VACCINE - INACTIVATED VIS   07-02-12   HISTORIC   1|14~INFLUENZA VACCINE - INACTIVATED VIS   07-26-13   HISTORIC   1|36~INFLUENZA VACCINE - INACTIVATED VIS   08-19-14   CURRENT   1^AFLURIA, PRESERVATIVE FREE|AGRIFLU|FLUARIX|FLUVIRIN-PRESERVATIVE FREE|FLUZONE-PRESERVATIVE FREE^' + '\r\n' +
                                '1141^INFLUENZA, SEASONAL, INJECTABLE^^141^^0^^^IIV3^Y^Influenza, seasonal, injectable^^13~INFLUENZA VACCINE - INACTIVATED VIS   07-02-12   HISTORIC   1|14~INFLUENZA VACCINE - INACTIVATED VIS   07-26-13   HISTORIC   1|36~INFLUENZA VACCINE - INACTIVATED VIS   08-19-14   CURRENT   1^AFLURIA|FLUAVAL|FLUVIRIN|FLUZONE^' + '\r\n' +
                                '1142^TETANUS TOXOID, NOT ADSORBED^^142^^0^^^^Y^Tetanus toxoid, not adsorbed^^34~TETANUS/DIPHTHERIA (TD) VIS   02-04-14   CURRENT   1^^' + '\r\n' +
                                '1143^ADENOVIRUS TYPES 4 AND 7^^143^^0^^^^Y^Adenovirus, type 4 and type 7, live, oral^^1~ADENOVIRUS VIS   07-14-11   HISTORIC   1|35~ADENOVIRUS VIS   06-11-14   CURRENT   1^ADENOVIRUS TYPES 4 AND 7^' + '\r\n' +
                                '1144^INFLUENZA, SEASONAL, INTRADERMAL, PRESERVATIVE FREE^^144^^0^^^IIV3^Y^Seasonal influenza, intradermal, preservative free^^13~INFLUENZA VACCINE - INACTIVATED VIS   07-02-12   HISTORIC   1|14~INFLUENZA VACCINE - INACTIVATED VIS   07-26-13   HISTORIC   1|36~INFLUENZA VACCINE - INACTIVATED VIS   08-19-14   CURRENT   1^FLUZONE, INTRADERMAL^' + '\r\n' +
                                '1147^MENINGOCOCCAL MCV4, UNSPECIFIED FORMULATION^^147^^1^^^^Y^Meningococcal, MCV4, unspecified formulation(groups A, C, Y and W-135)^^^^' + '\r\n' +
                                '1148^MENINGOCOCCAL C/Y-HIB PRP^^148^^0^^^Hib-MenCY^Y^Meningococcal Groups C and Y and Haemophilus b Tetanus Toxoid Conjugate Vaccine^^6~HAEMOPHILUS INFLUENZAE TYPE B VIS   12-16-98   HISTORIC   1|7~HAEMOPHILUS INFLUENZAE TYPE B VIS   02-04-14   CURRENT   1^MENHIBRIX^' + '\r\n' +
                                '1149^INFLUENZA, LIVE, INTRANASAL, QUADRIVALENT^^149^^0^^^LAIV4^Y^Influenza, live, intranasal, quadrivalent^^11~INFLUENZA VACCINE - LIVE, INTRANASAL VIS   07-02-12   HISTORIC   1|12~INFLUENZA VACCINE - LIVE, INTRANASAL VIS   07-26-13   HISTORIC   1|37~INFLUENZA VACCINE - LIVE, INTRANASAL VIS   08-19-14   CURRENT   1^FLUMIST QUADRIVALENT^' + '\r\n' +
                                '1150^INFLUENZA, INJECTABLE, QUADRIVALENT, PRESERVATIVE FREE^^150^^0^^^IIV4^Y^Influenza, injectable, quadrivalent, preservative free^^13~INFLUENZA VACCINE - INACTIVATED VIS   07-02-12   HISTORIC   1|14~INFLUENZA VACCINE - INACTIVATED VIS   07-26-13   HISTORIC   1|36~INFLUENZA VACCINE - INACTIVATED VIS   08-19-14   CURRENT   1^FLUARIX, QUADRIVALENT|FLUZONE, QUADRIVALENT^' + '\r\n' +
                                '1151^INFLUENZA NASAL, UNSPECIFIED FORMULATION^^151^^1^^^^Y^Influenza nasal, unspecified formulation^^^^' + '\r\n' +
                                '1152^PNEUMOCOCCAL CONJUGATE, UNSPECIFIED FORMULATION^^152^^1^^^^Y^Pneumococcal Conjugate, unspecified formulation^^^^' + '\r\n' +
                                '1153^INFLUENZA, INJECTABLE, MDCK, PRESERVATIVE FREE^^153^^0^^^ccIIV3^Y^Influenza, injectable, Madin Darby Canine Kidney, preservative free^^13~INFLUENZA VACCINE - INACTIVATED VIS   07-02-12   HISTORIC   1|14~INFLUENZA VACCINE - INACTIVATED VIS   07-26-13   HISTORIC   1|36~INFLUENZA VACCINE - INACTIVATED VIS   08-19-14   CURRENT   1^FLUCELVAX^' + '\r\n' +
                                '1155^INFLUENZA, RECOMBINANT, INJECTABLE, PRESERVATIVE FREE^^155^^0^^^RIV3^Y^Seasonal, trivalent, recombinant, injectable influenza vaccine, preservative free^^13~INFLUENZA VACCINE - INACTIVATED VIS   07-02-12   HISTORIC   1|14~INFLUENZA VACCINE - INACTIVATED VIS   07-26-13   HISTORIC   1|36~INFLUENZA VACCINE - INACTIVATED VIS   08-19-14   CURRENT   1^FLUBLOK^' + '\r\n' +
                                '1158^INFLUENZA, INJECTABLE, QUADRIVALENT^^158^^0^^^IIV4^Y^Influenza, injectable, quadrivalent, contains preservative^^13~INFLUENZA VACCINE - INACTIVATED VIS   07-02-12   HISTORIC   1|14~INFLUENZA VACCINE - INACTIVATED VIS   07-26-13   HISTORIC   1|36~INFLUENZA VACCINE - INACTIVATED VIS   08-19-14   CURRENT   1^FLULAVAL QUADRIVALENT^' + '\r\n' +
                                '1160^INFLUENZA A MONOVALENT (H5N1), ADJUVANTED-2013^^160^^0^^^^Y^Influenza A monovalent (H5N1), adjuvanted, National stockpile 2013^^^INFLUENZA A (H5N1) -2013^' + '\r\n' +
                                '1801^AS03 ADJUVANT^^801^^0^^^^Y^AS03 Adjuvant^^^^' + '\r\n' +
                                '500001^VOODOO SHOT (HISTORICAL)^V D SHOT^^^1^^^^N^^^^^' + '\r\n' +
                                '500002^VOODOO SHOT1 (HISTORICAL)^V D SHOT1^^^1^^^^N^^^^^' + '\r\n' +
                                '500003^HEPATITIS B (NEWBORN TO 12) (HISTORICAL)^HEPB NBORN^^^1^^^^N^^^^^' + '\r\n' +
                                '500004^SBY-INFLUENZA (HISTORICAL)^INFLUENZA^^^1^^^^N^^^^^' + '\r\n' +
                                '500005^FLU SHOT 2000 (HISTORICAL)^FLU2000^^^1^^^^N^^^^^' + '\r\n' +
                                '500006^MIKES INFLUENZA (HISTORICAL)^INFLUENZA^^^1^^^^N^^^^^' + '\r\n' +
                                '500007^PNEUMOCOCCAL POLYSACCHARIDE PPV23^PNEUMOVAX^33^^0^^^PPSV23^Y^Pneumococcal polysaccharide vaccine, 23 valent^^22~PNEUMOCOCCAL POLYSACCHARIDE VIS   10-06-09   CURRENT   1^PNEUMOVAX 23^' + '\r\n' +
                                '612006^NOVEL INFLUENZA-H1N1-09, ALL FORMULATIONS^FLU H1N1^128^^1^^^^Y^Novel influenza-H1N1-09, all formulations^^^^' + '\r\n' +
                                '612013^INFLUENZA, HIGH DOSE SEASONAL^FLU,HI DOS^135^^0^^FLU^IIV3^Y^Influenza, high dose seasonal, preservative-free^^13~INFLUENZA VACCINE - INACTIVATED VIS   07-02-12   HISTORIC   1|14~INFLUENZA VACCINE - INACTIVATED VIS   07-26-13   HISTORIC   1|36~INFLUENZA VACCINE - INACTIVATED VIS   08-19-14   CURRENT   1^FLUZONE-HIGH DOSE^' + '\r\n');

        /*expect(result).to.eql([
            {
                "ien": "1",
                "name": "VACCINIA (SMALLPOX)",
                "shortName": "SMALLPOX",
                "cvxCode": "75",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Vaccinia (smallpox) vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "ACAM2000|DRYVAX",
                "effectiveDate": ""
        },
            {
                "ien": "2",
                "name": "TD(ADULT) UNSPECIFIED FORMULATION",
                "shortName": "TD-ADULT",
                "cvxCode": "139",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Td(adult) unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "3",
                "name": "DTP",
                "shortName": "DPT",
                "cvxCode": "01",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Diphtheria, tetanus toxoids and pertussis vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "4",
                "name": "TETANUS TOXOID, UNSPECIFIED FORMULATION",
                "shortName": "TET TOX",
                "cvxCode": "112",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "TT",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Tetanus toxoid, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "5",
                "name": "TYPHOID, UNSPECIFIED FORMULATION",
                "shortName": "TYPHOID",
                "cvxCode": "91",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Typhoid vaccine, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "6",
                "name": "OPV",
                "shortName": "SABIN/OPV",
                "cvxCode": "02",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Poliovirus vaccine, live, oral",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "ORIMUNE",
                "effectiveDate": ""
        },
            {
                "ien": "7",
                "name": "POLIO, UNSPECIFIED FORMULATION",
                "shortName": "SALK",
                "cvxCode": "89",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Poliovirus vaccine, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "8",
                "name": "SWINE FLU MONO (HISTORICAL)",
                "shortName": "SWINE MONO",
                "cvxCode": "",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "9",
                "name": "SWINE FLU BIVAL (HISTORICAL)",
                "shortName": "SWINE BIVA",
                "cvxCode": "",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "10",
                "name": "HEP B, ADULT",
                "shortName": "HEP B",
                "cvxCode": "43",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "HepB",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Hepatitis B vaccine, adult dosage",
                "codingSystem": "",
                "vaccineInfoStmt": "5~HEPATITIS B VIS   02-02-12   CURRENT   1",
                "synonym": "",
                "cdcProductName": "ENGERIX-B-ADULT|RECOMBIVAX-ADULT",
                "effectiveDate": ""
        },
            {
                "ien": "11",
                "name": "MEASLES",
                "shortName": "MEASLES",
                "cvxCode": "05",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Measles virus vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "ATTENUVAX",
                "effectiveDate": ""
        },
            {
                "ien": "12",
                "name": "INFLUENZA, UNSPECIFIED FORMULATION",
                "shortName": "INFLUENZA",
                "cvxCode": "88",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Influenza virus vaccine, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "13",
                "name": "CHOLERA",
                "shortName": "CHOLERA",
                "cvxCode": "26",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Cholera vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "14",
                "name": "RUBELLA",
                "shortName": "RUBELLA",
                "cvxCode": "06",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Rubella virus vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "MERUVAX II",
                "effectiveDate": ""
        },
            {
                "ien": "15",
                "name": "MUMPS",
                "shortName": "MUMPS",
                "cvxCode": "07",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Mumps virus vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "MUMPSVAX",
                "effectiveDate": ""
        },
            {
                "ien": "16",
                "name": "BCG (HISTORICAL)",
                "shortName": "BCG",
                "cvxCode": "19",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "17",
                "name": "MMR",
                "shortName": "MMR",
                "cvxCode": "03",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "MMR",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Measles, mumps and rubella virus vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "17~MEASLES/MUMPS/RUBELLA VIS   04-20-12   CURRENT   1",
                "synonym": "",
                "cdcProductName": "M-M-R II",
                "effectiveDate": ""
        },
            {
                "ien": "18",
                "name": "M/R",
                "shortName": "MR",
                "cvxCode": "04",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Measles and rubella virus vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "19",
                "name": "PNEUMOCOCCAL, UNSPECIFIED FORMULATION",
                "shortName": "PNEUMO-VAC",
                "cvxCode": "109",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Pneumococcal vaccine, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "20",
                "name": "YELLOW FEVER",
                "shortName": "YELLOW FEV",
                "cvxCode": "37",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Yellow fever vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "31~YELLOW FEVER VIS   03-30-11   CURRENT   1",
                "synonym": "",
                "cdcProductName": "YF-VAX",
                "effectiveDate": ""
        },
            {
                "ien": "21",
                "name": "TYPHUS, HISTORICAL",
                "shortName": "TYPHUS",
                "cvxCode": "131",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Historical record of a typhus vaccination",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "22",
                "name": "RABIES, UNSPECIFIED FORMULATION",
                "shortName": "RABIES",
                "cvxCode": "90",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Rabies vaccine, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "23",
                "name": "DT (PEDIATRIC)",
                "shortName": "DT-PEDS",
                "cvxCode": "28",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "DT",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Diphtheria and tetanus toxoids, adsorbed for pediatric use",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "DT(GENERIC)",
                "effectiveDate": ""
        },
            {
                "ien": "24",
                "name": "INFLUENZA B (HISTORICAL)",
                "shortName": "HIB",
                "cvxCode": "",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "25",
                "name": "HEP A, UNSPECIFIED FORMULATION",
                "shortName": "HEP A",
                "cvxCode": "85",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "HepA",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Hepatitis A vaccine, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "26",
                "name": "MENINGOCOCCAL, UNSPECIFIED FORMULATION",
                "shortName": "MENING.",
                "cvxCode": "108",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Meningococcal vaccine, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "27",
                "name": "ENCEPHALITIS (HISTORICAL)",
                "shortName": "ENCEPH.",
                "cvxCode": "",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "28",
                "name": "ZOSTER",
                "shortName": "VARCELLA",
                "cvxCode": "121",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "HZV",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Zoster vaccine, live",
                "codingSystem": "",
                "vaccineInfoStmt": "30~VARICELLA (CHICKENPOX) VIS   03-13-08   CURRENT   1|27~SHINGLES VIS   10-06-09   CURRENT   1",
                "synonym": "",
                "cdcProductName": "VARIVAX|ZOSTAVAX",
                "effectiveDate": ""
        },
            {
                "ien": "29",
                "name": "DTAP, UNSPECIFIED FORMULATION",
                "shortName": "DTaP",
                "cvxCode": "107",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "DTaP",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Diphtheria, tetanus toxoids and acellular pertussis vaccine, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "30",
                "name": "RUBELLA/MUMPS",
                "shortName": "RUB-MUMPS",
                "cvxCode": "38",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Rubella and mumps virus vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "BIAVAX II",
                "effectiveDate": ""
        },
            {
                "ien": "31",
                "name": "DIPHTHERIA (HISTORICAL)",
                "shortName": "DIPHTHERIA",
                "cvxCode": "",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "32",
                "name": "DTB/HIB (HISTORICAL)",
                "shortName": "DTB/HIB",
                "cvxCode": "",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "33",
                "name": "DTP, POLIO (HISTORICAL)",
                "shortName": "DTP POLIO",
                "cvxCode": "",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "34",
                "name": "MMRV",
                "shortName": "MMR&V",
                "cvxCode": "94",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "MMRV",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Measles, mumps, rubella, and varicella virus vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "18~MEASLES/MUMPS/RUBELLA/VARICELLA VIS   05-21-10   CURRENT   1",
                "synonym": "",
                "cdcProductName": "PROQUAD",
                "effectiveDate": ""
        },
            {
                "ien": "35",
                "name": "PLAGUE",
                "shortName": "PLAGUE",
                "cvxCode": "23",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Plague vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "36",
                "name": "GAMMA GLOBULIN (HISTORICAL)",
                "shortName": "ISG",
                "cvxCode": "",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "37",
                "name": "UNLISTED (HISTORICAL)",
                "shortName": "UNLISTED",
                "cvxCode": "",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "38",
                "name": "SHERI (HISTORICAL)",
                "shortName": "SLB",
                "cvxCode": "",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "SLB",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "39",
                "name": "ADENOVIRUS, TYPE 4",
                "shortName": "ADEN TYP4",
                "cvxCode": "54",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Adenovirus vaccine, type 4, live, oral",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "cdcProductName": "",
                "synonym": "",
                "effectiveDate": ""
        },
            {
                "ien": "40",
                "name": "ADENOVIRUS, TYPE 7",
                "shortName": "ADEN TYP7",
                "cvxCode": "55",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Adenovirus vaccine, type 7, live, oral",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "cdcProductName": "",
                "synonym": "",
                "effectiveDate": ""
        },
            {
                "ien": "41",
                "name": "ANTHRAX",
                "shortName": "ANT SC",
                "cvxCode": "24",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Anthrax vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "BIOTHRAX",
                "effectiveDate": ""
        },
            {
                "ien": "42",
                "name": "BCG",
                "shortName": "BCG P",
                "cvxCode": "19",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Bacillus Calmette-Guerin vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "MYCOBAX|TICE BCG",
                "effectiveDate": ""
        },
            {
                "ien": "43",
                "name": "BCG,INTRAVESICAL (HISTORICAL)",
                "shortName": "BCG I",
                "cvxCode": "",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "44",
                "name": "CHOLERA, ORAL (HISTORICAL)",
                "shortName": "CHOL ORAL",
                "cvxCode": "26",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "45",
                "name": "HEP A, ADULT",
                "shortName": "HEPA AD",
                "cvxCode": "52",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "HepA",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Hepatitis A vaccine, adult dosage",
                "codingSystem": "",
                "vaccineInfoStmt": "4~HEPATITIS A VIS   10-25-11   CURRENT   1",
                "synonym": "",
                "cdcProductName": "HAVRIX-ADULT|VAQTA-ADULT",
                "effectiveDate": ""
        },
            {
                "ien": "46",
                "name": "HEP A, PED/ADOL, 2 DOSE",
                "shortName": "HEPA PED/ADOL-2",
                "cvxCode": "83",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "HepA",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Hepatitis A vaccine, pediatric/adolescent dosage, 2 dose schedule",
                "codingSystem": "",
                "vaccineInfoStmt": "4~HEPATITIS A VIS   10-25-11   CURRENT   1",
                "synonym": "",
                "cdcProductName": "HAVRIX-PEDS|VAQTA-PEDS",
                "effectiveDate": ""
        },
            {
                "ien": "47",
                "name": "HEP A, PED/ADOL, 3 DOSE",
                "shortName": "HEPA PED/ADOL-3",
                "cvxCode": "84",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "HepA",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Hepatitis A vaccine, pediatric/adolescent dosage, 3 dose schedule",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "48",
                "name": "HEP A-HEP B",
                "shortName": "HEPA/HEPB AD",
                "cvxCode": "104",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "HepA-HepB",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Hepatitis A and hepatitis B vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "4~HEPATITIS A VIS   10-25-11   CURRENT   1|5~HEPATITIS B VIS   02-02-12   CURRENT   1",
                "synonym": "",
                "cdcProductName": "TWINRIX",
                "effectiveDate": ""
        },
            {
                "ien": "49",
                "name": "HIB (HBOC)",
                "shortName": "HIB,HBOC",
                "cvxCode": "47",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Haemophilus influenzae type b vaccine, HbOC conjugate",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "HIBTITER",
                "effectiveDate": ""
        },
            {
                "ien": "50",
                "name": "HIB (PRP-D)",
                "shortName": "HIB PRP-D",
                "cvxCode": "46",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Haemophilus influenzae type b vaccine, PRP-D conjugate",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "PROHIBIT",
                "effectiveDate": ""
        },
            {
                "ien": "51",
                "name": "HIB (PRP-OMP)",
                "shortName": "HIB PRP-OMP",
                "cvxCode": "49",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "PRP-OMP",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Haemophilus influenzae type b vaccine, PRP-OMP conjugate",
                "codingSystem": "",
                "vaccineInfoStmt": "6~HAEMOPHILUS INFLUENZAE TYPE B VIS   12-16-98   HISTORIC   1|7~HAEMOPHILUS INFLUENZAE TYPE B VIS   02-04-14   CURRENT   1",
                "synonym": "",
                "cdcProductName": "PEDVAXHIB",
                "effectiveDate": ""
        },
            {
                "ien": "52",
                "name": "HIB (PRP-T)",
                "shortName": "HIB PRP-T",
                "cvxCode": "48",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "PRP-T",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Haemophilus influenzae type b vaccine, PRP-T conjugate",
                "codingSystem": "",
                "vaccineInfoStmt": "6~HAEMOPHILUS INFLUENZAE TYPE B VIS   12-16-98   HISTORIC   1|7~HAEMOPHILUS INFLUENZAE TYPE B VIS   02-04-14   CURRENT   1",
                "synonym": "",
                "cdcProductName": "ACTHIB|HIBERIX|OMNIHIB",
                "effectiveDate": ""
        },
            {
                "ien": "53",
                "name": "INFLUENZA, UNSPECIFIED FORMULATION (HISTORICAL)",
                "shortName": "INFLUENZA",
                "cvxCode": "88",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "Influenza virus vaccine, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "54",
                "name": "INFLUENZA, WHOLE",
                "shortName": "FLU WHOLE",
                "cvxCode": "16",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Influenza virus vaccine, whole virus",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "55",
                "name": "INFLUENZA, LIVE, INTRANASAL",
                "shortName": "FLU NAS",
                "cvxCode": "111",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "LAIV3",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Influenza virus vaccine, live, attenuated, for intranasal use",
                "codingSystem": "",
                "vaccineInfoStmt": "11~INFLUENZA VACCINE - LIVE, INTRANASAL VIS   07-02-12   HISTORIC   1|12~INFLUENZA VACCINE - LIVE, INTRANASAL VIS   07-26-13   HISTORIC   1|37~INFLUENZA VACCINE - LIVE, INTRANASAL VIS   08-19-14   CURRENT   1",
                "synonym": "",
                "cdcProductName": "FLUMIST",
                "effectiveDate": ""
        },
            {
                "ien": "56",
                "name": "LYME DISEASE",
                "shortName": "LYME",
                "cvxCode": "66",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Lyme disease vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "57",
                "name": "PNEUMOCOCCAL,PED (HISTORICAL)",
                "shortName": "PNEUMO-PED",
                "cvxCode": "109",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "58",
                "name": "RABIES, INTRAMUSCULAR INJECTION",
                "shortName": "RAB",
                "cvxCode": "18",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Rabies vaccine, for intramuscular injection",
                "codingSystem": "",
                "vaccineInfoStmt": "24~RABIES VIS   10-06-09   CURRENT   1",
                "synonym": "",
                "cdcProductName": "IMOVAX|RABAVERT",
                "effectiveDate": ""
        },
            {
                "ien": "59",
                "name": "RABIES, INTRADERMAL INJECTION",
                "shortName": "RAB ID",
                "cvxCode": "40",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Rabies vaccine, for intradermal injection",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "IMOVAX ID|RABAVERT",
                "effectiveDate": ""
        },
            {
                "ien": "60",
                "name": "ROTAVIRUS, UNSPECIFIED FORMULATION",
                "shortName": "ROTO ORAL",
                "cvxCode": "122",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Rotavirus vaccine, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "61",
                "name": "TYPHOID, ORAL",
                "shortName": "TYP ORAL",
                "cvxCode": "25",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Typhoid vaccine, live, oral",
                "codingSystem": "",
                "vaccineInfoStmt": "29~TYPHOID VIS   05-29-12   CURRENT   1",
                "synonym": "",
                "cdcProductName": "VIVOTIF BERNA",
                "effectiveDate": ""
        },
            {
                "ien": "62",
                "name": "TYPHOID, UNSPECIFIED FORMULATION (HISTORICAL)",
                "shortName": "TYP",
                "cvxCode": "91",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "Typhoid vaccine, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "63",
                "name": "TYPHOID, PARENTERAL",
                "shortName": "TYP H-P-SC/ID",
                "cvxCode": "41",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Typhoid vaccine, parenteral, other than acetone-killed, dried",
                "codingSystem": "",
                "vaccineInfoStmt": "29~TYPHOID VIS   05-29-12   CURRENT   1",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "64",
                "name": "TYPHOID, PARENTERAL, AKD (U.S. MILITARY)",
                "shortName": "TYP AKD-SC",
                "cvxCode": "53",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Typhoid vaccine, parenteral, acetone-killed, dried (U.S. military)",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "TYPHOID-AKD",
                "effectiveDate": ""
        },
            {
                "ien": "65",
                "name": "HEP B, UNSPECIFIED FORMULATION",
                "shortName": "HEPB ILL",
                "cvxCode": "45",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "HepB",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Hepatitis B vaccine, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "66",
                "name": "HIB-HEP B",
                "shortName": "HEPB/HIB",
                "cvxCode": "51",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "Hib-HepB",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Haemophilus influenzae type b conjugate and Hepatitis B vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "6~HAEMOPHILUS INFLUENZAE TYPE B VIS   12-16-98   HISTORIC   1|7~HAEMOPHILUS INFLUENZAE TYPE B VIS   02-04-14   CURRENT   1|5~HEPATITIS B VIS   02-02-12   CURRENT   1",
                "synonym": "",
                "cdcProductName": "COMVAX",
                "effectiveDate": ""
        },
            {
                "ien": "67",
                "name": "PNEUMOCOCCAL CONJUGATE PCV 13",
                "shortName": "PNEU PCV13",
                "cvxCode": "133",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "PCV13",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Pneumococcal conjugate vaccine, 13 valent",
                "codingSystem": "",
                "vaccineInfoStmt": "20~PNEUMOCOCCAL CONJUGATE (PCV13) VIS   04-16-10   HISTORIC   1|21~PNEUMOCOCCAL CONJUGATE (PCV13) VIS   02-27-13   CURRENT   1",
                "synonym": "",
                "cdcProductName": "PREVNAR 13",
                "effectiveDate": ""
        },
            {
                "ien": "1008",
                "name": "HEP B, ADOLESCENT OR PEDIATRIC",
                "shortName": "",
                "cvxCode": "08",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "HepB",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Hepatitis B vaccine, pediatric or pediatric/adolescent dosage",
                "codingSystem": "",
                "vaccineInfoStmt": "5~HEPATITIS B VIS   02-02-12   CURRENT   1",
                "synonym": "",
                "cdcProductName": "ENGERIX B-PEDS|RECOMBIVAX-PEDS",
                "effectiveDate": ""
        },
            {
                "ien": "1009",
                "name": "TD (ADULT), ADSORBED",
                "shortName": "",
                "cvxCode": "09",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "Td",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Tetanus and diphtheria toxoids, adsorbed, for adult use",
                "codingSystem": "",
                "vaccineInfoStmt": "34~TETANUS/DIPHTHERIA (TD) VIS   02-04-14   CURRENT   1",
                "synonym": "",
                "cdcProductName": "TD(GENERIC)|TD, (ADULT)",
                "effectiveDate": ""
        },
            {
                "ien": "1010",
                "name": "IPV",
                "shortName": "",
                "cvxCode": "10",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "IPV",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Poliovirus vaccine, inactivated",
                "codingSystem": "",
                "vaccineInfoStmt": "23~POLIO VIS   11-08-11   CURRENT   1",
                "synonym": "",
                "cdcProductName": "IPOL",
                "effectiveDate": ""
        },
            {
                "ien": "1011",
                "name": "PERTUSSIS",
                "shortName": "",
                "cvxCode": "11",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Pertussis vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1015",
                "name": "INFLUENZA, SPLIT (INCL. PURIFIED SURFACE ANTIGEN)",
                "shortName": "",
                "cvxCode": "15",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Influenza virus vaccine, split virus (incl. purified surface antigen)-retired CODE",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1017",
                "name": "HIB, UNSPECIFIED FORMULATION",
                "shortName": "",
                "cvxCode": "17",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Haemophilus influenzae type b vaccine, conjugate unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1020",
                "name": "DTAP",
                "shortName": "",
                "cvxCode": "20",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "DTaP",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Diphtheria, tetanus toxoids and acellular pertussis vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "3~DIPHTHERIA/TETANUS/PERTUSSIS (DTAP) VIS   05-17-07   CURRENT   1",
                "synonym": "",
                "cdcProductName": "ACEL-IMUNE|CERTIVA|INFANRIX|TRIPEDIA",
                "effectiveDate": ""
        },
            {
                "ien": "1022",
                "name": "DTP-HIB",
                "shortName": "",
                "cvxCode": "22",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "DTP-Haemophilus influenzae type b conjugate vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "TETRAMUNE",
                "effectiveDate": ""
        },
            {
                "ien": "1031",
                "name": "HEP A, PEDIATRIC, UNSPECIFIED FORMULATION",
                "shortName": "",
                "cvxCode": "31",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "HepA",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Hepatitis A vaccine, pediatric dosage, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1032",
                "name": "MENINGOCOCCAL MPSV4",
                "shortName": "",
                "cvxCode": "32",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "MPSV4",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Meningococcal polysaccharide vaccine (MPSV4)",
                "codingSystem": "",
                "vaccineInfoStmt": "19~MENINGOCOCCAL VIS   10-14-11   CURRENT   1",
                "synonym": "",
                "cdcProductName": "MENOMUNE",
                "effectiveDate": ""
        },
            {
                "ien": "1035",
                "name": "TETANUS TOXOID, ADSORBED",
                "shortName": "",
                "cvxCode": "35",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Tetanus toxoid, adsorbed",
                "codingSystem": "",
                "vaccineInfoStmt": "34~TETANUS/DIPHTHERIA (TD) VIS   02-04-14   CURRENT   1",
                "synonym": "",
                "cdcProductName": "TETANUS TOXOID (GENERIC)",
                "effectiveDate": ""
        },
            {
                "ien": "1039",
                "name": "JAPANESE ENCEPHALITIS SC",
                "shortName": "",
                "cvxCode": "39",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Japanese Encephalitis Vaccine SC",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "JE-VAX",
                "effectiveDate": ""
        },
            {
                "ien": "1042",
                "name": "HEP B, ADOLESCENT/HIGH RISK INFANT",
                "shortName": "",
                "cvxCode": "42",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "HepB",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Hepatitis B vaccine, adolescent/high risk infant dosage",
                "codingSystem": "",
                "vaccineInfoStmt": "5~HEPATITIS B VIS   02-02-12   CURRENT   1",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1044",
                "name": "HEP B, DIALYSIS",
                "shortName": "",
                "cvxCode": "44",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "HepB",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Hepatitis B vaccine, dialysis patient dosage",
                "codingSystem": "",
                "vaccineInfoStmt": "5~HEPATITIS B VIS   02-02-12   CURRENT   1",
                "synonym": "",
                "cdcProductName": "RECOMBIVAX-DIALYSIS",
                "effectiveDate": ""
        },
            {
                "ien": "1050",
                "name": "DTAP-HIB",
                "shortName": "",
                "cvxCode": "50",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "DTaP-Haemophilus influenzae type b conjugate vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "TRIHIBIT",
                "effectiveDate": ""
        },
            {
                "ien": "1062",
                "name": "HPV, QUADRIVALENT",
                "shortName": "",
                "cvxCode": "62",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "HPV4",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Human papilloma virus vaccine, quadrivalent",
                "codingSystem": "",
                "vaccineInfoStmt": "9~HUMAN PAPILLOMAVIRUS VACCINE (GARDASIL) VIS   02-22-12   HISTORIC   1|10~HUMAN PAPILLOMAVIRUS VACCINE (GARDASIL) VIS   05-17-13   CURRENT   1",
                "synonym": "",
                "cdcProductName": "GARDASIL",
                "effectiveDate": ""
        },
            {
                "ien": "1069",
                "name": "PARAINFLUENZA-3",
                "shortName": "",
                "cvxCode": "69",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Parainfluenza-3 virus vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1074",
                "name": "ROTAVIRUS, TETRAVALENT",
                "shortName": "",
                "cvxCode": "74",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Rotavirus, live, tetravalent vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1076",
                "name": "STAPHYLOCOCCUS BACTERIO LYSATE",
                "shortName": "",
                "cvxCode": "76",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Staphylococcus bacteriophage lysate",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1077",
                "name": "TICK-BORNE ENCEPHALITIS",
                "shortName": "",
                "cvxCode": "77",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Tick-borne encephalitis vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1078",
                "name": "TULAREMIA VACCINE",
                "shortName": "",
                "cvxCode": "78",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Tularemia vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1080",
                "name": "VEE, LIVE",
                "shortName": "",
                "cvxCode": "80",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Venezuelan equine encephalitis, live, attenuated",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1081",
                "name": "VEE, INACTIVATED",
                "shortName": "",
                "cvxCode": "81",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Venezuelan equine encephalitis, inactivated",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1082",
                "name": "ADENOVIRUS, UNSPECIFIED FORMULATION",
                "shortName": "",
                "cvxCode": "82",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Adenovirus vaccine, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1092",
                "name": "VEE, UNSPECIFIED FORMULATION",
                "shortName": "",
                "cvxCode": "92",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Venezuelan equine encephalitis vaccine, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1100",
                "name": "PNEUMOCOCCAL CONJUGATE PCV 7",
                "shortName": "",
                "cvxCode": "100",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "PCV7",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Pneumococcal conjugate vaccine, 7 valent",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "PREVNAR 7",
                "effectiveDate": ""
        },
            {
                "ien": "1101",
                "name": "TYPHOID, VICPS",
                "shortName": "",
                "cvxCode": "101",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Typhoid Vi capsular polysaccharide vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "29~TYPHOID VIS   05-29-12   CURRENT   1",
                "synonym": "",
                "cdcProductName": "TYPHIM VI",
                "effectiveDate": ""
        },
            {
                "ien": "1102",
                "name": "DTP-HIB-HEP B",
                "shortName": "",
                "cvxCode": "102",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "DTP- Haemophilus influenzae type b conjugate and hepatitis b vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1103",
                "name": "MENINGOCOCCAL C CONJUGATE",
                "shortName": "",
                "cvxCode": "103",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Meningococcal C conjugate vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1105",
                "name": "VACCINIA (SMALLPOX) DILUTED",
                "shortName": "",
                "cvxCode": "105",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Vaccinia (smallpox) vaccine, diluted",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1106",
                "name": "DTAP, 5 PERTUSSIS ANTIGENS",
                "shortName": "",
                "cvxCode": "106",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "DTaP",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Diphtheria, tetanus toxoids and acellular pertussis vaccine, 5 pertussis antigens",
                "codingSystem": "",
                "vaccineInfoStmt": "3~DIPHTHERIA/TETANUS/PERTUSSIS (DTAP) VIS   05-17-07   CURRENT   1",
                "synonym": "",
                "cdcProductName": "DAPTACEL",
                "effectiveDate": ""
        },
            {
                "ien": "1110",
                "name": "DTAP-HEP B-IPV",
                "shortName": "",
                "cvxCode": "110",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "DTaP-HepB-IPV",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "DTaP-hepatitis B and poliovirus vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "3~DIPHTHERIA/TETANUS/PERTUSSIS (DTAP) VIS   05-17-07   CURRENT   1|5~HEPATITIS B VIS   02-02-12   CURRENT   1|23~POLIO VIS   11-08-11   CURRENT   1",
                "synonym": "",
                "cdcProductName": "PEDIARIX",
                "effectiveDate": ""
        },
            {
                "ien": "1113",
                "name": "TD (ADULT) PRESERVATIVE FREE",
                "shortName": "",
                "cvxCode": "113",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "Td",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Tetanus and diphtheria toxoids, adsorbed, preservative free, for adult use",
                "codingSystem": "",
                "vaccineInfoStmt": "34~TETANUS/DIPHTHERIA (TD) VIS   02-04-14   CURRENT   1",
                "synonym": "",
                "cdcProductName": "DECAVAC|TENIVAC",
                "effectiveDate": ""
        },
            {
                "ien": "1114",
                "name": "MENINGOCOCCAL MCV4P",
                "shortName": "",
                "cvxCode": "114",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "MCV4P",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Meningococcal polysaccharide (groups A, C, Y and W-135) diphtheria toxoid conjugate vaccine (MCV4P)",
                "codingSystem": "",
                "vaccineInfoStmt": "19~MENINGOCOCCAL VIS   10-14-11   CURRENT   1",
                "synonym": "",
                "cdcProductName": "MENACTRA",
                "effectiveDate": ""
        },
            {
                "ien": "1115",
                "name": "TDAP",
                "shortName": "",
                "cvxCode": "115",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "Tdap",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Tetanus toxoid, reduced diphtheria toxoid, and acellular pertussis vaccine, adsorbed",
                "codingSystem": "",
                "vaccineInfoStmt": "33~TETANUS/DIPHTHERIA/PERTUSSIS (TDAP) VIS   05-09-13   CURRENT   1",
                "synonym": "",
                "cdcProductName": "ADACEL|BOOSTRIX",
                "effectiveDate": ""
        },
            {
                "ien": "1116",
                "name": "ROTAVIRUS, PENTAVALENT",
                "shortName": "",
                "cvxCode": "116",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "RV5",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Rotavirus, live, pentavalent vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "25~ROTAVIRUS VIS   12-06-10   HISTORIC   1|26~ROTAVIRUS VIS   08-26-13   CURRENT   1",
                "synonym": "",
                "cdcProductName": "ROTATEQ",
                "effectiveDate": ""
        },
            {
                "ien": "1118",
                "name": "HPV, BIVALENT",
                "shortName": "",
                "cvxCode": "118",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "HPV2",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Human papilloma virus vaccine, bivalent",
                "codingSystem": "",
                "vaccineInfoStmt": "8~HUMAN PAPILLOMAVIRUS VACCINE (CERVARIX) VIS   05-03-11   CURRENT   1",
                "synonym": "",
                "cdcProductName": "CERVARIX",
                "effectiveDate": ""
        },
            {
                "ien": "1119",
                "name": "ROTAVIRUS, MONOVALENT",
                "shortName": "",
                "cvxCode": "119",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "RV1",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Rotavirus, live, monovalent vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "25~ROTAVIRUS VIS   12-06-10   HISTORIC   1|26~ROTAVIRUS VIS   08-26-13   CURRENT   1",
                "synonym": "",
                "cdcProductName": "ROTARIX",
                "effectiveDate": ""
        },
            {
                "ien": "1120",
                "name": "DTAP-HIB-IPV",
                "shortName": "",
                "cvxCode": "120",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "DTaP-IPV/Hib",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Diphtheria, tetanus toxoids and acellular pertussis vaccine, Haemophilus influenzae type b conjugate, and poliovirus vaccine, inactivated (DTaP-Hib-IPV)",
                "codingSystem": "",
                "vaccineInfoStmt": "3~DIPHTHERIA/TETANUS/PERTUSSIS (DTAP) VIS   05-17-07   CURRENT   1|6~HAEMOPHILUS INFLUENZAE TYPE B VIS   12-16-98   HISTORIC   1|7~HAEMOPHILUS INFLUENZAE TYPE B VIS   02-04-14   CURRENT   1|23~POLIO VIS   11-08-11   CURRENT   1",
                "synonym": "",
                "cdcProductName": "PENTACEL",
                "effectiveDate": ""
        },
            {
                "ien": "1121",
                "name": "ZOSTER (HISTORICAL)",
                "shortName": "",
                "cvxCode": "121",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "HZV",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "Zoster vaccine, live",
                "codingSystem": "",
                "vaccineInfoStmt": "27~SHINGLES VIS   10-06-09   CURRENT   1",
                "synonym": "",
                "cdcProductName": "ZOSTAVAX",
                "effectiveDate": ""
        },
            {
                "ien": "1123",
                "name": "INFLUENZA, H5N1-1203",
                "shortName": "",
                "cvxCode": "123",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Influenza virus vaccine, H5N1, A/Vietnam/1203/2004 (national stockpile)",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1125",
                "name": "NOVEL INFLUENZA-H1N1-09, NASAL",
                "shortName": "",
                "cvxCode": "125",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Novel Influenza-H1N1-09, live virus for nasal administration",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1126",
                "name": "NOVEL INFLUENZA-H1N1-09, PRESERVATIVE-FREE",
                "shortName": "",
                "cvxCode": "126",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Novel influenza-H1N1-09, preservative-free, injectable",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1127",
                "name": "NOVEL INFLUENZA-H1N1-09",
                "shortName": "",
                "cvxCode": "127",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Novel influenza-H1N1-09, injectable",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1129",
                "name": "JAPANESE ENCEPHALITIS, UNSPECIFIED FORMULATION",
                "shortName": "",
                "cvxCode": "129",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Japanese Encephalitis vaccine, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1130",
                "name": "DTAP-IPV",
                "shortName": "",
                "cvxCode": "130",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "DTaP-IPV",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Diphtheria, tetanus toxoids and acellular pertussis vaccine, and poliovirus vaccine, inactivated",
                "codingSystem": "",
                "vaccineInfoStmt": "3~DIPHTHERIA/TETANUS/PERTUSSIS (DTAP) VIS   05-17-07   CURRENT   1|23~POLIO VIS   11-08-11   CURRENT   1",
                "synonym": "",
                "cdcProductName": "KINRIX",
                "effectiveDate": ""
        },
            {
                "ien": "1132",
                "name": "DTAP-IPV-HIB-HEP B, HISTORICAL",
                "shortName": "",
                "cvxCode": "132",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "DTaP-HepB-IPV",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Historical record of vaccine containing: * diphtheria, tetanus toxoids and acellular pertussis, * poliovirus, inactivated, * Haemophilus influenzae type b conjugate, * Hepatitis B (DTaP-Hib-IPV)",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1134",
                "name": "JAPANESE ENCEPHALITIS IM",
                "shortName": "",
                "cvxCode": "134",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Japanese Encephalitis vaccine for intramuscular administration",
                "codingSystem": "",
                "vaccineInfoStmt": "15~JAPANESE ENCEPHALITIS VIS   12-07-11   HISTORIC   1|16~JAPANESE ENCEPHALITIS VIS   01-24-14   CURRENT   1",
                "synonym": "",
                "cdcProductName": "IXIARO",
                "effectiveDate": ""
        },
            {
                "ien": "1136",
                "name": "MENINGOCOCCAL MCV4O",
                "shortName": "",
                "cvxCode": "136",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "MCV4O",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Meningococcal oligosaccharide (groups A, C, Y and W-135) diphtheria toxoid conjugate vaccine (MCV4O)",
                "codingSystem": "",
                "vaccineInfoStmt": "19~MENINGOCOCCAL VIS   10-14-11   CURRENT   1",
                "synonym": "",
                "cdcProductName": "MENVEO",
                "effectiveDate": ""
        },
            {
                "ien": "1137",
                "name": "HPV, UNSPECIFIED FORMULATION",
                "shortName": "",
                "cvxCode": "137",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "HPV, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1138",
                "name": "TD (ADULT)",
                "shortName": "",
                "cvxCode": "138",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Tetanus and diphtheria toxoids, not adsorbed, for adult use",
                "codingSystem": "",
                "vaccineInfoStmt": "34~TETANUS/DIPHTHERIA (TD) VIS   02-04-14   CURRENT   1",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1140",
                "name": "INFLUENZA, SEASONAL, INJECTABLE, PRESERVATIVE FREE",
                "shortName": "",
                "cvxCode": "140",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "IIV3",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Influenza, seasonal, injectable, preservative free",
                "codingSystem": "",
                "vaccineInfoStmt": "13~INFLUENZA VACCINE - INACTIVATED VIS   07-02-12   HISTORIC   1|14~INFLUENZA VACCINE - INACTIVATED VIS   07-26-13   HISTORIC   1|36~INFLUENZA VACCINE - INACTIVATED VIS   08-19-14   CURRENT   1",
                "synonym": "",
                "cdcProductName": "AFLURIA, PRESERVATIVE FREE|AGRIFLU|FLUARIX|FLUVIRIN-PRESERVATIVE FREE|FLUZONE-PRESERVATIVE FREE",
                "effectiveDate": ""
        },
            {
                "ien": "1141",
                "name": "INFLUENZA, SEASONAL, INJECTABLE",
                "shortName": "",
                "cvxCode": "141",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "IIV3",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Influenza, seasonal, injectable",
                "codingSystem": "",
                "vaccineInfoStmt": "13~INFLUENZA VACCINE - INACTIVATED VIS   07-02-12   HISTORIC   1|14~INFLUENZA VACCINE - INACTIVATED VIS   07-26-13   HISTORIC   1|36~INFLUENZA VACCINE - INACTIVATED VIS   08-19-14   CURRENT   1",
                "synonym": "",
                "cdcProductName": "AFLURIA|FLUAVAL|FLUVIRIN|FLUZONE",
                "effectiveDate": ""
        },
            {
                "ien": "1142",
                "name": "TETANUS TOXOID, NOT ADSORBED",
                "shortName": "",
                "cvxCode": "142",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Tetanus toxoid, not adsorbed",
                "codingSystem": "",
                "vaccineInfoStmt": "34~TETANUS/DIPHTHERIA (TD) VIS   02-04-14   CURRENT   1",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1143",
                "name": "ADENOVIRUS TYPES 4 AND 7",
                "shortName": "",
                "cvxCode": "143",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Adenovirus, type 4 and type 7, live, oral",
                "codingSystem": "",
                "vaccineInfoStmt": "1~ADENOVIRUS VIS   07-14-11   HISTORIC   1|35~ADENOVIRUS VIS   06-11-14   CURRENT   1",
                "synonym": "",
                "cdcProductName": "ADENOVIRUS TYPES 4 AND 7",
                "effectiveDate": ""
        },
            {
                "ien": "1144",
                "name": "INFLUENZA, SEASONAL, INTRADERMAL, PRESERVATIVE FREE",
                "shortName": "",
                "cvxCode": "144",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "IIV3",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Seasonal influenza, intradermal, preservative free",
                "codingSystem": "",
                "vaccineInfoStmt": "13~INFLUENZA VACCINE - INACTIVATED VIS   07-02-12   HISTORIC   1|14~INFLUENZA VACCINE - INACTIVATED VIS   07-26-13   HISTORIC   1|36~INFLUENZA VACCINE - INACTIVATED VIS   08-19-14   CURRENT   1",
                "synonym": "",
                "cdcProductName": "FLUZONE, INTRADERMAL",
                "effectiveDate": ""
        },
            {
                "ien": "1147",
                "name": "MENINGOCOCCAL MCV4, UNSPECIFIED FORMULATION",
                "shortName": "",
                "cvxCode": "147",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Meningococcal, MCV4, unspecified formulation(groups A, C, Y and W-135)",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1148",
                "name": "MENINGOCOCCAL C/Y-HIB PRP",
                "shortName": "",
                "cvxCode": "148",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "Hib-MenCY",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Meningococcal Groups C and Y and Haemophilus b Tetanus Toxoid Conjugate Vaccine",
                "codingSystem": "",
                "vaccineInfoStmt": "6~HAEMOPHILUS INFLUENZAE TYPE B VIS   12-16-98   HISTORIC   1|7~HAEMOPHILUS INFLUENZAE TYPE B VIS   02-04-14   CURRENT   1",
                "synonym": "",
                "cdcProductName": "MENHIBRIX",
                "effectiveDate": ""
        },
            {
                "ien": "1149",
                "name": "INFLUENZA, LIVE, INTRANASAL, QUADRIVALENT",
                "shortName": "",
                "cvxCode": "149",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "LAIV4",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Influenza, live, intranasal, quadrivalent",
                "codingSystem": "",
                "vaccineInfoStmt": "11~INFLUENZA VACCINE - LIVE, INTRANASAL VIS   07-02-12   HISTORIC   1|12~INFLUENZA VACCINE - LIVE, INTRANASAL VIS   07-26-13   HISTORIC   1|37~INFLUENZA VACCINE - LIVE, INTRANASAL VIS   08-19-14   CURRENT   1",
                "synonym": "",
                "cdcProductName": "FLUMIST QUADRIVALENT",
                "effectiveDate": ""
        },
            {
                "ien": "1150",
                "name": "INFLUENZA, INJECTABLE, QUADRIVALENT, PRESERVATIVE FREE",
                "shortName": "",
                "cvxCode": "150",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "IIV4",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Influenza, injectable, quadrivalent, preservative free",
                "codingSystem": "",
                "vaccineInfoStmt": "13~INFLUENZA VACCINE - INACTIVATED VIS   07-02-12   HISTORIC   1|14~INFLUENZA VACCINE - INACTIVATED VIS   07-26-13   HISTORIC   1|36~INFLUENZA VACCINE - INACTIVATED VIS   08-19-14   CURRENT   1",
                "synonym": "",
                "cdcProductName": "FLUARIX, QUADRIVALENT|FLUZONE, QUADRIVALENT",
                "effectiveDate": ""
        },
            {
                "ien": "1151",
                "name": "INFLUENZA NASAL, UNSPECIFIED FORMULATION",
                "shortName": "",
                "cvxCode": "151",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Influenza nasal, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1152",
                "name": "PNEUMOCOCCAL CONJUGATE, UNSPECIFIED FORMULATION",
                "shortName": "",
                "cvxCode": "152",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Pneumococcal Conjugate, unspecified formulation",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "1153",
                "name": "INFLUENZA, INJECTABLE, MDCK, PRESERVATIVE FREE",
                "shortName": "",
                "cvxCode": "153",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "ccIIV3",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Influenza, injectable, Madin Darby Canine Kidney, preservative free",
                "codingSystem": "",
                "vaccineInfoStmt": "13~INFLUENZA VACCINE - INACTIVATED VIS   07-02-12   HISTORIC   1|14~INFLUENZA VACCINE - INACTIVATED VIS   07-26-13   HISTORIC   1|36~INFLUENZA VACCINE - INACTIVATED VIS   08-19-14   CURRENT   1",
                "cdcProductName": "FLUCELVAX",
                "synonym": "",
                "effectiveDate": ""
        },
            {
                "ien": "1155",
                "name": "INFLUENZA, RECOMBINANT, INJECTABLE, PRESERVATIVE FREE",
                "shortName": "",
                "cvxCode": "155",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "RIV3",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Seasonal, trivalent, recombinant, injectable influenza vaccine, preservative free",
                "codingSystem": "",
                "vaccineInfoStmt": "13~INFLUENZA VACCINE - INACTIVATED VIS   07-02-12   HISTORIC   1|14~INFLUENZA VACCINE - INACTIVATED VIS   07-26-13   HISTORIC   1|36~INFLUENZA VACCINE - INACTIVATED VIS   08-19-14   CURRENT   1",
                "cdcProductName": "FLUBLOK",
                "synonym": "",
                "effectiveDate": ""
        },
            {
                "ien": "1158",
                "name": "INFLUENZA, INJECTABLE, QUADRIVALENT",
                "shortName": "",
                "cvxCode": "158",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "IIV4",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Influenza, injectable, quadrivalent, contains preservative",
                "codingSystem": "",
                "vaccineInfoStmt": "13~INFLUENZA VACCINE - INACTIVATED VIS   07-02-12   HISTORIC   1|14~INFLUENZA VACCINE - INACTIVATED VIS   07-26-13   HISTORIC   1|36~INFLUENZA VACCINE - INACTIVATED VIS   08-19-14   CURRENT   1",
                "synonym": "",
                "cdcProductName": "FLULAVAL QUADRIVALENT",
                "effectiveDate": ""
        },
            {
                "ien": "1160",
                "name": "INFLUENZA A MONOVALENT (H5N1), ADJUVANTED-2013",
                "shortName": "",
                "cvxCode": "160",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Influenza A monovalent (H5N1), adjuvanted, National stockpile 2013",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "INFLUENZA A (H5N1) -2013",
                "effectiveDate": ""
        },
            {
                "ien": "1801",
                "name": "AS03 ADJUVANT",
                "shortName": "",
                "cvxCode": "801",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "AS03 Adjuvant",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "500001",
                "name": "VOODOO SHOT (HISTORICAL)",
                "shortName": "V D SHOT",
                "cvxCode": "",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "500002",
                "name": "VOODOO SHOT1 (HISTORICAL)",
                "shortName": "V D SHOT1",
                "cvxCode": "",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "500003",
                "name": "HEPATITIS B (NEWBORN TO 12) (HISTORICAL)",
                "shortName": "HEPB NBORN",
                "cvxCode": "",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "500004",
                "name": "SBY-INFLUENZA (HISTORICAL)",
                "shortName": "INFLUENZA",
                "cvxCode": "",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "500005",
                "name": "FLU SHOT 2000 (HISTORICAL)",
                "shortName": "FLU2000",
                "cvxCode": "",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "500006",
                "name": "MIKES INFLUENZA (HISTORICAL)",
                "shortName": "INFLUENZA",
                "cvxCode": "",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "N",
                "cdcFullVaccineName": "",
                "codingSystem": "",
                "vaccineInfoStmt": "",
                "synonym": "",
                "cdcProductName": "",
                "effectiveDate": ""
        },
            {
                "ien": "500007",
                "name": "PNEUMOCOCCAL POLYSACCHARIDE PPV23",
                "shortName": "PNEUMOVAX",
                "cvxCode": "33",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "PPSV23",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Pneumococcal polysaccharide vaccine, 23 valent",
                "codingSystem": "",
                "vaccineInfoStmt": "22~PNEUMOCOCCAL POLYSACCHARIDE VIS   10-06-09   CURRENT   1",
                "synonym": "",
                "cdcProductName": "PNEUMOVAX 23",
                "effectiveDate": ""
        },
            {
                "ien": "612006",
                "name": "NOVEL INFLUENZA-H1N1-09, ALL FORMULATIONS",
                "shortName": "FLU H1N1",
                "cvxCode": "128",
                "maxInSeries": "",
                "inactiveFlag": "1",
                "vaccineGroup": "",
                "vaccineGroupName": "",
                "acronym": "",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Novel influenza-H1N1-09, all formulations",
                "cdcProductName": "",
                "codingSystem": "CPT~90663;;90470",
                "vaccineInfoStmt": "",
                "synonym": "",
                "effectiveDate": ""
        },
            {
                "ien": "612013",
                "name": "INFLUENZA, HIGH DOSE SEASONAL",
                "shortName": "FLU,HI DOS",
                "cdcProductName": "FLUZONE-HIGH DOSE",
                "cvxCode": "135",
                "maxInSeries": "",
                "inactiveFlag": "0",
                "vaccineGroup": "FLU",
                "vaccineGroupName": "",
                "acronym": "IIV3",
                "selectableHistoric": "Y",
                "cdcFullVaccineName": "Influenza, high dose seasonal, preservative-free",
                "codingSystem": "",
                "vaccineInfoStmt": "13~INFLUENZA VACCINE - INACTIVATED VIS   07-02-12   HISTORIC   1|14~INFLUENZA VACCINE - INACTIVATED VIS   07-26-13   HISTORIC   1|36~INFLUENZA VACCINE - INACTIVATED VIS   08-19-14   CURRENT   1",
                "synonym": "",
                "effectiveDate": ""
        }
        ]);
        /* jshint +W109 */
    });
});
