<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');
/**
* @package direct-project-innovation-initiative
* @subpackage models
*/

/** require parent */
require_model('mailbox');

require_model('api_user');

/**
* @package direct-project-innovation-initiative
* @subpackage models
*/
class User extends Mailbox {
	protected $_ldap_entry;
	
	protected static $_relationships = array( 'contact' => array( 'type' => 'has_many', 'foreign_key' => 'user_id'),
											  'global_contact' => array( 'type' => 'has_and_belongs_to_many', 'through' => 'global_contacts_shared', 'foreign_key' => 'user_id'), 
											  'theme' => array('type' => 'has_one',  'foreign_key' => 'theme_id', 'key_for_relationship' => 'user_theme'));
											  	

	protected $_readonly_fields = array( 'user_ep',
										  'user_created_by',
										  'user_created_date',
										  'user_is_group'); 	
	
	function given_name(){
		return element('givenname', $this->ldap_entry);
	}
	
	function family_name(){
		return $this->sn();
	}
	
	//full name, not including middle name
	//currently a wrapper for cn - in part to make it clearer to the non-ldap-savvy what this is, but also in case we ever want display name to be a modified version of the cn
	function display_name(){
		return $this->cn;
	}
	
	function cn(){
		return element('cn', $this->ldap_entry);
	}
	
	function country(){
		if(!$this->property_is_empty('user_locale')){
			return element('country', unserialize($this->user_locale), ENVIRONMENT_COUNTRY);
		}
		
		return ENVIRONMENT_COUNTRY;
	}
	
	function sn(){
		return element('sn', $this->ldap_entry);
	}
	
	function dn($include_user_id = true){
		if(!$this->is->boolean($include_user_id)) return $this->error->should_be_a_boolean($include_user_id);
		
		$dn = element('dn', $this->ldap_entry);
		if(!$include_user_id)
			$dn = strip_from_beginning('uid='.$this->user_name.',', $dn);
			
		return $dn;
	}
	
	/**
	* @return boolean
	*/
	function has_valid_ldap_credentials(){
		$CI = get_instance();
		
		//load the ldap library, but call it something new in case we've already connected
		$credentials = $this->ldap_credentials();
		$credentials['disable_fallback'] = true;

		$CI->load->library('ldap', $credentials, 'valid_ldap_credentials_test');
		
		//this will be true only if the provided credentials were used to bind successfully	
		return $CI->valid_ldap_credentials_test->authenticated();
	}

	function is_admin(){
		$CI = get_instance();
		if(!isset($CI->ldap)) $CI->load->library('ldap');
		return get_instance()->is->nonzero_unsigned_integer(element('count', $CI->ldap->get_admin_group_membership($this->dn())));
	}	
	
	function is_group_leader(){
		$roles = get_instance()->role_model->get_roles_membership($this->dn());
		return in_array('groupleader', $roles);
	}
	
	function is_facility_leader(){
		$roles = get_instance()->role_model->get_roles_membership($this->dn());
		return in_array('facilityleader', $roles);	
	}	
	
	/**
	* @boolean True if this user is the logged in user
	*/	
	function is_logged_in(){
		$CI = get_instance();
		
		if($CI->session->userdata('is_loggedin') !== 'true') 
			return false;
		
		if($CI->session->userdata('username') != $this->user_name)
			return false;
			
		if(!$this->is_active() || !$this->has_valid_ldap_credentials())
			return false;
		
		return ($CI->authentication->organization_id == $this->organization_id);
	}
	
	/**
	* Log in the user by storing  the user data in the session.
	* @return boolean
	*/
	public function log_in(){
		if($this->is_logged_in()) return true; //no need to log in again if they're already logged in
		
		$CI = get_instance();
		
		if(!$this->is_active()) 
			return $CI->error->warning($this->describe().' cannot be logged in because they are not active');
			
		 if(!$this->has_valid_ldap_credentials())
		 	return $CI->error->warning($this->describe().' cannot be logged in because they do not have valid ldap credentials');
		
		if($this->organization_id() != $CI->authentication->organization_id)
			return $CI->error->warning($this->describe().' cannot be logged in because their organization id does not match the organization id found by the authentication library');
							
		$userdata = array('is_loggedin' => 'true', 
						  'just_logged_in' => TRUE, //note the fact that the user just logged in within session
						  'username' => $this->user_name, 
						  'user_mail' => element('mail', $this->ldap_entry()),
						  'group_mailbox' => $this->group_names());		
						  			
		$CI->session->set_userdata($userdata); //set multiple userdata at once to reduce database writes
		
		session_regenerate_id(); 
		
		$user_ip_address = (isset($_SERVER['HTTP_X_REAL_IP'])) ? $_SERVER['HTTP_X_REAL_IP'] : $_SERVER['SERVER_ADDR'] ;
		$CI->audit->log_event('login',array(session_id(),$this->user_name,$user_ip_address,date('U'),TRUE, NULL));
		
		if(!empty($this->default_mailbox()))
			$CI->session->set_mailbox($this->default_mailbox());
		
		//if a site announcement has been set up, set up a banner to display it to the user
		$announcement = Announcement::find_one(array('enabled' => true));
		if(Announcement::is_an_entity($announcement) && !$announcement->property_is_empty('content'))
			$CI->session->display_banner('announcement', '<strong>Announcement:</strong> '.$announcement->content);
		
		//if an automatic replies are turned on, set up a banner to be shown to the user to let them know that replies are on
		$reply = Automatic_reply::find_one(array('mailbox' => $this->user_name));
		if(Automatic_reply::is_an_entity($reply) && $reply->is_active())
			Automatic_reply::enable_banner();
			
		$CI->user = $this; //make sure that find_from_session() won't have to do a separate lookup
		
		//as one last check, make sure that our model now believes the user to be logged in
		if(!$this->is_logged_in()) 
			return $this->error->warning($this->describe().' could not be successfully logged in');
		
		return true;
	}	
	
	function ldap_credentials(){
		$CI = get_instance();
		$credentials = array('user'=>$this->user_name,'pwd'=> $CI->encrypt->decode($this->user_ep));
		return $credentials;	
	}
	
	/**
	* Add user feedback to the Feedback Log
	* Note that this also sends email to Webmail Administrators
	* @param string
	* @param string
	* @return boolean
	*/
	function submit_feedback($type, $comments){	
		if(!in_array($type, array('General Feedback', 'Issue Report'))) return $this->error->should_be_a_known_feedback_type($type);
		if(!$this->is->nonempty_string($comments)) return $this->error->should_be_a_nonempty_string($comments);
		
		$success = $this->db->insert('feedback', array('user_id' => $this->id, 'feedback_type' => $type, 'feedback_comments' => $comments, 'feedback_datetime' => date('U')));
		if($success){
			$feedback_id = $this->db()->insert_id();
						
			$admin_emails = collect('user_mail', static::find_admins());
			if(empty($admin_emails))
				trigger_error('The email notification for feedback#'.$feedback_id.' could not be sent because there are no administrators to send it to, but the data has been recorded to the feedback log', E_USER_WARNING);
				
			// send and prevent showing all recipients
			$CI = get_instance();
			$CI->load->library('external_email');
			
			$subject = PORTAL_TITLE_PREFIX.'New '.(($type == 'Issue Report') ? 'Issue Reported' : 'Feedback Submitted');
			$message = $CI->load->view('feedback/_notification_email', compact('type'), TRUE);			
			$CI->external_email->to($admin_emails);
			$CI->external_email->subject($subject); 
			$CI->external_email->message($message); 
			if(!$CI->external_email->send()){
				trigger_error('The email notification for feedback#'.$feedback_id.' could not be sent, but the data has been recorded to the feedback log', E_USER_WARNING);
			}			
		}
		return $success;
	}

	function timezone(){
		if(!$this->property_is_empty('user_locale')){
			return element('timezone', unserialize($this->user_locale), get_instance()->locale->get_id_from_timezone($this->country));
		}
		
		return get_instance()->locale->get_id_from_timezone($this->country);
	}
	
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// ATTACHMENT CACHE FUNCTIONALITY
// Before they're attached to a message, attachments are stored in a cache specific to the logged-in user.  These functions deal with
// parsing/managing the attachments in the cache.
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////	
	
	
	function attachment_cache(){
		return APPPATH.'cache/'. $this->user_name.'/';
	}
	
	function attachment_from_cache($filename){
		if(!$this->file_name_exists_in_attachment_cache($filename)) return false;
		
		$binary_string = file_get_contents($this->attachment_cache().$filename);
		if($binary_string === FALSE) $this->error->warning('Could not read '.$path);
		
		return Attachment::create($filename, $binary_string, array('directory' => $this->attachment_cache()));
	}
	
	function attachments_from_cache(){
		require_library('attachment');
		
		if(!file_exists($this->attachment_cache())) return array();
		$filenames = get_filenames($this->attachment_cache());
		if(!is_array($filenames)) return $this->error->warning('Could not read '.$this->attachment_cache());
		
		natcasesort($filenames);

		$files = array();
		foreach($filenames as $filename){
			$attachment = $this->attachment_from_cache($filename);
			if(is_a($attachment, 'Attachment')) $files[$filename] = $attachment;
		}
		return $files;
	}	
	
	function attachment_cache_size_in_bytes(){
		$attachment_files = $this->attachments_from_cache();
		if(empty($attachment_files)) return 0;	
		return array_sum(collect('bytes', $attachment_files));
	}
	
	function file_name_exists_in_attachment_cache($file_name){
		if(empty($file_name) || !$this->is->string_like_a_file_path($file_name)) return $this->error->should_be_a_file_name($file_name);
		if(!directory_exists($this->attachment_cache())) return false;
		return file_exists($this->attachment_cache().$file_name);
	}
	
	function attachment_exists_in_cache($name, $binary_string){
		if(!file_exists($this->attachment_cache().$name)) return false;
		$current_binary_string = file_get_contents( $this->attachment_cache().$name );
		if($current_binary_string === FALSE){
			$this->error->warning('Could not read '.$this->attachment_cache().$name);
			return true; //assume that they're the same, since we can't determine otherwise
		}
		
		return ($current_binary_string == $binary_string);
	}
	
#TODO - WHAT IF THE USER HAS TWO DIFFERENT COMPOSE WINDOWS OPEN IN TWO DIFFERENT TABS?  WE NEED TO CACHE BY MESSAGE, NOT JUST BY USER
	function add_attachment_to_cache($name, $binary_string){			
		if(!$this->is->nonempty_string($name)) return $this->error->should_be_a_nonempty_string($name);
		if(!is_string($binary_string)) return $this->error->should_be_a_binary_string($binary_string);
		
		//first, make sure we have a cache 
		if(!directory_exists($this->attachment_cache())) mkdir($this->attachment_cache()); //attempt to create the cache if needed
		if(!directory_exists($this->attachment_cache())) return $this->error->warning('Could not create an attachment cache for '.$this->describe().' at '.$this->attachment_cache());
		
		if($this->attachment_exists_in_cache($name, $binary_string)) return true; //attachment is already in cache; this is the EXACT attachment, with the same binary string & everything
		
		
#TODO - MAKE THE FILENAME UNIQUE IF NEEDED
		$path = $this->attachment_cache().$name;		
		$success = file_put_contents($path, $binary_string);	
		if($success === FALSE)
			return $this->error->warning('Unable to write '.$path.' to attachment cache');
		return true;
	}
	
	function remove_attachment_from_cache($name){
		if(!directory_exists($this->attachment_cache())) return true; //we're done!		

		$path = $this->attachment_cache().$name;
		if(!file_exists($path)) return true; //we're done!
		if(!unlink($path)) return $this->error->warning('Failed to remove '.$path);
		return true;
	}
	
	
	function clear_attachment_cache(){
		if(!directory_exists($this->attachment_cache())) return true; //we're done!
		delete_files($this->attachment_cache());
		return rmdir($this->attachment_cache());
	}
	
//////////////////////////////////////////////////////////////////////////////////////////////////////////
// Creating a SYSTEM CACHE for Attachment Previews for now... - BH -1/06/2014
// This will be revisited to scale for other features, such as including sub-folders for messages,
// different types of attachement previews, etc.
//////////////////////////////////////////////////////////////////////////////////////////////////////////
	
	function system_cache(){
		return APPPATH.'cache/system/'. $this->user_name.'/';
	}
	
	function attachment_from_system_cache($filename){
		if(!$this->file_name_exists_in_system_cache($filename)) return false;
	
		$binary_string = file_get_contents($this->system_cache().$filename);
		if($binary_string === FALSE) $this->error->warning('Could not read '.$path);
	
		return Attachment::create($filename, $binary_string, array('directory' => $this->system_cache()));
	}
	
	function attachments_from_system_cache(){
		require_library('attachment');
	
		if(!file_exists($this->system_cache())) return array();
		$filenames = get_filenames($this->system_cache());
		if(!is_array($filenames)) return $this->error->warning('Could not read '.$this->system_cache());
	
		natcasesort($filenames);
	
		$files = array();
		foreach($filenames as $filename){
			$attachment = $this->attachment_from_system_cache($filename);
			if(is_a($attachment, 'Attachment')) $files[$filename] = $attachment;
		}
		return $files;
	}
	
	function system_cache_size_in_bytes(){
		$attachment_files = $this->attachments_from_system_cache();
		if(empty($attachment_files)) return 0;
		return array_sum(collect('bytes', $attachment_files));
	}
	
	function file_name_exists_in_system_cache($file_name){
		if(empty($file_name) || !$this->is->string_like_a_file_path($file_name)) return $this->error->should_be_a_file_name($file_name);
		if(!directory_exists($this->system_cache())) return false;
		return file_exists($this->system_cache().$file_name);
	}
	
	function file_exists_in_system_cache($name, $binary_string){
		if(!file_exists($this->system_cache().$name)) return false;
		$current_binary_string = file_get_contents( $this->system_cache().$name );
		if($current_binary_string === FALSE){
			$this->error->warning('Could not read '.$this->system_cache().$name);
			return true; //assume that they're the same, since we can't determine otherwise
		}
	
		return ($current_binary_string == $binary_string);
	}
	
	#TODO - WHAT IF THE USER HAS TWO DIFFERENT COMPOSE WINDOWS OPEN IN TWO DIFFERENT TABS?  WE NEED TO CACHE BY MESSAGE, NOT JUST BY USER
	function add_file_to_system_cache($name, $binary_string){
		if(!$this->is->nonempty_string($name)) return $this->error->should_be_a_nonempty_string($name);
		if(!is_string($binary_string)) return $this->error->should_be_a_binary_string($binary_string);
		
		//first, make sure we have a cache
		if(!directory_exists($this->system_cache())) mkdir($this->system_cache(),0777,true); //attempt to create the cache if needed
		if(!directory_exists($this->system_cache())) return $this->error->warning('Could not create an attachment cache for '.$this->describe().' at '.$this->system_cache());
		
		if($this->file_exists_in_system_cache($name, $binary_string)) return true; //attachment is already in cache; this is the EXACT attachment, with the same binary string & everything
		
		#TODO - MAKE THE FILENAME UNIQUE IF NEEDED
		$path = $this->system_cache().$name;
		$success = file_put_contents($path, $binary_string);
		if($success === FALSE)
			return $this->error->warning('Unable to write '.$path.' to attachment system cache');
					return true;
		}
		
		function remove_attachment_from_system_cache($name){
		if(!directory_exists($this->system_cache())) return true; //we're done!
		
		$path = $this->system_cache().$name;
		if(!file_exists($path)) return true; //we're done!
		if(!unlink($path)) return $this->error->warning('Failed to remove '.$path);
		return true;
	}
	
	
	function clear_system_cache(){
		if(!directory_exists($this->system_cache())) return true; //we're done!
		delete_files($this->system_cache(), TRUE);
		
		//catch any subdirectories not caught by delete_files - note that this is not recursive and only covers one level of subdirectories
		foreach(get_dir_file_info($this->system_cache()) as $file_info){
			if(is_dir($file_info['server_path']))
				rmdir($file_info['server_path']);
		}
		
		return rmdir($this->system_cache());
	}	
	
	
////////////////////////////
// GETTERS
////////////////////////////	

	public function default_mailbox(){
		$default_mailbox = element('default_mailbox', $this->_values);
		if(!empty($default_mailbox)) return $default_mailbox;		
		if(!$this->hide_personal_mailbox()) return $this->user_name;
		return array_first_key($this->mailbox_names());
	}

	public function hide_personal_mailbox(){
		if(!is_array($this->ldap_entry())) return false;
		return (element('employeetype', $this->ldap_entry()) == 'mailboxhidden');
	}
	
	public function group_names(){
		$CI = get_instance();
		if(!isset($CI->ldap)) $CI->load->library('ldap');
		$groups = $CI->ldap->get_group_membership('uid='.$this->user_name.','.LDAP_ACCOUNTS_DN);
		if(!is_array($groups)) return array();
		return $groups;
	}
	
	//get the info about this user from LDAP *once* - to force this info to refresh, set $_ldap_entry to null
	public function ldap_entry($attribute = null){
		if(!isset($this->ldap_entry)){
			$this->_ldap_entry = array(); //by default, set an empty array so that we don't keep trying to access LDAP if the call isn't working
			$CI = get_instance();
			if(!isset($CI->ldap)) $CI->load->library('ldap');
			
			$attributes = array('cn','sn','givenName','employeeType','initials','o','departmentnumber','telephonenumber','mobile','title','physicaldeliveryofficename');
			$result = $CI->ldap->search(null, 1, $attributes, '(uid='.$this->username.')', ($this->is_active ? null : LDAP_DELETED_ACCOUNTS_DN));
			
			//make sure that the result is an array
			if(!is_array($result)){
				$this->error->warning('No LDAP entries were found for user '.$this->error->describe($this->username));				
			}else{
				//let the developer know something's going wrong if we have more than one LDAP entry for this user
				if(count($result) > 1){
					$this->error->warning(count($result).' LDAP entries were found for user '.$this->error->describe($this->username).' when only one was expected.');
				}
				
				$this->_ldap_entry = first_element($result);
			}	
		}
		
		if(!is_null($attribute)){
			return element($attribute, $this->_ldap_entry);
		}
		
		return $this->_ldap_entry;
	}
	
	public function mailboxes(){
		$CI = get_instance();
		$CI->api->clear();
		$CI->api->call('direct/mailboxes', array('mailbox' => $this->username()), 'GET');
		return element('mailboxes', $CI->api->output_as_array());
	}

	public function mailbox_names(){
		$mailboxes = array($this->user_name => 'Personal (' .html_escape($this->user_name). ')');
		$groups = $this->group_names();
		
		if(!is_array($groups)) return $mailboxes;
		if($this->hide_personal_mailbox()) return $groups;
		
		return array_merge($mailboxes, $groups);
	}
	
	function organization_id(){
		if(USE_PIV_AUTH)
			return $this->user_piv_id;
		if(USE_CAC_AUTH)
			return $this->user_edipi;
		if(USE_SSO_AUTH)
			return $this->user_va_uid;

	}	
	
	public function username(){
		return $this->user_name;
	}
	
/////////////////////////////
// SETTERS
/////////////////////////////	

	protected function set_default_mailbox($value, $offset = 0){
		if(!is_null($value) && !array_key_exists($value, $this->mailbox_names))
			 return $this->error->property_value_should_be_the_name_of_a_mailbox_this_user_has_access_to('default_mailbox', get_class($this), $value, $offset+1);
		
		return $this->_set_field_value('default_mailbox', $value, $offset+1);
	}

////////////////////////////
// DATA MANGEMENT
////////////////////////////


	 protected function _run_before_create(){ 	 
	 	if($this->property_is_empty('user_name')) return $this->error->warning("I can't create a new user without a value for the username");
		
		$CI = get_instance();
		$CI->load->library('ldap', array('user' => LDAP_ADMIN_USERNAME, 'pwd' => LDAP_ADMIN_PASSWORD, 'disable_fallback' => true), 'admin_ldap');
		if(!$CI->admin_ldap->connected()) return $this->error->warning("I can't create new user ".$this->user_name.' without an LDAP connection');		
		
		$api_user = API_user::find_one(array('mailbox' => $this->user_name));
		if(!API_user::is_an_entity($api_user)){
			$message = 'I can\'t create a user with username '.$this->user_name.' when I am not able to retrieve a user with this user name from the API.';
			if($CI->api->http_status != 200)
				$message .= '  The API says: '.$CI->api->http_status.': '.element('message', $CI->api->output_as_array());
			else
				$message .= '  The API call appears to have been successful, so there is probably no active user with this username.';
			return $this->error->warning($message);
		}
		
		if(USE_PIV_AUTH)
			$this->user_piv_id = $api_user->organization_id;
		if(USE_CAC_AUTH)
			$this->user_edipi = $api_user->organization_id;
			
		if(!isset($this->user_mail)) $this->user_mail = $api_user->external_email;

		$password = static::random_password();
		
		$ldap_attributes = array('objectClass' => array('posixAccount', 'top', 'person', 'organizationalPerson', 'inetOrgPerson'),
								 'gidNumber' => 5000, 
								 'uidNumber' => 5000,
								 'uid' => $this->username,
								 'homeDirectory' => '/var/mailboxes/'.$this->username,
								 'mail' => $this->username . '@' . CLINICAL_DOMAIN,
								 'userPassword' => $CI->encrypt->ssha256_encode($password));
								 
		$ldap_attributes = array_merge($ldap_attributes, $api_user->values_for_ldap());
		foreach($ldap_attributes as $attribute => $value){
			if($value == '') unset($ldap_attributes[$attribute]);
		}
							 
		if(!$CI->admin_ldap->create_ldap_account($ldap_attributes)) { 
			return $this->error->warning('Could not add LDAP account for user '.$this->username);
		}
		
		$this->_set_field_value('user_ep', $CI->encrypt->encode($password), $error_offset = 0, $override_validation = TRUE);
		$this->_set_field_value('user_created_date', now(), $error_offset = 0, $override_validation = TRUE);
		$current_user = User::find_from_session();
		if(User::is_an_entity($current_user))
			$this->_set_field_value('user_created_by', $current_user->id, $error_offset = 0, $override_validation = TRUE);
		else
			$this->_set_field_value('user_created_by', 0, $error_offset = 0, $override_validation = TRUE);
		if(!isset($this->user_ext_notify_flag)) $this->user_ext_notify_flag = true;
		if(!isset($this->user_ext_group_notify_flag)) $this->user_ext_group_notify_flag = true;
		if(!isset($this->user_deleted_flag)) $this->user_deleted_flag = false;
		$this->_set_field_value('user_is_group', false, $error_offset = 0, $override_validation = TRUE);
		$this->user_theme = first_element($CI->db->query('SELECT theme_id FROM themes WHERE theme_is_default=1')->row_array());
		
		return true; 
	}   


	protected function _run_after_create(){ 
		get_instance()->load->library('audit');
		get_instance()->audit->log_event('edit', array($this->id, $this->user_created_by, 'Create User',date('U')));
		return true; 
	}

	
/////////////////////////////
// STATIC METHODS
/////////////////////////////	
	
	public static function find_admins($id_or_conditions = array(), $key_by = null){
		$CI = get_instance();
		if(!isset($CI->ldap)) $CI->load->library('ldap');
		if(!$CI->ldap->connected()){
			trigger_error('Unable to find admins without a valid LDAP connection', E_USER_WARNING);
			return array();
		}	
		
		$ldap_result = $CI->ldap->search(NULL,NULL,array('member'),'(&(ObjectClass=groupofNames)(ou=admins)(cn=admins))', LDAP_BASE_DOMAIN);
		if(empty($ldap_result)) return array();
	
		$usernames = array();
		foreach(element('member', first_element($ldap_result), array()) as $ldap_entry){
			if(string_begins_with('uid=', $ldap_entry))
				$usernames[] = strip_from_beginning('uid=', substr($ldap_entry, 0, strpos($ldap_entry, ',')));
		}
		
		User::db()->where('user_deleted_flag', 0);
		User::db()->where_in('user_name', $usernames);
		return User::find($id_or_conditions, $key_by);	
	}
	
	public static function find_by_organization_id($organization_id, $conditions = array()){
		if(USE_PIV_AUTH)
			User::db()->where('user_piv_id', $organization_id);
		if(USE_CAC_AUTH)
			User::db()->where('user_edipi', $organization_id);
		if(USE_SSO_AUTH)
			User::db()->where('user_va_uid', $organization_id);
		return User::find_one($conditions);
	}
	
	public static function find_from_session(){
		$username = get_instance()->session->userdata('username');
		if(empty($username)) return false;
		
		$CI = get_instance();
		if(isset($CI->user) && User::is_an_entity($CI->user) && $CI->user->username == $username)
			return $CI->user; //skip the db query
		
		$user = User::find_one( array('user_name' => $username) );
		if(User::is_an_entity($user)){
			$CI->user = $user;
		}
		
		return $user;
	}	

	public static function log_out($destination ='auth/logout'){
		if(empty($destination)) $destination = 'auth/logout';
			
		$CI = get_instance();
		$uri_before_logout = $CI->session->flashdata('uri_before_logout');		
		if(empty($uri_before_logout)){
			if(IS_AJAX)
				$uri_before_logout = $_SERVER['HTTP_REFERER'];
			else
				$uri_before_logout = $CI->config->site_url($CI->uri->uri_string());
		}
		
		//figure out where we are
		$controller = strtolower(get_class(get_instance($CI)));
		$method = strtolower($CI->router->method);
		if($method == 'index' && !string_ends_with('index', $destination)) $method = '';
		$action = implode_nonempty('/', array($controller, $method));
		
		//destroy the session
		session_destroy(); //note that as of CI 3.0, this destroys the CI session along with the PHP session
		
		//start a new session so we can set the previous URI			
		session_start(); //not 100% sure this is the correct CI3.0 practice to replace the old sess_create, but seems to match the session_destroy practice -- MG 2015-06-02
					
		//redirect to the logout page unless we're already there		
		$destination = strtolower(strip_from_beginning('/', strip_from_end('/', $destination))); //normalize the / for destination
		if(string_contains('?', $destination)) $destination = substr($destination, 0, strpos($destination, '?')); //make sure we're compensating for a query string
		if(string_begins_with(site_url(), $uri_before_logout)) $CI->session->set_flashdata('uri_before_logout', $uri_before_logout);
		
		//redirect to the logout page unless we're already there	
		if($destination != $action){
			redirect($destination);
		}
	}
	
}
