<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-project-innovation-initiative
* @subpackage models
*/

require_model('db_entity');

/**
* @author B. Holt <brandon.holt@bylight.com>
* @package direct-project-innovation-initiative
* @subpackage models
*/
class Global_contact extends DB_entity {
	static $table = 'admin_contact_list';	
	static $primary_key = 'contact_id';

	protected $_readonly_fields = array('modified_by', 'modified_at');
	
	protected $_property_validation_rules = array('first_name' => 'nonempty_string',
												  'last_name' => 'nonempty_string',
												  'direct_address' => 'string_like_a_direct_address',
												  'telephone' => 'string_like_a_phone_number',
												  'mobile' => 'string_like_a_phone_number');

	protected static $_relationships = array('user' => array('type' => 'has_and_belongs_to_many', 'through' => 'global_contacts_shared', 'related_foreign_key' => 'user_id'));


	

	//this will ensure that the global contact appears in the user's personal address book
	public function add_to_user_contact_list($user_id_or_entity){
		$user_id = (User::is_an_entity($user_id_or_entity)) ? $user_id_or_entity->id : $user_id_or_entity;
		if(!User::exists($user_id)) return $this->error->should_be_an_existing_user_id($user_id);
		
		$through_table = static::relationships('user', 'through');
		$values = array('user_id' => $user_id, 'contact_id' => $this->id);
		
		if($this->db->where($values)->count_all_results($through_table, $values) > 0) return true;
		return $this->db->insert($through_table, $values);	
	}
	
	public function remove_from_user_contact_list($user_id_or_entity){
		$user_id = (User::is_an_entity($user_id_or_entity)) ? $user_id_or_entity->id : $user_id_or_entity;
		if(!User::exists($user_id)) return $this->error->should_be_an_existing_user_id($user_id);
		
		return $this->db->delete(static::relationships('user', 'through'), array('user_id' => $user_id, 'contact_id' => $this->id));
	}

	//clear existing user relationships and add new ones
	public function set_users($user_ids){
		$user_ids = (is_null($user_ids)) ? array() : $user_ids;
		if(!$this->is->array_of_nonzero_unsigned_integer($user_ids)) return $this->error->should_be_an_array_of_nonzero_unsigned_integers($user_ids);
		$through_table = static::relationships('user', 'through');
		$this->db->delete($through_table, array('contact_id' => $this->id));
		if(empty($user_ids)) return true;
		$values = array();
		foreach($user_ids as $user_id)
			$values[] = array('user_id' => $user_id, 'contact_id' => $this->id);
		
		return $this->db->insert_batch($through_table, $values);
	}

	public function values_for_display(){
		return $this->values(array('first_name','middle_name','last_name','title','department','organization','telephone','mobile','direct_address'));
	}

	public function values_for_search(){
		$values = array('type' => 'admin_contact', 
						'cid' => $this->id,
						'displayname' => $this->last_name.', '.implode_nonempty(' ', array($this->first_name, $this->middle_name)),
						'mail' => $this->direct_address,
						'givenname' => $this->first_name,
						'sn' => $this->last_name,
						'initials' => $this->middle_name,
						'o' => $this->organization,
						'departmentnumber' => $this->department,
						'telephonenumber' => $this->telephone,
						'title' => $this->title);
								   
		foreach($values as $key => $value){
			if(!isset($value) || strlen($value) <= 0)
				unset($values[$key]);
		}						   
		
		return $values;
	}

////////////////////
// DATA MANAGEMENT
////////////////////
	
	/**
	* Sets the "modified_by" field to the current user id value.
	* Sets the "modified_at" field to the current timestamp to be stored when the site announcement is updated.
	* @return array
	*/
	protected function _values_for_save(){
		$values_for_save = parent::_values_for_save();
		
		$user = User::find_from_session();
		if(User::is_an_entity($user))
			$values_for_save['modified_by'] = $user->id;
		$values_for_save['modified_at'] = now();
		return $values_for_save;
	}

//the equivalent of ON DELETE CASCADE for the user relationship, since mssql doesn't have a way to do it automaticaly in the db
	protected static function _run_before_delete(&$entity){
		return get_instance()->db->delete(element('through', static::relationships('user')), array('contact_id' => $entity->id));
	} 

	
}

