<?php
/**
* @package direct-project-innovation-initiative
* @subpackage libraries
*/ /** */

load_library('audit');
require_library('form_markup_generator');
require_models('api_user', 'facility');

/**
* Extension of the Form Markup Generator, specifically for editing account settings for a user.
*
* Saving user information is complicated at the moment, since information about the user can be stored in the API, in LDAP, or in the database.
*
* @package direct-project-innovation-initiative
* @subpackage libraries
*/
class User_form_markup_generator extends Form_markup_generator{
	protected $_user;
	protected $_include_admin_fields = false;

	protected $field_names_to_db_columns= array('external_email' => 'user_mail',
												'send_notifications_for_personal_mailbox' => 'user_ext_notify_flag',
												'send_notifications_for_group_mailboxes' => 'user_ext_group_notify_flag',
												'default_mailbox' => 'default_mailbox');

	protected $field_names_to_ldap_attributes = array('first_name' => 'givenName',
													  'middle_name' => 'initials',
													  'last_name' => 'sn',
													  'job_title' => 'title', 
													  'department' => 'departmentnumber',
													  'organization' => 'o',
													  'telephone' => 'telephonenumber',
													  'mobile' => 'mobile',
													  'location' => 'physicaldeliveryofficename');
	
	protected $_field_settings = array(	'first_name' => array('type' => 'text_input', 'required' => true),
										'middle_name' => array('type' => 'text_input'),
										'last_name' => array('type' => 'text_input', 'required' => true),
										'default_mailbox' => array('type' => 'dropdown', 'required' => true, 'comment' => "You'll be directed to this mailbox each time you log in."),
										'job_title' => array('type' => 'text_input'),
										'department' => array('type' => 'text_input'),
										'organization' => array('type' => 'text_input'),
										'telephone' => array('type' => 'text_input', 'attributes' => array('size' => 12)),
										'mobile' => array('type' => 'text_input', 'attributes' => array('size' => 12), 'label_text' => 'Mobile Phone'),
										'location' => array('type' => 'text_input'),
										'facility_id' => array('type' => 'dropdown', 
															   'label_text' => 'Facility',
															   'required' => true, 
															   'attributes'=> array('data-placeholder' => 'Choose a Facility')),
										'external_email' => array('type' => 'text_input', 'required' => true),
										'send_notifications_for_personal_mailbox' => array('type' => 'checkbox'),
										'send_notifications_for_group_mailboxes' => array('type' => 'checkbox'),
										'is_admin' => array('type' => 'checkbox', 'label_text' => 'Administrative Privileges'),
										'personal_mailbox_access' => array('type' => 'checkbox'),		
										'is_group_leader' => array('type' => 'checkbox', 'label_text' => 'Group Leader'),
										'is_facility_leader' => array('type' => 'checkbox', 'label_text' => 'Facility Leader'),								
										);
										
										

										

	function __construct($user, $properties_to_set = array()){		
		parent::__construct();
		$this->set_user($user);
		
		foreach($properties_to_set as $property => $value)
			$this->$property = $value;
		
		$logged_in_user = User::find_from_session();
		
		//set up some configuration that we needed to generate dynamically
		$this->_field_settings['facility_id']['options'] = $this->facilities;
		
		//set up the default mailbox field as long as this is an active user
		if(!$user->is_active || empty($user->mailbox_names())){
			unset($this->_field_settings['default_mailbox']);
		}else{
			$this->_field_settings['default_mailbox']['options'] = $this->user->mailbox_names;
			if($logged_in_user->username != $this->user->username)
				$this->_field_settings['default_mailbox']['comment'] = 'The user will be directed to this mailbox each time they log in.';
		}
		
		//remove admin-only fields if this isn't the admin form or the user doesn't have access
		if(!$this->include_admin_fields || !$user->is_active || !$logged_in_user->is_active || !$logged_in_user->is_admin){
			$admin_only = array('external_email', 'send_notifications_for_personal_mailbox', 'send_notifications_for_group_mailboxes', 'is_admin', 'personal_mailbox_access', 'is_group_leader', 'is_facility_leader');
			foreach($admin_only as $admin_field){
				unset($this->_field_settings[$admin_field]);
			}
		}
		
		//make sure the logged-in user can't remove their own admin privileges, that tends to go badly
		if(array_key_exists('is_admin', $this->_field_settings) && $logged_in_user == $user){
			unset($this->_field_settings['is_admin']);
		}
		
		//since we might not be including some fields, make sure they're not still included in our mapping of field names to data fields
		foreach($this->field_names_to_db_columns as $field_name => $value){
			if(!array_key_exists($field_name, $this->_field_settings))
				unset($this->field_names_to_db_columns[$field_name]);
		}
		
		foreach($this->field_names_to_ldap_attributes as $field_name => $value){
			if(!array_key_exists($field_name, $this->_field_settings))
				unset($this->field_names_to_ldap_attributes[$field_name]);
		}	
				


		$this->set_fields($this->_field_settings);
	}

	
	/**
	* Loads values for this form from the various sources to which they have been saved.
	* This includes data from the database, the API, and LDAP. Returns true if we were able to save all o
	* @return boolean 
	*/
	public function load_saved_values(){
		$values_from_api = $this->values_from_api();
		if(empty($values_from_api)) return false; //this should only happen if the API call failed	
		
		$ldap_values = array();

		//add in some additional values that will only be turned on in certain cases
		if($this->field_exists('personal_mailbox_access'))
			$ldap_values['personal_mailbox_access'] = (string)(int)!$this->user->hide_personal_mailbox;

		foreach(array('is_admin', 'is_group_leader', 'is_facility_leader') as $field){
			if($this->field_exists($field))
				$ldap_values[$field] = (string)(int)$this->user->$field();
		}
		
		$values = array_intersect_key(array_merge($values_from_api, $ldap_values, $this->values_from_db()), $this->fields);	
		return $this->set_values($values);
	}
	
	/**
	* This form saves values to a weirdly large number of places, so let's have one method that understands how to saves all of them.
	* Currently, we save data to the database, the API, and LDAP.  Returns true if all aspects were saved successfully.
	* @return boolean 
	*/
	public function save_values(){
				
		foreach($this->values_for_db() as $field_name => $value){
			$this->user->$field_name = $value;
		}
			
		if(!$this->user->save()) 
			return $this->error->warning('Failed to save '.$this->user->describe().' account information to the database');
				
		if(!get_instance()->ldap->modify_ldap_account($this->user->user_name, $this->values_for_ldap(), $this->user->dn($include_user_id = false)))
			return $this->error->warning('Failed to save '.$this->user->describe().' account information to ldap');
				
		foreach(array('is_group_leader', 'is_facility_leader') as $field){
			if(!$this->field_exists($field)) 
				continue;
			if($this->value($field) && !$this->user->$field)
				get_instance()->role_model->add_roles_to_member(array(str_replace('_', '', strip_from_beginning('is_', $field))), $this->user->dn());
			elseif(!$this->value($field) && $this->user->$field)
				get_instance()->role_model->remove_roles_from_member(array(str_replace('_', '', strip_from_beginning('is_', $field))), $this->user->dn());
		}	
		
		if($this->field_exists('is_admin') && (($this->user->is_admin && !$this->value('is_admin')) || (!$this->user->is_admin && $this->value('is_admin')))){
			get_instance()->ldap->set_admin_group_membership($this->user->username, $this->value('is_admin'));
		}	
		
		$user = Api_user::find_one(array('mailbox' => $this->user->username));	
		foreach($this->values_for_api() as $field_name => $value)
			$user->$field_name = $value;
		if(!$user->save()) return $this->error->warning('Failed to save '.$this->user->describe().' account information to the api');
	
		$CI = get_instance();
		$CI->audit->log_event('edit', array($this->user->id, $CI->user->id, 'Update user information', date('U')));	
		return true;
	}

	function validates(){
		return parent::validates() && $this->default_mailbox_is_valid();
	}
	
	function validation_messages(){
		$messages = parent::validation_messages();
		if(!$this->default_mailbox_is_valid()){
			$messages[] = $this->field('personal_mailbox_access')->link_to_field().' cannot be disabled when the personal mailbox is selected as the '.$this->field('default_mailbox')->link_to_field().'.';
		}
		return $messages;
	}

	//make sure that the personal mailbox is not selected as the default if we're hiding the personal mailbox
	protected function default_mailbox_is_valid(){
		if(!$this->field_exists('default_mailbox') && !$this->field_exists('personal_mailbox_access')) return true; //we don't need to worry about it in this case.
		
		$default_mailbox = ($this->field_exists('default_mailbox')) ? $this->value('default_mailbox') : $this->user->default_mailbox;
		$hide_personal_mailbox = ($this->field_exists('personal_mailbox_access')) ? !$this->value('personal_mailbox_access') : $this->user->hide_personal_mailbox;
		
		$is_valid = (!$hide_personal_mailbox || $default_mailbox != $this->user->username);
		
		if(!$is_valid){	
			if($this->field_exists('default_mailbox')) $this->field('default_mailbox')->has_error = true;
			if($this->field_exists('personal_mailbox_access')) $this->field('personal_mailbox_access')->has_error = true;
		}
		
		return $is_valid;		
	}

		
///////////////////////////
// GETTERS FOR CLASS VARS
////////////////////////////
	
	function facilities(){
		$facilities = array();
		foreach(Facility::find() as $facility){
			$facilities[$facility->id] = $facility->name;
		}
		return $facilities;
	}
	
	public function values_for_api(){
		 return array_merge($this->values(array_keys($this->field_names_to_ldap_attributes)),array('mailbox' => $this->user->username, 'facility_id' => $this->value('facility_id')));	
	}
	
	public function values_for_db(){
						
		$values = array();
		foreach($this->field_names_to_db_columns as $field_name => $db_column){
			$values[$db_column] = $this->value($field_name);
			if($this->field($field_name)->type == 'checkbox')
				$values[$db_column] = (bool)$values[$db_column]; //unchecked checkbox has a null value instead of false
		}
				
		//if personal mailbox access is hidden, personal mailbox notifications need to be turned off and the default mailbox can't be the personal mailbox
		if($this->field_exists('personal_mailbox_access') && !$this->value('personal_mailbox_access')){
			$values[$this->field_names_to_db_columns['send_notifications_for_personal_mailbox']] = false;
			if($this->user->default_mailbox == $this->user->username || $values['default_mailbox'] == $this->user->username){
				$values['default_mailbox'] = array_first_key($this->user->group_names());
			}
		}			
				
		return $values;
	}	
	
	public function values_for_ldap(){		
		$values_for_ldap = array('cn' => implode_nonempty(' ', array($this->value('first_name'), $this->value('last_name'))),
								 'displayName' => implode_nonempty(', ', array($this->value('last_name'), implode_nonempty(' ', array($this->value('first_name'), $this->value('middle_name'))))));
		
		if($this->field_exists('personal_mailbox_access'))
			$values_for_ldap['employeeType'] = ($this->value('personal_mailbox_access') ? '' : 'mailboxhidden');
		
		foreach($this->field_names_to_ldap_attributes as $field_name => $ldap_attribute){
			$values_for_ldap[$ldap_attribute] = $this->value($field_name);
		}		
		
		return $values_for_ldap;
	}	
	
	/**
	*
	* @return array|false
	*/
	public function values_from_api(){
		$user = Api_user::find_one(array('mailbox' => $this->user->username));	
		if(!Api_user::is_an_entity($user)) return false;
		return $user->values($this->field_names);
	}
	
	public function values_from_db(){
		$values = array();
		
		foreach($this->field_names_to_db_columns as $field_name => $db_column){
			if(!$this->field_exists($field_name)) continue;
			$values[$field_name] = $this->user->$db_column;
		}		
		
		//if there isn't currently a default mailbox, we want the field value to just default to the first of the options
		if(array_key_exists('default_mailbox', $values) && empty($values['default_mailbox']))
			unset($values['default_mailbox']);
		
		return $values;
	}
	
	
///////////////////////////
// SETTERS FOR CLASS VARS
///////////////////////////
	
	protected function set_user($user){
		if(!User::is_an_entity($user)) return $this->error->property_value_should_be_a_message_entity('user', $this, $user);
		$this->_user = $user;
		return true;
	}

	protected function set_include_admin_fields($value){
		if(!$this->is->boolean($value)) 
			return $this->property_value_should_be_a_boolean('include_admin_fields', $this, $value);
			
		$user = get_instance()->user;
		if($value && !$user->is_admin()) 
			return $this->error->warning("I can't include admin fields for the user form markup generator because ".$user->describe().' is not an admin');
						
		$this->_include_admin_fields = $value;
	}
	
}