<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');
/**
* @package direct-project-innovation-initiative
* @subpackage controllers
*/
require_once APPPATH.'controllers/adminpanel.php';

/**
* Message all page within the Adminpanel.
* This controller is used to store functions related to
* messaging all active VA Direct users stored in the database,
* Routing is handled using CI routing functionality instead
* so that each action can be separated into a child function.
*
* @package direct-project-innovation-initiative
* @subpackage controllers
*/
class System_wide_message_controller extends Adminpanel {

	function __construct() {
		parent::__construct();
		$this->mailbox = Mailbox::find_from_session(); //set the mailbox class var, since we're not in the inbox (where it's handled in the constructor)
		$this->load->model('response_model');
		
#TODO - We need to set up a permission for messaging all users and make sure that only admins with that permission are able to send messages to all users		
	} 


#TODO script to check against mailbox, draft status, and flag for message_all
	public function compose() {		
		
		$data = array(); //values that we're passing on to the view

		//do we have POST values?  if so, parse them & validate; we'll either save or re-display
		if (!empty($_POST) && (array_key_exists('save', $_POST) || array_key_exists('send', $_POST))){
			$message = $this->_message_from_post();
			$action = (array_key_exists('send', $_POST)) ? 'send' : 'save';
			
			$this->form_validation->set_rules('message_subject', 'Message Subject', 'required|xss_clean');
			$this->form_validation->set_rules('message_body', 'Message Body', 'required|visible_content_required',
													array('visible_content_required' => 'The {field} field is required.'));
			
			//validate form data if true
			if(!$this->form_validation->run()){
				$validation_errors = explode("\n", trim(validation_errors()));
				$data['error_message'] = '<strong>We need you to make a few changes to this form.</strong>  '.ul_if_multiple($validation_errors);
			}else{
				//check to make sure we were able to find the message - if not, something has gone very wrong
				if(!Message::is_an_entity($message)){
					$this->session->set_error_message('An unexpected error occurred, and we were unable to '.$action.' your message.  Please try again, and contact an administrator if the problem persists.');
					return redirect('adminpanel/message_all');
				}
	
				//validate that the message is saveable - otherwise, don't go to the overhead of accessing the API
				$validation_errors = array();
				if($this->user->attachment_cache_size_in_bytes() > MAX_ATTACHMENTS_SIZE_IN_BYTES){
					$validation_errors[] = 'The attachments for this message exceed the limit of '.byte_format(MAX_ATTACHMENTS_SIZE_IN_BYTES).'. Please remove one or more attachments.';
				}
	
				foreach($this->user->attachments_from_cache() as $filename => $attachment){
					if (!Message::attachment_has_valid_extension($filename))
						$validation_errors[] = $filename.' is not a valid file type.';
				}
	
				if (!empty($validation_errors)){
					$data['feedback_message'] = implode_nonempty(' ', $validation_errors);
					$data['feedback_class'] = 'error';
				}else {
					if(!$message->save( $preserve_existing_attachments = FALSE )){
						//message shouldn't say the message failed to save/send, because attachment errors still save message updates
						$data['feedback_message'] = 'An error was encountered. '.$this->api->message();
						$data['feedback_class'] = 'error';
					}else{
						//PY 2015-10-08 - not sure if this is relevant
						//This case is saving a draft without sending - it's definitely relevant -- MG 2015-10-12
						if (!array_key_exists('send', $_POST)){
							$this->session->set_success_message('Success!  Your message has been saved.');
						 	return redirect(current_url()); //we need to redirect so that the message will be displayed and the POST data will be cleared
						}
	
						//we're not doing 7332 data or purpose of disclosures for messaging all users, so we can go straight to sending.
						$this->send($message);
					}
				}
			}
		}
		
		if(!isset($message) || !$message->draft || !Message::is_an_entity($message)){
			$message = Message::find_one(array('folder' => 'draft', 'to' => ALL_USERS_MOCK_ADDRESS, 'order_by' => 'id', 'order' => 'desc'));
			if(!Message::is_an_entity($message) || !$message->draft){
				$message = new Message(array('to' => ALL_USERS_MOCK_ADDRESS));
			}
		}
				
		//set up the attachments - unless we're redisplaying from post, in which case the attachment cache is set up correctly and we shouldn't mess with it
		$this->user->clear_attachment_cache(); //clear attachments when visiting compose for the first time in case any were not removed from previous messages
		if ($message->has_attachments()) {
			$message->add_existing_attachments_to_cache();
		}


		//set title and connection status
		$data['title']     = PORTAL_TITLE_PREFIX . "Message All Users";
		$data['mailboxes'] = element('folder_list', $this->mailformat->mailbox_list());
		$data['message']   = $message;
		$data['subject']   = $message->subject;
		$data['to']        = $message->addresses_for_display_for_send('to');

		//FIND THE ATTACHMENTS
		//We want to get this from the cache, not the message, since attachments may have changed since the last message save.
		//On the other hand, attachments from the message have more metadata (e.g. message_id), so if the same attachment is currently on the message, we'll make sure use the attachment
		$data['attachments'] = $this->user->attachments_from_cache();
		if(Message::formatted_like_an_id($message->id)){
			foreach($data['attachments'] as $name => $attachment){
				if ($message->has_attachment($name, $attachment->binary_string)){
					$data['attachments'][$name] = $message->attachment($name);
				}
			}
		}

		$this->load->view('adminpanel/message_all_users', $data);
  }

/////////////////////
// PROTECTED METHODS
/////////////////////


	protected function send(Message $message){
		if (!Message::is_an_entity($message) || !$message->draft) return $this->error->should_be_a_draft($message);

		//send the message!
		if (!$message->send()){

			//the API's message isn't so user-friendly, so use our own message
			$this->session->set_error_message('An error occurred and we were unable to send your message. It has been saved as a draft.');
			if ($this->api->http_status == '403' && string_contains('application is not authorized', $this->message())){
				$this->session->set_service_permission_error_message('Send', 'Failed to send message because Send Direct Service is disabled.');
			}
			return redirect(current_url()); //clear the POST and allow us to see flash feedback messages by redirecting
		}

		$this->response_model->check_response($message->id(), $this->mailbox->name);
		$this->session->set_success_message('Message successfully sent.');
		return redirect(current_url());
	}
	
	protected function _message_from_post(){		
		//FIND OR CREATE THE MESSAGE
		$id = $this->input->post('msg_id',TRUE);
		if(Message::formatted_like_an_id($id)){
			$message = Message::find_one(array('id' => $id, 'mailbox' => $this->user->user_name));
			if(!Message::is_an_entity($message)) return $this->error->should_be_a_message_id($id);
		}else{
			$message = new Message( array('sender' => $this->user->email_address()) );
			if(!Message::is_an_entity($message)) return $this->error->warning("I couldn't create a new message for ".$this->mailbox->describe());
		}
		
		//FIND THE VALUES FROM THE FORM
		$values = array('original_sender' => $this->user->username,	
						'to' => ALL_USERS_MOCK_ADDRESS, 
						'subject' => $this->input->post('message_subject',TRUE),
						'body' => $this->input->post('message_body', FALSE));
			 
		//SET THE FORM VALUES *IFF* THEY'RE DIFFERENT FROM WHAT WE CURRENTLY HAVE (we don't want to make an API call unless we need to)				 				 
		foreach($values as $field => $value){
			if($message->$field != $value)
				$message->$field = $value; 
		}
		
		return $message;
	}	
		
}