<?php 
//phpdocs requires a file docblock at the top of the file as well as a class docblock - please preserve this
/**
* @package vler
* @subpackage security
*/ 

/** grab purifier */
require_once VLERPATH.'third_party/HTMLPurifier.standalone.php';


/**
 * 
 * PHP version 5.2
 * 
 * LICENSE: This source file is subject to applicable Booz Allen IT policies.
 * 
 * @category	BoozAllen
 * @package	vler
 * @subpackage security
 * @author		Daniel L.Hunter <hunter_daniel@ne.bah.com
 * @copyright	2010 Booz Allen Hamilton
 * @license	http://ishare.bah.com/C15/C8/IT%20Policies/ITwebpages/IT%20Policies.aspx Booz Allen IT policies.
 * @link		http://www.bah.com Booz Allen Hamilton
 * @link		http://docs.src.bah.com/WebTeam_Wiki_Home_Page/Project_Documentation/hello.bah.com/security.class.php
 * 
 */
class Security{

	public static $default_validation_function = 'Security::isValidString';
	public static $default_sanitization_function = 'Security::sanitizeString';
	
	
	/**
	 * Secure an entire array such as $_POST or $_GET by defining specific functions or methods to validate and sanitize each element of the array.
	 * 
	 * This method requires a single array as input.  This array must contain data to be validated and sanitized.  
	 * It also allows for up to 4 optional arrays as arguments; an associative array of 
	 * functions to be used to perform the validation, an associative array of functions that are to be  
	 * used to sanitize the elements in the first array, an associative array that contains additional arguments 
	 * needed to run the functions in the second array, and an associative array that contains additional arguments 
	 * needed to run the functions in the third array.  
	 * 
	 * Note that if you want to define specific sanitization and validation methods for the items in the first array, all of
	 * the methods must be associative arrays.  Further, the key=>value pair in the $array_to_be_validated must have a corresponding
	 * key=>value pair (named using the same key) in the $validation_functions and/or $validation_functions arrays.  Otherwise, the default 
	 * validation function,defined in the static class properties $default_validation_function & $default_sanitization_function, will 
	 * be run for that element.
	 * 
	 * Detailed documentation can be found here:
	 * 
	 * http://docs.src.bah.com/WebTeam_Wiki_Home_Page/Project_Documentation/hello.bah.com/security.class.php
	 * 
	 * @param	mixed 		$array_to_be_validated 	Associative Array - Generally the $_POST or $_GET array 
	 * @param	string		$validation_functions 	(optional) - Array of Functions (or methods) to be used to validate first array  
	 * @param	string		$sanitization_functions	(optional) - Array of Functions (or methods) to be used to sanitize first array
     * @param	mixed		$validation_arguments 	(optional) - additional arguments that need to be passed to the validation_function 	 
     * @param	mixed		$sanitization_arguments 	(optional)- additional arguments that need to be passed to the sanitization_function
     *  
     * @return associative array		Original array is passed back
     * 
	 */
	
	public static function secureArray(array $array_to_be_validated, $validation_functions = null, $sanitization_functions = null, $validation_arguments = null, $sanitization_arguments = null)
	{
	
		$returnArray = array();
	
		foreach ($array_to_be_validated as $key=>$value)
		{
			//Get Val Function
			if( is_array($validation_functions) )
			{
				if( array_key_exists($key, $validation_functions) )
				{
					$val_function = $validation_functions[$key];
				} 
				else 
				{
					$val_function = null;	//passing a null value will result in the default validation method being run		
				}
			}
			else
			{
				$val_function = $validation_functions;
			}
		
			
			
			//Get San Function
			if( is_array($sanitization_functions) )
			{
				if(array_key_exists($key, $sanitization_functions))
				{
					$san_function = $sanitization_functions[$key];
				}
				else
				{
					$san_function = null;	//passing a null value will result in the default sanitization method being run
				}
			}
			else
			{
				$san_function = $sanitization_functions;
			}		

			

				
			
			//Check if we need to go deeper into array
			if( is_array($value) )
			{
				$returnArray[$key] = Security::secureArray($value, $val_function, $san_function, $validation_arguments, $sanitization_arguments);
				//$returnArray[$key] = secureArray($value, $validation_functions, $sanitization_functions, $validation_arguments, $sanitization_arguments);
			}
			else	//Otherwise, run function
			{

			
			
				//Check if additional arguments were passed for this method
				if( isset($validation_arguments[$key]) && is_array($validation_arguments[$key]) )
				{	
					$val_arguments = Security::prepareArguments($value, $validation_arguments[$key]);
				}
				else
				{
					$val_arguments = $value;
					//$val_arguments = NULL;
				}	
			
				//Check if additional arguments were passed for this method
				if( isset($sanitization_arguments[$key]) && is_array($sanitization_arguments[$key]) )
				{					
					$san_arguments = Security::prepareArguments($value, $sanitization_arguments[$key]);
				}
				else
				{
					$san_arguments = $value;
					//$san_arguments = NULL;
				}			
			
			
			
			
			
				//Run Passed Method Here
				if( !Security::execValidation($val_function, $val_arguments) )
				{
					$returnArray[$key] = false;
				}
				else
				{
					$returnArray[$key] = Security::execSanitization($san_function, $san_arguments);
				}
			}
		
		}
		return $returnArray;
	}
	

		
	
	
	/**
	 * Secure a variable by defining specific functions or methods to validate and sanitize it.
	 * 
	 * This method is really just a wrapper for the secureArray() method.  In addition to accepting arrays as arguments for 
	 * $variable_to_be_secured, $validation_function, & $sanitization_function, this method accepts individual variables as
	 * well (it has to accept both for flexibility because some variables can contain either arrays or single variables).
	 *  
	 * This method merely converts the arguments passed to it into arrays and then passes them to secureArray() which 
	 * is where the magic happens.
	 * 
	 * ***Although this method is a bit redundant since secureArray() has been made more flexible, it has been left here
	 * ***for backwards compatibility.
	 * 
	 * @param	mixed	$variable_to_be_secured - Variable that is to be validated and sanitized 
	 * @param	string	$validation_function (optional) - Function(s) (or method(s)) to be used to validate first array  
	 * @param	string	$sanitization_function (optional)- Function(s) (or method(s)) to be used to sanitize first array
     * @param	mixed	$validation_arguments (optional) - Additional arguments that need to be passed to the $validation_function 	 
     * @param	mixed	$sanitization_arguments (optional) - Additional arguments that need to be passed to the $sanitization_function
     * 
     * @return	mixed	Validated and Sanitized version of the variable_to_be_validated argument
     * 
	 */
	public static function secureVariable($variable_to_be_secured, $validation_function = null, $sanitization_function = null, $validation_arguments = null, $sanitization_arguments = null)
	{
		
		if(is_array($variable_to_be_secured))
		{
			//Get & Format data to be validated so that it can be passed to validateArray()
			$secure_this_data = $variable_to_be_secured;
			$input_is_array = true;
		}
		else
		{
			$secure_this_data = array('input'=>$variable_to_be_secured);
			$input_is_array = false;
		}
			
		//Get & Format Validation Functions so that it can be passed to validateArray()
		if(isset($validation_function))
		{
			if(is_array($validation_function))
			{
				$val_function = $validation_function;
			}
			else
			{
				$val_function = array('input'=>$validation_function);
			}
		}
		else
		{
			$val_function = null;
		}
		
		//Get & Format Sanitization functions so that it can be passed to validateArray()
		if(isset($sanitization_function))
		{
			if(is_array($sanitization_function))
			{
				$san_function = $sanitization_function;
			}
			else
			{
				$san_function = array('input'=>$sanitization_function);
			}
		}
		else
		{
			$san_function = null;
		}			
	
		
		
		//Get & Format Validation Arguments so that it can be passed to validateArray()
		if(isset($validation_arguments))
		{
			if(is_array($validation_arguments))
			{
				$validation_args = $validation_arguments;
			}
			else
			{
				$validation_args = array('input'=>$validation_arguments);
			}
		}
		else
		{
			$validation_args = null;
		}


		//Get & Format Sanitization Arguments so that it can be passed to validateArray()
		if(isset($sanitization_arguments))
		{
			if(is_array($sanitization_arguments))
			{
				$sanitization_args = $sanitization_arguments;
			}
			else
			{
				$sanitization_args = array('input'=>$sanitization_arguments);
			}
		}
		else
		{
			$sanitization_args = null;
		}
			
	
		//	Run the provided data through validateArray().  If the data provided was an array,
		//	return an array.  If the data provided was a single variable, return a single variable.
		if($input_is_array)
		{
			return Security::secureArray( $secure_this_data, $val_function, $san_function, $validation_args, $sanitization_args );
		}
		else
		{
			
			$retval = Security::secureArray( $secure_this_data, $val_function, $san_function, $validation_args, $sanitization_args );
		
			return $retval['input'];		
		}
		
	}

	/**
	 * Depreciated - Don't use this method.  It will be removed from future versions of this class.
	 * 
	 * Temporary placeholder - this function will be removed in the near future - DLH - 6/9/2010
	 * 
	 * @param	mixed 		$array_to_be_validated 	(optional) Associative Array - Generally the $_POST or $_GET array 
	 * @param	string		$validation_functions 	(optional) - Array of Functions (or methods) to be used to validate first array  
	 * @param	string		$sanitization_functions	(optional) - Array of Functions (or methods) to be used to sanitize first array
     * @param	mixed		$validation_arguments 	(optional) - additional arguments that need to be passed to the validation_function 	 
     * @param	mixed		$sanitization_arguments 	(optional)- additional arguments that need to be passed to the sanitization_function
     * 
     * @return	boolean	Returns false always.  This method has been depreciated.
	 */
	public static function validateArray($array_to_be_validated = null, $validation_functions = null, $sanitization_functions = null, $validation_arguments = null, array $sanitization_arguments = null)
	{
		echo "validateArray() has been renamed secureArray().  validateArray() will be removed from this class shortly.";
		return false;	
	}		
	
	
	
	
	
	
	
	
	
	
	
	
	/****************************************************************************************************/
	/**/
	/**/// Generic Verification Methods - These methods will only validate data and return a true or false
	/**/
	/****************************************************************************************************/
			
	
	/**
	 * Determine if input is a valid string.  Min and Max length are optional parameters.
	 * 
	 * This function checks to see if the value passed to it in the $string input variable
	 * is a valid string.  Further, it checks to see if the string's length is
	 * within a specific range (defined by $max_length and/or $min_length. 
	 * 
	 * @param	mixed	$string			Value we want to test to see if it is a string
	 * @param	int		$max_length		(optional) max length for the string to be considered valid
	 * @param	int		$min_length		(optional) min length for the string to be considered valid
	 * 
	 * @return 	bool
	 * 
	 */
	public static function isValidString($string, $max_length = null, $min_length = null)
	{
	
			
		if($max_length)
		{
			
			if (strlen($string) > $max_length)
			{
				return false;
			}	
		}
		
		if($min_length)
		{
			
			if (strlen($string) < $min_length)
			{
				return false;
			}				
		}
		
		return is_string($string);
		
	}

	
	/**
	 * Determine if input is a valid numeric string.  Min and Max length are optional parameters.
	 * 
	 * Checks to see if the $string variable contains a valid numeric   
	 * value.  $max_length & $min_length are optional arguments and will
	 * return false if string is longer than $max_length or shorter than 
	 * $min_length.
	 * 
	 * @param	string		$string			Value to be validated
	 * @param	integer		$max_length		(Optional) returns false if string longer than $max_length
	 * @param	integer		$min_length		(Optional) returns false if string shorter than $min_length
	 * 
	 * @return boolean		True if string's length is within $max_length and $min_length (if passed) and string is numeric, otherwise false
	 * 
	 */
	public static function isValidNumericString($string, $max_length = null, $min_length = null)
	{
		if($max_length)
		{
			if (strlen($string) > $max_length)
			{
				return false;
			}	
		}
		
		if($min_length)
		{
			if (strlen($string) < $min_length)
			{
				return false;
			}				
		}
		
		if( is_numeric($string) )
		{
			return true;
			
		} else {
		
			return false;
			
		}
		
	}

	
	/**
	 * Determine if input is a valid date.  
	 * 
	 * This method takes a date string and trys to determine if it represents a valid date.  Note that the year must 
	 * be in 4 digit format or false will be returned.
	 * 
	 * @param	string	$date_string - a string representing a date - almost any date format will work
	 * 
	 * @return	boolean	True if valid date was passed to method, false otherwise
	 * 
	 */
	public static function isValidDate($date_string)
	{
		if( @strtotime($date_string) )
		{
			return true;
		}
		
		return false;
	}
	
	/**
	 * Determine if input is a valid integer.  Min and Max values are optional parameters.
	 * 
	 * This function checks if the number provided is a valid integer.  Further, you can
	 * optionally check to see if the number provided is within a certain range.  If
	 * numeric values are provided for $min_val and / or $max_val, the number passed to 
	 * the function will be compared and, if not within the defined range, will return
	 * false.  Otherwise, the function will return true.
	 * 
	 * @param	mixed	$num		Value that we want to validate as an integer
	 * @param	int		$max_val	(optional) Maximum acceptable value for $num
	 * @param	int		$min_val	(optional)Minimum acceptable value for $num
	 * 
	 * @return	bool
	 * 
	 */	
	
	public static function isValidInt($num, $max_val = null, $min_val = null)
	{
		if( !is_numeric($num) )
		{
			return false;
		}
		
		if( is_numeric( $min_val ) )
		{
			if ($num < $min_val)
			{
				return false;
			}
		}
		
		if( is_numeric( $max_val ) )
		{
			if ($num > $max_val)
			{
				return false;
			}
		}
				
		$input = (int)$num;
		
		if( is_int($input) )
		{
			return true;
		}
		
		return false;
	}	

	/**
	 * Determine if input is a valid integer.  See isValidInt()
	 * 
	 * This is just an alias for isValidInt().  It does nothing but return
	 * the result of isValidInt().
	 * 
	 * @param	mixed	$num		Value that we want to validate as an integer
	 * @param	int		$max_val	(optional) Maximum acceptable value for $num
	 * @param	int		$min_val	(optional)Minimum acceptable value for $num 
	 * 
	 */
	public static function isValidInteger($num, $max_val = null, $min_val = null)
	{
	
		return Security::isValidInt($num, $max_val, $min_val);
	
	}	
	

	
	
	/**
	 * Determine if input is a valid float.  Min and Max value are optional parameters.
	 * 
	 * This function checks if the number provided is a valid float.  Further, you can
	 * optionally check to see if the number provided is within a certain range.  If
	 * numeric values are provided for $min_val and / or $max_val, the number passed to 
	 * the function will be compared and, if not within the defined range, will return
	 * false.  Otherwise, the function will return true.
	 * 
	 * @param	mixed	$num			Value that we want to validate as a float
	 * @param	int	$max_val	 	(optional) Maximum acceptable value for $num 
	 * @param	int	$min_val			(optional)Minimum acceptable value for $num
	 * 
	 * @return	bool
	 * 
	 */	
	public static function isValidFloat($num, $max_val = null, $min_val = null)
	{
		if( !is_numeric($num) )
		{
			return false;
		}
		
		if( is_numeric( $min_val ) )
		{
			if ($num < $min_val)
			{
				return false;
			}
		}
		
		if( is_numeric( $max_val ) )
		{
			if ($num > $max_val)
			{
				return false;
			}
		}
				
		$input = (float)$num;
		
		if( is_float($input) )
		{
			return true;
		}
		
		return false;
	}		
	
	
	/**
	 * Determine if input is a valid boolean.  
	 * 
	 * This method checks to see if a value is of type boolean.
	 * 
	 * @param	mixed	$val - Value that we want to validate as a boolean (true or false)
	 * 
	 * @return	bool
	 */
	public static function isValidBool($val)
	{
		return is_bool($val);
	}	

	
	
	
	
	/**
	 * Determine if input is a valid boolean.  See isValidBool()
	 * 
	 * This is just an alias for isValidBool().  It will do nothing more than
	 * return the result of isValidBool().
	 * 
	 * @param	mixed	$val - Value that we want to validate as a boolean (true or false)
	 * 
	 * @return	bool
	 * 
	 */
	public static function isValidBoolean($val)
	{
		return Security::isValidBool($val);
	}	
	
	
	/**
	 * Determine if input is a valid number.  Min and max values are optional parameters.  
	 * 
	 * This function checks if the number provided is a valid number (int or float).  Further, you can
	 * optionally check to see if the number provided is within a certain range.  If
	 * numeric values are provided for $min_val and / or $max_val, the number passed to 
	 * the function will be compared and, if not within the defined range, will return
	 * false.  Otherwise, the function will return true.
	 * 
	 * @param	mixed	$num		Value that we want to validate as a number (int or float)
	 * @param	int		$max_val	(optional) Maximum acceptable value for $num 
	 * @param	int		$min_val	(optional)Minimum acceptable value for $num
	 * 
	 * @return	bool
	 * 
	 */		
	public static function isValidNumber($num, $max_val = null, $min_val = null)
	{
		if( is_numeric($min_val) )
		{
			if($num < $min_val)
			{
				return false;
			}
		}
		
		if( is_numeric($max_val) )
		{
			if($num > $max_val)
			{
				return false;
			}
		}

		if(is_numeric($num) )
		{
			return true;
			
		} else {
		
			return false;
			
		}
	
	}
	
	
	/****************************************************************************************************/
	/**/
	/**/// Special Validation Methods - These methods will validate specific data types
	/**/	
	/****************************************************************************************************/	
	
	/**
	 * Determine if input is a valid employee ID.  
	 * 
	 * Validates an employeeId which is always a 6 character string containing all numeric values.
	 * Returns false if it contains any non-numeric values or is greater than or less than 6
	 * characters long. 
	 * 
	 * @param	string	$unvalidated - Unvalidated Employee ID
	 * 
	 * @return	boolean	$unvalidated - True if valid data, False if not
	 * 	
	 */
	public static function isValidEmployeeID($unvalidated)
	{
		return Security::isValidNumericString($unvalidated, 6, 6);
	}
	
		
	/**
	 * Determine if input is a valid e-mail address.  
	 * 
	 * Validates an e-mail address to ensure that it is syntactically correct
	 * 
	 * @param	string	$unvalidated -  e-mail address
	 * 
	 * @return	boolean	True if valid data, False if not
	 * 
	 */
	public static function isValidEmail($unvalidated)
	{
	
		 return filter_var( $unvalidated, FILTER_VALIDATE_EMAIL );
	
	}	
	
	
	/**
	 * Determine if input is a valid URL.  
	 * 
	 * Validates a URL to ensure that it is syntactically correct
	 * 
	 * @param	string	$unvalidated - URL
	 * 
	 * @return	boolean	True if valid data, False if not
	 * 
	 */	
	public static function isValidURL($unvalidated)
	{
		/*
		$pattern = '/^(([\w]+:)?\/\/)?(([\d\w]|%[a-fA-f\d]{2,2})+(:([\d\w]|%[a-fA-f\d]{2,2})+)?@)?([\d\w][-\d\w]{0,253}[\d\w]\.)+[\w]{2,4}(:[\d]+)?(\/([-+_~.\d\w]|%[a-fA-f\d]{2,2})*)*(\?(&amp;?([-+_~.\d\w]|%[a-fA-f\d]{2,2})=?)*)?(#([-+_~.\d\w]|%[a-fA-f\d]{2,2})*)?$/';
		$pattern = '/^([https]|[http]|[ftp]|[file])?://[a-z0-9-]+(.[a-z0-9-]+)*(:[0-9]+)?(/.*)?$/i';
		$pattern = '/^(http|https|ftp)://([A-Z0-9][A-Z0-9_-]*(?:.[A-Z0-9][A-Z0-9_-]*)+):?(d+)?/?/i';
		$pattern = "|^(((http|https|ftp):\/\/)?([[a-zA-Z0-9]\-\.])+(\.)([[a-zA-Z0-9]]){2,4}([[a-zA-Z0-9]\/+=%&_\.~?\-]*))*$|";
		$pattern = "/((http|https|ftp|file)\:\/\/)?([a-zA-Z0-9\-\.])+(\.)/i";
		//$pattern = "/((http|https|ftp|file)\:\/\/)?([a-zA-Z0-9\.\-\_])+(\.)/i";
		$pattern = "/((http|https|ftp|file)\:\/\/)?([a-z0-9\.\-]){1,1000}/";
		return preg_match ( $pattern, $unvalidated);	
		
		//return preg_match ( '|^http(s)?://[a-z0-9-]+(.[a-z0-9-]+)*(:[0-9]+)?(/.*)?$|i', $unvalidated);
		//return preg_match ( '/[^a-z\-_0-9\.:&?=#\/]/i' , $unvalidated );
		*/
	
		/*
		//Because FILTER_VALIDATE_URL returns false if the protocol is not included,
		//we'll append it before validating the URL
		if( !preg_match( '/^((http|https|ftp|file)\:\/\/)/', 'ftp://www.yahoo.com' ) )
		{
			$unvalidated = 'http://' . $unvalidated;
		}
		*/
	
		$unvalidated = str_replace(" ", "+", $unvalidated);
		
		if(filter_var($unvalidated, FILTER_VALIDATE_URL))
		{
			return true;
		}
		
		return false;
	}	
	
	
	/**
	 * Determine if input is a valid IP address.  
	 * 
	 * Validates an IP address to ensure that it is syntactically correct
	 * 
	 * @param	string	$unvalidated - IP Address
	 * 
	 * @return	boolean	True if valid data, False if not 
	 * 
	 */	
	public static function isValidIP($unvalidated)
	{
		return filter_var($unvalidated, FILTER_VALIDATE_IP);
	}	
		
	
	
	/**
	 * Determine if input is a valid serialized array.  
	 * 
	 * Validates a string to determine if it is a serialized array.
	 * 
	 * @param	string	$unvalidated string to be validated
	 * 
	 * @return	boolean	True if valid serialized array, False if not 
	 * 
	 */	
	public static function isValidSerializedArray($unvalidated)
	{
		// Prepend the "@" symbol to the unserialize function so that
		// notices are suppressed. 	
		if( !$ar = @unserialize($unvalidated) )
		{
			return false;
		}
		
		return is_array($ar);
	}	
	
	
	
	
	
	
	/****************************************************************************************************/
	/**/
	/**/// Generic Modification Methods - These methods will actually modify data
	/**/	
	/****************************************************************************************************/

	
	/**
	 * Wrapper for the sanitizeString method that defines by default the characters to be ignored by htmlentities()
	 * 
	 * 
	 */
	public static function sanitizeSearchString($to_be_sanitized)
	{
		$ignore_these_chars = array('&');
	
		return Security::sanitizeString( $to_be_sanitized, $ignore_these_chars ); 
	
	}
	
	/**
	 * 
	 * This function takes a string and sanitizes it so that it will be safe to use on a website or as part of a SQL string.
	 * 
	 * Currently it converts the charset to UTF-8 and then uses htmlentities() to clean up any html or script tags.  It then
	 * uess addcslashes to remove the same set of characters removed by mysql_escape_real_string.
	 * 
	 * @param	string	$to_be_sanitized - data to be sanitized
	 * @param	string	$ignore_these_chars - characters that shouldn't be entityized should be passed here as an simple array
	 * 
	 * @return	string	Returns sanitized version of $to_be_sanitized
	 * 
	 * 
	 */
	public static function sanitizeString($to_be_sanitized, $ignore_these_chars = null){
		
		if( is_numeric($to_be_sanitized) )
		{
			return $to_be_sanitized;
		}	
		
		//hokay, I'm not sure why we were sanitizing so crazily before, but we don't actually want to encode all the characters.  
		//for now, let's just default to running xss_clean on strings -- MG 2015-09-04
		return get_instance()->security->xss_clean($to_be_sanitized);
		
		
		
		
		// Default to all of the original chars of interest
		$targetChars = "\x00\x0A\x0D\x1A\x22\x27\x5C";
		
		if (preg_match('/[\\][\']/', $to_be_sanitized)){
		
			// add the single quote and backslash to the chars to ignore array so they don't get entityized
			$ignore_these_chars[] = '\\';
			$ignore_these_chars[] = '\'';

			// remove the single quote and backslash from the chars to run addcslashes against since those two chars are already escaped
			$targetChars = "\x00\x0A\x0D\x1A\x22";
		}
		
		$sanitized = mb_convert_encoding($to_be_sanitized, 'UTF-8', 'UTF-8');
		
		$sanitized = htmlentities($sanitized, ENT_QUOTES, 'UTF-8', false);
		
		// Old way which is now commented out
		//$sanitized = addcslashes($sanitized, "\x00\x0A\x0D\x1A\x22\x27\x5C");
		
		// New way 
		$sanitized = addcslashes($sanitized, $targetChars);

		/*
		$sanitized = addcslashes($sanitized, "\x00");	// 0 - NUL
		$sanitized = addcslashes($sanitized, "\x0A");	// 10 - new line
		$sanitized = addcslashes($sanitized, "\x0D");	// 13 - car ret
		$sanitized = addcslashes($sanitized, "\x1A");	// 26 - substitue
		$sanitized = addcslashes($sanitized, "\x22");	// 34 - double quote (")
		$sanitized = addcslashes($sanitized, "\x27");	// 39 - single quote (')
		$sanitized = addcslashes($sanitized, "\x5c");	// 92 - backslash (\)
		*/

		// If developer has passed an array of special characters that shouldn't be entityized,
		// they'll be replaced here.
		if( $ignore_these_chars ) 
		{
			if( !is_array($ignore_these_chars) )
			{
				$ignore_these_chars = array($ignore_these_chars);
			}

			foreach($ignore_these_chars as $char)
			{
				$sanitized = str_replace(Security::getEntity($char), $char, $sanitized);
			}	
		} 
		return $sanitized;
	}	
	
	
	
	/**
	 * 
	 * Sanitizes a string so that it will be safe to display on a website or run as part of a SQL string.
	 * 
	 * The difference between this method and sanitizeString() is that this one does allow HTML tags that aren't excluded by 
	 * default (<script>, <style>, <vbscript>). 
	 * 
	 * Currently it converts the charset of the $to_be_sanitized input variable to UTF-8 and then strips the default tags.
	 * 
	 * @param	string	$to_be_sanitized is the html string to be sanitized
	 * 
	 * @return string	$sanitized is the value after the original input variable has been sanitized
	 * 
	 */	
	public static function sanitizeHTMLString($to_be_sanitized)
	{
	
		if( is_numeric($to_be_sanitized) )
		{
			return $to_be_sanitized;
		}
	
		$sanitized = mb_convert_encoding($to_be_sanitized, 'UTF-8', 'UTF-8');
		
		$sanitized = Security::stripSelectedTags($sanitized);
		// $sanitized = htmlentities($sanitized, ENT_QUOTES, 'UTF-8');
			
		//$sanitized = addcslashes($sanitized, ";%_");
		//$sanitized = addcslashes($sanitized, "\0\10\13\34\39");

		
		return $sanitized;
	
	}


	/**
	 * 
	 * Sanitizes a string using HTMLPurifyer so that it will be safe to display on a website or run as part of a SQL string.
	 * 
	 * The difference between this method and sanitizeHTMLString() is that this one uses HTMPurifier which is an OS HTML sanitization library
	 * 
	 * that is significantly more powerful than sanitizeHTMLString().  It is also slower and more resourse intensive. 
	 * 
	 * 
	 * @param	string	$to_be_sanitized is the html string to be sanitized
	 * 
	 * @return string	$sanitized is the value after the original input variable has been sanitized
	 * 
	 */	
	 
	public static function sanitizeHTMLPurifierString($to_be_sanitized)
	{
	
		if( is_numeric($to_be_sanitized) )
		{
			return $to_be_sanitized;
		}
	
		$purifier = new HTMLPurifier();

		$sanitized = $purifier->purify( $to_be_sanitized );
		
		return $sanitized;
	
	}
	
	/**
	 * Sanitize a variable that is supposed to be an integer
	 * 
	 * Will return integer representation of whatever is passed to function.  If not a numeric value, "0" will probably be returned.
	 * 
	 * @param	string	$to_be_sanitized - data to be sanitized
	 * 
	 * @return	string	Returns sanitized version of $to_be_sanitized
	 *  
	 */
	public static function sanitizeInt($to_be_sanitized)
	{
		return (int)$to_be_sanitized;		
	}
	
	
	/**
	 * Sanitize a variable that is supposed to be an integer - See sanitizeInt()
	 * 
	 * Will return integer representation of whatever is passed to function.  If not a numeric value, "0" will probably be returned.
	 * ***Added to correct naming inconsistency - will just return the results of sanitizeInt() *** 
	 * 
	 * @param	string	$to_be_sanitized - data to be sanitized
	 * 
	 * @return	string	Returns sanitized version of $to_be_sanitized
	 *  
	 */	
	public static function sanitizeInteger($to_be_sanitized)
	{
		return Security::sanitizeInt($to_be_sanitized);		
	}	
	

	/**
	 * Sanitize a variable that is supposed to be an float
	 * 
	 * Will return float representation of whatever is passed to function.  If not a numeric value, "0" will probably be returned.
	 * 
	 * @param	string	$to_be_sanitized - data to be sanitized
	 * 
	 * @return	string	Returns sanitized version of $to_be_sanitized
	 *  
	 */
	public static function sanitizeFloat($to_be_sanitized)
	{
		return (float)$to_be_sanitized;		
	}
	
	
	/**
	 * Sanitize a variable that is supposed to be an number (regardless of whether it is an int or a float)
	 * 
	 * Will return numeric representation of whatever is passed to function.  If not a numeric value, "0" will probably be returned.
	 * 
	 * @param	mixed	$to_be_sanitized - data to be sanitized
	 * 
	 * @return	mixed	Returns sanitized version of $to_be_sanitized
	 *  
	 */
	public static function sanitizeNumber($to_be_sanitized)
	{
		if( !is_numeric($to_be_sanitized) )
		{
			return false;
		}
		
		if( filter_var($to_be_sanitized, FILTER_VALIDATE_INT) )
		{
			//echo 'returning int';
			return (int)$to_be_sanitized;
		}
		elseif (filter_var($to_be_sanitized, FILTER_VALIDATE_FLOAT))
		{
			//echo 'returning float';
			return (float)$to_be_sanitized;
		}
		else
		{
			return false;
		}
				
	}

	
	/**
	 * Sanitize a variable that is supposed to be a boolean value
	 * 
	 * Will return Boolean representation of whatever is passed to function.
	 * 
	 * @param	string	$to_be_sanitized - data to be sanitized
	 * 
	 * @return	string	Returns sanitized version of $to_be_sanitized
	 *  
	 */
	public static function sanitizeBool($to_be_sanitized)
	{
		return (boolean)$to_be_sanitized;		
	}	
	
	/**
	 * Sanitize a variable that is supposed to be a boolean value - see sanitizeBool()
	 * 
	 * Will return Boolean representation of whatever is passed to function.
	 * *** Added to fix naming inconsistency - Just returns the result of sanitizeBool() ***
	 * 
	 * @param	string	$to_be_sanitized - data to be sanitized
	 * 
	 * @return	string	Returns sanitized version of $to_be_sanitized
	 */
	public static function sanitizeBoolean($to_be_sanitized)
	{
		return Security::sanitizeBool($to_be_sanitized);		
	}	
	

	/**
	 * 
	 * "Sanitize" a serialized array.  
	 * 
	 * But we can't really sanitize serialized arrays because these data structures 
	 * are way too sensitive.  If modified incorrectly, it could easily become
	 * corrupted.  So, unlike the other sanitization methods, this method will
	 * check to ensure that the unvalidated string is indeed a serialized array.
	 * And if it is, by default, the serialized array will be returned without 
	 * modification.
	 * 
	 * However, if true is passed to the $sanitize argument, this method will unserialize
	 * the first argument, validate and sanitize each element of the array using the
	 * default sanitization & validation methods, and return the sanitized serialized 
	 * array.
	 * 
	 * ***Note that this method is a bit redundant because it is validating even
	 * though it is a sanitization method.  But to make it fit into our security
	 * strategy, this is how it has to be done.
	 * 
	 * @param	string	$unvalidated - Unvalidated serialized array
	 * @param	boolean	$sanitize - defaults to false.  If true, $unvalidated will be 
	 * 								unserialized and sanitized using the default 
	 * 								sanitization method
	 * 
	 * @return	string	Confirmed but unvalidated serialized array 
	 * 
	 */	
	public static function sanitizeSerializedArray($unvalidated, $sanitize = false)
	{
		if( Security::isValidSerializedArray($unvalidated) )
		{
			if( $sanitize )
			{
				//Unserialize the initial argument and copy it to a new array - $my_array
				$my_array = unserialize($unvalidated);

				//Apply the default sanitization function to all elements of this array
				if( $my_array = Security::secureArray($my_array) )
				{
					//array was successfully sanitized.  Reserialize and return array
					return serialize($my_array);
				}
				else
				{
					//Should we return the initial array if array_walk_recursive returns false?
					//Or should we return false because it wasn't walked successfully?
					return $unvalidated;
					//return false;
				}
			}
			else
			{
				return $unvalidated;
			}
		}
		else
		{
			return false;
		}
	}	
	
	


	
	
	/****************************************************************************************************/
	/**/
	/**/// Special Modification Methods - These methods will modify specific data types
	/**/	
	/****************************************************************************************************/	
	
	/**
	 * Sanitize a variable that is supposed to be a URL
	 * 
	 * This method will strip out any characters that are not considered valid in a URL.  It first checks to see if it is a valid URL
	 * and then strips out invalid characters.
	 * 
	 * @param	string	$to_be_sanitized - data to be sanitized
	 * 
	 * @return	string	Returns sanitized version of $to_be_sanitized
	 * 
	 */
	public static function sanitizeUrl($to_be_sanitized)
	{
		if( Security::isValidURL($to_be_sanitized) )
		{
			return preg_replace('/[^a-z\-\+\_0-9\.:~&%?=#\/\s]/i','',$to_be_sanitized);
		}
		
		return false;
	}  
		

	/**
	 * Sanitize a variable that is supposed to be an employee ID
	 * 
	 * Will strip out any non-numeric characters and truncate result to a 
	 * length of 6 characters.  This function does not guarantee a vaild
	 * employee ID but rather a syntactically correct employee ID;  6 
	 * characters, all numeric.
	 * 
	 * @param	string	$to_be_sanitized - data to be sanitized
	 * 
	 * @return	string	Returns sanitized version of $to_be_sanitized
	 * 
	 */
	public static function sanitizeEmployeeID($to_be_sanitized)
	{
		//return preg_replace('/[^a-z\-_0-9\.:\/]/i','',$to_be_sanitized);
		return substr(preg_replace('/\D/', '', $to_be_sanitized), 0, 6);
	}  
	
	
	/**
	 * Sanitize a variable that is supposed to be a numeric string
	 * 
	 * Will strip out any non-numeric characters and return whatever is
	 * left over as a string.  
	 * 
	 * @param	string	$to_be_sanitized - data to be sanitized
	 * 
	 * @return	string	Returns sanitized version of $to_be_sanitized
	 * 
	 */
	public static function sanitizeNumericString($to_be_sanitized)
	{
		if( is_numeric($to_be_sanitized) )
		{
			return $to_be_sanitized;
		}
		
		//return preg_replace('/[^a-z\-_0-9\.:\/]/i','',$to_be_sanitized);
		return preg_replace('/\D/', '', $to_be_sanitized);
	}  
	
	
	
	/**  
	 * Strip selected HTML tags from string.
	 * 
	 * This function strips $tags from $str and has the option to strip the internal content of the tags too
	 *     Use it like this:
	 *        $tagsAndContentToRemove="<script><style><vbscript>";		
	 *        $str = stripSelectedTags($str, $tagsAndContentToRemove, true);
	 *        
	 * @param string $str is the string to clean of tags
	 * @param string $tags are the tags to be striped from $str
	 * @param bool $stripContent specifies whether or not to strip the content between the tags too
	 * 
	 * @return string $str - return the string that has been cleaned of all tags
	 * 
	 */
	public static function stripSelectedTags($str, $tags = "<script><style><vbscript><link><meta><iframe>", $stripContent = false)
	{
		preg_match_all("/<([^>]+)>/i",$tags,$allTags,PREG_PATTERN_ORDER);
		
		foreach ($allTags[1] as $tag)
		{
			if ($stripContent) 
			{
				$str = preg_replace("/<".$tag."[^>]*>.*<\/".$tag.">/iUs","",$str);
				// Strip out encoded tags.  For example: &lt;script type="javascript"&gt;This is a test.&lt;/script&gt; | will be completely blanked out
				$str = preg_replace("/(&lt;)".$tag."*?(&gt;)?.*(&lt;)\/".$tag."(&gt;)/iUs","",$str);
			}
			$str = preg_replace("/<\/?".$tag."[^>]*>/iUs","",$str);
			// Strip out the encoded tags but not the content inbetween.  
			// For example: &lt;script type="javascript"&gt;This is a test.&lt;/script&gt; | will be replaced with: This is a test.
			$str = preg_replace("/(&lt;)\/?".$tag."?.*(&gt;)/iUs","",$str);
		}
		return $str;
	}
	
	

	




	
	/****************************************************************************************************/
	/**/
	/**/// Addtional Methods - These methods are used from within the class in order to simplify the code
	/**/	
	/****************************************************************************************************/	
	



	/**
	 * Executes the provided validation function against the provided input
	 * 
	 * This method executes the validation method passed to it and returns true or false
	 * depending on the results of the validation.
	 * 
	 * @param	string	$fx - method or function to be executed - If null or not valid function default method will be run
	 * @param	array	$input - All input variables for selected method
	 * 
	 * @return	boolean	True if validation passes, False if validation fails.
	 * 
	 */
	//private function execValidation( $fx, &$input )
	protected static function execValidation( $fx = null, $input ){

		
	
	
		// make sure our validation funtion exists
		if( is_null($fx) )
		{	
			$fx = Security::$default_validation_function;
		}
		
		if( !function_exists( $fx ) && !is_callable( $fx ) )
		{	
			$fx = Security::$default_validation_function;
		}
		
		//make sure $value is an array so that we can utilize call_user_func_array
		//rather than call_user_func()
		if ( !is_array($input) )
		{
			$input = array($input);
		}

		if( call_user_func_array($fx, $input) )
		{

			return true;
			
		}
		else
		{

			return false;
		}
			

			
	}	
	
	/**
	 * Executes the provided sanitization function against the provided input
	 * 
	 * This method executes the sanitization method passed to it and returns $value variable after 
	 * it has been run through the selected sanitization method.
	 * 
	 * @param	string	$fx - method or function to be executed - If null or not valid function default method will be run
	 * @param	string	$input - data to be validated
	 * 
	 * @return	mixed	Returns sanitized version of $value after being run through the selected method $fx 
	 * 
	 */	
	protected static function execSanitization( $fx = null, $input ){

		// make sure our validation funtion exists
		if( is_null($fx) )
		{	
			$fx = Security::$default_sanitization_function;
		}
			
		if( !function_exists( $fx ) && !is_callable( $fx ) )
		{	
			$fx = Security::$default_sanitization_function;
		}
		
	
		if ( !is_array($input) )
		{
			$input = array($input);
		}
		
		$value = call_user_func_array( $fx, $input );
		
	
		return $value;	
					
		
	}	
		

	/**
	 * Prepares supplied data so that it is ready to be run through execSanitization() or execValidation()
	 * 
	 * replaces the placeholder ('?') with the $subject if placeholder exists.  Otherwise, prepends $arguments
	 * array with $subject, making it the first element in the array.
	 * 
	 * @param	mixed	$subject	Item to be added to the $arguments array
	 * @param	array	$arguments	Array containing a list of arguments
	 * 
	 * @return	array	Modified array that now contains the $subject
	 * 
	 */
	private function prepareArguments($subject, array $arguments)
	{
	
	
		//	if the developer has explicitly defined the order of the arguments for this 
		//	method by passing a "?" as a placeholder in the arguments array, replace the 
		//	"?" with the value of the string to be validated.
		if(in_array('?', $arguments))
		{

			//Replace '?' with $subject
			$arguments[array_search('?', $arguments)] = $subject;
			
			//Sort array by key
			ksort($arguments);
		}
		else
		{
			//	The developer didn't explicitly define the order of the arguments for this method
			//	so we have to assume that the string to be validated should be the first argument.
			
			//Set the data to be validated as the first element in the arguments array
			array_unshift($arguments, $subject);						
			
		}
	
		
		return $arguments;		
	
	}			
	
	/**
	 * Get the HTML entity value of any ascii character
	 * 
	 * This method takes a character as input and returns the entityized representation of that character.  If
	 * HTML has a special HTML code defined for this character (i.e. &amp; instead of &#038;), the special code
	 * will be returned.  Otherwise, the generic HTML code (i.e. - &#038;) will be returned.  This is primarily
	 * being used in sanitizeString() as this method is using htmlentities() to sanitize the input.  In certain
	 * circumstances, characters should not be converted to their HTML entity so this function helps reverse
	 * the effects of htmlentities().  
	 * 
	 * @param	string	$char - Any valid ascii character.
	 * 
	 * @return	string	An HTML entity representation of $char
	 * 
	 */
	public static function getEntity($char)
	{
		$entities = array(
			"0"=>"&#000;",
			"1"=>"&#001;",
			"2"=>"&#002;",
			"3"=>"&#003;",
			"4"=>"&#004;",
			"5"=>"&#005;",
			"6"=>"&#006;",
			"7"=>"&#007;",
			"8"=>"&#008;",
			"9"=>"&#009;",
			"10"=>"&#010;",
			"11"=>"&#011;",
			"12"=>"&#012;",
			"13"=>"&#013;",
			"14"=>"&#014;",
			"15"=>"&#015;",
			"16"=>"&#016;",
			"17"=>"&#017;",
			"18"=>"&#018;",
			"19"=>"&#019;",
			"20"=>"&#020;",
			"21"=>"&#021;",
			"22"=>"&#022;",
			"23"=>"&#023;",
			"24"=>"&#024;",
			"25"=>"&#025;",
			"26"=>"&#026;",
			"27"=>"&#027;",
			"28"=>"&#028;",
			"29"=>"&#029;",
			"30"=>"&#030;",
			"31"=>"&#031;",
			"32"=>"&#032;",
			"33"=>"&#033;",
			"34"=>"&quot;",
			"35"=>"&#035;",
			"36"=>"&#036;",
			"37"=>"&#037;",
			"38"=>"&amp;",
			"39"=>"&#039;",
			"40"=>"&#040;",
			"41"=>"&#041;",
			"42"=>"&#042;",
			"43"=>"&#043;",
			"44"=>"&#044;",
			"45"=>"&#045;",
			"46"=>"&#046;",
			"47"=>"&#047;",
			"48"=>"&#048;",
			"49"=>"&#049;",
			"50"=>"&#050;",
			"51"=>"&#051;",
			"52"=>"&#052;",
			"53"=>"&#053;",
			"54"=>"&#054;",
			"55"=>"&#055;",
			"56"=>"&#056;",
			"57"=>"&#057;",
			"58"=>"&#058;",
			"59"=>"&#059;",
			"60"=>"&lt;",
			"61"=>"&#061;",
			"62"=>"&gt;",
			"63"=>"&#063;",
			"64"=>"&#064;",
			"65"=>"&#065;",
			"66"=>"&#066;",
			"67"=>"&#067;",
			"68"=>"&#068;",
			"69"=>"&#069;",
			"70"=>"&#070;",
			"71"=>"&#071;",
			"72"=>"&#072;",
			"73"=>"&#073;",
			"74"=>"&#074;",
			"75"=>"&#075;",
			"76"=>"&#076;",
			"77"=>"&#077;",
			"78"=>"&#078;",
			"79"=>"&#079;",
			"80"=>"&#080;",
			"81"=>"&#081;",
			"82"=>"&#082;",
			"83"=>"&#083;",
			"84"=>"&#084;",
			"85"=>"&#085;",
			"86"=>"&#086;",
			"87"=>"&#087;",
			"88"=>"&#088;",
			"89"=>"&#089;",
			"90"=>"&#090;",
			"91"=>"&#091;",
			"92"=>"&#092;",
			"93"=>"&#093;",
			"94"=>"&#094;",
			"95"=>"&#095;",
			"96"=>"&#096;",
			"97"=>"&#097;",
			"98"=>"&#098;",
			"99"=>"&#099;",
			"100"=>"&#100;",
			"101"=>"&#101;",
			"102"=>"&#102;",
			"103"=>"&#103;",
			"104"=>"&#104;",
			"105"=>"&#105;",
			"106"=>"&#106;",
			"107"=>"&#107;",
			"108"=>"&#108;",
			"109"=>"&#109;",
			"110"=>"&#110;",
			"111"=>"&#111;",
			"112"=>"&#112;",
			"113"=>"&#113;",
			"114"=>"&#114;",
			"115"=>"&#115;",
			"116"=>"&#116;",
			"117"=>"&#117;",
			"118"=>"&#118;",
			"119"=>"&#119;",
			"120"=>"&#120;",
			"121"=>"&#121;",
			"122"=>"&#122;",
			"123"=>"&#123;",
			"124"=>"&#124;",
			"125"=>"&#125;",
			"126"=>"&#126;",
			"127"=>"&#127;",
			"128"=>"&#128;",
			"129"=>"&#129;",
			"130"=>"&#130;",
			"131"=>"&#131;",
			"132"=>"&#132;",
			"133"=>"&#133;",
			"134"=>"&#134;",
			"135"=>"&#135;",
			"136"=>"&#136;",
			"137"=>"&#137;",
			"138"=>"&#138;",
			"139"=>"&#139;",
			"140"=>"&#140;",
			"141"=>"&#141;",
			"142"=>"&#142;",
			"143"=>"&#143;",
			"144"=>"&#144;",
			"145"=>"&#145;",
			"146"=>"&#146;",
			"147"=>"&#147;",
			"148"=>"&#148;",
			"149"=>"&#149;",
			"150"=>"&#150;",
			"151"=>"&#151;",
			"152"=>"&#152;",
			"153"=>"&#153;",
			"154"=>"&#154;",
			"155"=>"&#155;",
			"156"=>"&#156;",
			"157"=>"&#157;",
			"158"=>"&#158;",
			"159"=>"&#159;",
			"160"=>"&nbsp;",
			"161"=>"&iexcl;",
			"162"=>"&cent;",
			"163"=>"&pound;",
			"164"=>"&curren;",
			"165"=>"&yen;",
			"166"=>"&brvbar;",
			"167"=>"&sect;",
			"168"=>"&uml;",
			"169"=>"&copy;",
			"170"=>"&ordf;",
			"171"=>"&laquo;",
			"172"=>"&not;",
			"173"=>"&shy;",
			"174"=>"&reg;",
			"175"=>"&macr;",
			"176"=>"&deg;",
			"177"=>"&plusmn;",
			"178"=>"&sup2;",
			"179"=>"&sup3;",
			"180"=>"&acute;",
			"181"=>"&micro;",
			"182"=>"&para;",
			"183"=>"&middot;",
			"184"=>"&cedil;",
			"185"=>"&sup1;",
			"186"=>"&ordm;",
			"187"=>"&raquo;",
			"188"=>"&frac14;",
			"189"=>"&frac12;",
			"190"=>"&frac34;",
			"191"=>"&iquest;",
			"192"=>"&Agrave;",
			"193"=>"&Aacute;",
			"194"=>"&Acirc;",
			"195"=>"&Atilde;",
			"196"=>"&Auml;",
			"197"=>"&Aring;",
			"198"=>"&AElig;",
			"199"=>"&Ccedil;",
			"200"=>"&Egrave;",
			"201"=>"&Eacute;",
			"202"=>"&Ecirc;",
			"203"=>"&Euml;",
			"204"=>"&Igrave;",
			"205"=>"&Iacute;",
			"206"=>"&Icirc;",
			"207"=>"&Iuml;",
			"208"=>"&ETH;",
			"209"=>"&Ntilde;",
			"210"=>"&Ograve;",
			"211"=>"&Oacute;",
			"212"=>"&Ocirc;",
			"213"=>"&Otilde;",
			"214"=>"&Ouml;",
			"215"=>"&times;",
			"216"=>"&Oslash;",
			"217"=>"&Ugrave;",
			"218"=>"&Uacute;",
			"219"=>"&Ucirc;",
			"220"=>"&Uuml;",
			"221"=>"&Yacute;",
			"222"=>"&THORN;",
			"223"=>"&szlig;",
			"224"=>"&agrave;",
			"225"=>"&aacute;",
			"226"=>"&acirc;",
			"227"=>"&atilde;",
			"228"=>"&auml;",
			"229"=>"&aring;",
			"230"=>"&aelig;",
			"231"=>"&ccedil;",
			"232"=>"&egrave;",
			"233"=>"&eacute;",
			"234"=>"&ecirc;",
			"235"=>"&euml;",
			"236"=>"&igrave;",
			"237"=>"&iacute;",
			"238"=>"&icirc;",
			"239"=>"&iuml;",
			"240"=>"&eth;",
			"241"=>"&ntilde;",
			"242"=>"&ograve;",
			"243"=>"&oacute;",
			"244"=>"&ocirc;",
			"245"=>"&otilde;",
			"246"=>"&ouml;",
			"247"=>"&divide;",
			"248"=>"&oslash;",
			"249"=>"&ugrave;",
			"250"=>"&uacute;",
			"251"=>"&ucirc;",
			"252"=>"&uuml;",
			"253"=>"&yacute;",
			"254"=>"&thorn;",
			"255"=>"&#255;"
		);
		
		if( array_key_exists(ord($char), $entities) )
		{
			return $entities[ ord($char) ];
		}
		
		return false;
		
	}

	
	

} 

