<?php

//dependencies:  inflector helper, icarus router, url param helper

/**
* @package icarus
* @subpackage libraries
*/

load_helpers('date', 'html');
load_libraries('object');

/**
* @package icarus
* @subpackage libraries
* @author M. Gibbs <gibbs_margaret@bah.com>
* @todo Add a way to specify partials instead of having to create the file directory the same way each time
*/
class Lister extends Object{
	
	/**
	* Reference to the Codeigniter object, set in the constructor.
	* @var CodeIgniter
	*/
	protected $CI;
	
	/**
	* Location to look for custom partials.
	* This isn't required.  If you're overriding the default views or adding list-specific views for columns, you can give us the location here.
	* This should be located in the application's views directory.
	* @var string|array
	*/
	protected $_template_directory; //
	
	/**
	* The array of items that this lister will display.
	* This should be an array of entities, arrays, or strings if you're using the lister default views.  Otherwise, just make sure that your custom views can handle it.
	* @var array
	*/
	protected $_list_items;
	
	/**
	* The number of items included in the list.
	* This number will be used to determine the number of pages displayed in the pagination markup.  By default this will be the number of items in 
	* {@link _list_items}, but you may set it to a larger number if the array you passed for {@link _list_items} was for a single page.
	* @var int
	*/
	protected $_total_items;
	
	/**	@var int */	
	protected $_items_per_page;
	
	/**	
	* The current page.
	* If not set, this will be set using {@link load_property_from_url}.
	* @var int 
	*/	
	protected $_page;
	
	/**	
	* The property/column/array key that the list items should be sorted by.
	* This will not be applied if list items are not an array or an object.
	* @var string
	*/	
	protected $_order_by;
	
	/**
	* Used if {@link _order_by} has not been set.
	* @var string
	*/
	protected $_default_order_by;
	
	/**
	* Direction in which the items should be sorted (ASC or DESC).
	* If this value is not set, {@link _default_order} will be used.
	* @var string
	*/
	protected $_order;
	
	/** 
	* Default value to be used if {@link _default_order} has not been set.
	* @var string
	*/
	protected $_default_order = 'asc';
	
	/**
	* Whether or not the lister should attempt to sort list items on its own.
	* You should disable this if you're passing items that have already been sorted (e.g., a db query).  Ordering items using the db will be more efficient.
	* 
	* Note that items will only be successfully sorted if you have a valid value for {@link _order} and {@link _order_by}, and if the values of {@link _list_items}
	* are something that the lister understands how to sort (scalars, arrays, or objects).
	*
	* @see items_for_page
	* @see sort_list_items
	* @see list_items_are_sortable
	* 
	* @var boolean
	*/
	protected $_list_items_need_to_be_sorted;
	
	protected $_partials = array(); //currently not accessible outside of the class
	
	//////////// url param vars ///////////////////
	
	/** @var string */
	protected $_url_param_prefix = '';
	
	/** @var string */
	protected $_url_param_for_page = 'page';
	
	/** @var string */
	protected $_url_param_for_order = 'order';
	
	/** @var string */
	protected $_url_param_for_order_by = 'order_by';
	
	protected $_url_param_for_items_per_page = 'items_per_page';
	
	
	//////////// table-related vars ///////////////////
	
	/**
	* Indicates the properties of the list items that should be displayed in the markup.
	*
	* By default, this is used for table columns, and won't be used by the default partials for the markup for an unordered list.  However, if you're writing 
	* a custom list partial for an object or array, this might still be useful to you.
	*
	* You can set all of the column names at once, or use the {@link add_column} and {@link add_columns} methods.
	*
	* @var array 
	*/
	protected $_columns; 
	
	/**
	* Array of custom aliases for the columns of this lister.
	* Formatted $column_name => $human_readable_alias.  It is not required to manually set aliases for columns; if no alias is set, the code will use the {@link humanize}
	* function to auto-generate an alias for a column.
	* @var array
	*/
	protected $_column_aliases = array(); 
	
	/** @var array */
	protected $_property_validation_rules= array( 'template_directory' => 'nonempty_string_with_no_whitespace|array_of_nonempty_string_with_no_whitespaces',
																								'items_per_page' => 'nonzero_unsigned_integer',
																								'list_items_need_to_be_sorted' => 'boolean',
																								'url_param_prefix' => 'nonempty_string_with_no_whitespace',
																								'url_param_for_page' => 'nonempty_string_with_no_whitespace',
																								'url_param_for_order' => 'nonempty_string_with_no_whitespace',
																								'url_param_for_order_by' => 'nonempty_string_with_no_whitespace',
																								);
	
	/**
	* Extends the parent constructor to instantiate {@link CI}.
	*/
	function __construct($values = array()){
		$this->CI = &get_instance();
		parent::__construct($values);
	}
	
	/**
	* @param string
	* @return string 
	*/
	public function url_for_items_per_page($items_per_page=null){
		if(!is_null($items_per_page) && !$this->is->nonzero_unsigned_integer($items_per_page)) return $this->error->should_be_null_or_a_nonzero_unsigned_integer($items_per_page);
		
		$this->CI->load->helper('url_parameter_helper');
		
		if(empty($items_per_page) || $items_per_page=='0') return unset_url_param($this->url_param_for_items_per_page);
		return set_url_param($this->url_param_for_items_per_page, $items_per_page);
	}
	
	/**
	* @param string
	* @return string 
	*/
	public function url_for_order_by($order_by=null){
		if(!is_null($order_by) && !$this->is_a_valid_name_for_a_column($order_by))
				return $this->error->should_be_null_or_a_column_name($order_by);
		
		$this->CI->load->helper('url_parameter_helper');
		if(!empty($order_by)){
			$params = $this->CI->uri->url_params_to_assoc(); //get the current parameters
			$params[$this->url_param_for_order_by] = $order_by;
			$params[$this->url_param_for_page] = 1;
			if($order_by == $this->order_by)
				$params[$this->url_param_for_order] = $this->reverse_order();
			else
				unset($params[$this->url_param_for_order]); //in geneneral, sort should just go for a default order.
			
			return set_url_params($params);
		}
		
		return unset_url_param($this->url_param_for_order_by);
	}	
	
	/**
	* @param int Page number
	* @return string URL for page
	*/
	public function url_for_page($page=null){
		if(!is_null($page) && !$this->is->nonzero_unsigned_integer($page)) return $this->error->should_be_null_or_a_nonzero_unsigned_integer($page);
		
		$this->CI->load->helper('url_parameter_helper');
		
		if(empty($page) || $page=='0') return unset_url_param($this->url_param_for_page);
		return set_url_param($this->url_param_for_page, $page);
	}
	
	//makes use of the icarus router
	/**
	* @param string Name of the property that we're grabbing from the URL
	*/
	protected function load_property_from_url($property){
		if(!$this->property_exists($property)) return $this->error->property_does_not_exist($property);
		
		$url_param_name = 'url_param_for_'.$property;		
		if(!$this->property_exists($url_param_name)) return $this->error->property_does_not_exist($url_param_name);
		if(!$this->is->nonempty_string($this->$url_param_name)) return $this->error->should_be_a_nonempty_string($this->$url_param_name);
		
		$CI = &get_instance();
		$CI->load->helper('url_parameter_helper');
		$url_params = $CI->uri->url_params_to_assoc();
		
		if(!array_key_exists($this->$url_param_name, $url_params)) return;
		
		$property_value = element($this->$url_param_name, $CI->uri->url_params_to_assoc());
		
		if($this->property_has_validation($property) && !$this->value_is_valid_for_property($property, $property_value))
			return unset_url_param_and_redirect($this->$url_param_name); //avoid warnings by just unsetting the param
		
		$this->$property = $property_value;	
		if(!isset($this->$property))
			return unset_url_param_and_redirect($this->$url_param_name); //$property_value was invalid & there's no default value
				
		if(method_exists($this, $property))
			$value_after_being_set = $this->$property();		
		else
			$value_after_being_set = $this->$property;	
			
		if(is_null($value_after_being_set ))
			return unset_url_param_and_redirect($this->$url_param_name); //$property_value was invalid & there's no default value	
		
		if($value_after_being_set != $property_value)
				return set_url_param_and_redirect($this->$url_param_name, $this->$property); //$property_value was invalid & there's a default value
		
		return true;  //$property was correctly set
	}	
	
	
	/**
	* The reverse of the current value of {@link _order}.
	* @see _order
	* @return string Either 'asc' or 'desc'
	*/
	public function reverse_order(){
		if($this->order == 'asc') return 'desc';
		return 'asc';
	}
	
	/**
	* Returns just the list items for the current page.
	* This will also attempt to sort the list items before returning them, unless the items are not sortable by the {@link list_items_are_sortable} or 
	* {@link list_items_need_to_be_sorted} is turned off.
	*
	* @todo should sorting take place when list_items is set instead?
	* @todo some sort of warning/notice/error if items are not sortable but list_items_need_to_be_sorted is true?
	* @uses pagination_offset
	* @uses list_items_need_to_be_sorted
	* @uses list_items_are_sortable
	* @return array 
	*/
	public function items_for_page(){		
		if($this->list_items_need_to_be_sorted && $this->list_items_are_sortable())
				$this->sort_list_items();
		
		if(!isset($this->items_per_page) || count($this->list_items) <= $this->items_per_page) return $this->list_items;
	
		if($this->is->nonzero_unsigned_integer($this->items_per_page))
				return array_slice($this->list_items, $this->pagination_offset(), $this->items_per_page);
		else
				return array_slice($this->list_items, $this->pagination_offset());	
	}
	
	
	public function pagination_offset(){
		$page_number = $this->page;
		
		//only verify against total pages if we actually have the means to determine total pages at this point
		//this function may be used to generate list items, and we don't want to count the total items before running that - might wreck db conditions set up by dev
		if(isset($this->total_items) || isset($this->list_items)){
				$total_pages = $this->total_pages;
				if(!empty($total_pages) && $page_number > $total_pages){
						$page_number = $total_pages;
				}
		}
		
		if($this->is->nonzero_unsigned_integer($this->items_per_page))
				return $this->items_per_page * ($page_number - 1);
		
		return 0;	//start at the beginning
	}
	
	/**
	* Returns true if the list items appear to be sortable.
	* @todo warnings, error messages?
	* @todo should this be separated out by type for better extensibility?
	* @return boolean
	*/
	public function list_items_are_sortable(){		
		if(!isset($this->list_items)) return false;
		
		if($this->property_is_empty('order')) return false; //all sorting requires a direction, asc/desc
		if($this->is->array_of_scalars($this->list_items)) return true; 
		
		if($this->property_is_empty('order_by')) return false; //everything besides scalars requires an order_by value
		#TODO - what if the arrays or the objects don't have the order_by value?
		return ($this->is->array_of_arrays($this->list_items) || $this->is->array_of_objects($this->list_items));
	}
	
	/**
	* Attempts to sort {@link _list_items}, if possible. 
	* Note that this will only work if all the list items are of the same sortable type (scalar, arrays, or objects).  Sorting scalars requires that we have a value for
	* {@link _order}; sorting arrays and objects requires that we have a value for both {@link _order} and {@link _order_by}.
	*
	* @todo this method is gigantic.  can we divvy it up?
	*/
	public function sort_list_items(){
		if($this->property_is_empty('order')) return $this->error->warning("I can't sort ".get_class($this).'::list_items without a value for '.get_class($this).'::order');
		if(!isset($this->list_items)) return $this->error->warning("I can't sort ".get_class($this).'::list_items until a value has been set for it.');
		if($this->property_is_empty('list_items')) return true;			
		
		$first = first_element($this->list_items); //we're assuming that all the list items are of the same basic type.  let's hope that's the case.
		
		if(is_scalar($first)){
				if(!$this->is->array_of_scalars($this->list_items)) 
						return $this->error->warning("I can't sort ".get_class($this).'::list_items unless all the items are being sorted by the same criteria');	
				
				if($this->order == 'asc')
						$this->list_items = asort($this->list_items);
				else
						$this->list_items = arsort($this->list_items);
				return true;
		}
		
		$this->CI->load->helper('sort');
		
		if(is_array($first)){
				if(!$this->is->array_of_arrays($this->list_items)) 
						return $this->error->warning("I can't sort ".get_class($this).'::list_items unless all the items are being sorted by the same criteria');
				
				if($this->property_is_empty('order_by')) return $this->error->warning("I can't sort an array of arrays without a value for ".get_class($this).'::order_by');
				
				if($this->order == 'asc')
						$this->list_items = sort_arrays_by_key($this->list_items, $this->order_by);
				else
						$this->list_items = rsort_arrays_by_key($this->list_items, $this->order_by);
				return true;		
		}
		
		if(is_object($first)){
				if(!$this->is->array_of_objects($this->list_items)) 
						return $this->error->warning("I can't sort ".get_class($this).'::list_items unless all the items are being sorted by the same criteria');
				
				if($this->property_is_empty('order_by')) return $this->error->warning("I can't sort an array of objects without a value for ".get_class($this).'::order_by');
				
				if($this->order == 'asc')
						$this->list_items = sort_objects_by_property($this->list_items, $this->order_by);
				else
						$this->list_items = rsort_objects_by_property($this->list_items, $this->order_by);
				return true;		
		}
		
		return $this->error->should_be_a_type_that_i_know_how_to_sort($first);
	}
	
	
	/**
	* Calculates the total number of pages based on the value of {@link _items_per_page} and {@link _total_items}.
	* @return int
	*/
	public function total_pages(){
		if($this->items_per_page > 0 && $this->total_items > 0)
			return ceil($this->total_items / $this->items_per_page);
		
		return 1; //we're either showing all items, or no items exist
	}
	
	////////////////////////////////////////////////////////////////////
	// COLUMN FUNCTIONALITY
	////////////////////////////////////////////////////////////////////	
	/* This functionality will by default only apply to tables, but it could be applied to list items as well if you wrote views that took advantage of these methods.  These methods assume that your list items are some type of object or array that has properties that could be displayed as table columns. */	
	
	/**
	* Adds multiple columns to the {@link columns} array.
	* This is a wrapper for {@link add_column}.  Note that this differs from {@link set_columns} in that it adds additional entries to the {@link _columns} array, rather
	* than replacing it altogether.
	* @see _columns
	* @uses add_column
	* @param array $columns Array of column names.
	*/
	function add_columns($columns){
		if(!$this->is->array($columns)) return $this->error->should_be_an_array($columns);
		foreach($columns as $column){
			$this->add_column($column);
		}
	}
	
	/**
	* Adds a column to the {@link columns} array.
	* Note that column names should not have whitespace.  Preferably, these should be lower-case, using underscores instead of spaces.  This helps to facilitate
	* valid & consistent urls when used in conjunction with the order_by url param.  We'll use the {@link humanize} function to automatically generate a human-friendly
	* label for this column, or you can set one manually using {@link set_column_alias}.
	* @see _columns
	* @param string $column Name of the column.
	*/
	function add_column($column){
		if(!$this->is_a_valid_name_for_a_column($column)) return $this->error->should_be_a_valid_column_name($column);
		if(!isset($this->columns)) $this->_columns = array(); //in case this is the first time we've tried to add a column
		$this->_columns[] = $column;
	}
	
	/**
	* Removes a column from the {@link _columns} array.
	* @see _columns
	* @param string $column Name of the column.
	*/
	function remove_column($column){
		if(!$this->is_a_valid_name_for_a_column($column)) return $this->error->should_be_a_valid_column_name($column);
		if(!$this->column_exists($column)) return $this->error->should_be_the_name_of_an_existing_column($column);
		$this->remove_column_alias($column);
		unset($this->_columns[$column]);
	}
	
	/**
	* Returns true if a column of this name exists in the {@link _columns} array.
	* @see _columns
	* @param string $column
	* @return boolean
	*/
	function column_exists($column){
		if(!$this->is_a_valid_name_for_a_column($column)) return $this->error->should_be_a_valid_column_name($column);
		if($this->property_is_empty('columns')) return false;	//columns may not be set
		return in_array($column, $this->columns);
	}
	
	/**
	* Adds multiple column aliases at once.
	* This is a wrapper for {@link set_column_alias}.  Note that this differs from {@link set_column_aliases} in that it adds additional entries to the 
	* {@link _column_aliases} array, rather than replacing it altogether.
	* @see _columns
	* @uses add_column_column_alias
	* @param array $aliases Formatted ($column_name => $human_friendly_aliases)
	*/		
	function add_column_aliases($aliases){
		if(!$this->is->array($aliases)) return $this->error->should_be_an_array($aliases);
		foreach($aliases as $column => $alias){	$this->set_column_alias($column, $alias);	}	
	}
	
	/**
	* Sets a human-friendly alias for the given column.
	* Note that this is not required; if a column alias is not manually set, a human-friendly alias will be generated by running {@link humanize} on the column name.
	* @param string $colum Name of the column (as specified in {@link _columns})
	* @param string $alias Human-friendly alias to be used when displaying this column to the user.
	*/
	//developers - this is not an accessor for a protected variable, so don't give into the temptation to move it to the setters section.
	function set_column_alias($column, $alias){
		if(!$this->is_a_valid_name_for_a_column($column)) return $this->error->should_be_a_valid_column_name($column);
		if(!$this->is->string($alias)) return $this->error->should_be_a_string($alias); //empty string would be ok -- sometimes we don't label columns.
		if(!isset($this->column_aliases)) $this->_column_aliases = array();
		$this->_column_aliases[$column] = $alias;
	}
	
	/**
	* True if this column has a manually-set alias.
	* All columns will have an alias automatically generated for them, but this will return true if the developer has specified a particular alias for the column.
	* @param string $column
	* @return boolean
	*/
	function column_has_alias($column){
		if(!isset($this->column_aliases)) return false;
		return (array_key_exists($column, $this->_column_aliases));
	}
	
	/**
	* Returns a human-friendly alias for a column.
	* This alias may be manually set using {@link set_column_alias}, but an alias will be automatically generated using {@link humanize} if no alias is set.
	* @param string $column
	* @return string
	*/	
	function alias_for_column($column){
		if($this->column_has_alias($column)) return $this->_column_aliases[$column];
		return humanize($column);
	}
	
	/**
	* True if a column name fits our criteria for valid column names.
	* Since column names may be used as part of a url, we need to be sure that they won't cause problems.  Note that this is not checking to make sure that this
	* is an existing column - it's just checking to make sure that it's an acceptable name.
	* @todo Could probably use some more limitations -- puncuation marks, no forward slashes, etc.
	* @param string
	* @return boolean
	*/
	protected function is_a_valid_name_for_a_column($name){
		return $this->is->nonempty_string_with_no_whitespace($name);
	}
	
	protected function is_a_valid_column($name){
		if(!$this->is_a_valid_name_for_a_column($name)) return false;
		return array_key_exists($name, $this->columns);
	}
	
	#TODO
	//will format the column alias.  Also gives us the opportunity to alter the way that we generate column aliases.
	//function markup_for_column_label(){}
	
	
	////////////////////////////////////////////////////////////////////
	// MARKUP GENERATION METHODS
	////////////////////////////////////////////////////////////////////
	/* the markup for each component of the lister will be stored in a view/partial, by default in the ICARUS_DIRECTORY.libraries/lister/views/
	 can override each component by placing a view with the same name in APPPATH.views/lister/ for default overrides, or APPPATH.views/lister/.$this->_name/ otherwise.
	*/
	
	/** @return string */
	function markup_for_list(){	
		$markup = $this->markup_for_partial('list', array('lister' => &$this));
		if($markup !== FALSE) return $markup;
		return $this->error->warning('There is no view set up to display the list markup');
	}
	
	/** @return string */
	function markup_for_list_item($list_item){
		$markup = $this->markup_for_partial('list_item', array('list_item' => $list_item, 'lister' => &$this));
		if($markup !== FALSE) return $markup;
		if(is_string($list_item)) return $list_item;
		return $this->error->warning('The list item should either be a string or have a list item partial set up to display the table markup, but you gave me '.$this->error->describe($list_item));
	}
	
	/** @return string */
	function markup_for_spreadsheet(){
		$markup = $this->markup_for_partial('spreadsheet', array('lister' => &$this));
		if($markup !== FALSE) return $markup;
		return $this->error->warning('There is no view set up to display the spreadsheet markup');
	}	
	
	/** @return string */
	function markup_for_table(){
		$markup = $this->markup_for_partial('table', array('lister' => &$this));
		if($markup !== FALSE) return $markup;		
		return $this->error->warning('There is no view set up to display the table markup');
	}
	
	/** @return string */
	function markup_for_table_column($column, $list_item){		
		$partial_name = 'table_column_'.$column;
		if(array_key_exists('table_column_'.$column, $this->_partials))
			$view = $this->_partials[$partial_name];
		
		$path_possibilities = array('table/columns/_'.$column, 'columns/_'.$column, 'table/column', 'column');
		foreach($path_possibilities as $possibility){	
			$markup = $this->markup_for_partial($possibility, array('list_item' => $list_item, 'lister' => &$this));
			if($markup !== FALSE) return $markup;	
		}
				
		//if there's no view, check to see if $column is a key or a property of $list_item		
		if(is_array($list_item) && array_key_exists($column, $list_item))
			$value_to_display = $list_item[$column];
		if((is_object($list_item) && property_exists($list_item, $column)) || (method_exists($list_item, 'property_exists') && $list_item->property_exists($column)))
			$value_to_display = $list_item->$column;
		if(is_a($list_item, 'Entity') && $list_item->property_exists($column)){
			$value_to_display = $list_item->$column;
			if(!$value_to_display && !is_numeric($value_to_display)) $value_to_display = ''; // if the property exists but wasn't set, use a blank string so that isset() will show up as true.
		}
				
		//if the list item was a string, let's assume that it itself is the column & return it.  This doesn't seem like a great practice.
		if($this->is->string($list_item)) $value_to_display = $list_item;
		
		if(isset($value_to_display) && $this->is->string($value_to_display)){
			//if this is a recognizable mysql date, automatically format it
			if($this->is->mysql_date($value_to_display)) return '<span class="date">'.mysql_to_content_date($value_to_display).'</span>';
			if($this->is->mysql_datetime($value_to_display)) return '<span class="date date-time">'.mysql_to_content_datetime($value_to_display).'</span>';
			return $value_to_display;
		}
				
		$should_be_message = 'a string, an array with a key called '.$this->error->describe_as_text($column).', or an object with a property called '.$this->error->describe_as_text($column);
		return $this->error->should_be_an_x($list_item, $should_be_message);
	}
	
	/** @return string */
	function markup_for_table_column_label($column){
		$partial_name = 'table_column_label_'.$column;
		if(array_key_exists('table_column_label_'.$column, $this->_partials))
			$view = $this->_partials[$partial_name];
	
		$path_possibilities = array('table/column_labels/'.$column, 'table/column_label', 'column_label');
		foreach($path_possibilities as $possibility){	
			$markup = $this->markup_for_partial($possibility, array('column' => $column, 'lister' => &$this));
			if($markup !== FALSE) return $markup;	
		}

		return $this->error->warning('There is no view set up to display table column labels');
	}
	
	/** @return string */
	function markup_for_pagination(){
		if($this->total_pages() < 2) return '';	
		$markup = $this->markup_for_partial('pagination', array('lister' => &$this));
		if($markup !== FALSE) return $markup;

		return $this->error->warning('There is no view set up to display pagination markup');
	}
	
	/** @return string */
	function markup_for_pagination_link($page_number, $link_text = null, $attributes = array()){
		$params = compact('page_number', 'link_text', 'attributes');
		$markup = $this->markup_for_partial('pagination_link', array_merge($params, array('lister' => &$this)));
		if($markup !== FALSE) return $markup;
		
		return $this->error->warning('There is no view set up to display pagination link markup');
	}
	
	
    /**
    * Renders a partial and returns the content as a string.
    * This is analogous to using CI's {@link CI_loader::view} method with the optional $return_as_string parameter set to true.
    * @param string File path to the partial, in a format that will be understood by PHP's require() command.
    * @param array Array of variables to pass to the partial, formatted array(variable_name => variable_value)
    * @return string 
    */
    protected function markup_for_partial($partial, $variables_for_partial){
        if(!$this->is->string_like_a_file_path($partial)) return $this->error->should_be_a_string_like_a_file_path($partial_path);
        if(!$this->is->associative_array($variables_for_partial)) return $this->error->should_be_an_associative_array($variables_for_partial);
        
		$partial_path = $this->path_for_view($partial);
		
		if($this->is->view($partial_path))
			return $this->CI->load->view($partial_path, $variables_for_partial, TRUE);
		
		if(!file_exists($partial_path)) return false;	
			
        extract($variables_for_partial);
        ob_start();
        require $partial_path;
        $buffer = ob_get_contents();
        @ob_end_clean();
        return $buffer;
    }	
	
	
	/**
	* Returns a path to the given view that the Icarus Router's icarus_view() method will understand how to load.
	* Views for the lister can be located in a variety of locations; this allows you to override default views with a custom view for an application.  
	*
	* Possible locations for a view named $view, in order of priority:
	* - {@link APPPATH}/views/{@link _template_directory}/$view
	* - {@link APPPATH}/views/icarus/lister/{@link _template_directory}/$view
	* - {@link APPPATH}/views/lister/$view
	* - {@link ICARUS_DIRECTORY}/libraries/lister/$view
	*
	* @param string
	* @return string
	*/
	function path_for_view($view){
		if(!$this->is->string_like_a_view($view)) return $this->error->should_be_a_path_to_a_view($view);
	
		if(!$this->property_is_empty('template_directory')){
			if(is_string($this->template_directory))
				$directories = array($this->template_directory);
			else
				$directories = $this->template_directory;
	
			foreach($directories as $directory){
				if(!string_ends_with('/', $directory))
					$directory = $directory.'/';
				
				
				if($this->CI->load->view_exists($directory.$view))
					return $directory.$view;
			}	
			
			foreach($directories as $directory){
				
				if($this->CI->load->view_exists('lister/'.$directory.'/'.$view))
					return 'lister/'.$directory.'/'.$view;
			}
			
		}
		
		if($this->CI->load->view_exists('lister/'.$view))
			return 'lister/'.$view;
			
		foreach(array(APPPATH, VLERPATH) as $root){
			$path = $root.'libraries/lister/'.$view.'.php';
			if(file_exists($path)) return $path;
		}	
			
		return FALSE;
	}
	
	////////////////////////////////////////////////////////////////////
	// GETTERS
	////////////////////////////////////////////////////////////////////	
	
	//developers - the #@+ in the docblock below indicates that this docblock will be applied to every method until we hit the /**#@-*/ dockblock.
	//please only place getter methods for protected variables in this section, or the documentation will be inaccurate.
	
	/**#@+ 
	* Accessor for protected variable.
	*
	* This is a getter method for a protected variable that has the same name as this method, but prefixed by an underscore.  
	* 
	* In general, you should not call on this method directly.  Call on the variable, but without the underscore in the name -- the {@link __get} method will
	* call on any necessary accessor methods.
	*/
	
	/**
	* @see _columns
	* @uses _columns
	* @return array
	*/
	public function columns(){
		if(!isset($this->_columns)){
			if(isset($this->list_items) && !$this->property_is_empty('list_items')){
				if($this->is->array_of_arrays($this->list_items))
					$this->_columns = array_keys(first_element($this->list_items));
				elseif($this->is->array_of_objects($this->list_items)){
					$object = first_element($this->list_items);
					if(is_a($object, 'Entity') && method_exists($object, 'get_values'))
						$this->_columns = array_keys($object->get_values());
					else
						$this->_columns = get_class_vars(get_class($object));
				}
			}			
		}
		if(!isset($this->_columns)) return array();
		return $this->_columns;
	}
	
	protected function items_per_page(){
		if(!isset($this->_items_per_page)) $this->load_property_from_url('items_per_page');
		if(!isset($this->_items_per_page)) $this->_items_per_page = 10;
		return $this->_items_per_page;
	}
	
	/**
	* @see _list_items_need_to_be_sorted
	* @uses _list_items_need_to_be_sorted
	* @return boolean
	*/
	public function list_items_need_to_be_sorted(){
		if(isset($this->list_items_need_to_be_sorted)) return $this->_list_items_need_to_be_sorted;
		return (!$this->property_is_empty('order') && !$this->property_is_empty('order'));
	}
	
	/** 
	* @see _list_items
	* @uses _list_items
	* @return array 
	*/
	public function list_items(){
		if(!is_array($this->_list_items))	return array();
		return $this->_list_items;
	}
	
	/** 
	* @see _page
	* @uses _page
	* @return int 
	*/
	public function page(){
		if(empty($this->_page))	$this->load_property_from_url('page');
		if(!empty($this->_page)) return $this->_page;
		return 1;
	}	
	
	/** 
	* @see _order_by
	* @uses _order_by
	* @return string 
	*/
	public function order_by(){
		if(empty($this->_order_by))	$this->load_property_from_url('order_by');	
		if(!empty($this->_order_by)) return $this->_order_by;
		return $this->default_order_by;
	}
	
	/** 
	* @see _order
	* @uses _order
	* @return string 
	*/
	public function order(){
		if(empty($this->_order)) $this->load_property_from_url('order');
		if(!empty($this->_order))	return $this->_order;
		return $this->default_order;
	}
	
	/**
	* @see _total_items
	* @uses _total_items
	* @return int
	*/ 
	public function total_items(){
		if(isset($this->total_items)) return $this->_total_items;
		if(isset($this->list_items))	return count($this->list_items);
		return 0;
	}
	
	/**
	* @see _url_param_for_page
	* @uses _url_param_for_page
	* @return string
	*/ 
	function url_param_for_page(){
		if($this->is->nonempty_string($this->_url_param_for_page)) return $this->url_param_prefix.$this->_url_param_for_page;
		return '';
	}
	
	/**
	* @see _url_param_for_order
	* @uses _url_param_for_order
	* @return string
	*/ 
	function url_param_for_order(){
		if($this->is->nonempty_string($this->_url_param_for_order)) return $this->url_param_prefix.$this->_url_param_for_order;
		return '';
	}
	
	/**
	* @see _url_param_for_order_by
	* @uses _url_param_for_order_by
	* @return string
	*/ 
	function url_param_for_order_by(){
		if($this->is->nonempty_string($this->_url_param_for_order_by)) return $this->url_param_prefix.$this->_url_param_for_order_by;
		return '';
	}
	
	
	/**#@-*/		
	
	////////////////////////////////////////////////////////////////////
	// SETTERS
	////////////////////////////////////////////////////////////////////	
	
	//developers - the #@+ in the docblock below indicates that this docblock will be applied to every method until we hit the /**#@-*/ dockblock.
	//please only place setter methods for protected variables in this section, or the documentation will be inaccurate.
	
	/**#@+ 
	* Accessor for protected variable.
	*
	* This is a setter method for a protected variable that has the same name as this method, but prefixed by an underscore.  
	* 
	* In general, you should not call on this method directly.  Call on the variable, but without the underscore in the name -- the {@link __set} method will
	* call on any necessary accessor methods.
	*/
	
	//here mostly for easier extensibility
	/**
	* @see _list_items
	* @uses _list_items
	* @param array $list_items
	*/
	public function set_list_items($list_items){
		if(!$this->is->array($list_items)) return $this->error->should_be_an_array($list_items);
		$this->_list_items = $list_items;
	}
	
	//set the array of visible columns for this table/lister.
	/**
	* @see _columns
	* @uses _columns
	* @uses add_columns
	* @param array $columns
	*/
	function set_columns($columns){
		if(!$this->is->array($columns)) return $this->error->should_be_an_array($columns);
		$this->_columns = array();
		$this->add_columns($columns);
	}
	
	/**
	* Set the value of {@link _column_aliases} to the given array.
	*
	* Note that this will overwrite all current column aliases completely.  To add additional aliases rather than removing the current ones, use 
	* {@link add_column_aliases}.
	*
	* @see _column_aliases
	* @uses add_column_aliases
	* @param array $aliases Formatted ($column_name => $human_friendly_alias)
	*/
	function set_column_aliases($aliases){
		if(!$this->is->array($aliases)) return $this->error->should_be_an_array($aliases);
		$this->_column_aliases = array();
		$this->add_column_aliases($aliases);
	}
		
	/** 
	* @see _total_items
	* @uses _total_items
	* @param int $total_items 
	*/
	public function set_total_items($total_items){
		if(!$this->is->nonzero_unsigned_integer($total_items)) return $this->error->should_be_a_nonzero_unsigned_integer($total_items);
	
		if(isset($this->list_items)){		
			$item_count = count($this->list_items);
			if($total_items < $item_count)return $this->error->should_be_within_range($total_items, 1, ' the total number of list items ('.$item_count.')');
		}
			
		$this->_total_items = $total_items;
		//if we're setting total items manually, make sure it doesn't conflict with an already set page
		if(isset($this->page) && $this->page > $this->total_pages) set_url_param_and_redirect($this->url_param_for_page, $this->total_pages);
		return $this->_total_items;
	}
		
	/** 
	* @see _page
	& @uses _page
	* @param int $page 
	*/
	public function set_page($page){
		if(!$this->is->nonzero_unsigned_integer($page)) return $this->error->should_be_a_number_greater_than_zero($page);
		
		//check to see if we have the means to check what the total pages are
		if(isset($this->list_items) || isset($this->total_items)){
			if($page > $this->total_pages()){
				$this->_page = $this->total_pages();
				return $this->error->should_be_within_range($page, 1, $this->total_pages);
			}
		} 
				
		$this->_page = $page;
	}
	
	/** 
	* @see _order_by
	* @uses _order_by
	* @param string 
	*/	
	public function set_order_by($order_by){
		if(!$this->is->nonempty_string($order_by) || (!$this->property_is_empty('columns') && !in_array($order_by, $this->columns)))
			return $this->error->should_be_a_valid_column_name($order_by);
		$this->_order_by = $order_by;			
	}
	
	/** 
	* @see _default_order_by
	* @uses _default_order_by
	* @param string 
	*/	
	public function set_default_order_by($order_by){
		if(!$this->is->nonempty_string($order_by) || (!$this->property_is_empty('columns') && !in_array($order_by, $this->columns)))
			return $this->error->should_be_a_valid_column_name($order_by);
		$this->_default_order_by = $order_by;		
	}
	
	/** 
	* @see _order
	* @uses _order
	* @param string 
	*/
	public function set_order($order){
		if(!$this->is->nonempty_string($order)) return $this->error->should_be_a_nonempty_string($order);	
		$order = strtolower($order);	
		if($order != 'asc' && $order != 'desc') return $this->error->should_be_ASC_or_DESC($order);
		$this->_order = $order;	
	}
	
	/** 
	* @see _default_order
	* @uses _default_order
	* @param string 
	*/
	public function set_default_order($order){
		if(!$this->is->nonempty_string($order)) return $this->error->should_be_a_nonempty_string($order);		
		$order = strtolower($order);	
		if($order != 'asc' && $order != 'desc') return $this->error->should_be_ASC_or_DESC($order);
		$this->_default_order = $order;	
	}
	
/**#@-*/			
}