<?PHP  if ( ! defined('BASEPATH')) exit('No direct script access allowed');

require_library('object');
require_model('message');

/**
* Class Attachment
* @todo replace Message:: calls with something that checks for the model message class name first.
* @package vler
* @subpackage libraries
*/

class Attachment extends Object{

    // File Name: e.g. 'myfile.txt'
    protected $name;
    // Path to file, including the trailing slash.
	protected $_directory;
	// Content of the file
	protected $_binary_string;

	protected $_message_id;

	protected $_filetype;
	protected $_extension;
	protected $mime_type;
	protected $is_xdm;
	protected $_cid;
	protected $_url;

    // A version of the file name that will be appropriate for form names and css classes
	protected $_html_friendly_name;

	protected $_use_lightbox_preview = false;
	protected $_use_pdf_preview = false;

	//files inside of a zip/xdm/archive file will reference that zip/xdm/archive file as their parent
	protected $_parent; //reference to the actual zip/xdm/archive Attachment object
	protected $_path_in_parent; //if this file is within a subdirectory, this is the relative path
	protected $_index_in_parent; //the order in which this appears in the parent, so that we can reference it easily

	protected $_property_validation_rules= array(
        'binary_string' => 'string',
        'extension' => 'string_with_no_whitespace',
        'filetype' => 'string_with_no_whitespace',
        'name' => 'string_like_a_file_path',
        'cid' => 'nonempty_string',
        'url' => 'string_like_a_url',
        'path_in_parent' => 'string_like_a_file_path',
        'index_in_parent' => 'unsigned_integer',
        'mime_type' => 'string_with_no_whitespace',
        'is_xdm' => 'integer',
    );

    /**
     * @return array
     */
	function classes(){
		$classes = array_merge( array(get_class($this) => get_class($this)), class_parents($this));
		unset($classes['Object']);
		return array_map('make_string_css_friendly', $classes);
	}

    /**
     * @return string
     * note - in some places in the code, we ran base-64 on this in the past.
     */
    function describe(){
		if(Message::formatted_like_an_id($this->message_id)) {
		    return strtolower(humanize(get_class($this))).
                ' for message#'.$this->message_id.' '.
                $this->error->describe($this->name);
        }

		if(!$this->property_is_empty('directory')) {
		    return $this->error->describe($this->path());
        }
		return strtolower(humanize(get_class($this))).' '.$this->error->describe($this->name);
	}

    /**
     * @return false
     */
	function download(){
		if($this->property_is_empty('name')) {
		    return $this->error->warning("Can't set up download without a name for this file.");
        }
		header("Content-Type: application/force-download");
		header("Content-Disposition: attachment; filename=\"" . $this->name . "\"");
		header("Content-Transfer-Encoding: base64");
        // note - in some places in the code, we ran base-64 on this in the past.
        
        // Office XML documents are stored with an extra line terminator or two in the database
        // We need to remove them for MS Office to work.
        $isOfficeDoc = (strpos($this->mime_type,'openxmlformats') !== FALSE);
        if ($isOfficeDoc) {
            $this->binary_string = rtrim($this->binary_string,"\r\n");
            $this->binary_string = rtrim($this->binary_string,"\r");
            $this->binary_string = rtrim($this->binary_string,"\n");
        }
        header("Content-Length: " . string_length_in_bytes($this->binary_string));
        echo $this->binary_string;
	}

    /**
     * @return string
     */
	function url_for_download(){
		if(config_item('subclass_prefix') == 'DPII_'){
			$name = rawurlencode($this->name);
			if(isset($this->index_in_parent)) {
			    $name = rawurlencode($this->parent->name).'/'.$this->index_in_parent;
            }
			if(Message::formatted_like_an_id($this->message_id)) {
			    return site_url('inbox/message/'.$this->message_id.'/attachments/download/'.$name);
            }
			return site_url('inbox/compose/draft/attachments/download/'.$name);
		}else{
			$this->error->notice(
			    'Attachment::url_for_download() cannot be used until it is defined by an application extension'
            );
		}
	}

    /**
     * @return string
     * by default, this is the same as the view url, but may be different for specific attachment types
     */
    function url_for_preview(){
		return $this->url_for_view();
	}

    /**
     * @return string
     */
	function url_for_view(){
		if(config_item('subclass_prefix') == 'DPII_'){
			$name = rawurlencode($this->name);
			if(isset($this->index_in_parent)) {
			    $name = rawurlencode($this->parent->name).'/'.$this->index_in_parent;
            }
			if(Message::formatted_like_an_id($this->message_id)) {
			    return site_url('inbox/message/'.$this->message_id.'/attachments/'.$name);
            }
			return site_url('inbox/compose/draft/attachments/'.$name);
		}else{
			$this->error->notice(
			    'Attachment::url_for_view() cannot be used until it is defined by an application extension'
            );
		}
	}

	/**
	* @param string
	* @param array
	* @return string
	*/
	function link($text=null, $attributes = array()){
		if(is_null($text)) $text = $this->name;
		return link_to_url($this->_url_for_link(), $text, $this->_attributes_for_link($attributes));
	}

    /**
     * @return bool|false|string
     */
    function binary_string(){
		if(isset($this->_binary_string)) {
		    return $this->_binary_string;
        }
		if(!empty($this->path())){
			if(!file_exists($this->path())) {
			    $this->error->warning('Unable to find file: '.$this->path());
            }
			$this->_binary_string = file_get_contents($this->path());
			return $this->_binary_string;
		}
		return $this->error->warning('No binary string has been set');
	}

    /**
     * @return int
     */
	function bytes(){
		if(!empty($this->path()) || isset($this->binary_string)) {
		    return string_length_in_bytes($this->binary_string);
        }
		return 0;
	}

    /**
     * @return string
     */
	function extension(){
		if(!isset($this->_extension)) {
		    $this->_extension = strtolower(pathinfo($this->name, PATHINFO_EXTENSION));
        }
		return $this->_extension;
	}

    /**
     * @return string
     */
	function hash(){
		if(!$this->property_is_empty('binary_string')){
		    return sha1($this->binary_string);
        }
	}

    /**
     * @return string
     */
	function html_friendly_name(){
		return make_string_css_friendly($this->name);
	}

    /**
     * @return null
     */
	function filetype(){
		if(!isset($this->filetype)) return $this->extension;
		$this->_filetype;
	}

    /**
     * @return mixed
     */
	function filename(){
		return $this->name;
	}

    /**
     * @return array
     */
	function metadata(){
		$properties = array_keys(get_object_vars($this));
		foreach($properties as &$property) {
            $property = strip_from_beginning('_', $property);
        }

		$properties_to_exclude = array(
		    'is',
            'validator',
            'error',
            'property_validation_rules',
            'binary_string',
            'parent',
            'parser',
            'use_lightbox_preview',
            'use_pdf_preview'
        );

		$properties = array_merge(
		    array_diff(
		        $properties,
                $properties_to_exclude
            ),
            array('bytes', 'path', 'url', 'classes')
        );
		$metadata = array();
		$only_if_not_empty = array('message_id', 'cid', 'das_save_date', 'schema', 'description', 'url');
		foreach($properties as $property){
			$property = strip_from_beginning('_', $property);
			if(!in_array($property, $only_if_not_empty) || !$this->property_is_empty($property)){
				$metadata[$property] = $this->$property;
			}
		}

		if(is_a($this->parent, 'Attachment')) {
		    $metadata['parent'] = $this->parent->describe();
        }

		return $metadata;
	}

    /**
     * The full path of the directory + filename; will only be supplied if
     * both directory path and name has been populated
     * @return string
     */
    function path(){
		if(isset($this->directory) && isset($this->name)){
			return $this->directory.$this->name;
		}
	}

    /**
     * deprecated alias for binary_string
     * @return null
     */
	function string(){
		return $this->binary_string;
	}

    /**
     * @return string
     */
	function url(){
		return $this->url_for_view();
	}

    /**
     * @return mixed
     */
	function mime_type() {
        return $this->mime_type;
    }

    /**
     * @return mixed
     */
    function is_xdm() {
        return $this->is_xdm;
    }

    /**
     * Ensure that the path value always has a trailing slash && that the directory exists
     * @param $value
     * @return mixed
     */
    function set_directory($value){
		if(!$this->is->directory_path($value)) {
		    return $this->error->property_value_should_be_an_existing_directory_path('directory', $this, $value);
        }
        // make sure we always have a trailing slash, but only one
		$this->_directory = strip_from_end('/', $value).'/';
	}

    /**
     * @param $value
     * @return mixed
     */
	function set_message_id($value){
		if(!Message::formatted_like_an_id($value)) {
		    return $this->error->property_value_should_be_a_message_id('message_id', $this, $value);
        }
        /*
         * We could check to see if the message really exists,
         * but that would be expensive (API call)
         * - we'll trust the developers in this case
         *
         */
		$this->_message_id = $value;
	}

    /**
     * @param $value
     * @return mixed
     */
	function set_parent($value){
		if(!is_a($value, 'Attachment')) {
		    return $this->error->property_value_should_be_a_message_id('parent', $this, $value);
        }
		$this->_parent = $value;
	}

    /**
     * @param $value
     */
	function set_mime_type($value) {
        $this->mime_type = $value;
    }

    /**
     * @param $value
     */
    function set_is_xdm($value) {
        $this->is_xdm = $value;
    }

    /**
	* @param array
	* @return array
	*/
	protected function _attributes_for_link($attributes = array()){
		if(!empty($this->_source_class)) {
			$classes = array_merge($this->classes, $this->_source_class->classes);
		} else {
		    $classes = $this->classes;
		}
		$default_attributes = array('class' => implode(' ', $classes), 'title' => $this->name);

		if($this->use_lightbox_preview) {
            $default_attributes = merge_attributes($default_attributes, array('class' => 'lightbox'));
        } else if($this->use_pdf_preview) {
		    $default_attributes = merge_attributes($default_attributes, array('class' => 'lightbox iframe'));
        }

        if (!$this->property_is_empty('parent')) {
            $default_attributes = merge_attributes($default_attributes, array('class' => 'archived-attachment'));
        }

		return merge_attributes($default_attributes, $attributes);
	}

	/**
	* Returns the URL to be used by {@link}.
	*
	* By default, this is {@link url_for_preview} as long as {@link _use_lightbox_preview} or
    * {@link _use_pdf_preview} is enabled.  Will revert
	* to {@link url_for_download} if neither form of preview is available.  Child classes may
    * override or extend this behavior.
	*
	* @return string
	*/
	protected function _url_for_link(){
		if($this->use_lightbox_preview || $this->use_pdf_preview) {
            return $this->url_for_preview() . '?' .
                get_instance()->security->get_csrf_token_name() . '=' .
                get_instance()->security->get_csrf_hash();
        }
        if(method_exists($this, 'view')) {
            return $this->url_for_view();
        }
        return $this->url_for_download();
	}

	/**
	* Checks the content of a file and casts it as the appropriate child class.
	*
	* For example, if a C32 file is passed to this method, {@link matches_file} will be used to determine
    * that the content of the file
	* matches a child class and the file will be cast as a {@link C32_attachment}. Note that it would be
    * more efficient to just create
	* a new C32_attachment object if you were sure that a file was a C32; this method should be used
    * when you don't know how a file should be cast.
	*
	* @param string Name of the file
	* @param string Content of the file
	* @param array Any additional metadata, formatted array('message_id' => $message_id, 'directory' => $directory)
	* @return Attachment
	*/
	public static function create($name, $binary_string, $more_values = array()){
		log_message('debug','Who called this (Attachment::create) ' . debug_backtrace()[1]['class'].':'.debug_backtrace()[1]['function']);
        if ( empty($more_values['mime_type'])) {
            // if we can use the existing file in cache, we should
            if (!empty($more_values['directory'])) {
                $more_values['mime_type'] = mime_content_type($more_values['directory'] . $name);
            }

            // fallback to using the binary string
            if (empty($more_values['mime_type'])) {
                $finfo = new finfo(FILEINFO_MIME_TYPE);
                $more_values['mime_type'] = $finfo->buffer($binary_string);
                unset($finfo);
            }
        }
		log_message('debug','MIME Type of file ' . $name . ' is ' . $more_values['mime_type']);
        switch ($more_values['mime_type']) {
            case 'application/zip':
                // check to see if we are we an XDM
                require_library('attachments/Zip_attachment');
                $xdm_class = class_name_for_library('attachments/XDM_attachment');
                $more_values['is_xdm'] = $xdm_class::matches_file($name, $binary_string);
                if ($more_values['is_xdm'] == XDM_SCORE_VALID) {
                    $attachment_class_name = 'attachments/XDM_attachment';
                } else {
                    /**
                     * It looks like it failed validation. But, we can make a strong assumption that
                     * it is expected to be an XDM if the files name is formatted a certain way.
                     * We are going to check the filename. And, re-set the status if we think it is and XDM.
                     */
                     if ( $more_values['is_xdm'] != XDM_SCORE_INVALID &&
                         (stripos($name, 'xdm') !== FALSE || stripos($name, 'ccd') !== FALSE)
                     ) {
                         /**
                          * "XDM" or "CCD" is in the name of the file. We will assume that they are expecting
                          * this file to be an XDM.
                          */
                         $more_values['is_xdm'] = XDM_SCORE_INVALID;
                     }

                    $attachment_class_name = 'attachments/zip_attachment';
                }
                break;
            case 'text/xml':
            case 'application/xml':
                $attachment_class_name = 'attachments/xml_attachment';
                break;
            case 'text/csv':
                $attachment_class_name = 'attachments/csv_attachment';
                break;
            case 'application/pdf':
                $attachment_class_name = 'attachments/pdf_attachment';
                break;
            case 'application/msword':
            case 'application/vnd.openxmlformats-officedocument.wordprocessingml.document':
                $attachment_class_name = 'attachments/word_attachment';
                break;
            case 'text/plain':
            case 'text/css':
            case 'text/html':
                // wait. this could be a CSV file
                if ( strtolower(pathinfo($name, PATHINFO_EXTENSION)) === 'csv' ) {
                    $attachment_class_name = 'attachments/csv_attachment';
                } else {
                    $attachment_class_name = 'attachments/text_attachment';
                }
                break;
            case 'image/gif':
            case 'image/png':
            case 'image/jpeg':
            case 'image/svg+xml':
                $attachment_class_name = 'attachments/image_attachment';
                break;
            default:
                $attachment_class_name = null;
                break;
        }

        // we found a match. great!
        if (!empty($attachment_class_name)) {
            $attachment_class = class_name_for_library($attachment_class_name);
            $attachment = $attachment_class::create($name, $binary_string, $more_values);
            // if this attachment does not validattion, treat it as a generic attachment
            if ( is_a($attachment, $attachment_class) ) {
                return $attachment;
            }
        }
        // worst case, we use generic 'Attachment' type
        $values = array(
            'name' => $name,
            'binary_string' => $binary_string,
        );
        $values = array_merge($values, $more_values);
        return new Attachment($values);
	}

    /**
     * @param $path
     * @return Attachment|null
     */
	public static function from_session_cache($path){
		$CI = get_instance();
		if(!$CI->session->path_exists_in_cache($path)) {
		    return null;
        }
		$filename = $path;
		$directory = null;
		if(string_contains('/', $path)){
			$filename = substr($path, strrpos($path, '/') + 1);
			$directory = strip_from_end($filename, $path);
		}

		return static::create(
            $filename,
            $CI->session->file_from_cache($path),
            array(
                'directory' => $CI->session->cache_root($directory)
            )
        );

	}

	/**
	* True if the given file can be cast as this class.
	*
	* For example, if this is an extension of the default Attachment class for XML attachments, this method
	* will check to make sure that the content of the file is valid XML.
	*
	* @param string Name of the file (not including path)
	* @param string Contents of the file
	* @return boolean
	*/
	public static function matches_file($name, $binary_string){
		return (!empty($binary_string) && !empty($name));
	}

    /**
     * Generates the URL for downloading an attachment to Vista Imaging
     * @param bool $lookup
     * @return string
     */
    public function link_for_vista($lookup = true){
        $header = "Import Attachment to VistA Imaging";
        $text = "Save To VistA Imaging";
        $attributes = $this->_attributes_for_link();

        if ($lookup) {
            $attributes['class'] = ' btn-vista-download-lookup';
            $url = site_url('inbox/compose/draft/'
                . $this->message_id
                . '/attach/patient_document/' . $header);
        } else {
            $attributes['class'] .= ' btn-vista-download-patient';
            // Tempoary URL to show patient doc.  This should eventually redirect
            // to patient lookup results with icd
            $url = $this->_url_for_link();
        }

        return link_to_url($url, $text, $attributes);
     }
}
