<?php  if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* Extension of Codeigniter's {@link http://codeigniter.com/user_guide/helpers/html_helper.html html_helper}.
*
* Any changes to this library need to be made in both Webmail and the API.
*
* @see https://ellislab.com/codeigniter/user-guide/helpers/html_helper.html

* @package vler
* @subpackage helpers
*/

if(!function_exists('attributes_to_string')){
	/**
	* Takes an associative array of HTML attributes and returns the values as a string ready to be added to an HTML tag
	*
	* Example array: array('id' => 'display_name', 'class' => 'text-input numeric', 'required' => 'required').
	*
	* @param array
	* @return string
	*/
	function attributes_to_string($attributes){
		if(is_string($attributes)) return $attributes;
		if(!validates_as('associative_array', $attributes)) return should_be('string or an associative array', $attributes);
		$markup = '';
		foreach ($attributes as $key => $value){
			if(is_array($value))
				$value = implode(' ', $value);
			
			$markup .= ' '.$key.'="'.$value.'"';
		}
		return $markup;
	}
}

if(!function_exists('css_link_tag')){
	/**
	* Returns the markup for a CSS link tag.
	* Uses CodeIgniter's {@link http://codeigniter.com/user_guide/helpers/html_helper.html#link_tag link_tag()} function.
	*
	* @param string Path to the stylesheet.  This should be in the same format that you would use for the {@link http://codeigniter.com/user_guide/helpers/url_helper.html 
	*								site_url()} function
	* @param array Attributes to include in tag - e.g. array('class' => 'exciting_class', 'id' => 'my_favorite_div')
	* @return string
	*/
	function css_link_tag($path, $attributes = array()){
		if(!validates_as('nonempty_string_with_no_whitespace', $path)) return should_be('nonempty string with no whitespace', $path);
		if(!validates_as('associative_array', $attributes)) return should_be('associative array', $attributes);
		
		$default_attributes = array('href' => $path, 'rel' => 'stylesheet', 'type' => 'text/css', 'media' => 'all');
		return link_tag(merge_attributes($default_attributes, $attributes))."\n";
	}
}


if(!function_exists('dismissable_alert')){
	/**
	* Generates the markup for a Bootstrap dismissable alert. 
	* @see http://getbootstrap.com/components/#alerts
	*
	* @param string The text that the alert should display
	* @param string The type of alert to display (defaults to info - can also be success, danger, or info)
	* @param array Any additional attributes that should be applied to the div containing the alert.
	* @return string
	*/
	function dismissable_alert($content, $type='info', $attributes=array()){
		if(!validates_as('nonempty_string', $content)) return should_be('nonempty string', $content);
		if(!in_array($type, array('success', 'info', 'warning', 'danger'))) return should_be('known bootstrap type: success, info, warning or danger', $type);
		if(!is_array($attributes)) return should_be('array', $attributes);
		
		$markup = open_tag('div', merge_attributes($attributes, array('class' => 'alert alert-'.$type.' alert-dismissible', 'role' => 'alert')))."\n";
		$markup .= '<button type="button" class="close" data-dismiss="alert" aria-label="Close"><span aria-hidden="true">&times;</span></button>'."\n";
		$markup .= $content."\n";
		$markup .= '</div>';
		
		return $markup;
	}
}

if(!function_exists('link_to')){
	/**
	* Generates a full link tag for an internal link.
	* Similar to CI's {@link http://codeigniter.com/user_guide/helpers/url_helper.html site_url()} function, but generates a full link tag instead of just
	* the path.
	*
	* For example:
	* <code>
	* $attributes = array('title'=>'Log Out', 'onclick' => "return confirm('Are you sure you want to log out?');");
	* $markup = link_to('logout','Log Out',$attributes); 
	* echo $markup; //<a href="http://src.bah.com/logout" title="Log Out" onclick="return confirm('Are you sure you want to log out?');">Log Out</a> 
	* </code>
	* 
	* @uses link_to_url()
	*
	* @param string The path for the link. This can be either a full url or an internal path. If it's a path, it will be passed to CI's site_url() to generate the full url.
	* @param string (Optional) The text or HTML that should be the content of this link.  This will default to the URL if not supplied.
	* @param array (Optional) Any attributes that should be contained in the opening tag (e.g. class, title, onmouseover, etc.)
	* @return string HTML link tag
	*/
	function link_to($path, $text='',$attributes=array()){
		if(!string_begins_with('http', $path) && !string_begins_with('https', $path) && !string_begins_with('//', $path))
			$url = site_url($path);
		else
			$url = $path;
		return link_to_url($url, $text, $attributes);
	}
}

if(!function_exists('link_to_url')){
	function link_to_url($url, $text='', $attributes=array()){
		if(!is_numeric($text) && empty($text))
			$text = $url;
		return '<a href="'.$url.'"'.attributes_to_string($attributes).'>'.$text.'</a>';
	}
}

if(!function_exists('link_to_if')){
	/**
	* Link the text if $condition is true, otherwise return it in a span.
	* This uses {@link link_to()} in order to generate a link.
	* Highly useful for things like tabs and breadcrumbs.
	* @uses link_to()
	* @param boolean Whether or not the this should be displayed as a links.
	* @param string The path for the link.  This will be passed to codeigniter's site_url() to generate the full url.
	* @param string (Optional) The text or HTML that should be the content of this link.
	* @param array (Optional) Any attributes that should be contained in the opening tag (e.g. class, title, onmouseover, etc.) 
	* @return string HTML link tag if $condition was true, span containing $text otherwise
	* @todo Use tag() to generate the span so that we can pass the same attributes so that classes, ids, etc. could be the same.
	*/
	function link_to_if($condition, $path, $text, $attributes=array()){
		if($condition)
			return link_to($path, $text, $attributes);
		else
			return tag('span', $text, $attributes);
	}
}

if(!function_exists('link_to_unless')){
	function link_to_unless($condition, $path, $text, $attributes=array()){
		return link_to_if(!$condition, $path, $text, $attributes);
	}
}

if(!function_exists('make_string_css_friendly')){
	/**
	* Replaces spaces and underscores in a string with dashes to fit CSS standards
	* @param string
	* @return string
	*/
	function make_string_css_friendly($string){
		if(!validates_as('nonempty_string', $string)) return should_be('nonempty string', $string);
		return strip_from_beginning('-', strip_from_end('-', preg_replace('/-{2,}/', '-', preg_replace('/[^\p{L}\p{M}\p{N}]/u', '-',  mb_strtolower($string)))));
	}
}

if(!function_exists('merge_attributes')){
	/**
	* Merges two associative arrays of HTML attributes.
	*
	* Like {@link array_merge()}, but correctly merges classes. 
	* Example array: array('id' => 'display_name', 'class' => 'text-input numeric', 'required' => 'required').
	*
	* @param array
	* @param array
	* @return array
	*/
	function merge_attributes($one, $two){
		if(!validates_as('associative_array', $one)) return should_be('associative_array', $one);
		if(!validates_as('associative_array', $two)) return should_be('associative_array', $two);
		
		if(!empty($one['class']) && !empty($two['class'])){
			$classes_for_one = explode(' ', $one['class']);
			$classes_for_two = explode(' ', $two['class']);
			$two['class'] = implode(' ', array_merge($classes_for_one, $classes_for_two));
		}
	
		return array_merge($one, $two);
	}
}

if(!function_exists('open_tag')){
	/**
	* Returns the markup for the opening of a standard HTML tag.
	*
	* @todo Add option for self-closing tag
	* @todo Add option for $attributes to be either a string or an array
	*
	* @uses attributes_to_string()
	*
	* @param string Tag to open - e.g. 'a', 'p', 'div', etc.
	* @param array Attributes to include in tag - e.g. array('class' => 'exciting_class', 'id' => 'my_favorite_div')
	* @return string
	*/
	function open_tag($tag, $attributes=array()){
		if(!validates_as('nonempty_string_with_no_whitespace', $tag)) return should_be('name of an html tag', $tag);
		if(!validates_as('string', $attributes) && !validates_as('associative_array', $attributes)) 
			return should_be('string or an associative array', $attributes);
		
		$markup = '<'.$tag;
		if(!empty($attributes)){
			$markup .= attributes_to_string($attributes);
		}
		$markup .= '>';
		return $markup;
	}
}

if(!function_exists('tag')){
	/**
	* Generate an HTML tag.
	* Note that this function currently will only display the tag if the given content isn't empty.
	* @todo Make the whole "only display if not empty" feature an option, not a given.
	* @uses open_tag()
	* @param string The type of HTML tag to generate  -- e.g. 'p', 'span', 'div', etc.
	* @param string What should go in the tag
	* @param array (Optional) Any attributes for the opening tag (e.g., array('class' => 'supercoolhappyfunclass') would result in <div class="supercoolhappyfunclass">
	* @return string HTML tag
	*/
	function tag($tag, $content, $attributes=array()){
		if(!validates_as('nonempty_string_with_no_whitespace', $tag)) return should_be('name of an html tag', $tag);
		if(!validates_as('string', $content)) return should_be('string', $content);
		if(!validates_as('string', $attributes) && !validates_as('associative_array', $attributes)) 
			return should_be('string or an associative array', $attributes);
				
		if(!empty($content) || $content === '0'){
			$markup = open_tag($tag, $attributes).$content.'</'.$tag.'>';
			return $markup;
		}
		else return false;
	}
}

if(!function_exists('ul_if_multiple')){
	/**
	* Takes an array of strings and returns them as a ul tag if there is more than one and in a <span> if there is only one.
	* @param array Strings to be included in the list
	* @param array Any attributes to be applied to the containing tag, formatted array('class' => "my-class", 'id' => "my-id")
	* @return string 
	*/
	function ul_if_multiple($array, $attributes = array()){
		if(!validates_as('array_of_strings', $array)) return should_be('array_of_strings', $array);
		if(count($array) > 1) return ul($array, attributes_to_string($attributes));
		if(empty($attributes))
			return first_element($array);
		else
			return tag('span', first_element($array), $attributes);
	}
}