<?php  if ( ! defined('BASEPATH')) exit('No direct script access allowed');
/**
* @package vler
* @subpackage database
*/ /** */

load_library('validator');

/**
* Patches and extensions to the CI SQL Server database driver go here.
*
* The ability to extend database drivers is not native to Codeigniter; this extension relies on the Loader class being extended to look for database drivers.
* For more information, see {@link https://github.com/EllisLab/CodeIgniter/wiki/Extending-Database-Drivers Extending Database Drivers}.
*
* @author M. Gibbs <gibbs_margaret@bah.com>
*
* @package vler
* @subpackage database
*/
class VLER_DB_sqlsrv_driver extends CI_DB_sqlsrv_driver {
	
	/* IMPORTANT NOTE */
	/* CI loads database drivers in a REALLY weird way (see the DB() function in DB.php and the database() loader function) and any class vars from CI_DB_sqlsrv_driver that we try to load here 
	 will get overwritten by the loader database() function.  If you need to override a default class var value, set it up in the database config file.  $params['class_var_name'] => 		
	 $class_var_value will do the trick.  The params the config file are parsed by the DB() function and passed to the constructor for DB_driver class.  The constructor will simply set all of 
	 these as class vars.  */
	protected $_affected_rows; //the PHP function get affected rows can only be called once, so we'll store the value here so we can access it multiple tiems
	var $_escape_char = '[';
	var $_escape_char_close = ']';
	var $_like_wildcards = array(']', '[', '_', '%', '^');

	function __construct($params){
		parent::__construct($params);	

		//reduce memory requirements by making sure that save_queries isn't turned on unless the profiler is on - CI apparently leaves it on by default
		//this controls whether the actual SQL string gets saved - and if that's an INSERT statement with a 100MB file as data, that can be pretty big
		//at the time of this writing, we've confirmed that the saved queries are only used for the last_query() method and the profiler, so we should be safe turning it off.  -- MG 2016-09-19
		$this->save_queries = ($this->save_queries && $this->default_save_queries_value());	
		
		log_message('info', 'Extended DB driver class instantiated.');
	}	
	
	function default_save_queries_value(){
		return is_on_local() && get_instance()->output->profiler_is_enabled();
	}

	function _execute($sql){
		$this->_affected_rows = null; //reset the _affected_rows count before running the new query
		
		if($this->scrollable === FALSE OR $this->is_write_type($sql)){
			return sqlsrv_query($this->conn_id, $sql);
		}else{
			$config = array('Scrollable' => $this->scrollable);
			if($this->scrollable == SQLSRV_CURSOR_STATIC)
				$config['SendStreamParamsAtExec'] = true;
			return sqlsrv_query($this->conn_id, $sql, NULL, $config);
		}
	}
	
	
	/** 
	* Overrides parent to cache the affected rows result.
	* The PHP sql server driver only allows the affected rows function to be called once, so we cache the value to allow this method to be called multiple times.
	* Overrides parent to correct fatal error.
	* CI mispelled the name of the sqlsrv function in their version of this method, causing a fatal error.  Good job, CI.  In addition to correcting the fatal error,
	* this method will trigger warnings if anything happens that prevents it from calculating the affected rows, instead of failing silently. Also makes use of a class
	* var to store the number of affected rows so that we can safely call this method more than once per query.
	*/
	function affected_rows(){
		//affected rows is only available once, so save it to a protected var so that we can access it multiple times
		if(!isset($this->_affected_rows) || is_null($this->_affected_rows)){
			$this->_affected_rows = parent::affected_rows();
			if($this->_affected_rows === FALSE){
				foreach(sqlsrv_errors() as $error){
					trigger_error('Error encountered while calling sqlsrv_rows_affected(): '.$error['message'], E_USER_WARNING);
				}
			}
		}
			
		return $this->_affected_rows;
	}	
	
//NOTE - We originally only had this enabled in API, may cause issues in webmail
	/** 
	* Overrides parent to escape all strings with N'
	* Our default character set is UTF-8, but the database will only store UTF-8 strings correctly if we prefix string values
	* with N' and make sure that we use nvarchar/nchar/ntext fields instead varchar/char/test fields.  We are now assuming
	* that *all* of our text fields in the database are going to be nvarchar/nchar/ntext fields and automatically prefixing
	* all escaped strings accordingly; it's up to the developers to continue implementing the correct database field types.
	* @param string
	* @return string
	*/
	function escape($value, $assume_nvarchar=true){
		$value = parent::escape($value);
		if($assume_nvarchar && is_string($value) && string_begins_with("'", $value) && !is_numeric($value))
			$value = 'N'.$value;
		return $value;
	}

	/**
	 * Escape String
	 * Overrides parent to allow escaping of all LIKE wildcards (such as [] which define character sets and * which matches anything)
	 *
	 * @access	public
	 * @param	string
	 * @param	bool	whether or not the string will be used in a LIKE condition
	 * @return	string
	 */
	function escape_str($str, $like = FALSE) {
		if (is_array($str)){
			foreach ($str as $key => $val){
				$str[$key] = $this->escape_str($val, $like);
			}
			return $str;
		}
		
		$escaped_str = $str = $this->_escape_str($str, $like);

		/* VLER CUSTOMIZATION */
		//CI code only assumes one LIKE wildcard - this code looks for all possible
		//Note that this was written for 2.1.3, which didn't provide for LIKE at all.  The CI3 way of this may be preferred  - if we run into any trouble, comment this entire method out and see what happens.
		if($like){
			$escaped_str = '';
			//use mb_substr here to handle both unicode and other multibyte characters
			for($i = 0; $i < mb_strlen($str); $i++) {
				if(in_array(mb_substr($str, $i, 1), $this->_like_wildcards)) {
					$escaped_str .= $this->_like_escape_chr . mb_substr($str, $i, 1);
				}
				else {
					$escaped_str .= mb_substr($str, $i, 1);
				}
			}
		}
		/* END VLER CUSTOMIZATION */
		
		return $escaped_str;
	}

	
	 /**
     * An attempt to make _limit() work the same way for SQL Server that it would for more enlightened databases that allow offsets.
	 *
	 * Note that using this method with a non-empty offset will add an additional column called __daas_row_number to your database results.  
	 *
	 * When we switch to SQL Server 2012, try out using the CI _limit method without overloading - their existing code doesn't work for us, but the future version might.
     */
     function _limit($sql){
	
		$limit = $this->qb_limit;
		$offset = $this->qb_offset;
		
		if(empty($offset) || !is_numeric($offset) || $offset <= 0){
			return preg_replace('/(^\SELECT (DISTINCT)?)/i','\\1 TOP '.$limit.' ', $sql);     
		}	
		
		//HACKING A WAY TO DO OFFSETS IN SQL SERVER
		//Since SQL server < 2012 doesn't have a good way to do mysql offset, we're going to use the ROW_NUMBER OVER syntax to create row numbers in a derived table, 
		// and select the row numbers we need.  This may go poorly.  
		//Questions?  Ask Margaret.  She will probably deny all knowledge of ever having tried to cobble together a solution for this problem, but you never know.
			
		//hokay.  we need to figure out how this select is ordered, and remove the existing order by if it's there.  conveniently, order by is always the last part of the query if it exists.
		$order_by_position = mb_strpos($sql, 'ORDER BY ');
		if(empty($order_by_position)){
			//if they haven't a supplied an order by, we just can't apply the offset.  Let the developer know that they made a mistake.
			return $this->display_error( array('An ORDER BY value must be applied before a LIMIT offset can be applied to this query:', $sql), '', TRUE );
		}else{
			$order_by = trim(mb_substr($sql, $order_by_position - 1)); //grab the $order_by for our use
			$sql = mb_substr($sql, 0, $order_by_position); //remove the order by clause so that we can use it in ROW_NUMBER() instead
		}

		//now we need to add an additional column to the select query.  
		//the keyword after the columns for select is either INTO or FROM, so figure out where the select ends
		//note that this won't work if there are nested queries in the SELECT columns -- strrpos wouldn't work if we had nested queries in WHERE,
		//so we'll stick with this until we feel like trying to get more sophisticated.  You should be able to work around this limitation most of the time.  -- MG 2014-12-03
		$end_of_select = mb_strpos($sql, 'FROM ') - 1;
		$into_position = mb_strpos($sql, 'INTO '); 
		if(!empty($into_position) && $into_position < $end_of_select)
			$end_of_select = $into_position - 1;
		
		//note that because of the difference between our local SQL version (2008 R2) and the test/prod version (SQL Server 2008 SP3), we need to make sure we explicitly order by row number
		//see: http://dba.stackexchange.com/questions/27467/is-select-row-number-guaranteed-to-return-results-sorted-by-the-generated-row and ticket VAD-732
		$row_number_column = '__daas_row_number'; //ridiculous name, but descriptive & no one else is likely to need to use it		
		$sql = mb_substr($sql, 0, $end_of_select).', ROW_NUMBER() OVER ('.$order_by.') AS '.$row_number_column.mb_substr($sql, $end_of_select);
		$sql = 'SELECT * FROM ('."\n".trim($sql)."\n".') AS __daas_derived_table '."\n".'WHERE '.$row_number_column.' BETWEEN '.($offset+1).' AND '.($offset+$limit).' ORDER BY __daas_row_number';
		return $sql; //VICTORY.
    } 

	/**
	 * Escape the SQL Identifiers
	 *
	 * This function escapes column and table names.  Extended because CI assumes that there is a single escape char (like ` in mysql), but mssql uses [].
	 *
	 * @param	string
	 * @return	string
	 */
	function escape_identifiers($item){		
		if(empty($this->_escape_char))	return $item;
		if(empty($item) && $item !== 0) return $item;
				
		//handle expressions where there's no space between column name and equals sign
		if(string_contains('=', $item)){
			$position_of_equals_sign = mb_strpos($item, '=');
			return $this->_escape_identifiers( mb_substr($item, 0, $position_of_equals_sign) ). mb_substr($item, $position_of_equals_sign) ;			
		}

		foreach ($this->_reserved_identifiers as $id){
			if(string_contains('.'.$id, $item)){
				$str = $this->_escape_char. str_replace('.', $this->_escape_char_close.'.', $item);
				
				// remove duplicates if the user already included the escape		
				return $this->_remove_duplicate_escape_chars($str);
			}
		}

		if(string_contains('.', $item))
			$str = $this->_escape_char.str_replace('.', $this->_escape_char_close.'.'.$this->_escape_char, $item).$this->_escape_char_close;
		else
			$str = $this->_escape_char.$item.$this->_escape_char_close;
		

		// remove duplicates if the user already included the escape
		return $this->_remove_duplicate_escape_chars($str);
	} 


	//only nescessary for our override of _escape_identifiers - if we remove that, we can remove this.
	function _remove_duplicate_escape_chars($string){
		$escape_chars = array($this->_escape_char, $this->_escape_char_close);
		foreach($escape_chars as $escape_char){
			while(string_contains($escape_char.$escape_char, $string))
				$string = str_replace($escape_char.$escape_char, $escape_char, $string);
		}
		return $string;
	} 


	//extends parent to avoid trying to parse empty strings (was causing errors)
	public function protect_identifiers($item, $prefix_single = FALSE, $protect_identifiers = NULL, $field_exists = TRUE){
		if(empty($item) && $item !== 0) return '';
		return parent::protect_identifiers($item, $prefix_single, $protect_identifiers, $field_exists);
	}
	
	
	//extends parent to allow us to include a backtrace for db queries
	public function query($sql, $binds = FALSE, $return_object = NULL)
	{
		if ($sql === '')
		{
			log_message('error', 'Invalid query: '.$sql);
			return ($this->db_debug) ? $this->display_error('db_invalid_query') : FALSE;
		}
		elseif ( ! is_bool($return_object))
		{
			$return_object = ! $this->is_write_type($sql);
		}

		// Verify table prefix and replace if necessary
		if ($this->dbprefix !== '' && $this->swap_pre !== '' && $this->dbprefix !== $this->swap_pre)
		{
			$sql = preg_replace('/(\W)'.$this->swap_pre.'(\S+?)/', '\\1'.$this->dbprefix.'\\2', $sql);
		}

		// Compile binds if needed
		if ($binds !== FALSE)
		{
			$sql = $this->compile_binds($sql, $binds);
		}

		// Is query caching enabled? If the query is a "read type"
		// we will load the caching class and return the previously
		// cached query if it exists
		if ($this->cache_on === TRUE && $return_object === TRUE && $this->_cache_init())
		{
			$this->load_rdriver();
			if (FALSE !== ($cache = $this->CACHE->read($sql)))
			{
				return $cache;
			}
		}


		if ($this->save_queries === TRUE)
		{
			$this->queries[] = $sql;
############ VLER MODIFICATION STARTS HERE ###################################
			if(is_on_local()){						
				$backtrace = debug_backtrace();
				$query_backtrace = array();
				
				foreach($backtrace as $row => &$data){
					unset($data['object']);	//not necessary, just makes it easier to see things when you're debugging
					unset($data['args']);
					if(!isset($data['file'])) $data['file'] = false;
							
					$data['function_for_display'] = $data['function'];
					if(!empty($data['class'])){
						$data['function_for_display'] = $data['class'].$data['type'].$data['function'];
					}
					
					$data['short_file'] = '';
					if(is_string($data['file'])){
						$last_slash = strrpos($data['file'], '/');
						$data['short_file'] = '..'.substr($data['file'], $last_slash);
					}
					$query_backtrace[] = $data;
				}
				
				foreach($backtrace as $row => $data){
					$data['file'] = str_replace('\\', '/', $data['file']);
					
					if(strpos($data['file'], BASEPATH.'database/') !== 0 && $data['file'] != APPPATH.'models/Database_model.php' && $data['file'] != APPPATH.'models/DB_entity.php'
							&& $data['file'] != VLERPATH.'models/Entity.php' && array_key_exists('file', $data) && array_key_exists('line', $data)){
						$file = $data['file'];
						$line = $data['line'];
						$short_file = $data['short_file'];
						if(array_key_exists($row+1, $backtrace))
							$data = $backtrace[$row+1]; //to determine the function, go back one more in the backtrace	
						
						$function = element('class', $data).element('type', $data).element('function', $data);
						break;
					}
				}
				
				$this->queries_backtrace[count($this->queries) - 1] = array('sql' => $sql, 'line' => $line, 'file' => $file, 'short_file' => $short_file, 'backtrace' => $query_backtrace, 'function' => $function); 
			}
########### VLER MODIFICATION ENDS HERE ########################################			
		}

		// Start the Query Timer
		$time_start = microtime(TRUE);

		// Run the Query
		if (FALSE === ($this->result_id = $this->simple_query($sql)))
		{
			if ($this->save_queries === TRUE)
			{
				$this->query_times[] = 0;
			}

			// This will trigger a rollback if transactions are being used
			$this->_trans_status = FALSE;

			// Grab the error now, as we might run some additional queries before displaying the error
			$error = $this->error();

			// Log errors
			log_message('error', 'Query error: '.$error['message'].' - Invalid query: '.$sql);

			if ($this->db_debug)
			{
				// We call this function in order to roll-back queries
				// if transactions are enabled. If we don't call this here
				// the error message will trigger an exit, causing the
				// transactions to remain in limbo.
				if ($this->_trans_depth !== 0)
				{
					do
					{
						$this->trans_complete();
					}
					while ($this->_trans_depth !== 0);
				}

				// Display errors
				return $this->display_error(array('Error Number: '.$error['code'], $error['message'], $sql));
			}

			return FALSE;
		}

		// Stop and aggregate the query time results
		$time_end = microtime(TRUE);
		$this->benchmark += $time_end - $time_start;

		if ($this->save_queries === TRUE)
		{
			$this->query_times[] = $time_end - $time_start;
		}

		// Increment the query counter
		$this->query_count++;

		// Will we have a result object instantiated? If not - we'll simply return TRUE
		if ($return_object !== TRUE)
		{
			// If caching is enabled we'll auto-cleanup any existing files related to this particular URI
			if ($this->cache_on === TRUE && $this->cache_autodel === TRUE && $this->_cache_init())
			{
				$this->CACHE->delete();
			}

			return TRUE;
		}

		// Load and instantiate the result driver
		$driver		= $this->load_rdriver();
		$RES		= new $driver($this);

		// Is query caching enabled? If so, we'll serialize the
		// result object and save it to a cache file.
		if ($this->cache_on === TRUE && $this->_cache_init())
		{
			// We'll create a new instance of the result object
			// only without the platform specific driver since
			// we can't use it with cached data (the query result
			// resource ID won't be any good once we've cached the
			// result object, so we'll have to compile the data
			// and save it)
			$CR = new CI_DB_result($this);
			$CR->result_object	= $RES->result_object();
			$CR->result_array	= $RES->result_array();
			$CR->num_rows		= $RES->num_rows();

			// Reset these since cached objects can not utilize resource IDs.
			$CR->conn_id		= NULL;
			$CR->result_id		= NULL;

			$this->CACHE->write($sql, $CR);
		}

		return $RES;
	}
	
	//true if columns have been specified for the select for this query - note that CI defaults to * if you haven't specified anything yet
	function select_is_defined(){
		return !empty($this->qb_select);
	}
	
	
}