<?php  if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package vler
* @subpackage core
*//** */

require_once APPPATH.'helpers/environment_helper.php'; //can't rely on load if we're getting an error in the early loading process.

/**
* Extended from CI to include the following features:
* 	- On dev, PHP error messages now include backtraces instead of the less informative CI default.
* 	- If using the {@link Error} library to generate error messages, the line number will
* 		be corrected.
*
* @package vler
* @subpackage core
*/
class VLER_Exceptions extends CI_Exceptions {
	var $backtrace;
	//the CI code removes identifying info for security, but this is not helpful to us.  we don't show backtraces on prod, so we want full info on dev.
	var $original_filepath; 
	var $exception_offset; //previously the global $EXCEPTION_OFFSET, this marks how many rows in the backtrace we have to go back to find the actual error.
	
	/**
	* Extends parent to take into account any actions our custom code took after the error was triggered.
	* If this an error generated by {@link Error} or if we re-triggered the error within this class,
	* the filename and number will no longer reflect the original trigger point.  This method parses the 
	* backtrace to determine the fileline and number at which the error was originally triggered.
	*/
	public function log_exception($severity, $message, $filepath, $line){
		if(is_on_local()){ //only work out a backtrace if we're in a local development environment
			if(isset($this->exception_offset) && !empty($this->exception_offset) && is_numeric($this->exception_offset) && $this->exception_offset > 0){
				$backtrace = $this->backtrace($called_from = 'log_exception'); //don't set the class var, as we won't be needing this version of it again
				$target_index = count($backtrace) - 1 - $this->exception_offset;
				if(array_key_exists($target_index, $backtrace))
					$target = $backtrace[$target_index];

				if(!empty($target) && array_key_exists('file', $target) && array_key_exists('line', $target)){
					$line = $target['line'];
					$filepath = $target['file'];
				}
			}	
			$this->original_filepath = $filepath;
		}
		
		//parent doesn't do the log message right, so fix this
		$severity = isset($this->levels[$severity]) ? $this->levels[$severity] : $severity;
		log_message('error', 'Severity: '.$severity.' --> '.$message.' in '.$filepath.' on '.$line);
		
		//show_php_error is called after log_exception, so only reset the offset if we're not going to show the error
		if(!str_ireplace(array('off', 'none', 'no', 'false', 'null'), '', ini_get('display_errors')))
			$this->exception_offset = 0; 
	}	
		
	
	
	/**
	* On dev, bypass the usual CI_Exception::show_error method and show a php error with a backtrace instead.
	*/
	public function show_error($heading, $message, $template = 'error_general', $status_code = 500){
		if($template == 'error_general' && is_on_local()){ //only work out a backtrace if we're in a local development environment
			$this->backtrace = $this->backtrace();
			$this->exception_offset = $this->exception_offset + 1;
			trigger_error($message);
		}

		return parent::show_error($heading, $message, $template, $status_code);
	}
	
	/**
	* Sets up a backtrace for the error message to display & adjusts for any error messages triggered through multiple layers of code.
	* If this an error generated by {@link Error} or if we re-triggered the error within this class,
	* the filename and number will no longer reflect the original trigger point.  This method parses the 
	* backtrace to determine the fileline and number at which the error was originally triggered.
	*/
	public function show_php_error($severity, $message, $filepath, $line){	
		$this->truncate_message_if_needed($message);
			
		if(is_on_local()){ //only work out a backtrace if we're in a local development environment	
			$this->backtrace = $this->backtrace();				
			if(isset($this->exception_offset) && !empty($this->exception_offset) && is_numeric($this->exception_offset) && $this->exception_offset > 0){
				$target_index = count($this->backtrace) - 1 - $this->exception_offset;
				if(array_key_exists($target_index, $this->backtrace))
					$target = $this->backtrace[$target_index];
				
				if(!empty($target) && array_key_exists('file', $target) && array_key_exists('line', $target)){
					$line = $target['line'];
					$filepath = $target['file'];
				}
			}	
			
			$this->original_filepath = $filepath;
		}
		
		if(defined('PHPUNIT_TEST')){ //when we're running unit tests, use phpunit's error handling system
			$this->exception_offset = 0;
			return PHPUnit_Util_ErrorHandler::handleError($severity, $message, $filepath, $line);
		}
				
		parent::show_php_error($severity, $message, $filepath, $line);
		$this->exception_offset = 0; //show_php_error is the last thing we do, so reset the offset.
	}
	

	/**
	* Sets up a backtrace to be displayed by the error message.
	* Backtrace is generated by PHP's {@link debug_backtrace()} & formatted for display purposes.
	*/
	function backtrace($called_from = 'show_php_error'){			
		$backtrace = debug_backtrace();
		$post_trigger_error_functions = array($called_from, 'backtrace', 'MY_Exceptions', 'load_class', '_exception_handler', 'trigger_error');		
		
		//strip the backtrace back to the point at which the error was triggered for readability
		foreach($backtrace as $row => $data){
			if(!in_array($data['function'], $post_trigger_error_functions) || $data['function'] == 'trigger_error') break; 
			if(in_array($data['function'], $post_trigger_error_functions)) unset($backtrace[$row]); 
		} 
				
		//format the backtrace for the error handler		
		foreach($backtrace as $row=>&$data){ 
			unset($data['object']);	//not necessary, just makes it easier to see things when you're debugging
			if(!isset($data['file'])) $data['file'] = false;
						
			$data['function_for_display'] = $data['function'];
			if(!empty($data['class'])){
				$data['function_for_display'] = $data['class'].$data['type'].$data['function'];
			}
			
			$data['short_file'] = '';
			if(is_string($data['file'])){
				$last_slash = mb_strrpos($data['file'], '/');
				$data['short_file'] = '..'.mb_substr($data['file'], $last_slash);
			}
		}
		
		$backtrace = array_reverse($backtrace);		
		$this->backtrace = $backtrace;
		return $this->backtrace;
	}
	
	protected function truncate_message_if_needed(&$message){
		//being very careful to avoid errors when we're in the middle of error-handling
		//note that it should really be CI's problem to make sure that CI_Controller is available when get_instance is available, but apparently that's an issue, so we'll circumvent it
		if(!function_exists('get_instance') || !class_exists('CI_Controller')) return; 
		$CI = get_instance();
		if(!is_object($CI) && !isset($CI->log)) return;
		return $CI->log->truncate_message_if_needed($message);
	}
}