<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');

require_once 'Mail.php';
require_once 'Mail/mime.php';
require_once 'Mail/mimeDecode.php';

require_library('mime_generator');
require_model('entity');

/**
 * Class Direct_message
 * @package direct-as-a-service
 * @subpackage models
 */
class Direct_message extends DAAS_Entity {

//////////////////
// STATIC VARS
///////////////////
	static $database_group = 'mail_db';
	static $table = 'messages';
	protected static $_relationships = array( 'mailbox' => array('type' => 'belongs_to'),
											  'original_sender' => array('model' => 'mailbox',
											  							 'type' => 'belongs_to',
																		 'related_foreign_key' => 'original_sender_id'));

	public static $parts = array( 'flags' => array( 'seen',
													'draft',
													'sent',
													'archived',
													'flags',
													),

								  'headers' => array( 'sender',
								  					  'to',
													  'cc',
													  'bcc',
													  'subject',
													  'timestamp',
													  'size',
													  'seen',
													  'flags',
													  'draft',
								  					  'sent',
													  'archived',
													  'attachments',
													  'headers',
													  'priority',
								  					  'folder_id',
													  'mailtype',
													  'message_id'),
								  'raw_mime' => array('raw_mime'),
								  'xdm_attachments' => array('xdm_attachments'));


///////////////////////////
// INSTANCE VARS
///////////////////////////
	protected $_body; //either the html or the plain value, depending on mailtype
	protected $_attachment_files; //array of filename => binary_strings
	protected $_inline_attachment_files;
	protected $_mime_generator; //Mime_generator object for this message

	//each of these fields affect the mime - when they're changed, the mime needs to be regenerated
	protected $mime_fields = array( 'sender',
									'to',
									'cc',
									'bcc',
									'attachments',
									'subject',
									'body',
									//'flags', //I don't think this actually affects the mime, right?  this is just within our database
									'mailtype',
									'priority' );

	//readonly fields which are derived from the mime and may not be set by developers
	protected $mime_derived_fields = array( 'attachments',
											'headers',
											'html',
											'plain',
											'raw_mime',
											'recipients',
											'size' );

	protected $_readonly_fields = array( 'seen', //only readonly in the case of draft/sent messages - we'll unset this in readonly() for outgoing messages
										 'draft', //will be set internally for new messages
										 'sent', //will be set internally when send() is called
										 'message_id', //will be set by the RI for incoming messages
										 'timestamp', //will be generated by the model when fields that affect the mime are changed, and when the message is sent
										);

	//validation rules for settable properties on the method - these correspond to Validator methods
	protected $_property_validation_rules = array('priority' => 'nonzero_unsigned_integer',
												  'subject' => 'string',
												  'seen' => 'boolean');






///////////////////
// INSTANCE
///////////////////

	/**
	* @param string
	* @param string
	* @return boolean True on success
    * @todo Improve file validation here
	*/
	public function add_attachment($name, $binary_string){

		if(!$this->is->string_like_a_file_path($name)) {
		    return $this->error->should_be_a_string_like_a_file_path($name);
        }

		if(!$this->draft()) {
		    return $this->error->warning(
		        'Cannot attach '.$name.' to '.$this->describe().
                '; attachments can only be added to drafts.'
            );
        }

		if(!static::attachment_has_valid_extension($name)) {
		    return $this->error->warning('Cannot attach '.$name.' to '.
                $this->describe().'; invalid file extension.'
            );
        }

		if(!$this->attachment_is_within_size_limit($binary_string)) {
			return $this->error->warning(
			    'Adding file '.$name.' ('.byte_format(string_length_in_bytes($binary_string)).') to '.
                $this->describe().' would cause the message to exceed the limit of '.
                byte_format(DIRECT_ATTACHMENT_SIZE_LIMIT)
            );
		}

		$attachment_files = $this->attachment_files;
		if(array_key_exists($name, $attachment_files)){
			return $this->error->warning(
			    'Cannot attach '.$name.' to '.$this->describe().'; '.$name.
                ' has already been attached to this message.'
            );
		}

		$attachment_files[$name] = $binary_string;
		$this->_attachment_files = $attachment_files;
		// clean up
		unset($attachment_files);
        /**
         * Force the mime to regenerate the next time we call on a mime-derived field.
         * This will ensure that it includes the newly attached file.
         */
		$this->reload_mime();

		return true;
	}

    /**
     * @param $hash
     * @return array
     */
	public function attachments_matching_hash($hash){
		$attachments_matching_hash = array();
		foreach($this->attachment_files as $name => $binary_string){
			if(sha1($binary_string) == $hash)
				$attachments_matching_hash[$name] = $binary_string;
		}
		return $attachments_matching_hash;
	}

	public function attachment_is_within_size_limit($binary_string){
		/* $attachment_size_total = 0;
		foreach ($this->attachment_files as $name => $binary_string ) {
			$attachment_size_total += string_length_in_bytes($binary_string);
		}
		$new_total = ($attachment_size_total + string_length_in_bytes($binary_string));
		return ($new_total <= DIRECT_ATTACHMENT_SIZE_LIMIT); */
		$new_total = $this->size(true) + string_length_in_bytes($binary_string);
		return ($new_total <= DIRECT_ATTACHMENT_SIZE_LIMIT);
	}


	/**
	* True if the message belongs to the given mailbox.
	* @param Mailbox
	* @return boolean
    * @todo Make this a generalized relationship function for objects that belong to something?
    * @todo Allow mailbox ids instead of just mailbox objects?
	*/
	public function belongs_to_mailbox($mailbox){
		if(!Mailbox::is_an_entity($mailbox)) {
		    return $this->error->should_be_a_mailbox_entity($mailbox);
        }

		$related_foreign_key = static::related_foreign_key('mailbox');

		// if we have the mailbox id loaded, use it
		if(isset($this->$related_foreign_key)) {
		    return ($this->$related_foreign_key == $mailbox->id());
        }

		// if we have the sender loaded, use it in case we don't have the mailbox id
		if(isset($this->sender)) {
		    return ($this->sender == $mailbox->email_address());
        }

		/**
         * You're probably going to only run into this if message hasn't been saved,
         * or if you loaded the message using find_part()
         */
		return $this->error->warning(
		    'Cannot compare message to mailbox; no mailbox data is loaded into this message entity. '.
            'Run $message->load_field_values_from_db() to ensure the necessary values are loaded'
        );

	}

    /**
     * @return string
     */
    public function describe(){
		$description = parent::describe();
		if(!isset($this->id)) return $description;

		if($this->is_incoming()) {
		    return 'incoming '.$description;
        }

		if($this->sent) {
		    return 'sent '.$description;
        }

		return 'draft '.$description;
	}

    /**
     * @param $name
     * @param $binary_string
     * @return bool
     */
    public function has_attachment($name, $binary_string){
		if(!$this->is->nonempty_string($name)) {
		    return $this->error->should_be_a_nonempty_string($name);
        }

		if(!is_string($binary_string)) {
		    return $this->error->should_be_a_string($name);
        }

        $attachment_files = $this->attachment_files;
		if(!array_key_exists($name, $attachment_files)) {
		    return false;
        }

		return ($this->attachment_files[$name] == $binary_string);
	}

    /**
     * @return bool
     */
	public function has_xdm_attachment(){
		// if this has an id, then we determine the attachments from the attachments field.
		if($this->property_is_empty('attachments')) {
		    return false;
        }

		$attachments = get_instance()->json->decode($this->attachments, true);
		foreach($attachments as $attachment) {
			if(
			    (string_contains($this->subject,'XDM/1.0/DDM') || string_contains('xdm',strtolower($attachment)))
                && pathinfo($attachment, PATHINFO_EXTENSION) === 'zip'
            ) {
				return true;
			}
		}
		return false;
	}

    /**
     * @return bool
     */
	function has_attachments(){
		// if this has an id, then we determine the attachments from the attachments field.
		if($this->property_is_empty('attachments')) return false;
		$attachments = get_instance()->json->decode($this->attachments, true);
		return !empty($attachments);
	}

    /**
     * @return bool
     */
	public function has_recipients(){
		// checking actual recipients is not always useful, since it may have an empty json encoded value: '{}'
		return !$this->property_is_empty('to') || !$this->property_is_empty('cc') || !$this->property_is_empty('bcc');
	}

    /**
     * Patient documents (c32s, ccdas, possibly others in future) require a statement
     * about why patient information was disclosed system-wide messages should never
     * have patient documents attached to them, but if they DO we want to be able to identify that.
     * @return bool
     */
	public function has_required_attachment_disclosures(){
        $attachment_files = $this->attachment_files;
		foreach($attachment_files as $name => $binary_string){
			if(Attachment_disclosure::content_requires_disclosure($name, $binary_string)){
				$this->db->order_by('disclosed', 'desc')->limit(1);
				$disclosure = first_element($this->attachment_disclosures( array('hash' => sha1($binary_string))));
				if(!Attachment_disclosure::is_an_entity($disclosure)) return false;
			}
		}
		return true;
	}

	/**
	* True if this is a message that was sent to this mailbox.
	* @return boolean
	*/
	public function is_incoming(){
        /**
         * Any new messages we make, that don't have ids, are always
         * outgoing messages/incoming messages are created in the RI
         */
        if(!isset($this->id)) return false;
        // describe uses this method, can't call it or we could get an infinite loop
		$description = get_class($this).'#'.$this->id;
		if(!isset($this->draft)) {
		    return $this->error->warning(
		        'I can\'t determine if '.$description.' is incoming without a value for draft; '.
		        'run $message->load_field_values() to load all db values into this message'
            );
        }
		if(!isset($this->sent)) {
		    return $this->error->warning(
		        'I can\'t determine if '.$description.' is incoming without a value for sent; '.
                'run $message->load_field_values() to load all db values into this message'
            );
        }

		return !$this->is_outgoing();
	}

	/**
	* True if this is an outgoing message.
	* This will be true if the message has not yet been saved (all new messages *must* be outgoing), or if the sender matches the mailbox.
	* @return boolean
	*/
	public function is_outgoing(){
		if(!isset($this->id)) return true;//any new messages we make, that don't have ids, are always outgoing messages - incoming messages are created in the RI
		if(!isset($this->draft)) return $this->error->warning("I can't determine if ".$this->describe().' is outcoming without a value for draft; run $message->load_field_values() to load all db values into this message');
		if(!isset($this->sent)) return $this->error->warning("I can't determine if ".$this->describe().' is outgoing without a value for sent; run $message->load_field_values() to load all db values into this message');

		return ($this->draft || $this->sent);
	}

	/**
	* True if this message is in the inbox.
	* @return boolean
	*/
	public function is_in_inbox(){
		if(!$this->is_incoming()) return false;
		if(!isset($this->archived)) return $this->error->warning("I can't determine if ".$this->describe().' is an inbox message without a value for archived; run $message->load_field_values() to load all db values into this message');

		return !$this->archived && $this->property_is_empty('folder_id');
	}

    /**
     * @return bool|false
     */
	function mark_as_read(){
		if($this->is_outgoing()) return $this->error->warning( ucfirst($this->describe()).' is outgoing and cannot be marked as read' );
		if($this->seen) return true; //we're done
		$this->seen = true;
		$this->save();
	}

    /**
     * @return bool|false
     */
	function mark_as_unread(){
		if($this->is_outgoing()) return $this->error->warning( ucfirst($this->describe()).' is outgoing and cannot be marked as read' );
		if(!$this->seen) return true; //we're done
		$this->seen = false;
		$this->save();
	}


	/**
	* A Pear mime object representing this message.
	* Usually we use {@link mime_generator()} to get mime values, but certain of our derived values are easier to obtain using the PEAR Mime decoder class.
	* @return object
	*/
	public function mime($decode_bodies = true){
		$mime = Mail_mimeDecode::decode( array(  'include_bodies' => true,
												 'decode_bodies' => $decode_bodies,
												 'decode_headers' => true,
												 'input' => $this->raw_mime,
												 /*'crlf' => "\n" */)); //apparently we need to use \n if using PHP mail */
		if(Pear::isError($mime)){
			$this->error->warning('PEAR error encountered while decoding the mime for '.$this->describe().': '.$mime->getMessage());
			return null;
		}

		return $mime;
	}

	/**
	* Moves this message to the inbox.
	*
	* Only received messages may be moved to the inbox - attempting to move drafts or sent messages will trigger an error.
	*
	* @return boolean True on success
	*/
	public function move_to_inbox(){
#TODO - CHECK TO MAKE SURE THAT WE HAVE SEND, DRAFT, ARCHIVED VALUES LOADED
		if($this->sent) return $this->error->warning($this->model_alias.'#'.$this->id().' is a sent message and may not be moved to the inbox', 1);
		if($this->draft) return $this->error->warning($this->model_alias.'#'.$this->id().' is a draft and may not be moved to the inbox', 1);
		if($this->archived && !$this->restore()) return $this->error->warning('Could not restore archived message '.$this->model_alias.'#'.$this->id());
		$folder_foreign_key = static::related_foreign_key('folder');
		$this->$folder_foreign_key = null;
		return $this->save();
	}

	/**
	* Restores an archived message to its previous location (sent messages, drafts, inbox, folder, etc.).
	* @return boolean True on success
	*/
	public function restore(){
		$this->archived = false;
		return $this->save();
	}

	//formatting that applies to all values messages returned by the API goes here, so that we don't have to duplicate code
	public function values_for_api($part = null){
		if(!empty($part) && !Message::is_a_part($part)) return $this->error->should_be_a_message_part($part);

		if(empty($part))
			$values = $this->values();
		else
			$values = $this->values_for_part($part);

		if(array_key_exists('attachments', $values)){
			if(empty($values['attachments'])){
				$values['attachments'] = '{}';
			}
			$values['patients'] = array();
			if($this->has_relationship('patient') && ENABLE_PATIENT_PARSING_FOR_INBOX_MESSAGES){
				foreach($this->patients as $patient){
					$values['patients'][] = $patient->values_for_api();
				}
			}
		}

		if(array_key_exists('original_sender_id', $values)){
			$original_sender = $this->original_sender;
			$values['original_sender'] =  (Mailbox::is_an_entity($original_sender)) ? $original_sender->name : '';
			unset($values['original_sender_id']);
		}




#		if($this->sent && ($part === null || $part === "status" || $part === "headers" || $part=="")){
		if($this->sent && (empty($part) || $part == "status" || $part == "headers")){
/* NOTE - We should really rename this - shouldn't be prepended with "message_" since it's already part of a message (redundant).  Also, "status" is unclear and
   implies a singular value when this is actually a collection of rows from the database depicting the delivery history for each recipient -- MG 2014-05-16 */
			$values['message_status'] = get_instance()->messagestatusmodel->get_status($this->message_id);
		}

		//raw mime is pulled by default, but we also parse it by default unless raw mime is requested
		//^ anyone know what this comment is supposed to mean?  confusing, plus the $raw_mime variable wasn't ever set even when this was still in the controller
		if(array_key_exists('raw_mime', $values) && empty($raw_mime)) {
			#NOTE: parse_mime is meant to parse raw mime into an object which can then be
			#returned by the API as JSON, XML, etc. which can be helpful for people who want
			#the information in the raw mime without parsing it themselves.
			#If the raw_mime parameter is passed, they get the raw mime instead, but by
			#default they get the parsed object from this method - ATB 3/21/2014
			//moved this code out of parse_mime & into model; seemed like a better place for it than the controller -- MG 2014/05/16
			$mime = $this->mime(false);
			if(is_object($mime)){
				$values['mime'] = $mime;
				unset($values['raw_mime']);
			}
		}

		//There is no real xdm_attachments part, but if requested we can check if the message is likely to contain an xdm attachment and return
		//an assoc. array of it's zip contents through the API. Since this is extra work we do this through the API only if it is requested.
		if($part === 'xdm_attachments') {
			if($this->has_xdm_attachment()) {
				$xdm_attachments = array();
				foreach($this->attachment_files as $name => $binary) {
					if(pathinfo($name, PATHINFO_EXTENSION) === 'zip') {
						$xdm_file = tmpfile();
						$metadata = stream_get_meta_data($xdm_file);
						$tmp_filename = $metadata['uri'];
						fwrite($xdm_file, $binary);
						$zip = new ZipArchive;
						$zip->open($tmp_filename);
						$zip_items = array();
						for($i = 0; $i < $zip->numFiles; $i++){
						    $stat = $zip->statIndex($i);
						    $item = $zip->getFromIndex($i);
					    	$zip_items[$stat['name']] = array('content-transfer-encoding' => 'base64', 'name' => basename($stat['name']), 'body' => base64_encode($item));
					    }
					    array_push($xdm_attachments, array($name => explode_tree($zip_items, '/')));
					}
				}
				$values['xdm_attachments'] = $xdm_attachments;
			}
			else { $values['xdm_attachments'] = array(); }
		}

		if($this->draft && $this->has_relationship('attachment_disclosure')){
			$disclosures = array();
			foreach($this->attachment_disclosures as $key => $disclosure){
				$disclosures[$disclosure->hash] = $disclosure->values(array('purpose', 'ssn'));
			}
			$values['disclosures'] = $disclosures;
		}

		$values['file_transfers'] = array();
		foreach($this->file_transfers as $file_transfer) {
			$values['file_transfers'][$file_transfer->id] = array('name' => $file_transfer->name,
																	'size' => $file_transfer->size,
																	'url' => $file_transfer->url_for_view(),
																	'hash' => $file_transfer->hash,
																	'has_expired' => $file_transfer->has_expired(),
																	'downloaded_at' => $file_transfer->downloaded_at,
																	'downloaded_by' => $file_transfer->downloaded_by);
		}

		return $values;
	}

	/**
	* @param string
	* @return array
	*/
	public function values_for_part($part){
		if(!Message::is_a_part($part)) return $this->error->should_be_a_message_part($part);
		$primary_key = static::$primary_key;
		$part_fields = static::$parts[$part];
		return array_merge( array($primary_key => $this->id()), $this->values($part_fields));
	}


	/**
	* Helper method used to configure the Mime_generator object when editing/saving messages, and the Email class when sending.
	* @return boolean success
	*/
	protected function _configure_mail_object(&$mail_object){
		if(!is_a($mail_object, 'CI_Email')) return $this->error->should_be_an_object_that_inherits_from_the_codeigniter_email_library($mail_object);

		$mail_object->initialize( array( 'mailtype' => $this->mailtype(),
										 'priority' => $this->priority() ) );

#TODO - ARE WE SURE WE WANT \R\N?  ACCORDING TO CI'S COMMENTS, THEY DEFAULT TO \N BECAUSE IT'S THE ONLY ONE THAT ALL SERVERS SEEM TO ACCEPT (DESPITE SPECIFICATIONS TO THE CONTRARY)
		$mail_object->set_newline("\r\n");

		$mail_object->from($this->sender);  //the Email library needs a value for this even if it's an empty string

		foreach( array('to', 'cc', 'bcc', 'subject' ) as $field){
			if(!$this->property_is_empty($field))
				$mail_object->$field($this->$field);
		}
		$mail_object->set_header("storage-id",$this->id());
		if(REQUEST_DISPATCHED_MDN) {
			$mail_object->set_header("Disposition-Notification-Options","X-DIRECT-FINAL-DESTINATION-DELIVERY=optional,true");
		}
		if(isset($this->protected_data)){
			$mail_object->set_header("protected-data",$this->protected_data);
		}
                if(isset($this->non_va_referral)) {
                        $mail_object->set_header("non-va-referral", $this->non_va_referral);
                }
		if(!$this->property_is_empty('body')) {
			$mail_object->message($this->body);
                }
		//add attached files.  note that we'll check the size requirement when the files are passed to the model, so we don't need to check that here
		if(isset($this->_attachment_files)){
			foreach($this->attachment_files as $name => $binary_string)
				$mail_object->string_attach($binary_string, $name);
		}

		return true;
	}


	public function values($just_these_values=null){
		if(is_null($just_these_values)){
			$just_these_values = array();
			foreach($this->fields as $field){
				if(isset($this->$field) || array_key_exists($field, $this->_values) || !$this->property_is_empty($field))
					$just_these_values[] = $field;
			}
		}
        return parent::values($just_these_values);
    }


/////////////////////////////////////////////////////////////////////////////////////////////////////////
// GETTERS
// These are accessors for class variables prefaced by a _ and for database field values for this row
// of the message table.  You can access these values by just treating them as normal variables, and
// {@link __get()} will obtain the value for you.
///////////////////////////////////////////////////////////////////////////////////////////////////////////

	/**
	* A JSON-encoded array of the names of the attachments for this message.
	* This value will be automatically updated every time a value in {@link mime_fields} is altered
	* for this message.
	*
	* Getter for message.attachments, with a default derived from {@link mime()} if the message hasn't been
	* saved yet.
	*
	* @return string
	*/
	protected function attachments(){
		if(!isset($this->_values['attachments']) && !$this->property_is_empty('attachment_files')){
			return get_instance()->json->encode(array_keys($this->attachment_files));
		}
		return $this->_values['attachments'];
	}

	/**
	* An array of the message's attached files.
	* @return array Formatted array( $file_name => $file_binary_string)
	*/
	protected function attachment_files(){
		if(is_null($this->_attachment_files)){
			$raw_mime = $this->raw_mime();
			$mime = $this->mime();

			if(is_object($mime) && isset($mime->parts)){
				$this->_attachment_files = array();	//don't set this unless we've managed to get the $mime
					foreach($mime->parts as $part){
						if(isset($part->disposition) && $part->disposition == 'attachment'){
							$name = $part->ctype_parameters['name'];
							$this->_attachment_files[$name] = $part->body;
						}
					}
			}
		}
		return $this->_attachment_files;
	}

	public function encoded_attachment_files(){
		if(!is_array($this->attachment_files)) return array();
		$encoded_files = array();
		foreach($this->attachment_files as $filename => $file){
			$encoded_files[$filename] = base64_encode($file);
		}
		return $encoded_files;
	}

	/**
	* The body of the messsage.
	* This will either be the message.html value or the message.plain value depending on message.mailtype.
	* This is a getter for {@link _body}.
	* @return string
	*/
	protected function body(){
		if(!isset($this->_body)){
			if($this->mailtype() == 'html' && isset($this->html))
				$this->_body = $this->html;
			if($this->mailtype() == 'text' && isset($this->plain))
				$this->_body = $this->plain;
		}
		return $this->_body;
	}

	/**
	* True if this message is a draft.
	* Getter for message.draft, with a default of true if the message hasn't been saved yet.
	* @return boolean
	*/
	protected function draft(){
		if(!array_key_exists('draft', $this->_values)){
			//default new messages to draft, but ONLY new messages - all other messages, we need to rely on whatever was set in the databse
			if(isset($this->id))
				return $this->error->warning('The value for '.get_class($this).'::$draft has not been loaded from the database');
			else
				$this->_set_field_value('draft', true, 0, TRUE); //override validation because this is normally a read-only property
		}
		return $this->_values['draft'];
	}


	/**
	* A JSON-encoded array of the mime headers for this message.
	* This value will be automatically updated every time a value in {@link mime_fields} is altered
	* for this message.
	*
	* Getter for message.headers, with a default derived from {@link mime()} if the message hasn't been
	* saved yet.
	*
	* @return string
	*/
	protected function headers(){
		if(!isset($this->_values['headers'])){
			return get_instance()->json->encode($this->mime()->headers);
		}
		return $this->_values['headers'];
	}

	/**
	* The HTML body of the message.
	* This value will be empty when message.mailtype is not 'html'.  To reliably get the body of the
	* message regardless of mailtype, use {@link body()}
	*
	* Getter for message.html, with a default derived from {@link mime()} if the message hasn't been
	* saved yet.
	*
	* @return string
	*/
	protected function html(){
		if(!isset($this->_values['html']) && $this->mailtype == 'html'){
			$mime = $this->mime();
			if(is_object($mime) && property_exists($mime, 'parts')){
				$text = $this->_find_text_from_parts('html', $mime->parts);
				if(!empty($text)){
					$this->_set_field_value('html', $text, 0, TRUE); //override validation because this is normally a read-only property
				}
			}
		}
		if(isset($this->_values['html']))
			return $this->_values['html'];
	}

	function inline_attachment_files(){
		if(!isset($this->_inline_attachment_files)){
			$this->attachment_files(); //make sure we've parse the normal attachment files so that this array is set
			$this->_inline_attachment_files = $this->_attachment_files_of_type('inline');
		}
		return $this->_inline_attachment_files;
	}

	/**
	* The mailtype for this message ('html', 'text').
	* Getter for message.mailtype, with a default of 'html'.
	*/
	protected function mailtype(){
		if(!array_key_exists('mailtype', $this->_values)){
			if(isset($this->id))
				return $this->error->warning('The value for '.get_class($this).'::$mailtype has not been loaded from the database');
			else
				$this->_set_field_value('mailtype', 'html'); //set a default value for new messages, but ONLY for new messages - for existing messages, we need to get this value from the db
		}

		return $this->_values['mailtype'];
	}

	/**
	* A Mime_generator object that represents the message.
	* This value will be updated whenever a value {@link mime_fields} is altered.
	* @return Mime_generator
	*/
	protected function mime_generator(){
		//if this variable hasn't been loaded, derive it from the information available
		if(!isset($this->_mime_generator)){
			$mime_generator = new Mime_generator(); //we're not loading this the normal way because we don't want one singleton instance for the entire application, we want an individual one for this message
			if($this->_configure_mail_object($mime_generator)){ //configure this the same way that we would if we were sending the message
				$this->_mime_generator = $mime_generator; //as long as we didn't run into problems in the configuration, set the mime generator
				$this->_attachment_files = null; //clear _attachment_files to minimize the amount of data we have stored on the object.  We'll regenerate it before the next time that we wipe the raw mime
			}
		}
		return $this->_mime_generator;
	}


	/**
	* The plain-text body of this message.
	* This will always return the plain-text version of the message, even if it's an HTML message.
	* To get the primary body of the message regardless of {@link mailtype}, use {@link body()}.
	*
	* Getter for message.plain, with a value derived from {@link mime()} if changes to the message
	* have not been saved yet.
	*
	* @return string
	*/
	protected function plain(){
		if(!isset($this->_values['plain'])){
			if($this->mailtype == 'html'){
				$mime = $this->mime();
				if(is_object($mime) && property_exists($mime, 'parts')){
					$text = $this->_find_text_from_parts('plain', $mime->parts);
					if(!empty($text)){
						$this->_set_field_value('plain', $text, 0, TRUE); //override validation because this is normally a read-only property
					}
				}
			}else{
				$this->_set_field_value('plain', trim($this->mime()->body), 0, TRUE); //override validation, since this is normally a read-only property
			}
		}
		return $this->_values['plain'];
	}

	/**
	* A numerical value indicating the priority of this message.
	* Getter for message.priority, with a default of 3 (normal) if the message has not been saved yet.
	* @return int
	*/
	protected function priority(){
		if(!array_key_exists('priority', $this->_values)){
			if(isset($this->id))
				return $this->error->warning('The value for '.get_class($this).'::$priority has not been loaded from the database');
			else
				$this->_set_field_value('priority', 3);  //set a default value for new messages, but ONLY for new messages - for existing messages, we need to get this value from the db
		}
		return $this->_values['priority'];
	}

	/**
	* The raw text value of the mime for this message.
	* Getter for message.raw_mime, with a default derived from {@link mime_generator()} if the
	* message has not been saved yet.
	* @return string
	*/
	protected function raw_mime(){
		if(!isset($this->_values['raw_mime'])){
			$mime_generator = $this->mime_generator();
			if(!is_a($mime_generator, 'CI_Email')) return $this->error->warning('Could not generate mime for '.$this->describe());
			$this->_values['raw_mime'] = $mime_generator->raw_mime();
			//set the timestamp every time the raw_mime is re-formed - this ensures that the timestamp will get set when things are changed that affect the content of the message,
			//but not if it's archived or moved to a different folder, etc.
			$this->_set_field_value('timestamp', now(), 0, TRUE);
		}
		return $this->_values['raw_mime'];
	}


	/**
	* An array of all of the read-only class variables and database fields for this message.
	* Extends its parent method to include the {@link mime_derived_fields} as read-only fields.
	* Additionally, any fields which would change the mime may only be altered for drafts.
	* Getter for {@link _readonly_fields}.
	* @return array
	*/
	public function readonly_fields(){
		$readonly_fields = parent::readonly_fields();

		//the mime-derived fields are always readonly
		$readonly_fields = array_merge($readonly_fields, $this->mime_derived_fields);

		//any fields that would change the mime are readonly if this isn't a draft - sent messages can't be altered, incoming messages can't be altered
		if(isset($this->id) && !$this->draft) $readonly_fields = array_merge($readonly_fields, $this->mime_fields);

		$readonly_fields = array_values(array_unique($readonly_fields));

		//we can set the 'seen' flag if this is an incoming message.
		//(Need to remove this from array instead of adding in because class default is for outgoing messages - this will break in _merge_with_parent_array if we do this the other way around -- MG 2015-11-09)
		if($this->is_incoming())
			unset($readonly_fields[array_search('seen', $readonly_fields)]);

		return array_values(array_unique($readonly_fields));
	}


	/**
	* A JSON-encoded array of the email addresses of all recipients (to, cc, and bcc) of this message.
	* Getter for message.recipients, with a default determined from to, cc, and bcc values of this message
	* if it has not yet been saved.
	* @return string
	*/
	protected function recipients(){
		if(!isset($this->_values['recipients'])){
			$recipients = array_unique( array_filter( array_merge( explode(',', $this->to), explode(',', $this->cc), explode(',', $this->bcc) ) ) );
			if(!empty($recipients))
				$this->_values['recipients'] = get_instance()->json->encode( $recipients );
		}
		return element('recipients', $this->_values);
	}

	protected function seen(){
		if($this->is_outgoing()) return true;
		return $this->_values['seen'];
	}

	/**
	* The size of the entire message in bytes.
	* Getter for message.size, derived from {@link mime_generator} if the message has not been saved.
	* @return int
	*/
	protected function size($force_calculation = false){
		if($force_calculation || !isset($this->_values['size'])) $this->_values['size'] = $this->mime_generator->size_in_bytes();
		return $this->_values['size'];
	}

	/**
	* The subject of this message.
	* Getter for message.subject, defaults to '(No Subject)' if the message has not yet been saved.
	* @return string
	*/
# Adam says we should leave subject blanka nd let applications deal with it on their own.  I concur.
/*	protected function subject(){
		if(!array_key_exists('subject', $this->_values)){
			if(isset($this->id))
				return $this->error->warning('The value for '.get_class($this).'::$subject has not been loaded from the database');
#			else
#				$this->_set_field_value('subject', '(No Subject)');  //set a default value for new messages, but ONLY for new messages - for existing messages, we need to get this value from the db
		}
		return $this->_values['subject'];
	} */

	protected function _attachment_files_of_type($type){
		if(!in_array($type, array('inline', 'attachment'))) return $this->error->should_be_a_known_type($type);
		$files = array();
		if(isset($this->id) && is_object($this->mime) && isset($this->mime->parts)){
			foreach($this->mime->parts as $part){
				if(isset($part->headers->{'content-transfer-encoding'}) && !empty($part->headers->{'content-transfer-encoding'}))	{
					//decode part based on content transfer encoding, if necessary
					switch (strtolower($part->headers->{'content-transfer-encoding'})) {
						case 'base64':
							$part->body = base64_decode($part->body);
							break;
						case 'quoted-printable':
							$part->body = quoted_printable_decode($part->body);
						default: //if it's 8bit, 7bit, or binary, we should be able to handle it; for x-token, we don't handle custom encodings, so we'll still go to default
							break;
					}
				}

				//useful documentation for what the value of $part can be: http://users.dcc.uchile.cl/~xnoguer/peardoc2/package.mail.mail-mimedecode.decode.html
				if(isset($part->disposition) && $part->disposition == $type){
					if(isset($part->ctype_parameters) && is_array($part->ctype_parameters))
						$part->ctype_parameters = (object)$part->ctype_parameters;
					if(isset($part->d_parameters) && is_array($part->d_parameters))
						$part->d_parameters = (object)$part->d_parameters;

					//try to track down the name of this attachment if we can
					$name = 'Untitled';
					if(isset($part->ctype_parameters) && !empty($part->ctype_parameters->name))
						$name = $part->ctype_parameters->name;
					elseif(isset($part->ctype_parameters) && !empty($part->ctype_parameters->filename))
						$name = $part->ctype_parameters->filename;
					elseif(isset($part->d_parameters) && !empty($part->d_parameters->name))
						$name = $part->d_parameters->name;
					elseif(isset($part->d_parameters) && !empty($part->d_parameters->filename))
						$name = $part->d_parameters->filename;
					//figure out the extension, in case we need to modify the name at all
					$extension = '';
					if(string_contains('.', $name))
						$extension = substr($name, strpos($name, '.'));
					if($name == 'Untitled' && empty($extension) && !empty($part->ctype_secondary))
						$extension = '.'.$part->ctype_secondary; //note - this may not actually be a file type (eg "plain" or "x-unknown-content-type").  Could add a check, but risk falsely ruling out file types
					//if we have duplicate file names, make them unique by adding a number (#) to each one
					$name = strip_from_end($extension, $name);
					for($i = 1; array_key_exists($name.$extension, $files); $i++){
						//make sure to clear out any parentheses that we've added.  Note that we'll have to do this even for things we didn't add so that we don't mess things up while re-saving.
						if(preg_match('/\(\d+\)$/', $name, $matches)) {
							$name = trim(strip_from_end(first_element($matches), $name)); //clear out any numbers that we've added already
						}
						$name = $name.' ('.$i.')';
					}
					$files[$name.$extension] = $part->body;
				}
			}
		}

		return $files;
	}

	protected function _find_text_from_parts($text_to_search_for, $parts){
		if(!is_string($text_to_search_for) && !in_array($text_to_search_for, array('html', 'plain'))) return $this->error->should_be_a_known_text_type($text_to_search_for);
		if(empty($parts) || !is_array($parts)) return null;
		foreach($parts as $part){
			if(is_object($part) && isset($part->parts)){
				$text = $this->_find_text_from_parts($text_to_search_for, $part->parts);
				if(!is_null($text)) return $text;
			}
			if($part->ctype_primary == 'text' && $part->ctype_secondary == $text_to_search_for && (!isset($part->disposition) || $part->disposition != 'attachment')){
				return trim($part->body);
			}
		}
	}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// SETTERS
// These methods will be called by __set() when the corresponding class variable or database field
// value is set: e.g., calling $this->sender = 'gibbs_margaret@bah.com' calls $this->set_sender('gibbs_margaret@bah.com').
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	/**
	* Sets the attachment files for this message.
	* Overwrites any existing attachment files completely and regenerates the mime as needed.
	* @param array
	* @param offset
	*/
	public function set_attachment_files($value, $offset=0, $attempting_to_restore_original_files = false){
		if(!$this->is->associative_array_of_strings($value)) return $this->error->property_value_should_be_an_array_of_filenames_mapped_to_binary_strings('attachment_files', get_class($this), $value, $offset+1);

		//back up our current attachments in case this doesn't work
		$original_attachment_files = $this->attachment_files;
		$this->_attachment_files = array();
		$this->reload_mime(); //force the mime to regenerate so that we're not judging whether or not we can add this attachment based on the old size

		$success = true;
		foreach($value as $filename => $binary_string){
			$success = $success && $this->add_attachment($filename, $binary_string);
			$this->reload_mime(); //force the mime to regenerate so that we're not judging whether or not we can add this attachment based on the old size
		}

		//if we didn't succeed in attaching the files, restore our original files - but only try once, so that we avoid an infinite loop of failed attachments
		if(!$success){
			if($attempting_to_restore_original_files)
				return $this->error->warning('Unable to restore original attached files for '.$this->describe());
			return $this->set_attachment_files($original_attachment_files, $offset+1, $attempting_to_restore_original_files = true);
		}

		return $success;
	}

	/**
	* Sets the body for this message.
	* This will reload the {@link mime_generator()} and update the {@link mime_derived_fields} as
	* appropriate: message.plain and message.html will be updated as appropriate to {@link mailtype},
	* {@link size} will be updated with the new size of the message, etc.
	* Setter for {@link _body}.
	* @param string
	* @param int (Optional, internal use only) An integer offset to help {@link Error} build better backtraces
	*/
	public function set_body($value, $offset = 0){
		if(!is_string($value)) return $this->error->property_value_should_be_a_string('body', get_class($this), $value, $offset+1);
		$this->_body = $value;
		$this->reload_mime();
	}

	/**
	* Sets the mailbox id for this message.
	* Note that you cannot change the mailbox_id to a mailbox that doesn't match the message.sender
	* value; outgoing messages (which are the only messages we should be altering) must always
	* have matching values for sender and $mailbox->email_address().
	*
	* If message.sender is not already set, setting this value will automatically populate message.sender.
	*
	* Setter for messages.mailbox_id
	*
	* @param int
	* @param int (Optional, internal use only) An integer offset to help {@link Error} build better backtraces
	*/
	public function set_mailbox_id($value, $offset = 0){
		if(!Mailbox::formatted_like_an_id($value)) return $this->error->should_be_formatted_like_a_mailbox_id($value, $offset+1);

		$mailbox = Mailbox::find_one($value);
		if(!Mailbox::is_an_entity($mailbox)) return $this->error->should_be_a_mailbox_id($value, $offset+1);
		if(!$mailbox->is_active) return $this->error->warning('Cannot add '.$this->describe().' to '.$mailbox->describe().'; '.$mailbox->describe().' is not an active mailbox');

		if(!$this->property_is_empty('sender') && $mailbox->email_address != $this->sender)
			return $this->error->should_be_a_mailbox_id_that_matches_the_mailbox_for_the_message_sender($value, $offset+1);

		$this->_set_field_value('sender', $mailbox->email_address, $offset+1);
		return $this->_set_field_value('mailbox_id', $value, $offset+1);
	}

	/**
	* Sets the mailtype for this message.
	* Setter for message.mailtype.
	* @param string Valid values are 'text' and 'html'
	* @param int (Optional, internal use only) An integer offset to help {@link Error} build better backtraces
	*/
	public function set_mailtype($value, $offset=0){
		$valid_values = array('html', 'text');
		if(!in_array($value, $valid_values))
			return $this->error->property_value_should_be_an_x('mailtype', get_class($this), $value,
															   'valid value ('.array_to_human_readable_list(array_map(array($this->error, 'describe'), $valid_values), 'or').')', $offset+1);

		$this->_set_field_value('mailtype', $value, $offset+1);
	}

	/**
	* Sets the sender email address for this message.
	* Note that this cannot be set to a value that does not match the value of this mailbox's email address.
	* (For outgoing mail, sender and mailbox must match; for incoming mail, we can't change the value of sender.)
	* If the mailbox has not been set, it will automatically be set to the mailbox for this sender.
	*
	* Setter for message.sender
	*
	* @param string Email address of the sender
	* @param int (Optional, internal use only) An integer offset to help {@link Error} build better backtraces
	*/
	public function set_sender($sender, $offset = 0){
		if(!$this->is->nonempty_string($sender)) return $this->error->property_value_should_be_a_nonempty_string('sender', get_class($this), $sender, $offset+1);
		if(!$this->is->string_like_a_direct_address($sender)) return $this->error->property_value_should_be_a_direct_address('sender', get_class($this), $sender, $offset+1);

		$mailbox = Mailbox::find_by_email_address($sender);
		if(!Mailbox::is_an_entity($mailbox)) return $this->error->property_value_should_be_an_address_for_an_existing_mailbox('sender', get_class($this), $sender,  $offset+1);

		//make sure that sender isn't different than the mailbox's sender, if specified
		$mailbox_foreign_key = static::related_foreign_key('mailbox');
		if(!$this->property_is_empty($mailbox_foreign_key) && $this->$mailbox_foreign_key != $mailbox->id)
			return $this->error->property_value_should_be_an_x('sender', get_class($this), $sender, 'email address for mailbox#'.$mailbox->id, $offset+1);

		if(!$this->_set_field_value($mailbox_foreign_key, $mailbox->id()))
			$this->error->warning('Could not set '.$mailbox_foreign_key.' to '.$mailbox->id(), $offset+1);

		return $this->_set_field_value('sender', $sender, $offset+1);
	}

	/**
	* Helper method for setting database field values - for internal use only.
	* Extends parent method to ensure that the mime is regenerated every time a {@link mime_field}  is altered.
	* Do NOT set override_validation to true without careful thought - this should be used only in special circumstances, like setting a default value for a read-only value.
	* @return boolean
	*/
	protected function _set_field_value($field, $value, $error_offset=0, $override_validation = FALSE){
		if(!parent::_set_field_value($field, $value, $error_offset+1, $override_validation)) return false;

		//if something that affects the mime got changed, let's make sure that the mime & all derived get wiped
		//they'll get regenerated the next time someone calls on them
		if(in_array($field, $this->mime_fields)) $this->reload_mime();

		return true;
	}

	/**
	* Clears the value for the mime and the mime-derived fields, forcing them to be reloaded when next called.
	* This method will be called whenever the value of a field in {@link mime_fields} is altered.
	* The next time that these values are called upon, their getters will re-derive their value
	* from the mime.
	*/
	protected function reload_mime(){
		$this->_mime_generator = null;
		$this->attachment_files(); //make sure that the attachment_files gets saved to the _attachment_files so that it won't get overwritten when the mime gets regenerated
		$this->body(); //make sure that the body gets saved to the _body variable so that it won't get overwritten when the mime gets regenerated
		foreach($this->mime_derived_fields as $mime_derived_field){
			$this->_values[$mime_derived_field] = null;
			$this->_altered_fields[] = $mime_derived_field;
		}
	}



//////////////////////
// DATA MANAGEMENT
//////////////////////

	protected function _values_for_save(){
		$values_for_save = array_merge(parent::_values_for_save(), $this->values( $this->mime_derived_fields ) ); //force mime_derived_values to be saved

		if(!isset($this->id)){
			$values_for_save['draft'] = $this->draft; //draft isn't normally a writeable field, so make sure that the default value gets saved
			$values_for_save['seen'] = $this->seen; //seen isn't normally a writeable field, so make sure that the default value gets saved
			$values_for_save['timestamp'] = $this->timestamp;
			$values_for_save['seen'] = $this->seen;
		}
		return $values_for_save;
	}

#TODO - DO SOME SORT OF CHECK TO MAKE SURE THAT WE'RE NOT SAVING CHANGES TO A MESSAGE THAT BELONGS TO AN ARCHIVED MAILBOX
	protected function _values_are_valid_for_create_and_update(){
		if(!parent::_values_are_valid_for_create_and_update()) return false;
		$required_fields = array( static::related_foreign_key('mailbox'), 'sender' );
		foreach($required_fields as $required_field){
			if($this->property_is_empty($required_field)) return $this->error->warning(get_class($this).'::$'.$required_field.' is a required field, and must be set before saving');
		}
		return true;
	}



////////////////////
// STATIC
////////////////////

	/**
	* Checks to make sure that the given file has a valid filetype.
	*
	* @todo It would be better to whitelist these.
	*
	* @param string
	* @param string
	* @return boolean
	*/
	public static function attachment_has_valid_extension($name){
#TODO - WE SHOULD REALLY BE CHECKING THIS AGAINST A WHITELIST, NOT JUST BLACKLISTING EXE
		$extension = pathinfo($name, PATHINFO_EXTENSION);
		return !empty($extension) && mb_strtolower($extension) != 'exe';
	}

	public static function find_part($part, $additional_conditions=array(), $key_results_by = null){
		if(!static::is_a_part($part)) return get_instance()->error->should_be_a_part($part, 1);
		static::set_up_select_for_part($part);
		return static::find($additional_conditions, $key_results_by);
	}

	public static function order_by_subject_without_prefix($direction){
		static::db()->order_by("replace(replace(replace(replace(replace(replace(replace(replace(replace(replace(replace(replace(replace(subject,'RE: ', ''), 'RE:', ''), 'Re: ', ''), 're: ', ''), 'Re:', ''), 're:', ''), 'FWD: ', ''), 'FWD:', ''), 'Fwd: ', ''), 'fwd: ', ''), 'Fwd:', ''), 'fwd:', ''), 'Undeliverable: ', '')", $direction, FALSE);
	}

	public static function select_subject_without_prefix($column_name='_subject_without_prefix'){
		$subject_without_prefix = 'subject';
		$reply_forward_prefixes = array('RE: ', 'RE:', 'Re: ', 're: ', 'Re:', 're:', 'FWD: ', 'FWD:' , 'Fwd: ', 'fwd: ', 'Fwd:', 'fwd:', 'Undeliverable: ');
		foreach($reply_forward_prefixes as $prefix) {
			$subject_without_prefix = "replace(" . $subject_without_prefix . ",'" . $prefix . "','')";
		}
		static::db()->select($subject_without_prefix . ' AS '.$column_name, $escape = FALSE);
	}

	public static function set_up_select_for_part($part, $table_alias = null){
		if(!static::is_a_part($part)) return get_instance()->error->should_be_a_message_part($part, 1);
		if(!is_null($table_alias) && !validates_as('nonempty_string', $table_alias)) return should_be('table alias', $table_prefix, 1);
		if(is_null($table_alias)) $table_alias = static::$table;

		$opening_brace = '[';
		if(!empty($table_alias)) $opening_brace .= '['.$table_alias.'].[';

		static::db()->select($opening_brace.implode('],'.$opening_brace, array_merge( array(static::$primary_key), static::$parts[$part])).']');
		return true;
	}

	public static function is_a_part($name){
		if(!is_string($name) || empty($name)) return get_instance()->error->should_be_a_nonempty_string($name, 1);
		return array_key_exists($name, static::$parts);
	}

	//needs to override default pattern on this because we were lazy and switched the usual pattern (parent class should be Message, child class should be Direct_message)
	protected static function default_foreign_key($include_table=false){
		$table = static::$table; //unnesc., but makes phpdocumentor happier
		$primary_key = static::$primary_key;

		if($primary_key == 'id')
			$foreign_key = 'message_'.$primary_key;
		else
			$foreign_key = $primary_key;

		if($include_table) $foreign_key = $table.'.'.$foreign_key;
		return $foreign_key;
	}

}
