<?php defined('BASEPATH') OR exit('No direct script access allowed');
/**
* @package direct-as-a-service
* @subpackage controllers
*//** */
 
require_once APPPATH.'controllers/services/mailbox_controller.php';

/**
* API methods to manage a mailbox's messages.
*
* Inherits mailbox validation from the mailbox controller.  Mailbox name is a required field for all actions on this controller, with the exception of messages to send or save a messagage; these actions require the sender's email address to be provided.
*
* @author Adam Bell <bell_adam@bah.com>
* @author M. Gibbs <gibbs_margaret@bah.com>
* @author Elan Jaffee <jaffee_elan@bah.com>
*
* @package direct-as-a-service
* @subpackage controllers
*/
class Message_retrieve_controller extends Mailbox_controller{	
	
	/**
	* Everything that applies to all actions should go here.
	*/
	function __construct(){
		parent::__construct();
		$this->load->helper(array('mail', 'validate'));
		$this->load->model('messagestatusmodel');
	}	
	
	public function count_get(){
		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('retrieve');
		
		//SET OPTIONAL FIELDS && CHECK FOR INVALID FIELD VALUES
		$folder = element('folder', $this->get(), 'all');
		
		if(!$this->is->nonempty_string($folder)) $this->invalid_fields[] = 'folder';
		$this->respond_with_error_if_fields_are_invalid();
		
		$total = 0;
		foreach(explode(',', $folder) as $index => $folder) {
			//folder should either be a Folder entity or the name of a known location
			if(Folder::formatted_like_an_id($folder))
				$folder = first_element($this->mailbox->folders( array('id' => $folder))) ;
			if(!Folder::is_an_entity($folder) && !in_array($folder, $this->valid_locations())){
				$this->invalid_fields[] = 'folder';
			}
			
			$filter = $this->decode_filter($this->get('filter'));
			
			//if there is an error with the passed in fields create an error message
			$this->respond_with_error_if_fields_are_invalid();
			$this->apply_filter($filter);
			
			if(Folder::is_an_entity($folder)){
				$total += $folder->message_count();
			}else{
				$count_method = $folder.'_message_count';
				$count_method = strip_from_beginning('all_', $count_method); //all is a weird case, adjust accordingly
				$count_method = str_replace('draft_message', 'drafts', $count_method); //drafts -> also a weird case
				$total += $this->mailbox->$count_method;
			}
		}
		
		
		$this->response_message['count'] = $total;
		$this->response($this->response_message, 200);
	}
	
	/**
	* Retrieve a single message.
	*
	* This action is available to both active and inactive mailboxes.
	*
	* Required fields:
	* 	- id (message id)
	* 	- mailbox (mailbox name)
	* 
	* Optional fields:
	*	- mark ('read', 'unread')
	* 	- part (see the array keys of Message::$parts for options)
	*
	*/
	public function message_get(){
		if(!array_key_exists('id', $this->get()))
			$this->missing_required_fields[] = 'id';
		
		$this->respond_with_error_if_fields_are_missing(); //mailbox is only the required parameter - it will be set in the constructor
		$this->respond_with_error_if_user_is_unauthorized('retrieve');		
	
		$id = $this->get('id');
		if(!Message::formatted_like_an_id($id)) $this->invalid_fields[] = 'id';	
	
		//validate input		
		$mark = $this->get('mark');
		if($mark && $mark != 'read' && $mark != 'unread') $this->invalid_fields[] = 'mark';
		
		$part = $this->get('part');
		if(!empty($part) && !Message::is_a_part($part)) $this->invalid_fields[] = 'part';	
		
		$this->respond_with_error_if_fields_are_invalid();

		//find the message
		$message = Message::find_one($id);
		if(!Message::is_an_entity($message)){
			$message = System_wide_message::find_one($id);
			if(System_wide_message::is_an_entity($message) && !$this->request->admin_api_authorized){
				$this->response( 'Access Denied. Use Not Authorized.', 403);
			}
		}
		
		if(!Direct_message::is_an_entity($message) || !$message->belongs_to_mailbox($this->mailbox)){
			$this->response('Message not found.', 422);
		}
	
		//if needed, mark the value read/unread	
		if($message->is_incoming()){
			if($mark == 'read') $message->mark_as_read();
			if($mark == 'unread') $message->mark_as_unread();
		}

#TODO - CHECK TO SEE IF IT WILL CAUSE WEBMAIL PROBLEMS IF WE SWITCH THIS TO A SINGULAR ID			
		$this->response_message['ids'] = array($id);
		
#TODO - IF WE KEEP PLURAL IDS, THIS SHOULD BE ARRAY($VALUES) FOR CONSISTENCY.  BUT WE PROBABLY SHOULDN'T KEEP THIS TO PLURAL IDS	
		$this->response_message['mail'] = $message->values_for_api($part);
		$this->response($this->response_message, 200);	
	}
	
	/**
	* Retrieve messages from a mailbox.	
	*
	* This action is available to both active and inactive mailboxes.
	*
	* Required parameters:
	* 	- mailbox (the mailbox name: this will be verified in the constructor & the mailbox object will be saved to {@link mailbox})
	*
	*/
	public function messages_get(){
		$this->respond_with_error_if_fields_are_missing(); //mailbox is only the required parameter - it will be set in the constructor
		$this->respond_with_error_if_user_is_unauthorized('retrieve');		
		
		if(array_key_exists('id', $this->get()))
			return $this->message_get(); //essentially, redirect to message_get
		
		//FIND THE OPTIONAL FIELD VALUES
		//listing & showing the optional fields with their default values makes it easier to see what options are available
		$optional_fields = array( 'folder' => 'inbox', 
								  'mark' => false,
								  'part' => '',
								  'filter' => null,
								  'limit' => MAX_MESSAGES,
								  'order_by' => 'timestamp',
								  'start' => 0,
								  'order' => 'desc');			  
		
		//set variables for each optional field						  
		foreach($optional_fields as $optional_field => $default_value){
			$$optional_field = element($optional_field, $this->get(), $default_value); //checks to see if $field is in $get - if not, sets to $default_value
		}
		
		//VALIDATE THE OPTIONAL FIELDS AS NEEDED
		$filter = $this->decode_filter($filter); 		
		
		//check if we have multiple folders specified
		
		//get part of message.  If left not set returns all
		if(!empty($part) && !Direct_message::is_a_part($part)){
			$this->invalid_fields[] = 'part';
		}
		
		if($mark && $mark !== 'read' && $mark !== 'unread'){
			$this->invalid_fields[] = 'mark';
		}
				
		//get limit for amount of messages returned.  can't be over constant MAX_MESSAGES
		if(!$this->is->unsigned_integer($limit) || $limit > MAX_MESSAGES)
			$this->invalid_fields[] = 'limit';
			
		//get the offset.  0 = no offset
		if(!$this->is->unsigned_integer($start)){
			$this->invalid_fields[] = 'start';
		}

		//order by.  default is timestamp.  can only order by set parameters
		if(!in_array($order_by, array('timestamp', 'priority', 'id', 'sender', 'to', 'cc', 'bcc', 'attachments', 'size', 'subject', 'plain'))){
			$this->invalid_fields[] =  'order_by';
		}
		
		//order.  
		if($order !== 'desc' && $order !== 'asc'){
			$this->invalid_fields[] = 'order';
		}
		
		//make sure valid folders are specified
		$folders_to_search = explode(',', trim($folder));
		foreach($folders_to_search as $folder_id){
			if(!in_array($folder_id, $this->valid_locations()) && (!Folder::formatted_like_an_id($folder_id) || !$this->mailbox->has_folder($folder_id))){
				$this->invalid_fields[] = 'folder';
				break;
			} 
		}
		
		$this->respond_with_error_if_fields_are_invalid();
		
		//FIND THE MESSAGES
		$messages = array();
		$total = $unseen = 0;
		foreach($folders_to_search as $index => $folder_id) {
			$conditions = array();
			if(Folder::formatted_like_an_id($folder_id))
				$conditions = compact('folder_id');

			// ASSEMBLE THE DATABASE QUERY
			$this->apply_filter($filter);
			if ($order_by === 'subject') {
				Direct_message::select_subject_without_prefix();
				Direct_message::order_by_subject_without_prefix($order);
			}
			
//			Do only if the message has already been read.
//			Direct_message::set_up_select_for_part($part);
			Direct_message::db()->order_by($order_by,$order);
			Direct_message::db()->limit($limit,$start);

			//message methods follow a common pattern, so we can generate them instead of needing to hard code separate cases
			//following a DRY approach will make the code easier to maintain if we need to make changes later on
			$relationship_name = 'message';
			if(!empty($filter['to']) && $filter['to'] == ALL_USERS_MOCK_ADDRESS)
				$relationship_name = 'system_wide_message';
			elseif($folder_id == 'draft')
				$relationship_name = 'draft';
			elseif(!Folder::formatted_like_an_id($folder_id) && $folder_id != 'all') 
				$relationship_name = implode_nonempty( '_', array($folder_id, 'message') );
			
			$message_method = plural($relationship_name);					
			$messages = array_merge($messages, $this->mailbox->$message_method($conditions));

			$message_count_method = $relationship_name.'_count';
			$this->apply_filter($filter);
			$total += $this->mailbox->$message_count_method($conditions);
			
			$this->apply_filter($filter);
			$unseen += $this->mailbox->$message_count_method( array_merge($conditions, array('seen' => false)));		
		}	
		
		$this->response_message['total'] = $total;
		$this->response_message['total_unseen'] = $unseen;
		$this->response_message['count'] = count($messages);  //todo - how is this different from 'total'?
		$this->response_message['ids'] = array_keys($messages);
		
		//todo - in a future version of the API, this should definitely be called 'messages' instead of 'mail'
		$this->response_message['mail'] = array();
		foreach($messages as $message){
			if($message->is_incoming()){
				if($mark == 'read') $message->mark_as_read();
				if($mark == 'unread') $message->mark_as_unread();
			}
				
			$this->response_message['mail'][$message->id] = $message->values_for_api($part);
		}
		$this->response($this->response_message, 200);
	}
	
///////////////////////////////////////////////
// HELPER METHODS
///////////////////////////////////////////////	

	//original message retrieve code forced the $filter to be a base-64 encoded json-encoded array, probably because the dev didn't know you could pass arrays via POST
	//we're preserving that original functionality, but also allowing $filter to simply be an array, straight-up, no fancy footwork needed.
	protected function decode_filter($filter = null){
		if(is_null($filter) && array_key_exists('filter', $_GET)) 
			$filter = $this->get('filter');
		
		//if the filter wasn't passed, return an array
		if(empty($filter) && !is_numeric($filter)) 
			return array();
			
		if(!is_array($filter)){
			$filter = base64_decode(rawurldecode($filter));
		
			if(!$this->json->is_valid($filter)){
				$this->invalid_fields[] = 'filter';
				return array();
			}
			
			$filter = $this->json->decode_as_array($filter);
			
			if(!is_array($filter)){
				$this->invalid_fields[] = 'filter';
				return array();
			}
		}
		
		return $filter;
	}
	
	protected function apply_filter($filter){
		if(!is_array($filter)) return $this->error->should_be_an_array($filter); //should have been converted to an array before we get this far
		if(empty($filter)) return true; //we're done	
		
		$where = array('seen','flags','size','sender','to','cc','bcc','subject','plain','html','priority');
		foreach($where as $field){
			if(array_key_exists($field, $filter)){
				Direct_message::db()->where($field , $filter[$field]);
			}
		}
		
		if(!empty($filter['original_folder'])){
			$possible_folders = array('inbox','sent','draft');
			$original_folder_array = explode(',',  $filter['original_folder']);
			$allowed_folders = array_intersect($possible_folders, $original_folder_array);
			$num_folders = count($allowed_folders);
			if($num_folders !== 3){ // not have all
				if($num_folders === 2){// two folders
					if(in_array('inbox', $allowed_folders)){
						if(in_array('sent', $allowed_folders))
							Direct_message::db()->where('draft',0);
						if(in_array('draft', $allowed_folders))
							Direct_message::db()->where('sent',0);
					}
					else{ //sent and draft
						Direct_message::db()->where("(([draft] = '0' AND [sent] = '1') OR ([draft] = '1' AND [sent] = '0'))");
					}
				}
				else if ($num_folders === 1){ // only one
					switch (reset($allowed_folders)){
						case 'inbox':
							Direct_message::db()->where("sent","0");
							Direct_message::db()->where("draft","0");
							break;
						case 'draft':
							Direct_message::db()->where("sent","0");
							Direct_message::db()->where("draft","1");
							break;
						case 'sent':
							Direct_message::db()->where("sent","1");
							Direct_message::db()->where("draft","0");
							break;
					}
				}
			}
		}
		
		if(isset($filter['first_date'])){
			Direct_message::db()->where('timestamp >=', $filter['first_date']);
		}
		if(isset($filter['end_date'])){
			Direct_message::db()->where('timestamp <=', $filter['end_date']);
		}
		if(isset($filter['date'])){
			Direct_message::db()->where('timestamp' , $filter['date']);
		}
		if(array_key_exists('has_attachment', $filter)){
			if($filter['has_attachment']){
				Direct_message::db()->where('attachments !=',  "{}");
			}
			else{
				Direct_message::db()->where('attachments',  "{}");
			}
		}
		if(isset($filter['smallest'])){
			Direct_message::db()->where('size >=', $filter['smallest']);
		}
		if(isset($filter['largest'])){
			Direct_message::db()->where('size <=', $filter['largest']);
		}
		if(isset($filter['body'])){
			$v = Direct_message::db()->escape_like_str($filter['body']);
			Direct_message::db()->where('([plain]=\''.$v.'\' OR [html]=\''.$v.'\')');
		}
		
		$like = array('flags','sender','to','cc','bcc','subject','plain','html','attachments');
		foreach($like as $field){
			$contains = "contains_". $field;
			if(isset($filter[$contains])){
				Direct_message::db()->like($field , $filter[$contains]);
			}
		}
		if(isset($filter['contains_body'])){
			$v = Direct_message::db()->escape_like_str($filter['contains_body']);
			Direct_message::db()->where('([plain]=\''.$v.'\' OR [html]=\''.$v.'\')');
		}
		if(isset($filter['contains_recipients'])){
			$v = Direct_message::db()->escape_like_str($filter['contains_recipients']);
			Direct_message::db()->where("([to] like'%".$v."%' OR [cc] like'%".$v."%' OR [bcc] like'%".$v."%')");
		}
		if(isset($filter['fuzzy'])){
			$v = Direct_message::db()->escape_like_str($filter['fuzzy']);
			Direct_message::db()->where("([to] like'%".$v."%' OR [cc] like'%".$v."%' OR [bcc] like'%".$v."%' OR [subject] like'%".$v."%' OR [html] like'%".$v."%' OR [plain] like'%".$v."%' OR [sender] like'%".$v."%' OR [attachments] like '%".$v."%')");
		}
		
		return $filter;
	}
	
}
?>