<?php  if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-project-innovation-initiative
* @subpackage libraries
*/ /** */

require_library('patient_data/soap_call');

/**
* Looks up patients in the Master Veterans Index and returns a set of matching patients.
*
* @package direct-project-innovation-initiative
* @subpackage libraries
*/ 
class Patient_id extends Soap_call {
#TODO - NEED A WSDL URI	
	var $wsdl_uri = MVI_PATIENT_SEARCH_WSDL;
	var $endpoint_uri = MVI_PATIENT_SEARCH_ENDPOINT;
	var $processing_code = MVI_PROCESSING_CODE; //indicates why we're making this call - will be D for debugging, T for training, P for production systems
	var $mvi_id_code = '2.16.840.1.113883.4.349'; //may change depending on dev/prod system - make sure to check with MVI
	var $vler_direct_id_code = '1.2.840.114350.1.13.0.1.7.1.1'; //an ID code indicating our system. I'm guessing we'll be assigned this when we're approved to interact with them?  this should probably be a constant.  
	var $vler_direct_site_key = '200DSM'; //key for Direct Secure Messaging
	var $interaction_id = '2.16.840.1.113883.1.6'; //seems to be another id that we'll be assigned when we're authorized to interact wtih the system, but I'm not sure on this.  
	
	protected $unique_transaction_id; //autogenerated every time the request is generated
	protected $unique_query_id; //autogenerated every time the request is generated
	protected $user; //set in the constructor
	
	public function __construct(){
		parent::__construct();
		$this->user = User::find_from_session();
		$this->curl_options = array(
			CURLOPT_SSLCERT => TLS_SERVER_CERT_PATH,
			CURLOPT_SSLKEY => TLS_SERVER_KEY_PATH,
			CURLOPT_SSL_VERIFYHOST => TLS_SERVER_VERIFYHOST,
			CURLOPT_SSL_VERIFYPEER => TLS_SERVER_VERIFYPEER,
		);
	}	
	
	//returns false if it encountered an error, or an array of patient data if the call was successful
	function patients_matching_criteria($criteria){			
		if(empty($criteria) || !is_array($criteria)) return $this->CI->error->should_be_a_nonempty_array($criteria);
		
		//check that the required fields have been provided
		foreach( array('given_name', 'family_name', 'social_security_number') as $required_field){
			if(!$this->CI->is->nonempty_string(trim($criteria[$required_field])))
				return $this->CI->error->should_be_a_nonempty_string($criteria[$required_field]);
			$this->$required_field = $criteria[$required_field];
		}
		
		//make sure that the ssn really is an ssn and format it for our query
		#TODO - Test patients probably will not have valid SSNs, we need the ability to turn this off via some sort of testing mode, etc.		
		if(!$this->CI->is->string_like_a_social_security_number($criteria['social_security_number'])) 
			return $this->CI->error->should_be_a_social_security_number($criteria['social_security_number']);
		$this->social_security_number = str_replace('-', '', $this->social_security_number);
		
		//check for optional fields fields
		if(!empty($criteria['gender']) && in_array(strtoupper($criteria['gender']), array('M', 'F', 'U')))
			$this->gender = $criteria['gender'];
		if(!empty($criteria['date_of_birth'])){
			if(!$this->CI->is->mysql_date($criteria['date_of_birth'])) return $this->CI->error->should_be_an_x($criteria['date_of_birth'], 'date formatted YYYY-MM-DD');
			$this->date_of_birth = str_replace('-', '', $criteria['date_of_birth']);
		}
		
		return $this->_find_patients();
	}
	
	//returns false if it encountered an error, or a Patient object if the call was successful
	function patient_matching_id($patient_id){
		if(!$this->CI->is->nonempty_string($patient_id)) return $this->CI->error->should_be_a_nonempty_string($patient_id);
		$this->patient_id = $patient_id;
		$this->xml_template = '_patient_demographics';
		
		$patients = $this->_find_patients();
		if(!is_array($patients)) return false;
		
		if(count($patients) > 1) {
			log_message('error', 'MVI returned more than one result for search by patient id for '.$this->CI->error->describe($patient_id).', expecting only one'); 
		}
		
		return element($patient_id, $patients);	
	}	
	
	/** Extends parent to ensure that {@link unique_query_id} is generated every time that we generate the request */	
	function request_as_xml(){
		$this->unique_transaction_id = sha1(sha1($this->vler_direct_site_key . microtime()));
		$this->unique_query_id = sha1($this->vler_direct_site_key . microtime());
		return parent::request_as_xml();
	}
	
//////////////////////////////////////////
//  PROTECTED METHODS
//////////////////////////////////////////
	
	protected function _find_patients(){
#		if(is_on_local()) return $this->mock_patients();
		
		//make the actual call
		$response_data = $this->call();
		if(empty($response_data)) return false; //encountered error while trying to make the call; call() will provide error message
		
#TODO - confirm encoding
#TODO - recognize too many patients
#TODO - recognize no patients		
		//set up object to parse the response data as xml
		$dom = new DOMDocument('1.0', 'UTF-8');
		$dom->encoding = 'utf-8';
		$dom->formatOutput = FALSE;
		$dom->preserveWhiteSpace = TRUE;
		$dom->substituteEntities = TRUE;
		$dom->loadXML($response_data);
		
		//search for the <subject> elements that represent patients.
		#TODO - RECOGNIZE A RESPONSE WITH NO PATIENST
		$xpath = new DOMXPath($dom);	
		$xpath->registerNamespace('x', 'urn:hl7-org:v3'); //argsblargle.  Looks like PHP won't let us add a namespace without an identifier, so we'll use the all-purposeful x.
		$xpath->registerNamespace('idm', 'http://URL         .DOMAIN.EX');
		
#		$xpath->registerNamespace('xsi', 'http://www.w3.org/2001/XMLSchema-instance');
		$patient_nodes = $xpath->query('//idm:PRPA_IN201306UV02/x:controlActProcess//x:patient');		

		if(!is_a($patient_nodes, 'DOMNodeList') || !$this->CI->is->nonzero_unsigned_integer($patient_nodes->length)){				
			//check to see if we have an AA acknowledgment - if not, there was something wrong with our query
			$acceptability_acknowledgement = $xpath->query('//idm:PRPA_IN201306UV02/x:acknowledgement/x:typeCode[@code="AA"]');	
			if(!$this->CI->is->nonzero_unsigned_integer($acceptability_acknowledgement->length)){
				log_message('debug', 'MVI indicates that our search query was malformed');
				return false;
			}
			
			//we might not have patient nodes just because there are no results
			$not_found = $xpath->query('//idm:PRPA_IN201306UV02/x:controlActProcess/x:queryAck/x:queryResponseCode[@code="NF"]');
			if($this->CI->is->nonzero_unsigned_integer($not_found->length)){
				return array(); //no patients found, no need for an error message
			}				
			
			//check to see if we have a QE query error - if we have AA + QE, then there were too many patients returned
			$query_error = $xpath->query('//idm:PRPA_IN201306UV02/x:controlActProcess/x:queryAck/x:queryResponseCode[@code="QE"]');
			if($this->CI->is->nonzero_unsigned_integer($query_error->length)){
				$this->feedback_messages[] = 'Your search matched more patients than could be returned.  Please narrow your search terms and try again.';
				return false;
			}	

			if(!is_a($patient_nodes, 'DOMNodeList')) return $this->CI->error->warning('Unknown error: unable to parse MVI response');			
		}

		$patients = array();
		foreach($patient_nodes as $patient_node){
			#TODO - prefix, suffix
			#TODO - assigning facility?
			#TODO - Legal names vs aliases/maiden names
			#TODO - Include current address in results?
			
			$patient = new Patient($patient_node, $xpath);
			if(!empty($patient->icn()))
				$patients[$patient->icn()] = $patient; //patients aren't really patients if we can't find their ICN

		}
		
		return $patients;
	}
		
	//for testing purposes
	protected function mock_patients(){
# CHECK WITH MARGARET BEFORE DELETING - SHE HAS SPECIFIC CASES SHE'S TESTING USING THIS DATA		
		$mock_data = array( '1012663662V729834' => array( 'given_names' => array('NWHINELVEN'), 
														  'family_name' => 'NWHINZZZTESTPATIENT',
														  'ssn' => '111223333',
														 ),
												  		
							'123456789' => array( 'given_names' => array('Cecilia', 'Helena'), 
												  'family_name' => 'Payne-Gaposchkin',
												  'prefix' => 'Dr.',
												  'gender' => 'f',
												  'assigning_facility' => 'Harvard University', //Need to find out what the actual format for this will be
												  'id_state' => 'Deactivated', //format?
												  'date_of_birth' => '19000510', // 5/10/1900 //note that this can apparently be a range of dates - we'll need to know how to parse it.  Also, dates before 1970 - do they use negatives?
												  'date_of_death' => '19791207', // 12/7/1979
												  'ssn' => '111111111',
												 ),
												  
					  		'987654321' => array( 'given_names' => array('Bonnie', 'Jeanne'), 
												  'family_name' => 'Dunbar',
												  'prefix' => 'Dr.',
												  'gender' => 'f',
												  'assigning_facility' => 'NASA', //Need to find out what the actual format for this will be
												  'id_state' => 'Temporary', //format?
												  'date_of_birth' => '19490303', // 3/3/1949
												  'ssn' => '222222222',
												 ),
							'654321987' => array( 'given_names' => array('Eugene', 'Francis'), 
												  'family_name' => 'Kranz',
												  'prefix' => 'Mr.',
												  'gender' => 'm',
												  'assigning_facility' => 'NASA', //Need to find out what the actual format for this will be
												  'id_state' => 'Permanent', //format?
												  'date_of_birth' => '19330817', // 8/17/1933
												  'ssn' => '33333333',
												 ),
							'754321987' => array( 'given_names' => array('Edward'), 
												  'family_name' => 'Sabine',
												  'prefix' => 'Sir',
												  'suffix' => 'KCB FRS',
												  'gender' => 'm',
												  'assigning_facility' => 'Royal Military Academy', //Need to find out what the actual format for this will be
												  'id_state' => 'Deactivated', //format?
												  'date_of_birth' => '17881014', // 10/14/1788
												  'date_of_death' => '18830626', //6/26/1883
												  'ssn' => '4444444444'
												 ),
							'754321887' => array( 'given_names' => array('Eden'), 
												  'family_name' => 'Atwood',
												  'gender' => 'u',
												  'assigning_facility' => 'Concord Records', //Need to find out what the actual format for this will be
												  'id_state' => 'Permanent', //format?
												  'date_of_birth' => '19690111', // 1/11/1969
												  'ssn' => '5555555555'
												 ),
							'754321888' => array( 'given_names' => array('Marie', 'Charlotte', 'Amalie', 'Ernestine', 'Wilhelmine', 'Philippine'), 
												  'suffix' => 'H.R.H.',
												  'gender' => 'f',
												  'assigning_facility' => 'House of Saxe-Gotha-Altenburg', 
												  'id_state' => 'Deactivated', //format?
												  'date_of_birth' => -6889030019, // 9/11/1751
												  'date_of_death' => -4502748419, // 4/25/1827
												 ),
							'794321888' => array( 'given_names' => array('Johannes Müller'), 
												  'family_name' => 'von Königsberg',
												  'gender' => 'm',
												  'assigning_facility' => 'Holy Roman Empire', 
												  'id_state' => 'Deactivated', //format?
												  'date_of_birth' => -16837817219, // 6/6/1436
												  'date_of_death' => -15572921219, // 7/6/1476
												 ),														 														 
												 		
												 												 
					 ); 

		
		foreach($mock_data as $icn => $data){
			$data['icn'] = $icn;
			$mock_data[$icn] = new Patient($data);
		}
		
		return $mock_data;		
	}	
	

}

#TODO - abandon this class or put it in its own library (model?)
class Patient{
	protected $dom_element;	
	
	//raw vars are exactly the value given by ICN; these will be formatted elsewhere
	protected $raw_icn;
	protected $raw_ssn;
	
	protected $raw_prefix;
	protected $raw_given_names;
	protected $raw_family_name;
	protected $raw_suffix;
	
	protected $raw_gender;
	protected $raw_assigning_facility;
	protected $raw_date_of_birth; //formatted YYYYMMDD or YYYYMMDDHHMMSS
	protected $raw_date_of_death; //formatted YYYYMMDD or YYYYMMDDHHMMSS
	protected $raw_id_state; 
	
	
	function __construct($patient_node_or_values = array(), $xpath=null){
		$this->is = get_instance()->is;
		$this->error = get_instance()->error;
		
		if(is_a($patient_node_or_values, 'DOMElement')){
			$this->xpath = $xpath;
			$this->load_values_from_dom_element($patient_node_or_values);
		}elseif(is_array($patient_node_or_values))
			$this->load_values_from_array($patient_node_or_values);
		else 
			return $this->error->should_be_an_array_or_DOMElement_object($patient_node_or_values);
			
	}
	
	public function hidden_field_markup($root_tag_name = 'patient'){
		if(!$this->is->nonempty_string_with_no_whitespace($root_tag_name)) return $this->error->should_be_a_nonempty_string_with_no_whitespace($root_tag_name);
		
		$markup = '';
		foreach($this->raw_values as $property => $value){
			if(!is_null($value)) $markup .= form_hidden($root_tag_name.'['.$property.']', $value);
		}
		
		return $markup;
	}
	
	public function raw_values(){
		$values = get_object_vars($this);
		unset($values['is'], $values['error'], $values['dom_element'], $values['xpath']); //these are not useful values to output
		foreach($values as $property => $value){
			if(string_begins_with('raw_', $property)){
				$values[strip_from_beginning('raw_', $property)] = $value;
				unset($values[$property]);
			}
		}
		return $values;
	}
	
	public function values(){
		$values = get_object_vars($this);
		unset($values['is'], $values['error'], $values['dom_element'], $values['xpath']); //these are not useful values to output
		foreach($values as $property => $value){
			if(string_begins_with('raw_', $property)){	
				$non_raw_property = strip_from_beginning('raw_', $property);			
				if(!array_key_exists($non_raw_property, $values))
					$values[$non_raw_property] = $this->$non_raw_property;
				unset($values[$property]);
			}
		}
		return $values;
	}
		
	///////////////////////
	// DOM-PARSING METHODS
	///////////////////////
	
	
	protected function load_values_from_dom_element($dom_element){
		if(!is_a($dom_element, 'DOMElement')) return get_instance()->error->should_be_a_DOMElement_object($dom_element);
		if($dom_element->tagName != 'patient') return get_instance()->error->should_be_a_patient_DOMElement_object($dom_element);
		
		$this->dom_element = $dom_element;

		$this->raw_icn = $this->value_for_attribute('extension', 'x:id[@root="'.get_instance()->patient_id->mvi_id_code.'"]');
#		$this->raw_ssn = $this->value_for_attribute('extension', 'x:id[@root="2.16.840.1.113883.4.1"]');
		$this->raw_ssn = $this->value_for_attribute('extension', 'x:patientPerson/x:asOtherIDs[@classCode="SSN"]/x:id[@root="2.16.840.1.113883.4.1"]');
		$this->raw_id_state = $this->value_for_attribute('code', 'x:statusCode[@code]');
		$this->raw_date_of_birth = $this->value_for_attribute('value', 'x:patientPerson/x:birthTime[@value]');
		$this->raw_date_of_death = $this->value_for_attribute('value', 'x:patientPerson/x:deceasedTime[@value]');
		$this->raw_gender = $this->value_for_attribute('code', 'x:patientPerson/x:administrativeGenderCode[@code]');
		$this->raw_given_names = $this->values_for_tag('x:patientPerson/x:name[@use="L"]/x:given');
		$this->raw_family_name = $this->value_for_tag('x:patientPerson/x:name[@use="L"]/x:family');
		$this->raw_prefix = $this->value_for_tag('x:patientPerson/x:name[@use="L"]/x:prefix');	
		$this->raw_suffix = $this->value_for_tag('x:patientPerson/x:name[@use="L"]/x:suffix');		
	}
	
	protected function value_for_attribute($attribute, $xpath_for_tag){
		$values = $this->values_for_attribute($attribute, $xpath_for_tag);
		if(!is_array($values)) return false; //we encountered an error
		if(count($values) > 1) $this->error->warning('More than one tag with attribute '.$attribute.' matched '.$xpath_for_tag.'; ignoring all but the first');
		return first_element($values);
	}
	
	protected function values_for_attribute($attribute, $xpath_for_tag){
		if(!$this->is->nonempty_string($attribute)) return $this->error->should_be_a_nonempty_string($attribute);
		if(!$this->is->nonempty_string($xpath_for_tag)) return $this->error->should_be_a_nonempty_string($xpath_for_tag);
	
		$xpath_results = $this->xpath->query($xpath_for_tag, $this->dom_element);
		
		if(!is_a($xpath_results, 'DOMNodeList') || $xpath_results->length < 1) return array();
		
		$values = array();
		foreach($xpath_results as $xpath_result){					
			if($xpath_result->hasAttribute($attribute)) 
				$values[] = $xpath_result->getAttribute($attribute);
		}
		return $values;
	}
	
	protected function value_for_tag($xpath_for_tag){
		$values = $this->values_for_tag($xpath_for_tag);
		if(!is_array($values)) return false; //we encountered an error
		if(count($values) > 1) $this->error->warning('More than one tag matched '.$xpath_for_tag.'; ignoring all but the first');
		return first_element($values);
	
	}
	
	protected function values_for_tag($xpath_for_tag){
		if(!$this->is->nonempty_string($xpath_for_tag)) return $this->error->should_be_a_nonempty_string($xpath_for_tag);
	
		$xpath_results = $this->xpath->query($xpath_for_tag, $this->dom_element);
		if(!is_a($xpath_results, 'DOMNodeList') || $xpath_results->length < 1) return array();
		
		$values = array();
		foreach($xpath_results as $xpath_result){
			$values[] = $xpath_result->textContent;
		}
	
		return $values;
	}
	
	function full_name(){
		$name = implode_nonempty(' ', array($this->prefix, implode(' ', $this->given_names), $this->family_name));
		if(!empty($this->raw_suffix)) $name .= ', '.$this->suffix;
		return $name;
	}
	
	function date_of_birth(){
		if(empty($this->raw_date_of_birth)) return 'Unknown';
		return $this->human_readable_date($this->raw_date_of_birth);
	}
	
	function date_of_death(){
		if(empty($this->raw_date_of_death)) return '&mdash;';
		return $this->human_readable_date($this->raw_date_of_death);
	}
	
	function gender(){
		return element(strtoupper($this->raw_gender), array('F' => 'Female', 'M' => 'Male', 'U' => 'Undifferentiated'));
	}

	function icn(){
		if(!string_contains('^', (string)$this->raw_icn)) return $this->raw_icn; //just in case they ever start formatting this differently
		return substr($this->raw_icn, 0, strpos($this->raw_icn, '^'));
	}

	function id_state(){
		return ucfirst($this->raw_id_state);
	}	
	
	function ssn(){
		$ssn = $this->raw_ssn;
		if(empty($ssn) || !is_string($ssn)) return '&mdash;';
		
		if(string_contains('^', $ssn))
			$ssn = substr($ssn, 0, strpos($ssn, '^'));
		return substr($ssn, 0, 3).'-'.substr($ssn, 3, 2).'-'.substr($ssn, 5, 4);
	}
	
	protected function human_readable_date($timestamp){
		if(empty($timestamp) || !is_numeric($timestamp) || strlen($timestamp) < 8) return false; //we don't know what to do with it if we don't have YYYYMMDD
		
		//MVI doesn't use unix timestamps - they return dates as YYYYMMDD or YYYYMMDDHHMMSSS
		$year = substr($timestamp, 0, 4);
		$month = substr($timestamp, 4, 2);
		$day = substr($timestamp, 6, 2);
		
		return $month.'/'.$day.'/'.$year;
	}
	
	protected function load_values_from_array($values){
		foreach($values as $property => $value){
			$raw_property = 'raw_'.$property;
			if(property_exists(get_class($this), $raw_property))
				$this->$raw_property = $value;
			else
				$this->$property = $value;
		}
	}
	
	function __get($property){
		$raw_property = 'raw_'.$property;	
		
		if(method_exists(get_class($this), $property))
			return $this->$property();
		elseif(property_exists(get_class($this), $property))
			return $this->$property; //we're hitting this because we're out of scope, but we don't mind giving read-only access		
		elseif(property_exists(get_class($this), $raw_property))
			return $this->$raw_property;	

		get_instance()->error->property_does_not_exist($property, get_class($this), 1);
		return null;
	}	
	
}T